/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.kyuubi

import java.nio.charset.StandardCharsets
import java.nio.file.{Files, Path, StandardOpenOption}

import scala.collection.JavaConverters._
import scala.collection.mutable.ArrayBuffer
import scala.compat.Platform.EOL

import com.vladsch.flexmark.formatter.Formatter
import com.vladsch.flexmark.parser.{Parser, ParserEmulationProfile, PegdownExtensions}
import com.vladsch.flexmark.profile.pegdown.PegdownOptionsAdapter
import com.vladsch.flexmark.util.data.{MutableDataHolder, MutableDataSet}
import org.scalatest.Assertions.{assertResult, withClue}

object MarkdownUtils {

  def verifyOutput(
      markdown: Path,
      newOutput: MarkdownBuilder,
      agent: String,
      module: String): Unit = {
    val formatted = newOutput.formatMarkdown()
    if (System.getenv("KYUUBI_UPDATE") == "1") {
      Files.write(
        markdown,
        formatted.asJava,
        StandardOpenOption.CREATE,
        StandardOpenOption.TRUNCATE_EXISTING)
    } else {
      val linesInFile = Files.readAllLines(markdown, StandardCharsets.UTF_8)
      linesInFile.asScala.zipWithIndex.zip(formatted).foreach { case ((str1, index), str2) =>
        withClue(s"$markdown out of date, as line ${index + 1} is not expected." +
          " Please update doc with KYUUBI_UPDATE=1 build/mvn clean test" +
          s" -pl $module -am -Pflink-provided,spark-provided,hive-provided" +
          s" -DwildcardSuites=$agent") {
          assertResult(str2)(str1)
        }
      }
    }
  }

  def line(str: String): String = {
    str.stripMargin.replaceAll(EOL, "")
  }

  def appendBlankLine(buffer: ArrayBuffer[String]): Unit = buffer += ""

  def appendFileContent(buffer: ArrayBuffer[String], path: Path): Unit = {
    buffer += "```bash"
    buffer ++= Files.readAllLines(path).asScala
    buffer += "```"
  }
}

class MarkdownBuilder {
  private val buffer = new ArrayBuffer[String]()

  /**
   * append a single line
   * with replacing EOL to empty string
   * @param str single line
   * @return
   */
  def line(str: String = ""): MarkdownBuilder = {
    buffer += str.stripMargin.replaceAll(EOL, "")
    this
  }

  /**
   * append the multiline
   * with splitting EOL into single lines
   * @param multiline multiline with default line margin "|"
   * @return
   */
  def lines(multiline: String): MarkdownBuilder = {
    buffer ++= multiline.stripMargin.split(EOL)
    this
  }

  /**
   * append the licence
   * @return
   */
  def licence(): MarkdownBuilder = {
    lines("""
        |<!--
        |- Licensed to the Apache Software Foundation (ASF) under one or more
        |- contributor license agreements.  See the NOTICE file distributed with
        |- this work for additional information regarding copyright ownership.
        |- The ASF licenses this file to You under the Apache License, Version 2.0
        |- (the "License"); you may not use this file except in compliance with
        |- the License.  You may obtain a copy of the License at
        |-
        |-   http://www.apache.org/licenses/LICENSE-2.0
        |-
        |- Unless required by applicable law or agreed to in writing, software
        |- distributed under the License is distributed on an "AS IS" BASIS,
        |- WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
        |- See the License for the specific language governing permissions and
        |- limitations under the License.
        |-->
        |""")
  }

  /**
   * append the auto-generation hint
   * @param className the full class name of agent suite
   * @return
   */
  def generationHint(className: String): MarkdownBuilder = {
    lines(s"""
        |<!-- DO NOT MODIFY THIS FILE DIRECTLY, IT IS AUTO-GENERATED BY [$className] -->
        |
        |""")
  }

  /**
   * append file content
   * @param path file path
   * @return
   */
  def file(path: Path): MarkdownBuilder = {
    buffer ++= Files.readAllLines(path).asScala
    this
  }

  /**
   * append file content with code block quote
   * @param path path to file
   * @param language language of codeblock
   * @return
   */
  def fileWithBlock(path: Path, language: String = "bash"): MarkdownBuilder = {
    buffer += s"```$language"
    file(path)
    buffer += "```"
    this
  }

  def formatMarkdown(): Stream[String] = {
    def createParserOptions(emulationProfile: ParserEmulationProfile): MutableDataHolder = {
      PegdownOptionsAdapter.flexmarkOptions(PegdownExtensions.ALL).toMutable
        .set(Parser.PARSER_EMULATION_PROFILE, emulationProfile)
    }

    def createFormatterOptions(
        parserOptions: MutableDataHolder,
        emulationProfile: ParserEmulationProfile): MutableDataSet = {
      new MutableDataSet()
        .set(Parser.EXTENSIONS, Parser.EXTENSIONS.get(parserOptions))
        .set(Formatter.FORMATTER_EMULATION_PROFILE, emulationProfile)
    }

    val emulationProfile = ParserEmulationProfile.COMMONMARK
    val parserOptions = createParserOptions(emulationProfile)
    val formatterOptions = createFormatterOptions(parserOptions, emulationProfile)
    val parser = Parser.builder(parserOptions).build
    val renderer = Formatter.builder(formatterOptions).build
    val document = parser.parse(buffer.mkString(EOL))
    val formattedLines = new ArrayBuffer[String](buffer.length)
    val formattedLinesAppendable = new Appendable {
      override def append(csq: CharSequence): Appendable = {
        if (csq.length() > 0) {
          formattedLines.append(csq.toString)
        }
        this
      }

      override def append(csq: CharSequence, start: Int, end: Int): Appendable = {
        append(csq.toString.substring(start, end))
      }

      override def append(c: Char): Appendable = {
        append(c.toString)
      }
    }
    renderer.render(document, formattedLinesAppendable)
    // trim the ending EOL appended by renderer for each line
    formattedLines.toStream.map(str =>
      if (str.endsWith(EOL)) {
        str.substring(0, str.length - 1)
      } else {
        str
      })
  }
}

object MarkdownBuilder {
  def apply(licenced: Boolean = true, className: String = null): MarkdownBuilder = {
    val builder = new MarkdownBuilder
    if (licenced) { builder.licence() }
    if (className != null) { builder.generationHint(className) }
    builder
  }
}
