/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.spark.sql.execution.datasources.v2.jdbc

import org.apache.spark.sql.SparkSession
import org.apache.spark.sql.connector.read._
import org.apache.spark.sql.execution.datasources.jdbc.{JDBCOptions, ShardJDBCUtil, ShardOptions}
import org.apache.spark.sql.types.StructType

class ShardJDBCScanBuilder(
    session: SparkSession,
    schema: StructType,
    jdbcOptions: JDBCOptions)
  extends JDBCScanBuilder(session, schema, jdbcOptions) {

  override def build(): Scan = {
    val scan = super.build().asInstanceOf[JDBCScan]
    val relation = scan.relation

    // TODO supports logical partitions and shard partitions together
    val shards = ShardOptions.create(jdbcOptions)
    if (shards.shards.length == 1) {
      scan
    } else {
      // Replace logical partitions of JDBC to shard partitions of ClickHouse.
      val newParts = ShardJDBCUtil.shardPartition(shards)
      val newRelation = relation.copy(parts = newParts)(relation.sparkSession)
      scan.copy(relation = newRelation)
    }
  }
}
