/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.kylin.common.persistence;

import java.io.Serializable;
import java.util.List;

import org.apache.kylin.common.KylinVersion;
import org.apache.kylin.common.util.JsonUtil;
import org.apache.kylin.common.util.RandomUtil;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonAutoDetect.Visibility;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonView;
import org.apache.kylin.guava30.shaded.common.collect.Lists;

import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * Marks the root entity of JSON persistence. Unit of read, write, cache, and
 * refresh.
 * <p>
 * - CubeInstance - CubeDesc - SourceTable - JobMeta - Dictionary (not JSON but
 * also top level persistence)
 *
 * @author yangli9
 */
@SuppressWarnings("serial")
@JsonAutoDetect(fieldVisibility = Visibility.NONE, getterVisibility = Visibility.NONE, isGetterVisibility = Visibility.NONE, setterVisibility = Visibility.NONE)
@Slf4j
public abstract class RootPersistentEntity implements AclEntity, Serializable {

    // for spring session save serializable object(ManagerUser), do not modify
    private static final long serialVersionUID = 0L;

    @JsonProperty("uuid")
    protected String uuid = RandomUtil.randomUUIDStr();

    @JsonProperty("last_modified")
    protected long lastModified;

    @Getter
    @Setter
    @JsonProperty("create_time")
    protected long createTime = System.currentTimeMillis();

    // if cached and shared, the object MUST NOT be modified (call setXXX() for example)
    protected boolean isCachedAndShared = false;

    /**
     * Metadata model version
     * <p>
     * User info only, we don't do version control
     */
    @JsonProperty("version")
    protected String version = KylinVersion.getCurrentVersion().toString();

    @Getter
    @JsonProperty("mvcc")
    @JsonView(JsonUtil.PublicView.class)
    private long mvcc = -1;

    @Getter
    @Setter
    private boolean isBroken = false;

    @Getter
    @Setter
    private List<RootPersistentEntity> dependencies;

    public List<RootPersistentEntity> calcDependencies() {
        return Lists.newArrayList();
    }

    public String getVersion() {
        return version;
    }

    public void setVersion(String version) {
        checkIsNotCachedAndShared();
        this.version = version;
    }

    public String getUuid() {
        return uuid;
    }

    public void setUuid(String uuid) {
        checkIsNotCachedAndShared();
        this.uuid = uuid;
    }

    public String getId() {
        return uuid;
    }

    public long getLastModified() {
        return lastModified;
    }

    public void setLastModified(long lastModified) {
        //checkIsNotCachedAndShared(); // comment out due to let pass legacy tests, like StreamingManagerTest
        this.lastModified = lastModified;
    }

    public void updateRandomUuid() {
        setUuid(RandomUtil.randomUUIDStr());
    }

    public boolean isCachedAndShared() {
        return isCachedAndShared;
    }

    public void setCachedAndShared(boolean isCachedAndShared) {
        if (this.isCachedAndShared && !isCachedAndShared)
            throw new IllegalStateException();

        this.isCachedAndShared = isCachedAndShared;
    }

    public void checkIsNotCachedAndShared() {
        if (isCachedAndShared)
            throw new IllegalStateException();
    }

    public void setMvcc(long mvcc) {
        if (isCachedAndShared) {
            log.warn("[UNEXPECTED_THINGS_HAPPENED]update mvcc for isCachedAndShared object " + this.getClass()
                    + ", from " + this.mvcc + " to " + mvcc, new IllegalStateException());
        }
        this.mvcc = mvcc;
    }

    /**
     * The name as a part of the resource path used to save the entity.
     * <p>
     * E.g. /resource-root-dir/{RESOURCE_NAME}.json
     */
    public String resourceName() {
        return uuid;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + (int) (lastModified ^ (lastModified >>> 32));
        result = prime * result + ((uuid == null) ? 0 : uuid.hashCode());
        result = prime * result + ((version == null) ? 0 : version.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        RootPersistentEntity other = (RootPersistentEntity) obj;
        if (lastModified != other.lastModified || !(version == null || version.equals(other.getVersion())))
            return false;
        if (uuid == null) {
            if (other.uuid != null)
                return false;
        } else if (!uuid.equals(other.uuid))
            return false;
        return true;
    }

    public String getResourcePath() {
        return "";
    }

    @Override
    public String toString() {
        return getClass().getName() + "@" + uuid;
    }
}