/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.connect.mirror;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.kafka.clients.admin.Admin;
import org.apache.kafka.clients.admin.Config;
import org.apache.kafka.clients.admin.ConfigEntry;
import org.apache.kafka.clients.admin.CreateTopicsOptions;
import org.apache.kafka.clients.admin.NewPartitions;
import org.apache.kafka.clients.admin.NewTopic;
import org.apache.kafka.clients.admin.TopicDescription;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.acl.AccessControlEntry;
import org.apache.kafka.common.acl.AccessControlEntryFilter;
import org.apache.kafka.common.acl.AclBinding;
import org.apache.kafka.common.acl.AclBindingFilter;
import org.apache.kafka.common.acl.AclOperation;
import org.apache.kafka.common.acl.AclPermissionType;
import org.apache.kafka.common.config.ConfigDef;
import org.apache.kafka.common.config.ConfigResource;
import org.apache.kafka.common.errors.InvalidPartitionsException;
import org.apache.kafka.common.errors.SecurityDisabledException;
import org.apache.kafka.common.resource.PatternType;
import org.apache.kafka.common.resource.ResourcePattern;
import org.apache.kafka.common.resource.ResourcePatternFilter;
import org.apache.kafka.common.resource.ResourceType;
import org.apache.kafka.common.utils.AppInfoParser;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.connect.connector.Task;
import org.apache.kafka.connect.mirror.ConfigPropertyFilter;
import org.apache.kafka.connect.mirror.MirrorSourceConfig;
import org.apache.kafka.connect.mirror.MirrorSourceTask;
import org.apache.kafka.connect.mirror.MirrorUtils;
import org.apache.kafka.connect.mirror.ReplicationPolicy;
import org.apache.kafka.connect.mirror.Scheduler;
import org.apache.kafka.connect.mirror.SourceAndTarget;
import org.apache.kafka.connect.mirror.TopicFilter;
import org.apache.kafka.connect.source.SourceConnector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MirrorSourceConnector
extends SourceConnector {
    private static final Logger log = LoggerFactory.getLogger(MirrorSourceConnector.class);
    private static final ResourcePatternFilter ANY_TOPIC = new ResourcePatternFilter(ResourceType.TOPIC, null, PatternType.ANY);
    private static final AclBindingFilter ANY_TOPIC_ACL = new AclBindingFilter(ANY_TOPIC, AccessControlEntryFilter.ANY);
    private Scheduler scheduler;
    private MirrorSourceConfig config;
    private SourceAndTarget sourceAndTarget;
    private String connectorName;
    private TopicFilter topicFilter;
    private ConfigPropertyFilter configPropertyFilter;
    private List<TopicPartition> knownSourceTopicPartitions = Collections.emptyList();
    private List<TopicPartition> knownTargetTopicPartitions = Collections.emptyList();
    private ReplicationPolicy replicationPolicy;
    private int replicationFactor;
    private Admin sourceAdminClient;
    private Admin targetAdminClient;
    private Admin offsetSyncsAdminClient;
    private AtomicBoolean noAclAuthorizer = new AtomicBoolean(false);

    public MirrorSourceConnector() {
    }

    MirrorSourceConnector(List<TopicPartition> knownSourceTopicPartitions, MirrorSourceConfig config) {
        this.knownSourceTopicPartitions = knownSourceTopicPartitions;
        this.config = config;
    }

    MirrorSourceConnector(SourceAndTarget sourceAndTarget, ReplicationPolicy replicationPolicy, TopicFilter topicFilter, ConfigPropertyFilter configPropertyFilter) {
        this.sourceAndTarget = sourceAndTarget;
        this.replicationPolicy = replicationPolicy;
        this.topicFilter = topicFilter;
        this.configPropertyFilter = configPropertyFilter;
    }

    MirrorSourceConnector(Admin sourceAdminClient, Admin targetAdminClient) {
        this.sourceAdminClient = sourceAdminClient;
        this.targetAdminClient = targetAdminClient;
    }

    public void start(Map<String, String> props) {
        long start = System.currentTimeMillis();
        this.config = new MirrorSourceConfig(props);
        if (!this.config.enabled()) {
            return;
        }
        this.connectorName = this.config.connectorName();
        this.sourceAndTarget = new SourceAndTarget(this.config.sourceClusterAlias(), this.config.targetClusterAlias());
        this.topicFilter = this.config.topicFilter();
        this.configPropertyFilter = this.config.configPropertyFilter();
        this.replicationPolicy = this.config.replicationPolicy();
        this.replicationFactor = this.config.replicationFactor();
        this.sourceAdminClient = this.config.forwardingAdmin(this.config.sourceAdminConfig());
        this.targetAdminClient = this.config.forwardingAdmin(this.config.targetAdminConfig());
        this.offsetSyncsAdminClient = this.config.forwardingAdmin(this.config.offsetSyncsTopicAdminConfig());
        this.scheduler = new Scheduler(MirrorSourceConnector.class, this.config.adminTimeout());
        this.scheduler.execute(this::createOffsetSyncsTopic, "creating upstream offset-syncs topic");
        this.scheduler.execute(this::loadTopicPartitions, "loading initial set of topic-partitions");
        this.scheduler.execute(this::computeAndCreateTopicPartitions, "creating downstream topic-partitions");
        this.scheduler.execute(this::refreshKnownTargetTopics, "refreshing known target topics");
        this.scheduler.scheduleRepeating(this::syncTopicAcls, this.config.syncTopicAclsInterval(), "syncing topic ACLs");
        this.scheduler.scheduleRepeating(this::syncTopicConfigs, this.config.syncTopicConfigsInterval(), "syncing topic configs");
        this.scheduler.scheduleRepeatingDelayed(this::refreshTopicPartitions, this.config.refreshTopicsInterval(), "refreshing topics");
        log.info("Started {} with {} topic-partitions.", (Object)this.connectorName, (Object)this.knownSourceTopicPartitions.size());
        log.info("Starting {} took {} ms.", (Object)this.connectorName, (Object)(System.currentTimeMillis() - start));
    }

    public void stop() {
        long start = System.currentTimeMillis();
        if (!this.config.enabled()) {
            return;
        }
        Utils.closeQuietly((AutoCloseable)this.scheduler, (String)"scheduler");
        Utils.closeQuietly((AutoCloseable)this.topicFilter, (String)"topic filter");
        Utils.closeQuietly((AutoCloseable)this.configPropertyFilter, (String)"config property filter");
        Utils.closeQuietly((AutoCloseable)this.sourceAdminClient, (String)"source admin client");
        Utils.closeQuietly((AutoCloseable)this.targetAdminClient, (String)"target admin client");
        Utils.closeQuietly((AutoCloseable)this.offsetSyncsAdminClient, (String)"offset syncs admin client");
        log.info("Stopping {} took {} ms.", (Object)this.connectorName, (Object)(System.currentTimeMillis() - start));
    }

    public Class<? extends Task> taskClass() {
        return MirrorSourceTask.class;
    }

    public List<Map<String, String>> taskConfigs(int maxTasks) {
        if (!this.config.enabled() || this.knownSourceTopicPartitions.isEmpty()) {
            return Collections.emptyList();
        }
        int numTasks = Math.min(maxTasks, this.knownSourceTopicPartitions.size());
        ArrayList roundRobinByTask = new ArrayList(numTasks);
        for (int i = 0; i < numTasks; ++i) {
            roundRobinByTask.add(new ArrayList());
        }
        int count = 0;
        for (TopicPartition partition : this.knownSourceTopicPartitions) {
            int index = count % numTasks;
            ((List)roundRobinByTask.get(index)).add(partition);
            ++count;
        }
        return roundRobinByTask.stream().map(this.config::taskConfigForTopicPartitions).collect(Collectors.toList());
    }

    public ConfigDef config() {
        return MirrorSourceConfig.CONNECTOR_CONFIG_DEF;
    }

    public String version() {
        return AppInfoParser.getVersion();
    }

    List<TopicPartition> findSourceTopicPartitions() throws InterruptedException, ExecutionException {
        Set<String> topics = this.listTopics(this.sourceAdminClient).stream().filter(this::shouldReplicateTopic).collect(Collectors.toSet());
        return MirrorSourceConnector.describeTopics(this.sourceAdminClient, topics).stream().flatMap(MirrorSourceConnector::expandTopicDescription).collect(Collectors.toList());
    }

    List<TopicPartition> findTargetTopicPartitions() throws InterruptedException, ExecutionException {
        Set<String> topics = this.listTopics(this.targetAdminClient).stream().filter(t -> this.sourceAndTarget.source().equals(this.replicationPolicy.topicSource(t))).filter(t -> !t.equals(this.config.checkpointsTopic())).collect(Collectors.toSet());
        return MirrorSourceConnector.describeTopics(this.targetAdminClient, topics).stream().flatMap(MirrorSourceConnector::expandTopicDescription).collect(Collectors.toList());
    }

    void refreshTopicPartitions() throws InterruptedException, ExecutionException {
        List<TopicPartition> sourceTopicPartitions = this.findSourceTopicPartitions();
        List<TopicPartition> targetTopicPartitions = this.findTargetTopicPartitions();
        HashSet<TopicPartition> sourceTopicPartitionsSet = new HashSet<TopicPartition>(sourceTopicPartitions);
        HashSet<TopicPartition> knownSourceTopicPartitionsSet = new HashSet<TopicPartition>(this.knownSourceTopicPartitions);
        Set upstreamTargetTopicPartitions = targetTopicPartitions.stream().map(x -> new TopicPartition(this.replicationPolicy.upstreamTopic(x.topic()), x.partition())).collect(Collectors.toSet());
        HashSet<TopicPartition> missingInTarget = new HashSet<TopicPartition>(sourceTopicPartitions);
        missingInTarget.removeAll(upstreamTargetTopicPartitions);
        this.knownTargetTopicPartitions = targetTopicPartitions;
        if (!knownSourceTopicPartitionsSet.equals(sourceTopicPartitionsSet) || !missingInTarget.isEmpty()) {
            HashSet<TopicPartition> newTopicPartitions = sourceTopicPartitionsSet;
            newTopicPartitions.removeAll(this.knownSourceTopicPartitions);
            HashSet<TopicPartition> deletedTopicPartitions = knownSourceTopicPartitionsSet;
            deletedTopicPartitions.removeAll(sourceTopicPartitions);
            log.info("Found {} new topic-partitions on {}. Found {} deleted topic-partitions on {}. Found {} topic-partitions missing on {}.", new Object[]{newTopicPartitions.size(), this.sourceAndTarget.source(), deletedTopicPartitions.size(), this.sourceAndTarget.source(), missingInTarget.size(), this.sourceAndTarget.target()});
            log.trace("Found new topic-partitions on {}: {}", (Object)this.sourceAndTarget.source(), newTopicPartitions);
            log.trace("Found deleted topic-partitions on {}: {}", (Object)this.sourceAndTarget.source(), deletedTopicPartitions);
            log.trace("Found missing topic-partitions on {}: {}", (Object)this.sourceAndTarget.target(), missingInTarget);
            this.knownSourceTopicPartitions = sourceTopicPartitions;
            this.computeAndCreateTopicPartitions();
            this.context.requestTaskReconfiguration();
        }
    }

    private void loadTopicPartitions() throws InterruptedException, ExecutionException {
        this.knownSourceTopicPartitions = this.findSourceTopicPartitions();
        this.knownTargetTopicPartitions = this.findTargetTopicPartitions();
    }

    private void refreshKnownTargetTopics() throws InterruptedException, ExecutionException {
        this.knownTargetTopicPartitions = this.findTargetTopicPartitions();
    }

    private Set<String> topicsBeingReplicated() {
        Set<String> knownTargetTopics = this.toTopics(this.knownTargetTopicPartitions);
        return this.knownSourceTopicPartitions.stream().map(TopicPartition::topic).distinct().filter(x -> knownTargetTopics.contains(this.formatRemoteTopic((String)x))).collect(Collectors.toSet());
    }

    private Set<String> toTopics(Collection<TopicPartition> tps) {
        return tps.stream().map(TopicPartition::topic).collect(Collectors.toSet());
    }

    void syncTopicAcls() throws InterruptedException, ExecutionException {
        Optional<Collection<AclBinding>> rawBindings = this.listTopicAclBindings();
        if (!rawBindings.isPresent()) {
            return;
        }
        List<AclBinding> filteredBindings = rawBindings.get().stream().filter(x -> x.pattern().resourceType() == ResourceType.TOPIC).filter(x -> x.pattern().patternType() == PatternType.LITERAL).filter(this::shouldReplicateAcl).filter(x -> this.shouldReplicateTopic(x.pattern().name())).map(this::targetAclBinding).collect(Collectors.toList());
        this.updateTopicAcls(filteredBindings);
    }

    private void syncTopicConfigs() throws InterruptedException, ExecutionException {
        Map<String, Config> sourceConfigs = this.describeTopicConfigs(this.topicsBeingReplicated());
        Map<String, Config> targetConfigs = sourceConfigs.entrySet().stream().collect(Collectors.toMap(x -> this.formatRemoteTopic((String)x.getKey()), x -> this.targetConfig((Config)x.getValue())));
        this.updateTopicConfigs(targetConfigs);
    }

    private void createOffsetSyncsTopic() {
        MirrorUtils.createSinglePartitionCompactedTopic(this.config.offsetSyncsTopic(), this.config.offsetSyncsTopicReplicationFactor(), this.offsetSyncsAdminClient);
    }

    void computeAndCreateTopicPartitions() throws ExecutionException, InterruptedException {
        Map<String, Long> sourceTopicToPartitionCounts = this.knownSourceTopicPartitions.stream().collect(Collectors.groupingBy(TopicPartition::topic, Collectors.counting())).entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        Map<String, Long> targetTopicToPartitionCounts = this.knownTargetTopicPartitions.stream().collect(Collectors.groupingBy(TopicPartition::topic, Collectors.counting())).entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        Set<String> knownSourceTopics = sourceTopicToPartitionCounts.keySet();
        Set<String> knownTargetTopics = targetTopicToPartitionCounts.keySet();
        Map sourceToRemoteTopics = knownSourceTopics.stream().collect(Collectors.toMap(Function.identity(), this::formatRemoteTopic));
        Map partitionedSourceTopics = knownSourceTopics.stream().collect(Collectors.partitioningBy(sourceTopic -> knownTargetTopics.contains(sourceToRemoteTopics.get(sourceTopic)), Collectors.toSet()));
        Set existingSourceTopics = partitionedSourceTopics.get(true);
        Set<String> newSourceTopics = partitionedSourceTopics.get(false);
        if (!newSourceTopics.isEmpty()) {
            this.createNewTopics(newSourceTopics, sourceTopicToPartitionCounts);
        }
        Map sourceTopicsWithNewPartitions = existingSourceTopics.stream().filter(sourceTopic -> {
            String targetTopic = (String)sourceToRemoteTopics.get(sourceTopic);
            return (Long)sourceTopicToPartitionCounts.get(sourceTopic) > (Long)targetTopicToPartitionCounts.get(targetTopic);
        }).collect(Collectors.toMap(Function.identity(), sourceTopicToPartitionCounts::get));
        if (!sourceTopicsWithNewPartitions.isEmpty()) {
            Map<String, NewPartitions> newTargetPartitions = sourceTopicsWithNewPartitions.entrySet().stream().collect(Collectors.toMap(sourceTopicAndPartitionCount -> (String)sourceToRemoteTopics.get(sourceTopicAndPartitionCount.getKey()), sourceTopicAndPartitionCount -> NewPartitions.increaseTo((int)((Long)sourceTopicAndPartitionCount.getValue()).intValue())));
            this.createNewPartitions(newTargetPartitions);
        }
    }

    void createNewTopics(Set<String> newSourceTopics, Map<String, Long> sourceTopicToPartitionCounts) throws ExecutionException, InterruptedException {
        Map<String, Config> sourceTopicToConfig = this.describeTopicConfigs(newSourceTopics);
        Map<String, NewTopic> newTopics = newSourceTopics.stream().map(sourceTopic -> {
            String remoteTopic = this.formatRemoteTopic((String)sourceTopic);
            int partitionCount = ((Long)sourceTopicToPartitionCounts.get(sourceTopic)).intValue();
            Map<String, String> configs = MirrorSourceConnector.configToMap(this.targetConfig((Config)sourceTopicToConfig.get(sourceTopic)));
            return new NewTopic(remoteTopic, partitionCount, (short)this.replicationFactor).configs(configs);
        }).collect(Collectors.toMap(NewTopic::name, Function.identity()));
        this.createNewTopics(newTopics);
    }

    void createNewTopics(Map<String, NewTopic> newTopics) {
        this.targetAdminClient.createTopics(newTopics.values(), new CreateTopicsOptions()).values().forEach((k, v) -> v.whenComplete((x, e) -> {
            if (e != null) {
                log.warn("Could not create topic {}.", k, e);
            } else {
                log.info("Created remote topic {} with {} partitions.", k, (Object)((NewTopic)newTopics.get(k)).numPartitions());
            }
        }));
    }

    void createNewPartitions(Map<String, NewPartitions> newPartitions) {
        this.targetAdminClient.createPartitions(newPartitions).values().forEach((k, v) -> v.whenComplete((x, e) -> {
            if (!(e instanceof InvalidPartitionsException)) {
                if (e != null) {
                    log.warn("Could not create topic-partitions for {}.", k, e);
                } else {
                    log.info("Increased size of {} to {} partitions.", k, (Object)((NewPartitions)newPartitions.get(k)).totalCount());
                }
            }
        }));
    }

    private Set<String> listTopics(Admin adminClient) throws InterruptedException, ExecutionException {
        return (Set)adminClient.listTopics().names().get();
    }

    private Optional<Collection<AclBinding>> listTopicAclBindings() throws InterruptedException, ExecutionException {
        Collection bindings;
        try {
            bindings = (Collection)this.sourceAdminClient.describeAcls(ANY_TOPIC_ACL).values().get();
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof SecurityDisabledException) {
                if (this.noAclAuthorizer.compareAndSet(false, true)) {
                    log.info("No ACL authorizer is configured on the source Kafka cluster, so no topic ACL syncing will take place. Consider disabling topic ACL syncing by setting sync.topic.acls.enabled to 'false'.");
                } else {
                    log.debug("Source-side ACL authorizer still not found; skipping topic ACL sync");
                }
                return Optional.empty();
            }
            throw e;
        }
        return Optional.of(bindings);
    }

    private static Collection<TopicDescription> describeTopics(Admin adminClient, Collection<String> topics) throws InterruptedException, ExecutionException {
        return ((Map)adminClient.describeTopics(topics).allTopicNames().get()).values();
    }

    static Map<String, String> configToMap(Config config) {
        return config.entries().stream().collect(Collectors.toMap(ConfigEntry::name, ConfigEntry::value));
    }

    private void updateTopicConfigs(Map<String, Config> topicConfigs) {
        Map<ConfigResource, Config> configs = topicConfigs.entrySet().stream().collect(Collectors.toMap(x -> new ConfigResource(ConfigResource.Type.TOPIC, (String)x.getKey()), Map.Entry::getValue));
        log.trace("Syncing configs for {} topics.", (Object)configs.size());
        this.targetAdminClient.alterConfigs(configs).values().forEach((k, v) -> v.whenComplete((x, e) -> {
            if (e != null) {
                log.warn("Could not alter configuration of topic {}.", (Object)k.name(), e);
            }
        }));
    }

    private void updateTopicAcls(List<AclBinding> bindings) {
        log.trace("Syncing {} topic ACL bindings.", (Object)bindings.size());
        this.targetAdminClient.createAcls(bindings).values().forEach((k, v) -> v.whenComplete((x, e) -> {
            if (e != null) {
                log.warn("Could not sync ACL of topic {}.", (Object)k.pattern().name(), e);
            }
        }));
    }

    private static Stream<TopicPartition> expandTopicDescription(TopicDescription description) {
        String topic = description.name();
        return description.partitions().stream().map(x -> new TopicPartition(topic, x.partition()));
    }

    Map<String, Config> describeTopicConfigs(Set<String> topics) throws InterruptedException, ExecutionException {
        Set resources = topics.stream().map(x -> new ConfigResource(ConfigResource.Type.TOPIC, x)).collect(Collectors.toSet());
        return ((Map)this.sourceAdminClient.describeConfigs(resources).all().get()).entrySet().stream().collect(Collectors.toMap(x -> ((ConfigResource)x.getKey()).name(), Map.Entry::getValue));
    }

    Config targetConfig(Config sourceConfig) {
        List entries = sourceConfig.entries().stream().filter(x -> !x.isDefault() && !x.isReadOnly() && !x.isSensitive()).filter(x -> x.source() != ConfigEntry.ConfigSource.STATIC_BROKER_CONFIG).filter(x -> this.shouldReplicateTopicConfigurationProperty(x.name())).collect(Collectors.toList());
        return new Config(entries);
    }

    private static AccessControlEntry downgradeAllowAllACL(AccessControlEntry entry) {
        return new AccessControlEntry(entry.principal(), entry.host(), AclOperation.READ, entry.permissionType());
    }

    AclBinding targetAclBinding(AclBinding sourceAclBinding) {
        String targetTopic = this.formatRemoteTopic(sourceAclBinding.pattern().name());
        AccessControlEntry entry = sourceAclBinding.entry().permissionType() == AclPermissionType.ALLOW && sourceAclBinding.entry().operation() == AclOperation.ALL ? MirrorSourceConnector.downgradeAllowAllACL(sourceAclBinding.entry()) : sourceAclBinding.entry();
        return new AclBinding(new ResourcePattern(ResourceType.TOPIC, targetTopic, PatternType.LITERAL), entry);
    }

    boolean shouldReplicateTopic(String topic) {
        return (this.topicFilter.shouldReplicateTopic(topic) || this.replicationPolicy.isHeartbeatsTopic(topic)) && !this.replicationPolicy.isInternalTopic(topic) && !this.isCycle(topic);
    }

    boolean shouldReplicateAcl(AclBinding aclBinding) {
        return aclBinding.entry().permissionType() != AclPermissionType.ALLOW || aclBinding.entry().operation() != AclOperation.WRITE;
    }

    boolean shouldReplicateTopicConfigurationProperty(String property) {
        return this.configPropertyFilter.shouldReplicateConfigProperty(property);
    }

    boolean isCycle(String topic) {
        String source = this.replicationPolicy.topicSource(topic);
        if (source == null) {
            return false;
        }
        if (source.equals(this.sourceAndTarget.target())) {
            return true;
        }
        String upstreamTopic = this.replicationPolicy.upstreamTopic(topic);
        if (upstreamTopic == null || upstreamTopic.equals(topic)) {
            return false;
        }
        return this.isCycle(upstreamTopic);
    }

    String formatRemoteTopic(String topic) {
        return this.replicationPolicy.formatRemoteTopic(this.sourceAndTarget.source(), topic);
    }
}

