/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.connect.mirror;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Semaphore;
import java.util.stream.Collectors;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.WakeupException;
import org.apache.kafka.common.header.Header;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.header.ConnectHeaders;
import org.apache.kafka.connect.header.Headers;
import org.apache.kafka.connect.mirror.MirrorSourceConnector;
import org.apache.kafka.connect.mirror.MirrorSourceMetrics;
import org.apache.kafka.connect.mirror.MirrorSourceTaskConfig;
import org.apache.kafka.connect.mirror.MirrorUtils;
import org.apache.kafka.connect.mirror.OffsetSync;
import org.apache.kafka.connect.mirror.ReplicationPolicy;
import org.apache.kafka.connect.source.SourceRecord;
import org.apache.kafka.connect.source.SourceTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MirrorSourceTask
extends SourceTask {
    private static final Logger log = LoggerFactory.getLogger(MirrorSourceTask.class);
    private static final int MAX_OUTSTANDING_OFFSET_SYNCS = 10;
    private KafkaConsumer<byte[], byte[]> consumer;
    private KafkaProducer<byte[], byte[]> offsetProducer;
    private String sourceClusterAlias;
    private String offsetSyncsTopic;
    private Duration pollTimeout;
    private long maxOffsetLag;
    private Map<TopicPartition, PartitionState> partitionStates;
    private ReplicationPolicy replicationPolicy;
    private MirrorSourceMetrics metrics;
    private boolean stopping = false;
    private Semaphore outstandingOffsetSyncs;
    private Semaphore consumerAccess;

    public MirrorSourceTask() {
    }

    MirrorSourceTask(KafkaConsumer<byte[], byte[]> consumer, MirrorSourceMetrics metrics, String sourceClusterAlias, ReplicationPolicy replicationPolicy, long maxOffsetLag, KafkaProducer<byte[], byte[]> producer) {
        this.consumer = consumer;
        this.metrics = metrics;
        this.sourceClusterAlias = sourceClusterAlias;
        this.replicationPolicy = replicationPolicy;
        this.maxOffsetLag = maxOffsetLag;
        this.consumerAccess = new Semaphore(1);
        this.offsetProducer = producer;
    }

    public void start(Map<String, String> props) {
        MirrorSourceTaskConfig config = new MirrorSourceTaskConfig(props);
        this.outstandingOffsetSyncs = new Semaphore(10);
        this.consumerAccess = new Semaphore(1);
        this.sourceClusterAlias = config.sourceClusterAlias();
        this.metrics = config.metrics();
        this.pollTimeout = config.consumerPollTimeout();
        this.maxOffsetLag = config.maxOffsetLag();
        this.replicationPolicy = config.replicationPolicy();
        this.partitionStates = new HashMap<TopicPartition, PartitionState>();
        this.offsetSyncsTopic = config.offsetSyncsTopic();
        this.consumer = MirrorUtils.newConsumer(config.sourceConsumerConfig());
        this.offsetProducer = MirrorUtils.newProducer(config.offsetSyncsTopicProducerConfig());
        Set<TopicPartition> taskTopicPartitions = config.taskTopicPartitions();
        Map<TopicPartition, Long> topicPartitionOffsets = this.loadOffsets(taskTopicPartitions);
        this.consumer.assign(topicPartitionOffsets.keySet());
        log.info("Starting with {} previously uncommitted partitions.", (Object)topicPartitionOffsets.entrySet().stream().filter(x -> (Long)x.getValue() == 0L).count());
        log.trace("Seeking offsets: {}", topicPartitionOffsets);
        topicPartitionOffsets.forEach((arg_0, arg_1) -> this.consumer.seek(arg_0, arg_1));
        log.info("{} replicating {} topic-partitions {}->{}: {}.", new Object[]{Thread.currentThread().getName(), taskTopicPartitions.size(), this.sourceClusterAlias, config.targetClusterAlias(), taskTopicPartitions});
    }

    public void commit() {
    }

    public void stop() {
        long start = System.currentTimeMillis();
        this.stopping = true;
        this.consumer.wakeup();
        try {
            this.consumerAccess.acquire();
        }
        catch (InterruptedException e) {
            log.warn("Interrupted waiting for access to consumer. Will try closing anyway.");
        }
        Utils.closeQuietly(this.consumer, (String)"source consumer");
        Utils.closeQuietly(this.offsetProducer, (String)"offset producer");
        Utils.closeQuietly((AutoCloseable)this.metrics, (String)"metrics");
        log.info("Stopping {} took {} ms.", (Object)Thread.currentThread().getName(), (Object)(System.currentTimeMillis() - start));
    }

    public String version() {
        return new MirrorSourceConnector().version();
    }

    public List<SourceRecord> poll() {
        List<SourceRecord> list;
        if (!this.consumerAccess.tryAcquire()) {
            return null;
        }
        if (this.stopping) {
            return null;
        }
        try {
            Object object;
            ConsumerRecords records = this.consumer.poll(this.pollTimeout);
            ArrayList<SourceRecord> sourceRecords = new ArrayList<SourceRecord>(records.count());
            for (ConsumerRecord record : records) {
                SourceRecord converted = this.convertRecord((ConsumerRecord<byte[], byte[]>)record);
                sourceRecords.add(converted);
                TopicPartition topicPartition = new TopicPartition(converted.topic(), converted.kafkaPartition().intValue());
                this.metrics.recordAge(topicPartition, System.currentTimeMillis() - record.timestamp());
                this.metrics.recordBytes(topicPartition, MirrorSourceTask.byteSize((byte[])record.value()));
            }
            if (sourceRecords.isEmpty()) {
                object = null;
                return object;
            }
            log.trace("Polled {} records from {}.", (Object)sourceRecords.size(), (Object)records.partitions());
            object = sourceRecords;
            return object;
        }
        catch (WakeupException e) {
            list = null;
            return list;
        }
        catch (KafkaException e) {
            log.warn("Failure during poll.", (Throwable)e);
            list = null;
            return list;
        }
        catch (Throwable e) {
            log.error("Failure during poll.", e);
            throw e;
        }
        finally {
            this.consumerAccess.release();
        }
    }

    public void commitRecord(SourceRecord record, RecordMetadata metadata) {
        if (this.stopping) {
            return;
        }
        if (metadata == null) {
            log.debug("No RecordMetadata (source record was probably filtered out during transformation) -- can't sync offsets for {}.", (Object)record.topic());
            return;
        }
        if (!metadata.hasOffset()) {
            log.error("RecordMetadata has no offset -- can't sync offsets for {}.", (Object)record.topic());
            return;
        }
        TopicPartition topicPartition = new TopicPartition(record.topic(), record.kafkaPartition().intValue());
        long latency = System.currentTimeMillis() - record.timestamp();
        this.metrics.countRecord(topicPartition);
        this.metrics.replicationLatency(topicPartition, latency);
        TopicPartition sourceTopicPartition = MirrorUtils.unwrapPartition(record.sourcePartition());
        long upstreamOffset = MirrorUtils.unwrapOffset(record.sourceOffset());
        long downstreamOffset = metadata.offset();
        this.maybeSyncOffsets(sourceTopicPartition, upstreamOffset, downstreamOffset);
    }

    private void maybeSyncOffsets(TopicPartition topicPartition, long upstreamOffset, long downstreamOffset) {
        PartitionState partitionState = this.partitionStates.computeIfAbsent(topicPartition, x -> new PartitionState(this.maxOffsetLag));
        if (partitionState.update(upstreamOffset, downstreamOffset)) {
            this.sendOffsetSync(topicPartition, upstreamOffset, downstreamOffset);
        }
    }

    private void sendOffsetSync(TopicPartition topicPartition, long upstreamOffset, long downstreamOffset) {
        if (!this.outstandingOffsetSyncs.tryAcquire()) {
            return;
        }
        OffsetSync offsetSync = new OffsetSync(topicPartition, upstreamOffset, downstreamOffset);
        ProducerRecord record = new ProducerRecord(this.offsetSyncsTopic, Integer.valueOf(0), (Object)offsetSync.recordKey(), (Object)offsetSync.recordValue());
        this.offsetProducer.send(record, (x, e) -> {
            if (e != null) {
                log.error("Failure sending offset sync.", (Throwable)e);
            } else {
                log.trace("Sync'd offsets for {}: {}=={}", new Object[]{topicPartition, upstreamOffset, downstreamOffset});
            }
            this.outstandingOffsetSyncs.release();
        });
    }

    private Map<TopicPartition, Long> loadOffsets(Set<TopicPartition> topicPartitions) {
        return topicPartitions.stream().collect(Collectors.toMap(x -> x, this::loadOffset));
    }

    private Long loadOffset(TopicPartition topicPartition) {
        Map<String, Object> wrappedPartition = MirrorUtils.wrapPartition(topicPartition, this.sourceClusterAlias);
        Map wrappedOffset = this.context.offsetStorageReader().offset(wrappedPartition);
        return MirrorUtils.unwrapOffset(wrappedOffset) + 1L;
    }

    SourceRecord convertRecord(ConsumerRecord<byte[], byte[]> record) {
        String targetTopic = this.formatRemoteTopic(record.topic());
        Headers headers = this.convertHeaders(record);
        return new SourceRecord(MirrorUtils.wrapPartition(new TopicPartition(record.topic(), record.partition()), this.sourceClusterAlias), MirrorUtils.wrapOffset(record.offset()), targetTopic, Integer.valueOf(record.partition()), Schema.OPTIONAL_BYTES_SCHEMA, record.key(), Schema.BYTES_SCHEMA, record.value(), Long.valueOf(record.timestamp()), (Iterable)headers);
    }

    private Headers convertHeaders(ConsumerRecord<byte[], byte[]> record) {
        ConnectHeaders headers = new ConnectHeaders();
        for (Header header : record.headers()) {
            headers.addBytes(header.key(), header.value());
        }
        return headers;
    }

    private String formatRemoteTopic(String topic) {
        return this.replicationPolicy.formatRemoteTopic(this.sourceClusterAlias, topic);
    }

    private static int byteSize(byte[] bytes) {
        if (bytes == null) {
            return 0;
        }
        return bytes.length;
    }

    static class PartitionState {
        long previousUpstreamOffset = -1L;
        long previousDownstreamOffset = -1L;
        long lastSyncUpstreamOffset = -1L;
        long lastSyncDownstreamOffset = -1L;
        long maxOffsetLag;

        PartitionState(long maxOffsetLag) {
            this.maxOffsetLag = maxOffsetLag;
        }

        boolean update(long upstreamOffset, long downstreamOffset) {
            boolean shouldSyncOffsets = false;
            long upstreamStep = upstreamOffset - this.lastSyncUpstreamOffset;
            long downstreamTargetOffset = this.lastSyncDownstreamOffset + upstreamStep;
            if (this.lastSyncDownstreamOffset == -1L || downstreamOffset - downstreamTargetOffset >= this.maxOffsetLag || upstreamOffset - this.previousUpstreamOffset != 1L || downstreamOffset < this.previousDownstreamOffset) {
                this.lastSyncUpstreamOffset = upstreamOffset;
                this.lastSyncDownstreamOffset = downstreamOffset;
                shouldSyncOffsets = true;
            }
            this.previousUpstreamOffset = upstreamOffset;
            this.previousDownstreamOffset = downstreamOffset;
            return shouldSyncOffsets;
        }
    }
}

