/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.processor.internals;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import org.apache.kafka.clients.admin.Admin;
import org.apache.kafka.clients.admin.CreateTopicsResult;
import org.apache.kafka.clients.admin.DescribeTopicsResult;
import org.apache.kafka.clients.admin.NewTopic;
import org.apache.kafka.clients.admin.TopicDescription;
import org.apache.kafka.common.KafkaFuture;
import org.apache.kafka.common.errors.LeaderNotAvailableException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.errors.TopicExistsException;
import org.apache.kafka.common.errors.UnknownTopicOrPartitionException;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.streams.StreamsConfig;
import org.apache.kafka.streams.errors.StreamsException;
import org.apache.kafka.streams.processor.internals.ClientUtils;
import org.apache.kafka.streams.processor.internals.InternalTopicConfig;
import org.slf4j.Logger;

public class InternalTopicManager {
    private static final String INTERRUPTED_ERROR_MESSAGE = "Thread got interrupted. This indicates a bug. Please report at https://issues.apache.org/jira/projects/KAFKA or dev-mailing list (https://kafka.apache.org/contact).";
    private final Logger log;
    private final Time time;
    private final Admin adminClient;
    private final short replicationFactor;
    private final long windowChangeLogAdditionalRetention;
    private final long retryBackOffMs;
    private final long retryTimeoutMs;
    private final Map<String, String> defaultTopicConfigs = new HashMap<String, String>();

    public InternalTopicManager(Time time, Admin adminClient, StreamsConfig streamsConfig) {
        this.time = time;
        this.adminClient = adminClient;
        LogContext logContext = new LogContext(String.format("stream-thread [%s] ", Thread.currentThread().getName()));
        this.log = logContext.logger(this.getClass());
        this.replicationFactor = streamsConfig.getInt("replication.factor").shortValue();
        this.windowChangeLogAdditionalRetention = streamsConfig.getLong("windowstore.changelog.additional.retention.ms");
        this.retryBackOffMs = streamsConfig.getLong("retry.backoff.ms");
        Map<String, Object> consumerConfig = streamsConfig.getMainConsumerConfigs("dummy", "dummy", -1);
        consumerConfig.put("key.deserializer", ByteArrayDeserializer.class);
        consumerConfig.put("value.deserializer", ByteArrayDeserializer.class);
        this.retryTimeoutMs = (long)new ClientUtils.QuietConsumerConfig(consumerConfig).getInt("max.poll.interval.ms").intValue() / 2L;
        this.log.debug("Configs:" + Utils.NL + "\t{} = {}" + Utils.NL + "\t{} = {}", new Object[]{"replication.factor", this.replicationFactor, "windowstore.changelog.additional.retention.ms", this.windowChangeLogAdditionalRetention});
        for (Map.Entry entry : streamsConfig.originalsWithPrefix("topic.").entrySet()) {
            if (entry.getValue() == null) continue;
            this.defaultTopicConfigs.put((String)entry.getKey(), entry.getValue().toString());
        }
    }

    public Set<String> makeReady(Map<String, InternalTopicConfig> topics) {
        this.log.debug("Starting to validate internal topics {} in partition assignor.", topics);
        long currentWallClockMs = this.time.milliseconds();
        long deadlineMs = currentWallClockMs + this.retryTimeoutMs;
        Set<String> topicsNotReady = new HashSet<String>(topics.keySet());
        HashSet<String> newlyCreatedTopics = new HashSet<String>();
        while (!topicsNotReady.isEmpty()) {
            HashSet<String> tempUnknownTopics = new HashSet<String>();
            topicsNotReady = this.validateTopics(topicsNotReady, topics, tempUnknownTopics);
            newlyCreatedTopics.addAll(topicsNotReady);
            if (!topicsNotReady.isEmpty()) {
                HashSet<NewTopic> newTopics = new HashSet<NewTopic>();
                for (String topicName : topicsNotReady) {
                    if (tempUnknownTopics.contains(topicName)) continue;
                    InternalTopicConfig internalTopicConfig = Objects.requireNonNull(topics.get(topicName));
                    Map<String, String> topicConfig = internalTopicConfig.getProperties(this.defaultTopicConfigs, this.windowChangeLogAdditionalRetention);
                    this.log.debug("Going to create topic {} with {} partitions and config {}.", new Object[]{internalTopicConfig.name(), internalTopicConfig.numberOfPartitions(), topicConfig});
                    newTopics.add(new NewTopic(internalTopicConfig.name(), internalTopicConfig.numberOfPartitions(), Optional.of(this.replicationFactor)).configs(topicConfig));
                }
                CreateTopicsResult createTopicsResult = this.adminClient.createTopics(newTopics);
                for (Map.Entry createTopicResult : createTopicsResult.values().entrySet()) {
                    String topicName = (String)createTopicResult.getKey();
                    try {
                        ((KafkaFuture)createTopicResult.getValue()).get();
                        topicsNotReady.remove(topicName);
                    }
                    catch (InterruptedException fatalException) {
                        Thread.currentThread().interrupt();
                        this.log.error(INTERRUPTED_ERROR_MESSAGE, (Throwable)fatalException);
                        throw new IllegalStateException(INTERRUPTED_ERROR_MESSAGE, fatalException);
                    }
                    catch (ExecutionException executionException) {
                        Throwable cause = executionException.getCause();
                        if (cause instanceof TopicExistsException) {
                            this.log.info("Could not create topic {}. Topic is probably marked for deletion (number of partitions is unknown).\nWill retry to create this topic in {} ms (to let broker finish async delete operation first).\nError message was: {}", new Object[]{topicName, this.retryBackOffMs, cause.toString()});
                            continue;
                        }
                        this.log.error("Unexpected error during topic creation for {}.\nError message was: {}", (Object)topicName, (Object)cause.toString());
                        throw new StreamsException(String.format("Could not create topic %s.", topicName), cause);
                    }
                    catch (TimeoutException retryableException) {
                        this.log.error("Creating topic {} timed out.\nError message was: {}", (Object)topicName, (Object)retryableException.toString());
                    }
                }
            }
            if (topicsNotReady.isEmpty()) continue;
            currentWallClockMs = this.time.milliseconds();
            if (currentWallClockMs >= deadlineMs) {
                String timeoutError = String.format("Could not create topics within %d milliseconds. This can happen if the Kafka cluster is temporarily not available.", this.retryTimeoutMs);
                this.log.error(timeoutError);
                throw new TimeoutException(timeoutError);
            }
            this.log.info("Topics {} could not be made ready. Will retry in {} milliseconds. Remaining time in milliseconds: {}", new Object[]{topicsNotReady, this.retryBackOffMs, deadlineMs - currentWallClockMs});
            Utils.sleep((long)this.retryBackOffMs);
        }
        this.log.debug("Completed validating internal topics and created {}", newlyCreatedTopics);
        return newlyCreatedTopics;
    }

    protected Map<String, Integer> getNumPartitions(Set<String> topics, Set<String> tempUnknownTopics) {
        this.log.debug("Trying to check if topics {} have been created with expected number of partitions.", topics);
        DescribeTopicsResult describeTopicsResult = this.adminClient.describeTopics(topics);
        Map futures = describeTopicsResult.values();
        HashMap<String, Integer> existedTopicPartition = new HashMap<String, Integer>();
        for (Map.Entry topicFuture : futures.entrySet()) {
            String topicName = (String)topicFuture.getKey();
            try {
                TopicDescription topicDescription = (TopicDescription)((KafkaFuture)topicFuture.getValue()).get();
                existedTopicPartition.put(topicName, topicDescription.partitions().size());
            }
            catch (InterruptedException fatalException) {
                Thread.currentThread().interrupt();
                this.log.error(INTERRUPTED_ERROR_MESSAGE, (Throwable)fatalException);
                throw new IllegalStateException(INTERRUPTED_ERROR_MESSAGE, fatalException);
            }
            catch (ExecutionException couldNotDescribeTopicException) {
                Throwable cause = couldNotDescribeTopicException.getCause();
                if (cause instanceof UnknownTopicOrPartitionException) {
                    this.log.debug("Topic {} is unknown or not found, hence not existed yet.\nError message was: {}", (Object)topicName, (Object)cause.toString());
                    continue;
                }
                if (cause instanceof LeaderNotAvailableException) {
                    tempUnknownTopics.add(topicName);
                    this.log.debug("The leader of topic {} is not available.\nError message was: {}", (Object)topicName, (Object)cause.toString());
                    continue;
                }
                this.log.error("Unexpected error during topic description for {}.\nError message was: {}", (Object)topicName, (Object)cause.toString());
                throw new StreamsException(String.format("Could not create topic %s.", topicName), cause);
            }
            catch (TimeoutException retryableException) {
                tempUnknownTopics.add(topicName);
                this.log.debug("Describing topic {} (to get number of partitions) timed out.\nError message was: {}", (Object)topicName, (Object)retryableException.toString());
            }
        }
        return existedTopicPartition;
    }

    private Set<String> validateTopics(Set<String> topicsToValidate, Map<String, InternalTopicConfig> topicsMap, Set<String> tempUnknownTopics) {
        if (!topicsMap.keySet().containsAll(topicsToValidate)) {
            throw new IllegalStateException("The topics map " + topicsMap.keySet() + " does not contain all the topics " + topicsToValidate + " trying to validate.");
        }
        Map<String, Integer> existedTopicPartition = this.getNumPartitions(topicsToValidate, tempUnknownTopics);
        HashSet<String> topicsToCreate = new HashSet<String>();
        for (String topicName : topicsToValidate) {
            Optional<Integer> numberOfPartitions = topicsMap.get(topicName).numberOfPartitions();
            if (!numberOfPartitions.isPresent()) {
                this.log.error("Found undefined number of partitions for topic {}", (Object)topicName);
                throw new StreamsException("Topic " + topicName + " number of partitions not defined");
            }
            if (existedTopicPartition.containsKey(topicName)) {
                if (existedTopicPartition.get(topicName).equals(numberOfPartitions.get())) continue;
                String errorMsg = String.format("Existing internal topic %s has invalid partitions: expected: %d; actual: %d. Use 'kafka.tools.StreamsResetter' tool to clean up invalid topics before processing.", topicName, numberOfPartitions.get(), existedTopicPartition.get(topicName));
                this.log.error(errorMsg);
                throw new StreamsException(errorMsg);
            }
            topicsToCreate.add(topicName);
        }
        return topicsToCreate;
    }
}

