// ***************************************************************************************************************************
// * Licensed to the Apache Software Foundation (ASF) under one or more contributor license agreements.  See the NOTICE file *
// * distributed with this work for additional information regarding copyright ownership.  The ASF licenses this file        *
// * to you under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance            *
// * with the License.  You may obtain a copy of the License at                                                              *
// *                                                                                                                         *
// *  http://www.apache.org/licenses/LICENSE-2.0                                                                             *
// *                                                                                                                         *
// * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on an  *
// * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the License for the        *
// * specific language governing permissions and limitations under the License.                                              *
// ***************************************************************************************************************************
package org.apache.juneau.serializer;

import static org.apache.juneau.assertions.Assertions.*;
import static org.junit.runners.MethodSorters.*;

import java.util.*;

import org.apache.juneau.*;
import org.apache.juneau.html.*;
import org.apache.juneau.json.*;
import org.apache.juneau.msgpack.*;
import org.apache.juneau.parser.*;
import org.apache.juneau.uon.*;
import org.apache.juneau.urlencoding.*;
import org.apache.juneau.xml.*;
import org.junit.*;
import org.junit.runner.*;
import org.junit.runners.*;

/**
 * Superclass for tests that verify results against all supported content types.
 */
@FixMethodOrder(NAME_ASCENDING)
@RunWith(Parameterized.class)
public class UriResolutionTest {

	private final String label;
	private final Input input;
	private final Results results;

	//------------------------------------------------------------------------------------------------------------------
	// Input
	//------------------------------------------------------------------------------------------------------------------
	private static class Input {
		private final UriContext context;
		private final UriResolution resolution;
		private final UriRelativity relativity;

		public Input(UriResolution resolution, UriRelativity relativity, String authority, String context, String resource, String path) {
			this.resolution = resolution;
			this.relativity = relativity;
			this.context = UriContext.of(authority, context, resource, path);
		}
	}

	private static Input input(UriResolution uriResolution, UriRelativity uriRelativity, String authority, String context, String resource, String path) {
		return new Input(uriResolution, uriRelativity, authority, context, resource, path);
	}

	//------------------------------------------------------------------------------------------------------------------
	// Output
	//------------------------------------------------------------------------------------------------------------------
	private static class Results {
		final String json, xml, html, uon, urlEncoding, msgPack;

		Results(String json, String xml, String html, String uon, String urlEncoding, String msgPack, String rdfXml) {
			this.json = json;
			this.xml = xml;
			this.html = html;
			this.uon = uon;
			this.urlEncoding = urlEncoding;
			this.msgPack = msgPack;
		}
	}

	private static Results results(String json, String xml, String html, String uon, String urlEncoding, String msgPack, String rdfXml) {
		return new Results(json, xml, html, uon, urlEncoding, msgPack, rdfXml);
	}

	public UriResolutionTest(String label, Input input, Results results) {
		this.label = label;
		this.input = input;
		this.results = results;
	}

	@Parameterized.Parameters
	public static Collection<Object[]> getParameters() {
		return Arrays.asList(new Object[][] {

			// Happy cases - All URL parts known.
			{	/* 0 */
				"Happy-1-ABSOLUTE",
				input(
					UriResolution.ABSOLUTE, UriRelativity.RESOURCE, "http://foo.com:123","/context","/resource","/path"
				),
				results(
					"{f0:'http://foo.com:123/context/resource/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'http://foo.com:123/f1b',f1c:'http://foo.com:123/f1c/x/y',f1d:'http://foo.com:123/context/resource/f1d',f1e:'http://foo.com:123/context/resource/f1e/x/y',f1f:'http://foo.com:123/context/resource',f1g:'http://foo.com:123/context/resource/f1g/x',f1h:'http://foo.com:123/context/resource/f1h',f1i:'http://foo.com:123/context/resource',f1j:'http://foo.com:123/context',f1k:'http://foo.com:123/context/f1j/x',f1l:'http://foo.com:123/context/f1k',f1m:'http://foo.com:123/context',f1n:'http://foo.com:123',f2a:'http://www.apache.org/f2a',f2b:'http://foo.com:123/f2b',f2c:'http://foo.com:123/f2c/x/y',f2d:'http://foo.com:123/context/resource/f2d',f2e:'http://foo.com:123/context/resource/f2e/x/y',f2f:'http://foo.com:123/context/resource',f2g:'http://foo.com:123/context/resource/f2g/x',f2h:'http://foo.com:123/context/resource/f2h',f2i:'http://foo.com:123/context/resource',f2j:'http://foo.com:123/context',f2k:'http://foo.com:123/context/f2j/x',f2l:'http://foo.com:123/context/f2k',f2m:'http://foo.com:123/context',f2n:'http://foo.com:123',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'http://foo.com:123/context/resource/%3C%3E%26%27%22',f3c:'http://foo.com:123/context/resource/<>&\\'\"',f4:'http://foo.com:123/context/resource/test/uri/b',f5:'http://foo.com:123/context/resource/f5/x'}",
					"<object f0='http://foo.com:123/context/resource/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>http://foo.com:123/f1b</f1b><f1c>http://foo.com:123/f1c/x/y</f1c><f1d>http://foo.com:123/context/resource/f1d</f1d><f1e>http://foo.com:123/context/resource/f1e/x/y</f1e><f1f>http://foo.com:123/context/resource</f1f><f1g>http://foo.com:123/context/resource/f1g/x</f1g><f1h>http://foo.com:123/context/resource/f1h</f1h><f1i>http://foo.com:123/context/resource</f1i><f1j>http://foo.com:123/context</f1j><f1k>http://foo.com:123/context/f1j/x</f1k><f1l>http://foo.com:123/context/f1k</f1l><f1m>http://foo.com:123/context</f1m><f1n>http://foo.com:123</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>http://foo.com:123/f2b</f2b><f2c>http://foo.com:123/f2c/x/y</f2c><f2d>http://foo.com:123/context/resource/f2d</f2d><f2e>http://foo.com:123/context/resource/f2e/x/y</f2e><f2f>http://foo.com:123/context/resource</f2f><f2g>http://foo.com:123/context/resource/f2g/x</f2g><f2h>http://foo.com:123/context/resource/f2h</f2h><f2i>http://foo.com:123/context/resource</f2i><f2j>http://foo.com:123/context</f2j><f2k>http://foo.com:123/context/f2j/x</f2k><f2l>http://foo.com:123/context/f2k</f2l><f2m>http://foo.com:123/context</f2m><f2n>http://foo.com:123</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>http://foo.com:123/context/resource/%3C%3E%26%27%22</f3b><f3c>http://foo.com:123/context/resource/&lt;&gt;&amp;'\"</f3c><f4>http://foo.com:123/context/resource/test/uri/b</f4><f5>http://foo.com:123/context/resource/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='http://foo.com:123/context/resource/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='http://foo.com:123/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='http://foo.com:123/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='http://foo.com:123/context/resource/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='http://foo.com:123/context/resource/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='http://foo.com:123/context/resource'>resource</a></td></tr><tr><td>f1g</td><td><a href='http://foo.com:123/context/resource/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='http://foo.com:123/context/resource/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='http://foo.com:123/context/resource'>resource</a></td></tr><tr><td>f1j</td><td><a href='http://foo.com:123/context'>context</a></td></tr><tr><td>f1k</td><td><a href='http://foo.com:123/context/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='http://foo.com:123/context/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='http://foo.com:123/context'>context</a></td></tr><tr><td>f1n</td><td><a href='http://foo.com:123'>foo.com:123</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='http://foo.com:123/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='http://foo.com:123/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='http://foo.com:123/context/resource/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='http://foo.com:123/context/resource/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='http://foo.com:123/context/resource'>resource</a></td></tr><tr><td>f2g</td><td><a href='http://foo.com:123/context/resource/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='http://foo.com:123/context/resource/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='http://foo.com:123/context/resource'>resource</a></td></tr><tr><td>f2j</td><td><a href='http://foo.com:123/context'>context</a></td></tr><tr><td>f2k</td><td><a href='http://foo.com:123/context/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='http://foo.com:123/context/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='http://foo.com:123/context'>context</a></td></tr><tr><td>f2n</td><td><a href='http://foo.com:123'>foo.com:123</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='http://foo.com:123/context/resource/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='http://foo.com:123/context/resource/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='http://foo.com:123/context/resource/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='http://foo.com:123/context/resource/f5/x'>x</a></td></tr></table>",
					"(f0=http://foo.com:123/context/resource/f0/x0,f1a=http://www.apache.org/f1a,f1b=http://foo.com:123/f1b,f1c=http://foo.com:123/f1c/x/y,f1d=http://foo.com:123/context/resource/f1d,f1e=http://foo.com:123/context/resource/f1e/x/y,f1f=http://foo.com:123/context/resource,f1g=http://foo.com:123/context/resource/f1g/x,f1h=http://foo.com:123/context/resource/f1h,f1i=http://foo.com:123/context/resource,f1j=http://foo.com:123/context,f1k=http://foo.com:123/context/f1j/x,f1l=http://foo.com:123/context/f1k,f1m=http://foo.com:123/context,f1n=http://foo.com:123,f2a=http://www.apache.org/f2a,f2b=http://foo.com:123/f2b,f2c=http://foo.com:123/f2c/x/y,f2d=http://foo.com:123/context/resource/f2d,f2e=http://foo.com:123/context/resource/f2e/x/y,f2f=http://foo.com:123/context/resource,f2g=http://foo.com:123/context/resource/f2g/x,f2h=http://foo.com:123/context/resource/f2h,f2i=http://foo.com:123/context/resource,f2j=http://foo.com:123/context,f2k=http://foo.com:123/context/f2j/x,f2l=http://foo.com:123/context/f2k,f2m=http://foo.com:123/context,f2n=http://foo.com:123,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=http://foo.com:123/context/resource/%3C%3E%26%27%22,f3c=http://foo.com:123/context/resource/<>&~'\",f4=http://foo.com:123/context/resource/test/uri/b,f5=http://foo.com:123/context/resource/f5/x)",
					"f0=http://foo.com:123/context/resource/f0/x0&f1a=http://www.apache.org/f1a&f1b=http://foo.com:123/f1b&f1c=http://foo.com:123/f1c/x/y&f1d=http://foo.com:123/context/resource/f1d&f1e=http://foo.com:123/context/resource/f1e/x/y&f1f=http://foo.com:123/context/resource&f1g=http://foo.com:123/context/resource/f1g/x&f1h=http://foo.com:123/context/resource/f1h&f1i=http://foo.com:123/context/resource&f1j=http://foo.com:123/context&f1k=http://foo.com:123/context/f1j/x&f1l=http://foo.com:123/context/f1k&f1m=http://foo.com:123/context&f1n=http://foo.com:123&f2a=http://www.apache.org/f2a&f2b=http://foo.com:123/f2b&f2c=http://foo.com:123/f2c/x/y&f2d=http://foo.com:123/context/resource/f2d&f2e=http://foo.com:123/context/resource/f2e/x/y&f2f=http://foo.com:123/context/resource&f2g=http://foo.com:123/context/resource/f2g/x&f2h=http://foo.com:123/context/resource/f2h&f2i=http://foo.com:123/context/resource&f2j=http://foo.com:123/context&f2k=http://foo.com:123/context/f2j/x&f2l=http://foo.com:123/context/f2k&f2m=http://foo.com:123/context&f2n=http://foo.com:123&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=http://foo.com:123/context/resource/%253C%253E%2526%2527%2522&f3c=http://foo.com:123/context/resource/%3C%3E%26~'%22&f4=http://foo.com:123/context/resource/test/uri/b&f5=http://foo.com:123/context/resource/f5/x",
					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
					"<rdf:RDF>\n<rdf:Description rdf:about='http://foo.com:123/context/resource/f0/x0'>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f1g rdf:resource='http://foo.com:123/context/resource/f1g/x'/>\n<jp:f2m rdf:resource='http://foo.com:123/context'/>\n<jp:f2i rdf:resource='http://foo.com:123/context/resource'/>\n<jp:f5 rdf:resource='http://foo.com:123/context/resource/f5/x'/>\n<jp:f1k rdf:resource='http://foo.com:123/context/f1j/x'/>\n<jp:f2f rdf:resource='http://foo.com:123/context/resource'/>\n<jp:f4 rdf:resource='http://foo.com:123/context/resource/test/uri/b'/>\n<jp:f1f rdf:resource='http://foo.com:123/context/resource'/>\n<jp:f1b rdf:resource='http://foo.com:123/f1b'/>\n<jp:f1c rdf:resource='http://foo.com:123/f1c/x/y'/>\n<jp:f1l rdf:resource='http://foo.com:123/context/f1k'/>\n<jp:f2d rdf:resource='http://foo.com:123/context/resource/f2d'/>\n<jp:f1e rdf:resource='http://foo.com:123/context/resource/f1e/x/y'/>\n<jp:f2k rdf:resource='http://foo.com:123/context/f2j/x'/>\n<jp:f2e rdf:resource='http://foo.com:123/context/resource/f2e/x/y'/>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f2j rdf:resource='http://foo.com:123/context'/>\n<jp:f2n rdf:resource='http://foo.com:123'/>\n<jp:f2b rdf:resource='http://foo.com:123/f2b'/>\n<jp:f2h rdf:resource='http://foo.com:123/context/resource/f2h'/>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f1h rdf:resource='http://foo.com:123/context/resource/f1h'/>\n<jp:f1i rdf:resource='http://foo.com:123/context/resource'/>\n<jp:f1j rdf:resource='http://foo.com:123/context'/>\n<jp:f2g rdf:resource='http://foo.com:123/context/resource/f2g/x'/>\n<jp:f1n rdf:resource='http://foo.com:123'/>\n<jp:f1d rdf:resource='http://foo.com:123/context/resource/f1d'/>\n<jp:f3c rdf:resource='http://foo.com:123/context/resource/&lt;&gt;&amp;&apos;&quot;'/>\n<jp:f2l rdf:resource='http://foo.com:123/context/f2k'/>\n<jp:f2c rdf:resource='http://foo.com:123/f2c/x/y'/>\n<jp:f3b rdf:resource='http://foo.com:123/context/resource/%3C%3E%26%27%22'/>\n<jp:f1m rdf:resource='http://foo.com:123/context'/>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 1 */
				"Happy-1-ROOT_RELATIVE",
				input(
					UriResolution.ROOT_RELATIVE, UriRelativity.RESOURCE, "http://foo.com:123","/context","/resource","/path"
				),
				results(
					"{f0:'/context/resource/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'/context/resource/f1d',f1e:'/context/resource/f1e/x/y',f1f:'/context/resource',f1g:'/context/resource/f1g/x',f1h:'/context/resource/f1h',f1i:'/context/resource',f1j:'/context',f1k:'/context/f1j/x',f1l:'/context/f1k',f1m:'/context',f1n:'/',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'/context/resource/f2d',f2e:'/context/resource/f2e/x/y',f2f:'/context/resource',f2g:'/context/resource/f2g/x',f2h:'/context/resource/f2h',f2i:'/context/resource',f2j:'/context',f2k:'/context/f2j/x',f2l:'/context/f2k',f2m:'/context',f2n:'/',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'/context/resource/%3C%3E%26%27%22',f3c:'/context/resource/<>&\\'\"',f4:'/context/resource/test/uri/b',f5:'/context/resource/f5/x'}",
					"<object f0='/context/resource/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>/context/resource/f1d</f1d><f1e>/context/resource/f1e/x/y</f1e><f1f>/context/resource</f1f><f1g>/context/resource/f1g/x</f1g><f1h>/context/resource/f1h</f1h><f1i>/context/resource</f1i><f1j>/context</f1j><f1k>/context/f1j/x</f1k><f1l>/context/f1k</f1l><f1m>/context</f1m><f1n>/</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>/context/resource/f2d</f2d><f2e>/context/resource/f2e/x/y</f2e><f2f>/context/resource</f2f><f2g>/context/resource/f2g/x</f2g><f2h>/context/resource/f2h</f2h><f2i>/context/resource</f2i><f2j>/context</f2j><f2k>/context/f2j/x</f2k><f2l>/context/f2k</f2l><f2m>/context</f2m><f2n>/</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>/context/resource/%3C%3E%26%27%22</f3b><f3c>/context/resource/&lt;&gt;&amp;'\"</f3c><f4>/context/resource/test/uri/b</f4><f5>/context/resource/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='/context/resource/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='/context/resource/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='/context/resource/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f1g</td><td><a href='/context/resource/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/context/resource/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f1j</td><td><a href='/context'>context</a></td></tr><tr><td>f1k</td><td><a href='/context/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/context/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/context'>context</a></td></tr><tr><td>f1n</td><td><a href='/'>/</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='/context/resource/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='/context/resource/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f2g</td><td><a href='/context/resource/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/context/resource/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f2j</td><td><a href='/context'>context</a></td></tr><tr><td>f2k</td><td><a href='/context/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/context/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/context'>context</a></td></tr><tr><td>f2n</td><td><a href='/'>/</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='/context/resource/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='/context/resource/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='/context/resource/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='/context/resource/f5/x'>x</a></td></tr></table>",
					"(f0=/context/resource/f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=/context/resource/f1d,f1e=/context/resource/f1e/x/y,f1f=/context/resource,f1g=/context/resource/f1g/x,f1h=/context/resource/f1h,f1i=/context/resource,f1j=/context,f1k=/context/f1j/x,f1l=/context/f1k,f1m=/context,f1n=/,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=/context/resource/f2d,f2e=/context/resource/f2e/x/y,f2f=/context/resource,f2g=/context/resource/f2g/x,f2h=/context/resource/f2h,f2i=/context/resource,f2j=/context,f2k=/context/f2j/x,f2l=/context/f2k,f2m=/context,f2n=/,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=/context/resource/%3C%3E%26%27%22,f3c=/context/resource/<>&~'\",f4=/context/resource/test/uri/b,f5=/context/resource/f5/x)",
					"f0=/context/resource/f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=/context/resource/f1d&f1e=/context/resource/f1e/x/y&f1f=/context/resource&f1g=/context/resource/f1g/x&f1h=/context/resource/f1h&f1i=/context/resource&f1j=/context&f1k=/context/f1j/x&f1l=/context/f1k&f1m=/context&f1n=/&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=/context/resource/f2d&f2e=/context/resource/f2e/x/y&f2f=/context/resource&f2g=/context/resource/f2g/x&f2h=/context/resource/f2h&f2i=/context/resource&f2j=/context&f2k=/context/f2j/x&f2l=/context/f2k&f2m=/context&f2n=/&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=/context/resource/%253C%253E%2526%2527%2522&f3c=/context/resource/%3C%3E%26~'%22&f4=/context/resource/test/uri/b&f5=/context/resource/f5/x",
					"DE0022A26630B72F636F6E746578742F7265736F757263652F66302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164B52F636F6E746578742F7265736F757263652F663164A3663165B92F636F6E746578742F7265736F757263652F6631652F782F79A3663166B12F636F6E746578742F7265736F75726365A3663167B72F636F6E746578742F7265736F757263652F6631672F78A3663168B52F636F6E746578742F7265736F757263652F663168A3663169B12F636F6E746578742F7265736F75726365A366316AA82F636F6E74657874A366316BAE2F636F6E746578742F66316A2F78A366316CAC2F636F6E746578742F66316BA366316DA82F636F6E74657874A366316EA12FA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264B52F636F6E746578742F7265736F757263652F663264A3663265B92F636F6E746578742F7265736F757263652F6632652F782F79A3663266B12F636F6E746578742F7265736F75726365A3663267B72F636F6E746578742F7265736F757263652F6632672F78A3663268B52F636F6E746578742F7265736F757263652F663268A3663269B12F636F6E746578742F7265736F75726365A366326AA82F636F6E74657874A366326BAE2F636F6E746578742F66326A2F78A366326CAC2F636F6E746578742F66326BA366326DA82F636F6E74657874A366326EA12FA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362D9212F636F6E746578742F7265736F757263652F253343253345253236253237253232A3663363B72F636F6E746578742F7265736F757263652F3C3E262722A26634BC2F636F6E746578742F7265736F757263652F746573742F7572692F62A26635B62F636F6E746578742F7265736F757263652F66352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='/context/resource/f0/x0'>\n<jp:f2h>/context/resource/f2h</jp:f2h>\n<jp:f2f>/context/resource</jp:f2f>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1h>/context/resource/f1h</jp:f1h>\n<jp:f5>/context/resource/f5/x</jp:f5>\n<jp:f1e>/context/resource/f1e/x/y</jp:f1e>\n<jp:f3c>/context/resource/&lt;&gt;&amp;''</jp:f3c>\n<jp:f2k>/context/f2j/x</jp:f2k>\n<jp:f1j>/context</jp:f1j>\n<jp:f2m>/context</jp:f2m>\n<jp:f2i>/context/resource</jp:f2i>\n<jp:f1d>/context/resource/f1d</jp:f1d>\n<jp:f4>/context/resource/test/uri/b</jp:f4>\n<jp:f1n>/</jp:f1n>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f1k>/context/f1j/x</jp:f1k>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f2e>/context/resource/f2e/x/y</jp:f2e>\n<jp:f3b>/context/resource/%3C%3E%26%27%22</jp:f3b>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f1i>/context/resource</jp:f1i>\n<jp:f1g>/context/resource/f1g/x</jp:f1g>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f1m>/context</jp:f1m>\n<jp:f2d>/context/resource/f2d</jp:f2d>\n<jp:f2g>/context/resource/f2g/x</jp:f2g>\n<jp:f2n>/</jp:f2n>\n<jp:f2j>/context</jp:f2j>\n<jp:f1l>/context/f1k</jp:f1l>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f1f>/context/resource</jp:f1f>\n<jp:f2l>/context/f2k</jp:f2l>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 2 */
				"Happy-1-NONE",
				input(
					UriResolution.NONE, UriRelativity.RESOURCE, "http://foo.com:123","/context","/resource","/path"
				),
				results(
					"{f0:'f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'f1d',f1e:'f1e/x/y',f1f:'',f1g:'/context/resource/f1g/x',f1h:'/context/resource/f1h',f1i:'/context/resource',f1j:'/context',f1k:'/context/f1j/x',f1l:'/context/f1k',f1m:'/context',f1n:'/',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'f2d',f2e:'f2e/x/y',f2f:'',f2g:'/context/resource/f2g/x',f2h:'/context/resource/f2h',f2i:'/context/resource',f2j:'/context',f2k:'/context/f2j/x',f2l:'/context/f2k',f2m:'/context',f2n:'/',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'%3C%3E%26%27%22',f3c:'<>&\\'\"',f4:'test/uri/b',f5:'f5/x'}",
					"<object f0='f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>f1d</f1d><f1e>f1e/x/y</f1e><f1f>_xE000_</f1f><f1g>/context/resource/f1g/x</f1g><f1h>/context/resource/f1h</f1h><f1i>/context/resource</f1i><f1j>/context</f1j><f1k>/context/f1j/x</f1k><f1l>/context/f1k</f1l><f1m>/context</f1m><f1n>/</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>f2d</f2d><f2e>f2e/x/y</f2e><f2f>_xE000_</f2f><f2g>/context/resource/f2g/x</f2g><f2h>/context/resource/f2h</f2h><f2i>/context/resource</f2i><f2j>/context</f2j><f2k>/context/f2j/x</f2k><f2l>/context/f2k</f2l><f2m>/context</f2m><f2n>/</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>%3C%3E%26%27%22</f3b><f3c>&lt;&gt;&amp;'\"</f3c><f4>test/uri/b</f4><f5>f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href=''>/</a></td></tr><tr><td>f1g</td><td><a href='/context/resource/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/context/resource/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f1j</td><td><a href='/context'>context</a></td></tr><tr><td>f1k</td><td><a href='/context/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/context/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/context'>context</a></td></tr><tr><td>f1n</td><td><a href='/'>/</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href=''>/</a></td></tr><tr><td>f2g</td><td><a href='/context/resource/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/context/resource/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f2j</td><td><a href='/context'>context</a></td></tr><tr><td>f2k</td><td><a href='/context/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/context/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/context'>context</a></td></tr><tr><td>f2n</td><td><a href='/'>/</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='f5/x'>x</a></td></tr></table>",
					"(f0=f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=f1d,f1e=f1e/x/y,f1f='',f1g=/context/resource/f1g/x,f1h=/context/resource/f1h,f1i=/context/resource,f1j=/context,f1k=/context/f1j/x,f1l=/context/f1k,f1m=/context,f1n=/,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=f2d,f2e=f2e/x/y,f2f='',f2g=/context/resource/f2g/x,f2h=/context/resource/f2h,f2i=/context/resource,f2j=/context,f2k=/context/f2j/x,f2l=/context/f2k,f2m=/context,f2n=/,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=%3C%3E%26%27%22,f3c=<>&~'\",f4=test/uri/b,f5=f5/x)",
					"f0=f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=f1d&f1e=f1e/x/y&f1f=''&f1g=/context/resource/f1g/x&f1h=/context/resource/f1h&f1i=/context/resource&f1j=/context&f1k=/context/f1j/x&f1l=/context/f1k&f1m=/context&f1n=/&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=f2d&f2e=f2e/x/y&f2f=''&f2g=/context/resource/f2g/x&f2h=/context/resource/f2h&f2i=/context/resource&f2j=/context&f2k=/context/f2j/x&f2l=/context/f2k&f2m=/context&f2n=/&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=%253C%253E%2526%2527%2522&f3c=%3C%3E%26~'%22&f4=test/uri/b&f5=f5/x",
					"DE0022A26630A566302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164A3663164A3663165A76631652F782F79A3663166A0A3663167B72F636F6E746578742F7265736F757263652F6631672F78A3663168B52F636F6E746578742F7265736F757263652F663168A3663169B12F636F6E746578742F7265736F75726365A366316AA82F636F6E74657874A366316BAE2F636F6E746578742F66316A2F78A366316CAC2F636F6E746578742F66316BA366316DA82F636F6E74657874A366316EA12FA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264A3663264A3663265A76632652F782F79A3663266A0A3663267B72F636F6E746578742F7265736F757263652F6632672F78A3663268B52F636F6E746578742F7265736F757263652F663268A3663269B12F636F6E746578742F7265736F75726365A366326AA82F636F6E74657874A366326BAE2F636F6E746578742F66326A2F78A366326CAC2F636F6E746578742F66326BA366326DA82F636F6E74657874A366326EA12FA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362AF253343253345253236253237253232A3663363A53C3E262722A26634AA746573742F7572692F62A26635A466352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='f0/x0'>\n<jp:f2l>/context/f2k</jp:f2l>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f2g>/context/resource/f2g/x</jp:f2g>\n<jp:f2f></jp:f2f>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f3b>%3C%3E%26%27%22</jp:f3b>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f5>f5/x</jp:f5>\n<jp:f2k>/context/f2j/x</jp:f2k>\n<jp:f1e>f1e/x/y</jp:f1e>\n<jp:f2i>/context/resource</jp:f2i>\n<jp:f2n>/</jp:f2n>\n<jp:f2m>/context</jp:f2m>\n<jp:f1m>/context</jp:f1m>\n<jp:f2d>f2d</jp:f2d>\n<jp:f1l>/context/f1k</jp:f1l>\n<jp:f3c>&lt;&gt;&amp;''</jp:f3c>\n<jp:f1n>/</jp:f1n>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f1k>/context/f1j/x</jp:f1k>\n<jp:f1h>/context/resource/f1h</jp:f1h>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f2h>/context/resource/f2h</jp:f2h>\n<jp:f1d>f1d</jp:f1d>\n<jp:f1i>/context/resource</jp:f1i>\n<jp:f2e>f2e/x/y</jp:f2e>\n<jp:f2j>/context</jp:f2j>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1j>/context</jp:f1j>\n<jp:f1f></jp:f1f>\n<jp:f1g>/context/resource/f1g/x</jp:f1g>\n<jp:f4>test/uri/b</jp:f4>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 3 */
				"Happy-2-ABSOLUTE",
				input(
					UriResolution.ABSOLUTE, UriRelativity.RESOURCE, "http://foo.com:123","/c1/c2","/r1/r2","/p1/p2"
				),
				results(
					"{f0:'http://foo.com:123/c1/c2/r1/r2/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'http://foo.com:123/f1b',f1c:'http://foo.com:123/f1c/x/y',f1d:'http://foo.com:123/c1/c2/r1/r2/f1d',f1e:'http://foo.com:123/c1/c2/r1/r2/f1e/x/y',f1f:'http://foo.com:123/c1/c2/r1/r2',f1g:'http://foo.com:123/c1/c2/r1/r2/f1g/x',f1h:'http://foo.com:123/c1/c2/r1/r2/f1h',f1i:'http://foo.com:123/c1/c2/r1/r2',f1j:'http://foo.com:123/c1/c2/r1',f1k:'http://foo.com:123/c1/c2/f1j/x',f1l:'http://foo.com:123/c1/c2/f1k',f1m:'http://foo.com:123/c1/c2',f1n:'http://foo.com:123/c1',f2a:'http://www.apache.org/f2a',f2b:'http://foo.com:123/f2b',f2c:'http://foo.com:123/f2c/x/y',f2d:'http://foo.com:123/c1/c2/r1/r2/f2d',f2e:'http://foo.com:123/c1/c2/r1/r2/f2e/x/y',f2f:'http://foo.com:123/c1/c2/r1/r2',f2g:'http://foo.com:123/c1/c2/r1/r2/f2g/x',f2h:'http://foo.com:123/c1/c2/r1/r2/f2h',f2i:'http://foo.com:123/c1/c2/r1/r2',f2j:'http://foo.com:123/c1/c2/r1',f2k:'http://foo.com:123/c1/c2/f2j/x',f2l:'http://foo.com:123/c1/c2/f2k',f2m:'http://foo.com:123/c1/c2',f2n:'http://foo.com:123/c1',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'http://foo.com:123/c1/c2/r1/r2/%3C%3E%26%27%22',f3c:'http://foo.com:123/c1/c2/r1/r2/<>&\\'\"',f4:'http://foo.com:123/c1/c2/r1/r2/test/uri/b',f5:'http://foo.com:123/c1/c2/r1/r2/f5/x'}",
					"<object f0='http://foo.com:123/c1/c2/r1/r2/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>http://foo.com:123/f1b</f1b><f1c>http://foo.com:123/f1c/x/y</f1c><f1d>http://foo.com:123/c1/c2/r1/r2/f1d</f1d><f1e>http://foo.com:123/c1/c2/r1/r2/f1e/x/y</f1e><f1f>http://foo.com:123/c1/c2/r1/r2</f1f><f1g>http://foo.com:123/c1/c2/r1/r2/f1g/x</f1g><f1h>http://foo.com:123/c1/c2/r1/r2/f1h</f1h><f1i>http://foo.com:123/c1/c2/r1/r2</f1i><f1j>http://foo.com:123/c1/c2/r1</f1j><f1k>http://foo.com:123/c1/c2/f1j/x</f1k><f1l>http://foo.com:123/c1/c2/f1k</f1l><f1m>http://foo.com:123/c1/c2</f1m><f1n>http://foo.com:123/c1</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>http://foo.com:123/f2b</f2b><f2c>http://foo.com:123/f2c/x/y</f2c><f2d>http://foo.com:123/c1/c2/r1/r2/f2d</f2d><f2e>http://foo.com:123/c1/c2/r1/r2/f2e/x/y</f2e><f2f>http://foo.com:123/c1/c2/r1/r2</f2f><f2g>http://foo.com:123/c1/c2/r1/r2/f2g/x</f2g><f2h>http://foo.com:123/c1/c2/r1/r2/f2h</f2h><f2i>http://foo.com:123/c1/c2/r1/r2</f2i><f2j>http://foo.com:123/c1/c2/r1</f2j><f2k>http://foo.com:123/c1/c2/f2j/x</f2k><f2l>http://foo.com:123/c1/c2/f2k</f2l><f2m>http://foo.com:123/c1/c2</f2m><f2n>http://foo.com:123/c1</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>http://foo.com:123/c1/c2/r1/r2/%3C%3E%26%27%22</f3b><f3c>http://foo.com:123/c1/c2/r1/r2/&lt;&gt;&amp;'\"</f3c><f4>http://foo.com:123/c1/c2/r1/r2/test/uri/b</f4><f5>http://foo.com:123/c1/c2/r1/r2/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='http://foo.com:123/c1/c2/r1/r2/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='http://foo.com:123/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='http://foo.com:123/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='http://foo.com:123/c1/c2/r1/r2/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='http://foo.com:123/c1/c2/r1/r2/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='http://foo.com:123/c1/c2/r1/r2'>r2</a></td></tr><tr><td>f1g</td><td><a href='http://foo.com:123/c1/c2/r1/r2/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='http://foo.com:123/c1/c2/r1/r2/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='http://foo.com:123/c1/c2/r1/r2'>r2</a></td></tr><tr><td>f1j</td><td><a href='http://foo.com:123/c1/c2/r1'>r1</a></td></tr><tr><td>f1k</td><td><a href='http://foo.com:123/c1/c2/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='http://foo.com:123/c1/c2/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='http://foo.com:123/c1/c2'>c2</a></td></tr><tr><td>f1n</td><td><a href='http://foo.com:123/c1'>c1</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='http://foo.com:123/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='http://foo.com:123/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='http://foo.com:123/c1/c2/r1/r2/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='http://foo.com:123/c1/c2/r1/r2/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='http://foo.com:123/c1/c2/r1/r2'>r2</a></td></tr><tr><td>f2g</td><td><a href='http://foo.com:123/c1/c2/r1/r2/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='http://foo.com:123/c1/c2/r1/r2/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='http://foo.com:123/c1/c2/r1/r2'>r2</a></td></tr><tr><td>f2j</td><td><a href='http://foo.com:123/c1/c2/r1'>r1</a></td></tr><tr><td>f2k</td><td><a href='http://foo.com:123/c1/c2/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='http://foo.com:123/c1/c2/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='http://foo.com:123/c1/c2'>c2</a></td></tr><tr><td>f2n</td><td><a href='http://foo.com:123/c1'>c1</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='http://foo.com:123/c1/c2/r1/r2/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='http://foo.com:123/c1/c2/r1/r2/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='http://foo.com:123/c1/c2/r1/r2/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='http://foo.com:123/c1/c2/r1/r2/f5/x'>x</a></td></tr></table>",
					"(f0=http://foo.com:123/c1/c2/r1/r2/f0/x0,f1a=http://www.apache.org/f1a,f1b=http://foo.com:123/f1b,f1c=http://foo.com:123/f1c/x/y,f1d=http://foo.com:123/c1/c2/r1/r2/f1d,f1e=http://foo.com:123/c1/c2/r1/r2/f1e/x/y,f1f=http://foo.com:123/c1/c2/r1/r2,f1g=http://foo.com:123/c1/c2/r1/r2/f1g/x,f1h=http://foo.com:123/c1/c2/r1/r2/f1h,f1i=http://foo.com:123/c1/c2/r1/r2,f1j=http://foo.com:123/c1/c2/r1,f1k=http://foo.com:123/c1/c2/f1j/x,f1l=http://foo.com:123/c1/c2/f1k,f1m=http://foo.com:123/c1/c2,f1n=http://foo.com:123/c1,f2a=http://www.apache.org/f2a,f2b=http://foo.com:123/f2b,f2c=http://foo.com:123/f2c/x/y,f2d=http://foo.com:123/c1/c2/r1/r2/f2d,f2e=http://foo.com:123/c1/c2/r1/r2/f2e/x/y,f2f=http://foo.com:123/c1/c2/r1/r2,f2g=http://foo.com:123/c1/c2/r1/r2/f2g/x,f2h=http://foo.com:123/c1/c2/r1/r2/f2h,f2i=http://foo.com:123/c1/c2/r1/r2,f2j=http://foo.com:123/c1/c2/r1,f2k=http://foo.com:123/c1/c2/f2j/x,f2l=http://foo.com:123/c1/c2/f2k,f2m=http://foo.com:123/c1/c2,f2n=http://foo.com:123/c1,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=http://foo.com:123/c1/c2/r1/r2/%3C%3E%26%27%22,f3c=http://foo.com:123/c1/c2/r1/r2/<>&~'\",f4=http://foo.com:123/c1/c2/r1/r2/test/uri/b,f5=http://foo.com:123/c1/c2/r1/r2/f5/x)",
					"f0=http://foo.com:123/c1/c2/r1/r2/f0/x0&f1a=http://www.apache.org/f1a&f1b=http://foo.com:123/f1b&f1c=http://foo.com:123/f1c/x/y&f1d=http://foo.com:123/c1/c2/r1/r2/f1d&f1e=http://foo.com:123/c1/c2/r1/r2/f1e/x/y&f1f=http://foo.com:123/c1/c2/r1/r2&f1g=http://foo.com:123/c1/c2/r1/r2/f1g/x&f1h=http://foo.com:123/c1/c2/r1/r2/f1h&f1i=http://foo.com:123/c1/c2/r1/r2&f1j=http://foo.com:123/c1/c2/r1&f1k=http://foo.com:123/c1/c2/f1j/x&f1l=http://foo.com:123/c1/c2/f1k&f1m=http://foo.com:123/c1/c2&f1n=http://foo.com:123/c1&f2a=http://www.apache.org/f2a&f2b=http://foo.com:123/f2b&f2c=http://foo.com:123/f2c/x/y&f2d=http://foo.com:123/c1/c2/r1/r2/f2d&f2e=http://foo.com:123/c1/c2/r1/r2/f2e/x/y&f2f=http://foo.com:123/c1/c2/r1/r2&f2g=http://foo.com:123/c1/c2/r1/r2/f2g/x&f2h=http://foo.com:123/c1/c2/r1/r2/f2h&f2i=http://foo.com:123/c1/c2/r1/r2&f2j=http://foo.com:123/c1/c2/r1&f2k=http://foo.com:123/c1/c2/f2j/x&f2l=http://foo.com:123/c1/c2/f2k&f2m=http://foo.com:123/c1/c2&f2n=http://foo.com:123/c1&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=http://foo.com:123/c1/c2/r1/r2/%253C%253E%2526%2527%2522&f3c=http://foo.com:123/c1/c2/r1/r2/%3C%3E%26~'%22&f4=http://foo.com:123/c1/c2/r1/r2/test/uri/b&f5=http://foo.com:123/c1/c2/r1/r2/f5/x",
					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
					"<rdf:RDF>\n<rdf:Description rdf:about='http://foo.com:123/c1/c2/r1/r2/f0/x0'>\n<jp:f2m rdf:resource='http://foo.com:123/c1/c2'/>\n<jp:f1n rdf:resource='http://foo.com:123/c1'/>\n<jp:f2e rdf:resource='http://foo.com:123/c1/c2/r1/r2/f2e/x/y'/>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f1j rdf:resource='http://foo.com:123/c1/c2/r1'/>\n<jp:f1c rdf:resource='http://foo.com:123/f1c/x/y'/>\n<jp:f1f rdf:resource='http://foo.com:123/c1/c2/r1/r2'/>\n<jp:f1l rdf:resource='http://foo.com:123/c1/c2/f1k'/>\n<jp:f1k rdf:resource='http://foo.com:123/c1/c2/f1j/x'/>\n<jp:f3b rdf:resource='http://foo.com:123/c1/c2/r1/r2/%3C%3E%26%27%22'/>\n<jp:f1e rdf:resource='http://foo.com:123/c1/c2/r1/r2/f1e/x/y'/>\n<jp:f2n rdf:resource='http://foo.com:123/c1'/>\n<jp:f2k rdf:resource='http://foo.com:123/c1/c2/f2j/x'/>\n<jp:f2c rdf:resource='http://foo.com:123/f2c/x/y'/>\n<jp:f1i rdf:resource='http://foo.com:123/c1/c2/r1/r2'/>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f1b rdf:resource='http://foo.com:123/f1b'/>\n<jp:f1g rdf:resource='http://foo.com:123/c1/c2/r1/r2/f1g/x'/>\n<jp:f2i rdf:resource='http://foo.com:123/c1/c2/r1/r2'/>\n<jp:f1h rdf:resource='http://foo.com:123/c1/c2/r1/r2/f1h'/>\n<jp:f2f rdf:resource='http://foo.com:123/c1/c2/r1/r2'/>\n<jp:f1d rdf:resource='http://foo.com:123/c1/c2/r1/r2/f1d'/>\n<jp:f2d rdf:resource='http://foo.com:123/c1/c2/r1/r2/f2d'/>\n<jp:f4 rdf:resource='http://foo.com:123/c1/c2/r1/r2/test/uri/b'/>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f5 rdf:resource='http://foo.com:123/c1/c2/r1/r2/f5/x'/>\n<jp:f2h rdf:resource='http://foo.com:123/c1/c2/r1/r2/f2h'/>\n<jp:f2l rdf:resource='http://foo.com:123/c1/c2/f2k'/>\n<jp:f3c rdf:resource='http://foo.com:123/c1/c2/r1/r2/&lt;&gt;&amp;&apos;&quot;'/>\n<jp:f2b rdf:resource='http://foo.com:123/f2b'/>\n<jp:f2g rdf:resource='http://foo.com:123/c1/c2/r1/r2/f2g/x'/>\n<jp:f1m rdf:resource='http://foo.com:123/c1/c2'/>\n<jp:f2j rdf:resource='http://foo.com:123/c1/c2/r1'/>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 4 */
				"Happy-2-ROOT_RELATIVE",
				input(
					UriResolution.ROOT_RELATIVE, UriRelativity.RESOURCE, "http://foo.com:123","/c1/c2","/r1/r2","/p1/p2"
				),
				results(
					"{f0:'/c1/c2/r1/r2/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'/c1/c2/r1/r2/f1d',f1e:'/c1/c2/r1/r2/f1e/x/y',f1f:'/c1/c2/r1/r2',f1g:'/c1/c2/r1/r2/f1g/x',f1h:'/c1/c2/r1/r2/f1h',f1i:'/c1/c2/r1/r2',f1j:'/c1/c2/r1',f1k:'/c1/c2/f1j/x',f1l:'/c1/c2/f1k',f1m:'/c1/c2',f1n:'/c1',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'/c1/c2/r1/r2/f2d',f2e:'/c1/c2/r1/r2/f2e/x/y',f2f:'/c1/c2/r1/r2',f2g:'/c1/c2/r1/r2/f2g/x',f2h:'/c1/c2/r1/r2/f2h',f2i:'/c1/c2/r1/r2',f2j:'/c1/c2/r1',f2k:'/c1/c2/f2j/x',f2l:'/c1/c2/f2k',f2m:'/c1/c2',f2n:'/c1',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'/c1/c2/r1/r2/%3C%3E%26%27%22',f3c:'/c1/c2/r1/r2/<>&\\'\"',f4:'/c1/c2/r1/r2/test/uri/b',f5:'/c1/c2/r1/r2/f5/x'}",
					"<object f0='/c1/c2/r1/r2/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>/c1/c2/r1/r2/f1d</f1d><f1e>/c1/c2/r1/r2/f1e/x/y</f1e><f1f>/c1/c2/r1/r2</f1f><f1g>/c1/c2/r1/r2/f1g/x</f1g><f1h>/c1/c2/r1/r2/f1h</f1h><f1i>/c1/c2/r1/r2</f1i><f1j>/c1/c2/r1</f1j><f1k>/c1/c2/f1j/x</f1k><f1l>/c1/c2/f1k</f1l><f1m>/c1/c2</f1m><f1n>/c1</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>/c1/c2/r1/r2/f2d</f2d><f2e>/c1/c2/r1/r2/f2e/x/y</f2e><f2f>/c1/c2/r1/r2</f2f><f2g>/c1/c2/r1/r2/f2g/x</f2g><f2h>/c1/c2/r1/r2/f2h</f2h><f2i>/c1/c2/r1/r2</f2i><f2j>/c1/c2/r1</f2j><f2k>/c1/c2/f2j/x</f2k><f2l>/c1/c2/f2k</f2l><f2m>/c1/c2</f2m><f2n>/c1</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>/c1/c2/r1/r2/%3C%3E%26%27%22</f3b><f3c>/c1/c2/r1/r2/&lt;&gt;&amp;'\"</f3c><f4>/c1/c2/r1/r2/test/uri/b</f4><f5>/c1/c2/r1/r2/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='/c1/c2/r1/r2/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='/c1/c2/r1/r2/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='/c1/c2/r1/r2/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='/c1/c2/r1/r2'>r2</a></td></tr><tr><td>f1g</td><td><a href='/c1/c2/r1/r2/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/c1/c2/r1/r2/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/c1/c2/r1/r2'>r2</a></td></tr><tr><td>f1j</td><td><a href='/c1/c2/r1'>r1</a></td></tr><tr><td>f1k</td><td><a href='/c1/c2/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/c1/c2/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/c1/c2'>c2</a></td></tr><tr><td>f1n</td><td><a href='/c1'>c1</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='/c1/c2/r1/r2/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='/c1/c2/r1/r2/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='/c1/c2/r1/r2'>r2</a></td></tr><tr><td>f2g</td><td><a href='/c1/c2/r1/r2/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/c1/c2/r1/r2/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/c1/c2/r1/r2'>r2</a></td></tr><tr><td>f2j</td><td><a href='/c1/c2/r1'>r1</a></td></tr><tr><td>f2k</td><td><a href='/c1/c2/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/c1/c2/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/c1/c2'>c2</a></td></tr><tr><td>f2n</td><td><a href='/c1'>c1</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='/c1/c2/r1/r2/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='/c1/c2/r1/r2/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='/c1/c2/r1/r2/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='/c1/c2/r1/r2/f5/x'>x</a></td></tr></table>",
					"(f0=/c1/c2/r1/r2/f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=/c1/c2/r1/r2/f1d,f1e=/c1/c2/r1/r2/f1e/x/y,f1f=/c1/c2/r1/r2,f1g=/c1/c2/r1/r2/f1g/x,f1h=/c1/c2/r1/r2/f1h,f1i=/c1/c2/r1/r2,f1j=/c1/c2/r1,f1k=/c1/c2/f1j/x,f1l=/c1/c2/f1k,f1m=/c1/c2,f1n=/c1,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=/c1/c2/r1/r2/f2d,f2e=/c1/c2/r1/r2/f2e/x/y,f2f=/c1/c2/r1/r2,f2g=/c1/c2/r1/r2/f2g/x,f2h=/c1/c2/r1/r2/f2h,f2i=/c1/c2/r1/r2,f2j=/c1/c2/r1,f2k=/c1/c2/f2j/x,f2l=/c1/c2/f2k,f2m=/c1/c2,f2n=/c1,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=/c1/c2/r1/r2/%3C%3E%26%27%22,f3c=/c1/c2/r1/r2/<>&~'\",f4=/c1/c2/r1/r2/test/uri/b,f5=/c1/c2/r1/r2/f5/x)",
					"f0=/c1/c2/r1/r2/f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=/c1/c2/r1/r2/f1d&f1e=/c1/c2/r1/r2/f1e/x/y&f1f=/c1/c2/r1/r2&f1g=/c1/c2/r1/r2/f1g/x&f1h=/c1/c2/r1/r2/f1h&f1i=/c1/c2/r1/r2&f1j=/c1/c2/r1&f1k=/c1/c2/f1j/x&f1l=/c1/c2/f1k&f1m=/c1/c2&f1n=/c1&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=/c1/c2/r1/r2/f2d&f2e=/c1/c2/r1/r2/f2e/x/y&f2f=/c1/c2/r1/r2&f2g=/c1/c2/r1/r2/f2g/x&f2h=/c1/c2/r1/r2/f2h&f2i=/c1/c2/r1/r2&f2j=/c1/c2/r1&f2k=/c1/c2/f2j/x&f2l=/c1/c2/f2k&f2m=/c1/c2&f2n=/c1&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=/c1/c2/r1/r2/%253C%253E%2526%2527%2522&f3c=/c1/c2/r1/r2/%3C%3E%26~'%22&f4=/c1/c2/r1/r2/test/uri/b&f5=/c1/c2/r1/r2/f5/x",
					"DE0022A26630B22F63312F63322F72312F72322F66302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164B02F63312F63322F72312F72322F663164A3663165B42F63312F63322F72312F72322F6631652F782F79A3663166AC2F63312F63322F72312F7232A3663167B22F63312F63322F72312F72322F6631672F78A3663168B02F63312F63322F72312F72322F663168A3663169AC2F63312F63322F72312F7232A366316AA92F63312F63322F7231A366316BAC2F63312F63322F66316A2F78A366316CAA2F63312F63322F66316BA366316DA62F63312F6332A366316EA32F6331A3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264B02F63312F63322F72312F72322F663264A3663265B42F63312F63322F72312F72322F6632652F782F79A3663266AC2F63312F63322F72312F7232A3663267B22F63312F63322F72312F72322F6632672F78A3663268B02F63312F63322F72312F72322F663268A3663269AC2F63312F63322F72312F7232A366326AA92F63312F63322F7231A366326BAC2F63312F63322F66326A2F78A366326CAA2F63312F63322F66326BA366326DA62F63312F6332A366326EA32F6331A3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362BC2F63312F63322F72312F72322F253343253345253236253237253232A3663363B22F63312F63322F72312F72322F3C3E262722A26634B72F63312F63322F72312F72322F746573742F7572692F62A26635B12F63312F63322F72312F72322F66352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='/c1/c2/r1/r2/f0/x0'>\n<jp:f2j>/c1/c2/r1</jp:f2j>\n<jp:f2l>/c1/c2/f2k</jp:f2l>\n<jp:f2m>/c1/c2</jp:f2m>\n<jp:f1i>/c1/c2/r1/r2</jp:f1i>\n<jp:f2h>/c1/c2/r1/r2/f2h</jp:f2h>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f1e>/c1/c2/r1/r2/f1e/x/y</jp:f1e>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f2d>/c1/c2/r1/r2/f2d</jp:f2d>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f1l>/c1/c2/f1k</jp:f1l>\n<jp:f3c>/c1/c2/r1/r2/&lt;&gt;&amp;''</jp:f3c>\n<jp:f1d>/c1/c2/r1/r2/f1d</jp:f1d>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f1f>/c1/c2/r1/r2</jp:f1f>\n<jp:f2g>/c1/c2/r1/r2/f2g/x</jp:f2g>\n<jp:f5>/c1/c2/r1/r2/f5/x</jp:f5>\n<jp:f3b>/c1/c2/r1/r2/%3C%3E%26%27%22</jp:f3b>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f2i>/c1/c2/r1/r2</jp:f2i>\n<jp:f1n>/c1</jp:f1n>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f1h>/c1/c2/r1/r2/f1h</jp:f1h>\n<jp:f4>/c1/c2/r1/r2/test/uri/b</jp:f4>\n<jp:f1j>/c1/c2/r1</jp:f1j>\n<jp:f2f>/c1/c2/r1/r2</jp:f2f>\n<jp:f2n>/c1</jp:f2n>\n<jp:f2k>/c1/c2/f2j/x</jp:f2k>\n<jp:f1m>/c1/c2</jp:f1m>\n<jp:f2e>/c1/c2/r1/r2/f2e/x/y</jp:f2e>\n<jp:f1g>/c1/c2/r1/r2/f1g/x</jp:f1g>\n<jp:f1k>/c1/c2/f1j/x</jp:f1k>\n<jp:f2b>/f2b</jp:f2b>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 5 */
				"Happy-2-NONE",
				input(
					UriResolution.NONE, UriRelativity.RESOURCE, "http://foo.com:123","/c1/c2","/r1/r2","/p1/p2"
				),
				results(
					"{f0:'f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'f1d',f1e:'f1e/x/y',f1f:'',f1g:'/c1/c2/r1/r2/f1g/x',f1h:'/c1/c2/r1/r2/f1h',f1i:'/c1/c2/r1/r2',f1j:'/c1/c2/r1',f1k:'/c1/c2/f1j/x',f1l:'/c1/c2/f1k',f1m:'/c1/c2',f1n:'/c1',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'f2d',f2e:'f2e/x/y',f2f:'',f2g:'/c1/c2/r1/r2/f2g/x',f2h:'/c1/c2/r1/r2/f2h',f2i:'/c1/c2/r1/r2',f2j:'/c1/c2/r1',f2k:'/c1/c2/f2j/x',f2l:'/c1/c2/f2k',f2m:'/c1/c2',f2n:'/c1',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'%3C%3E%26%27%22',f3c:'<>&\\'\"',f4:'test/uri/b',f5:'f5/x'}",
					"<object f0='f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>f1d</f1d><f1e>f1e/x/y</f1e><f1f>_xE000_</f1f><f1g>/c1/c2/r1/r2/f1g/x</f1g><f1h>/c1/c2/r1/r2/f1h</f1h><f1i>/c1/c2/r1/r2</f1i><f1j>/c1/c2/r1</f1j><f1k>/c1/c2/f1j/x</f1k><f1l>/c1/c2/f1k</f1l><f1m>/c1/c2</f1m><f1n>/c1</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>f2d</f2d><f2e>f2e/x/y</f2e><f2f>_xE000_</f2f><f2g>/c1/c2/r1/r2/f2g/x</f2g><f2h>/c1/c2/r1/r2/f2h</f2h><f2i>/c1/c2/r1/r2</f2i><f2j>/c1/c2/r1</f2j><f2k>/c1/c2/f2j/x</f2k><f2l>/c1/c2/f2k</f2l><f2m>/c1/c2</f2m><f2n>/c1</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>%3C%3E%26%27%22</f3b><f3c>&lt;&gt;&amp;'\"</f3c><f4>test/uri/b</f4><f5>f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href=''>/</a></td></tr><tr><td>f1g</td><td><a href='/c1/c2/r1/r2/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/c1/c2/r1/r2/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/c1/c2/r1/r2'>r2</a></td></tr><tr><td>f1j</td><td><a href='/c1/c2/r1'>r1</a></td></tr><tr><td>f1k</td><td><a href='/c1/c2/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/c1/c2/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/c1/c2'>c2</a></td></tr><tr><td>f1n</td><td><a href='/c1'>c1</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href=''>/</a></td></tr><tr><td>f2g</td><td><a href='/c1/c2/r1/r2/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/c1/c2/r1/r2/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/c1/c2/r1/r2'>r2</a></td></tr><tr><td>f2j</td><td><a href='/c1/c2/r1'>r1</a></td></tr><tr><td>f2k</td><td><a href='/c1/c2/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/c1/c2/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/c1/c2'>c2</a></td></tr><tr><td>f2n</td><td><a href='/c1'>c1</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='f5/x'>x</a></td></tr></table>",
					"(f0=f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=f1d,f1e=f1e/x/y,f1f='',f1g=/c1/c2/r1/r2/f1g/x,f1h=/c1/c2/r1/r2/f1h,f1i=/c1/c2/r1/r2,f1j=/c1/c2/r1,f1k=/c1/c2/f1j/x,f1l=/c1/c2/f1k,f1m=/c1/c2,f1n=/c1,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=f2d,f2e=f2e/x/y,f2f='',f2g=/c1/c2/r1/r2/f2g/x,f2h=/c1/c2/r1/r2/f2h,f2i=/c1/c2/r1/r2,f2j=/c1/c2/r1,f2k=/c1/c2/f2j/x,f2l=/c1/c2/f2k,f2m=/c1/c2,f2n=/c1,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=%3C%3E%26%27%22,f3c=<>&~'\",f4=test/uri/b,f5=f5/x)",
					"f0=f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=f1d&f1e=f1e/x/y&f1f=''&f1g=/c1/c2/r1/r2/f1g/x&f1h=/c1/c2/r1/r2/f1h&f1i=/c1/c2/r1/r2&f1j=/c1/c2/r1&f1k=/c1/c2/f1j/x&f1l=/c1/c2/f1k&f1m=/c1/c2&f1n=/c1&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=f2d&f2e=f2e/x/y&f2f=''&f2g=/c1/c2/r1/r2/f2g/x&f2h=/c1/c2/r1/r2/f2h&f2i=/c1/c2/r1/r2&f2j=/c1/c2/r1&f2k=/c1/c2/f2j/x&f2l=/c1/c2/f2k&f2m=/c1/c2&f2n=/c1&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=%253C%253E%2526%2527%2522&f3c=%3C%3E%26~'%22&f4=test/uri/b&f5=f5/x",
					"DE0022A26630A566302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164A3663164A3663165A76631652F782F79A3663166A0A3663167B22F63312F63322F72312F72322F6631672F78A3663168B02F63312F63322F72312F72322F663168A3663169AC2F63312F63322F72312F7232A366316AA92F63312F63322F7231A366316BAC2F63312F63322F66316A2F78A366316CAA2F63312F63322F66316BA366316DA62F63312F6332A366316EA32F6331A3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264A3663264A3663265A76632652F782F79A3663266A0A3663267B22F63312F63322F72312F72322F6632672F78A3663268B02F63312F63322F72312F72322F663268A3663269AC2F63312F63322F72312F7232A366326AA92F63312F63322F7231A366326BAC2F63312F63322F66326A2F78A366326CAA2F63312F63322F66326BA366326DA62F63312F6332A366326EA32F6331A3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362AF253343253345253236253237253232A3663363A53C3E262722A26634AA746573742F7572692F62A26635A466352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='f0/x0'>\n<jp:f2n>/c1</jp:f2n>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f2l>/c1/c2/f2k</jp:f2l>\n<jp:f2f></jp:f2f>\n<jp:f1m>/c1/c2</jp:f1m>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f3b>%3C%3E%26%27%22</jp:f3b>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f2h>/c1/c2/r1/r2/f2h</jp:f2h>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f2i>/c1/c2/r1/r2</jp:f2i>\n<jp:f2g>/c1/c2/r1/r2/f2g/x</jp:f2g>\n<jp:f5>f5/x</jp:f5>\n<jp:f2k>/c1/c2/f2j/x</jp:f2k>\n<jp:f1i>/c1/c2/r1/r2</jp:f1i>\n<jp:f1e>f1e/x/y</jp:f1e>\n<jp:f1l>/c1/c2/f1k</jp:f1l>\n<jp:f2d>f2d</jp:f2d>\n<jp:f1k>/c1/c2/f1j/x</jp:f1k>\n<jp:f3c>&lt;&gt;&amp;''</jp:f3c>\n<jp:f2m>/c1/c2</jp:f2m>\n<jp:f2j>/c1/c2/r1</jp:f2j>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f1h>/c1/c2/r1/r2/f1h</jp:f1h>\n<jp:f1d>f1d</jp:f1d>\n<jp:f2e>f2e/x/y</jp:f2e>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1n>/c1</jp:f1n>\n<jp:f1f></jp:f1f>\n<jp:f1g>/c1/c2/r1/r2/f1g/x</jp:f1g>\n<jp:f4>test/uri/b</jp:f4>\n<jp:f1j>/c1/c2/r1</jp:f1j>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 6 */
				"NoAuthority-1-ABSOLUTE",
				input(
					UriResolution.ABSOLUTE, UriRelativity.RESOURCE, "","/context","/resource","/path"
				),
				results(
					"{f0:'/context/resource/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'/context/resource/f1d',f1e:'/context/resource/f1e/x/y',f1f:'/context/resource',f1g:'/context/resource/f1g/x',f1h:'/context/resource/f1h',f1i:'/context/resource',f1j:'/context',f1k:'/context/f1j/x',f1l:'/context/f1k',f1m:'/context',f1n:'/',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'/context/resource/f2d',f2e:'/context/resource/f2e/x/y',f2f:'/context/resource',f2g:'/context/resource/f2g/x',f2h:'/context/resource/f2h',f2i:'/context/resource',f2j:'/context',f2k:'/context/f2j/x',f2l:'/context/f2k',f2m:'/context',f2n:'/',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'/context/resource/%3C%3E%26%27%22',f3c:'/context/resource/<>&\\'\"',f4:'/context/resource/test/uri/b',f5:'/context/resource/f5/x'}",
					"<object f0='/context/resource/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>/context/resource/f1d</f1d><f1e>/context/resource/f1e/x/y</f1e><f1f>/context/resource</f1f><f1g>/context/resource/f1g/x</f1g><f1h>/context/resource/f1h</f1h><f1i>/context/resource</f1i><f1j>/context</f1j><f1k>/context/f1j/x</f1k><f1l>/context/f1k</f1l><f1m>/context</f1m><f1n>/</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>/context/resource/f2d</f2d><f2e>/context/resource/f2e/x/y</f2e><f2f>/context/resource</f2f><f2g>/context/resource/f2g/x</f2g><f2h>/context/resource/f2h</f2h><f2i>/context/resource</f2i><f2j>/context</f2j><f2k>/context/f2j/x</f2k><f2l>/context/f2k</f2l><f2m>/context</f2m><f2n>/</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>/context/resource/%3C%3E%26%27%22</f3b><f3c>/context/resource/&lt;&gt;&amp;'\"</f3c><f4>/context/resource/test/uri/b</f4><f5>/context/resource/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='/context/resource/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='/context/resource/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='/context/resource/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f1g</td><td><a href='/context/resource/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/context/resource/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f1j</td><td><a href='/context'>context</a></td></tr><tr><td>f1k</td><td><a href='/context/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/context/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/context'>context</a></td></tr><tr><td>f1n</td><td><a href='/'>/</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='/context/resource/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='/context/resource/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f2g</td><td><a href='/context/resource/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/context/resource/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f2j</td><td><a href='/context'>context</a></td></tr><tr><td>f2k</td><td><a href='/context/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/context/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/context'>context</a></td></tr><tr><td>f2n</td><td><a href='/'>/</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='/context/resource/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='/context/resource/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='/context/resource/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='/context/resource/f5/x'>x</a></td></tr></table>",
					"(f0=/context/resource/f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=/context/resource/f1d,f1e=/context/resource/f1e/x/y,f1f=/context/resource,f1g=/context/resource/f1g/x,f1h=/context/resource/f1h,f1i=/context/resource,f1j=/context,f1k=/context/f1j/x,f1l=/context/f1k,f1m=/context,f1n=/,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=/context/resource/f2d,f2e=/context/resource/f2e/x/y,f2f=/context/resource,f2g=/context/resource/f2g/x,f2h=/context/resource/f2h,f2i=/context/resource,f2j=/context,f2k=/context/f2j/x,f2l=/context/f2k,f2m=/context,f2n=/,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=/context/resource/%3C%3E%26%27%22,f3c=/context/resource/<>&~'\",f4=/context/resource/test/uri/b,f5=/context/resource/f5/x)",
					"f0=/context/resource/f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=/context/resource/f1d&f1e=/context/resource/f1e/x/y&f1f=/context/resource&f1g=/context/resource/f1g/x&f1h=/context/resource/f1h&f1i=/context/resource&f1j=/context&f1k=/context/f1j/x&f1l=/context/f1k&f1m=/context&f1n=/&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=/context/resource/f2d&f2e=/context/resource/f2e/x/y&f2f=/context/resource&f2g=/context/resource/f2g/x&f2h=/context/resource/f2h&f2i=/context/resource&f2j=/context&f2k=/context/f2j/x&f2l=/context/f2k&f2m=/context&f2n=/&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=/context/resource/%253C%253E%2526%2527%2522&f3c=/context/resource/%3C%3E%26~'%22&f4=/context/resource/test/uri/b&f5=/context/resource/f5/x",
					"DE0022A26630B72F636F6E746578742F7265736F757263652F66302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164B52F636F6E746578742F7265736F757263652F663164A3663165B92F636F6E746578742F7265736F757263652F6631652F782F79A3663166B12F636F6E746578742F7265736F75726365A3663167B72F636F6E746578742F7265736F757263652F6631672F78A3663168B52F636F6E746578742F7265736F757263652F663168A3663169B12F636F6E746578742F7265736F75726365A366316AA82F636F6E74657874A366316BAE2F636F6E746578742F66316A2F78A366316CAC2F636F6E746578742F66316BA366316DA82F636F6E74657874A366316EA12FA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264B52F636F6E746578742F7265736F757263652F663264A3663265B92F636F6E746578742F7265736F757263652F6632652F782F79A3663266B12F636F6E746578742F7265736F75726365A3663267B72F636F6E746578742F7265736F757263652F6632672F78A3663268B52F636F6E746578742F7265736F757263652F663268A3663269B12F636F6E746578742F7265736F75726365A366326AA82F636F6E74657874A366326BAE2F636F6E746578742F66326A2F78A366326CAC2F636F6E746578742F66326BA366326DA82F636F6E74657874A366326EA12FA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362D9212F636F6E746578742F7265736F757263652F253343253345253236253237253232A3663363B72F636F6E746578742F7265736F757263652F3C3E262722A26634BC2F636F6E746578742F7265736F757263652F746573742F7572692F62A26635B62F636F6E746578742F7265736F757263652F66352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='/context/resource/f0/x0'>\n<jp:f2h>/context/resource/f2h</jp:f2h>\n<jp:f2f>/context/resource</jp:f2f>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1h>/context/resource/f1h</jp:f1h>\n<jp:f5>/context/resource/f5/x</jp:f5>\n<jp:f1e>/context/resource/f1e/x/y</jp:f1e>\n<jp:f3c>/context/resource/&lt;&gt;&amp;''</jp:f3c>\n<jp:f2k>/context/f2j/x</jp:f2k>\n<jp:f1j>/context</jp:f1j>\n<jp:f2m>/context</jp:f2m>\n<jp:f2i>/context/resource</jp:f2i>\n<jp:f1d>/context/resource/f1d</jp:f1d>\n<jp:f4>/context/resource/test/uri/b</jp:f4>\n<jp:f1n>/</jp:f1n>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f1k>/context/f1j/x</jp:f1k>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f2e>/context/resource/f2e/x/y</jp:f2e>\n<jp:f3b>/context/resource/%3C%3E%26%27%22</jp:f3b>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f1i>/context/resource</jp:f1i>\n<jp:f1g>/context/resource/f1g/x</jp:f1g>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f1m>/context</jp:f1m>\n<jp:f2d>/context/resource/f2d</jp:f2d>\n<jp:f2g>/context/resource/f2g/x</jp:f2g>\n<jp:f2n>/</jp:f2n>\n<jp:f2j>/context</jp:f2j>\n<jp:f1l>/context/f1k</jp:f1l>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f1f>/context/resource</jp:f1f>\n<jp:f2l>/context/f2k</jp:f2l>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 7 */
				"NoAuthority-1-ROOT_RELATIVE",
				input(
					UriResolution.ROOT_RELATIVE, UriRelativity.RESOURCE, "","/context","/resource","/path"
				),
				results(
					"{f0:'/context/resource/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'/context/resource/f1d',f1e:'/context/resource/f1e/x/y',f1f:'/context/resource',f1g:'/context/resource/f1g/x',f1h:'/context/resource/f1h',f1i:'/context/resource',f1j:'/context',f1k:'/context/f1j/x',f1l:'/context/f1k',f1m:'/context',f1n:'/',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'/context/resource/f2d',f2e:'/context/resource/f2e/x/y',f2f:'/context/resource',f2g:'/context/resource/f2g/x',f2h:'/context/resource/f2h',f2i:'/context/resource',f2j:'/context',f2k:'/context/f2j/x',f2l:'/context/f2k',f2m:'/context',f2n:'/',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'/context/resource/%3C%3E%26%27%22',f3c:'/context/resource/<>&\\'\"',f4:'/context/resource/test/uri/b',f5:'/context/resource/f5/x'}",
					"<object f0='/context/resource/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>/context/resource/f1d</f1d><f1e>/context/resource/f1e/x/y</f1e><f1f>/context/resource</f1f><f1g>/context/resource/f1g/x</f1g><f1h>/context/resource/f1h</f1h><f1i>/context/resource</f1i><f1j>/context</f1j><f1k>/context/f1j/x</f1k><f1l>/context/f1k</f1l><f1m>/context</f1m><f1n>/</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>/context/resource/f2d</f2d><f2e>/context/resource/f2e/x/y</f2e><f2f>/context/resource</f2f><f2g>/context/resource/f2g/x</f2g><f2h>/context/resource/f2h</f2h><f2i>/context/resource</f2i><f2j>/context</f2j><f2k>/context/f2j/x</f2k><f2l>/context/f2k</f2l><f2m>/context</f2m><f2n>/</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>/context/resource/%3C%3E%26%27%22</f3b><f3c>/context/resource/&lt;&gt;&amp;'\"</f3c><f4>/context/resource/test/uri/b</f4><f5>/context/resource/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='/context/resource/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='/context/resource/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='/context/resource/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f1g</td><td><a href='/context/resource/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/context/resource/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f1j</td><td><a href='/context'>context</a></td></tr><tr><td>f1k</td><td><a href='/context/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/context/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/context'>context</a></td></tr><tr><td>f1n</td><td><a href='/'>/</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='/context/resource/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='/context/resource/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f2g</td><td><a href='/context/resource/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/context/resource/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f2j</td><td><a href='/context'>context</a></td></tr><tr><td>f2k</td><td><a href='/context/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/context/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/context'>context</a></td></tr><tr><td>f2n</td><td><a href='/'>/</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='/context/resource/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='/context/resource/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='/context/resource/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='/context/resource/f5/x'>x</a></td></tr></table>",
					"(f0=/context/resource/f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=/context/resource/f1d,f1e=/context/resource/f1e/x/y,f1f=/context/resource,f1g=/context/resource/f1g/x,f1h=/context/resource/f1h,f1i=/context/resource,f1j=/context,f1k=/context/f1j/x,f1l=/context/f1k,f1m=/context,f1n=/,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=/context/resource/f2d,f2e=/context/resource/f2e/x/y,f2f=/context/resource,f2g=/context/resource/f2g/x,f2h=/context/resource/f2h,f2i=/context/resource,f2j=/context,f2k=/context/f2j/x,f2l=/context/f2k,f2m=/context,f2n=/,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=/context/resource/%3C%3E%26%27%22,f3c=/context/resource/<>&~'\",f4=/context/resource/test/uri/b,f5=/context/resource/f5/x)",
					"f0=/context/resource/f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=/context/resource/f1d&f1e=/context/resource/f1e/x/y&f1f=/context/resource&f1g=/context/resource/f1g/x&f1h=/context/resource/f1h&f1i=/context/resource&f1j=/context&f1k=/context/f1j/x&f1l=/context/f1k&f1m=/context&f1n=/&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=/context/resource/f2d&f2e=/context/resource/f2e/x/y&f2f=/context/resource&f2g=/context/resource/f2g/x&f2h=/context/resource/f2h&f2i=/context/resource&f2j=/context&f2k=/context/f2j/x&f2l=/context/f2k&f2m=/context&f2n=/&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=/context/resource/%253C%253E%2526%2527%2522&f3c=/context/resource/%3C%3E%26~'%22&f4=/context/resource/test/uri/b&f5=/context/resource/f5/x",
					"DE0022A26630B72F636F6E746578742F7265736F757263652F66302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164B52F636F6E746578742F7265736F757263652F663164A3663165B92F636F6E746578742F7265736F757263652F6631652F782F79A3663166B12F636F6E746578742F7265736F75726365A3663167B72F636F6E746578742F7265736F757263652F6631672F78A3663168B52F636F6E746578742F7265736F757263652F663168A3663169B12F636F6E746578742F7265736F75726365A366316AA82F636F6E74657874A366316BAE2F636F6E746578742F66316A2F78A366316CAC2F636F6E746578742F66316BA366316DA82F636F6E74657874A366316EA12FA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264B52F636F6E746578742F7265736F757263652F663264A3663265B92F636F6E746578742F7265736F757263652F6632652F782F79A3663266B12F636F6E746578742F7265736F75726365A3663267B72F636F6E746578742F7265736F757263652F6632672F78A3663268B52F636F6E746578742F7265736F757263652F663268A3663269B12F636F6E746578742F7265736F75726365A366326AA82F636F6E74657874A366326BAE2F636F6E746578742F66326A2F78A366326CAC2F636F6E746578742F66326BA366326DA82F636F6E74657874A366326EA12FA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362D9212F636F6E746578742F7265736F757263652F253343253345253236253237253232A3663363B72F636F6E746578742F7265736F757263652F3C3E262722A26634BC2F636F6E746578742F7265736F757263652F746573742F7572692F62A26635B62F636F6E746578742F7265736F757263652F66352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='/context/resource/f0/x0'>\n<jp:f2h>/context/resource/f2h</jp:f2h>\n<jp:f2f>/context/resource</jp:f2f>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1h>/context/resource/f1h</jp:f1h>\n<jp:f5>/context/resource/f5/x</jp:f5>\n<jp:f1e>/context/resource/f1e/x/y</jp:f1e>\n<jp:f3c>/context/resource/&lt;&gt;&amp;''</jp:f3c>\n<jp:f2k>/context/f2j/x</jp:f2k>\n<jp:f1j>/context</jp:f1j>\n<jp:f2m>/context</jp:f2m>\n<jp:f2i>/context/resource</jp:f2i>\n<jp:f1d>/context/resource/f1d</jp:f1d>\n<jp:f4>/context/resource/test/uri/b</jp:f4>\n<jp:f1n>/</jp:f1n>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f1k>/context/f1j/x</jp:f1k>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f2e>/context/resource/f2e/x/y</jp:f2e>\n<jp:f3b>/context/resource/%3C%3E%26%27%22</jp:f3b>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f1i>/context/resource</jp:f1i>\n<jp:f1g>/context/resource/f1g/x</jp:f1g>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f1m>/context</jp:f1m>\n<jp:f2d>/context/resource/f2d</jp:f2d>\n<jp:f2g>/context/resource/f2g/x</jp:f2g>\n<jp:f2n>/</jp:f2n>\n<jp:f2j>/context</jp:f2j>\n<jp:f1l>/context/f1k</jp:f1l>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f1f>/context/resource</jp:f1f>\n<jp:f2l>/context/f2k</jp:f2l>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 8 */
				"NoAuthority-1-NONE",
				input(
					UriResolution.NONE, UriRelativity.RESOURCE, "","/context","/resource","/path"
				),
				results(
					"{f0:'f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'f1d',f1e:'f1e/x/y',f1f:'',f1g:'/context/resource/f1g/x',f1h:'/context/resource/f1h',f1i:'/context/resource',f1j:'/context',f1k:'/context/f1j/x',f1l:'/context/f1k',f1m:'/context',f1n:'/',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'f2d',f2e:'f2e/x/y',f2f:'',f2g:'/context/resource/f2g/x',f2h:'/context/resource/f2h',f2i:'/context/resource',f2j:'/context',f2k:'/context/f2j/x',f2l:'/context/f2k',f2m:'/context',f2n:'/',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'%3C%3E%26%27%22',f3c:'<>&\\'\"',f4:'test/uri/b',f5:'f5/x'}",
					"<object f0='f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>f1d</f1d><f1e>f1e/x/y</f1e><f1f>_xE000_</f1f><f1g>/context/resource/f1g/x</f1g><f1h>/context/resource/f1h</f1h><f1i>/context/resource</f1i><f1j>/context</f1j><f1k>/context/f1j/x</f1k><f1l>/context/f1k</f1l><f1m>/context</f1m><f1n>/</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>f2d</f2d><f2e>f2e/x/y</f2e><f2f>_xE000_</f2f><f2g>/context/resource/f2g/x</f2g><f2h>/context/resource/f2h</f2h><f2i>/context/resource</f2i><f2j>/context</f2j><f2k>/context/f2j/x</f2k><f2l>/context/f2k</f2l><f2m>/context</f2m><f2n>/</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>%3C%3E%26%27%22</f3b><f3c>&lt;&gt;&amp;'\"</f3c><f4>test/uri/b</f4><f5>f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href=''>/</a></td></tr><tr><td>f1g</td><td><a href='/context/resource/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/context/resource/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f1j</td><td><a href='/context'>context</a></td></tr><tr><td>f1k</td><td><a href='/context/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/context/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/context'>context</a></td></tr><tr><td>f1n</td><td><a href='/'>/</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href=''>/</a></td></tr><tr><td>f2g</td><td><a href='/context/resource/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/context/resource/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f2j</td><td><a href='/context'>context</a></td></tr><tr><td>f2k</td><td><a href='/context/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/context/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/context'>context</a></td></tr><tr><td>f2n</td><td><a href='/'>/</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='f5/x'>x</a></td></tr></table>",
					"(f0=f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=f1d,f1e=f1e/x/y,f1f='',f1g=/context/resource/f1g/x,f1h=/context/resource/f1h,f1i=/context/resource,f1j=/context,f1k=/context/f1j/x,f1l=/context/f1k,f1m=/context,f1n=/,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=f2d,f2e=f2e/x/y,f2f='',f2g=/context/resource/f2g/x,f2h=/context/resource/f2h,f2i=/context/resource,f2j=/context,f2k=/context/f2j/x,f2l=/context/f2k,f2m=/context,f2n=/,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=%3C%3E%26%27%22,f3c=<>&~'\",f4=test/uri/b,f5=f5/x)",
					"f0=f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=f1d&f1e=f1e/x/y&f1f=''&f1g=/context/resource/f1g/x&f1h=/context/resource/f1h&f1i=/context/resource&f1j=/context&f1k=/context/f1j/x&f1l=/context/f1k&f1m=/context&f1n=/&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=f2d&f2e=f2e/x/y&f2f=''&f2g=/context/resource/f2g/x&f2h=/context/resource/f2h&f2i=/context/resource&f2j=/context&f2k=/context/f2j/x&f2l=/context/f2k&f2m=/context&f2n=/&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=%253C%253E%2526%2527%2522&f3c=%3C%3E%26~'%22&f4=test/uri/b&f5=f5/x",
					"DE0022A26630A566302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164A3663164A3663165A76631652F782F79A3663166A0A3663167B72F636F6E746578742F7265736F757263652F6631672F78A3663168B52F636F6E746578742F7265736F757263652F663168A3663169B12F636F6E746578742F7265736F75726365A366316AA82F636F6E74657874A366316BAE2F636F6E746578742F66316A2F78A366316CAC2F636F6E746578742F66316BA366316DA82F636F6E74657874A366316EA12FA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264A3663264A3663265A76632652F782F79A3663266A0A3663267B72F636F6E746578742F7265736F757263652F6632672F78A3663268B52F636F6E746578742F7265736F757263652F663268A3663269B12F636F6E746578742F7265736F75726365A366326AA82F636F6E74657874A366326BAE2F636F6E746578742F66326A2F78A366326CAC2F636F6E746578742F66326BA366326DA82F636F6E74657874A366326EA12FA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362AF253343253345253236253237253232A3663363A53C3E262722A26634AA746573742F7572692F62A26635A466352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='f0/x0'>\n<jp:f2l>/context/f2k</jp:f2l>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f2g>/context/resource/f2g/x</jp:f2g>\n<jp:f2f></jp:f2f>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f3b>%3C%3E%26%27%22</jp:f3b>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f5>f5/x</jp:f5>\n<jp:f2k>/context/f2j/x</jp:f2k>\n<jp:f1e>f1e/x/y</jp:f1e>\n<jp:f2i>/context/resource</jp:f2i>\n<jp:f2n>/</jp:f2n>\n<jp:f2m>/context</jp:f2m>\n<jp:f1m>/context</jp:f1m>\n<jp:f2d>f2d</jp:f2d>\n<jp:f1l>/context/f1k</jp:f1l>\n<jp:f3c>&lt;&gt;&amp;''</jp:f3c>\n<jp:f1n>/</jp:f1n>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f1k>/context/f1j/x</jp:f1k>\n<jp:f1h>/context/resource/f1h</jp:f1h>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f2h>/context/resource/f2h</jp:f2h>\n<jp:f1d>f1d</jp:f1d>\n<jp:f1i>/context/resource</jp:f1i>\n<jp:f2e>f2e/x/y</jp:f2e>\n<jp:f2j>/context</jp:f2j>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1j>/context</jp:f1j>\n<jp:f1f></jp:f1f>\n<jp:f1g>/context/resource/f1g/x</jp:f1g>\n<jp:f4>test/uri/b</jp:f4>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 9 */
				"NoContext-1-ABSOLUTE",
				input(
					UriResolution.ABSOLUTE, UriRelativity.RESOURCE, "http://foo.com:123","","/resource","/path"
				),
				results(
					"{f0:'http://foo.com:123/resource/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'http://foo.com:123/f1b',f1c:'http://foo.com:123/f1c/x/y',f1d:'http://foo.com:123/resource/f1d',f1e:'http://foo.com:123/resource/f1e/x/y',f1f:'http://foo.com:123/resource',f1g:'http://foo.com:123/resource/f1g/x',f1h:'http://foo.com:123/resource/f1h',f1i:'http://foo.com:123/resource',f1j:'http://foo.com:123',f1k:'http://foo.com:123/f1j/x',f1l:'http://foo.com:123/f1k',f1m:'http://foo.com:123',f1n:'http://foo.com:123/..',f2a:'http://www.apache.org/f2a',f2b:'http://foo.com:123/f2b',f2c:'http://foo.com:123/f2c/x/y',f2d:'http://foo.com:123/resource/f2d',f2e:'http://foo.com:123/resource/f2e/x/y',f2f:'http://foo.com:123/resource',f2g:'http://foo.com:123/resource/f2g/x',f2h:'http://foo.com:123/resource/f2h',f2i:'http://foo.com:123/resource',f2j:'http://foo.com:123',f2k:'http://foo.com:123/f2j/x',f2l:'http://foo.com:123/f2k',f2m:'http://foo.com:123',f2n:'http://foo.com:123/..',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'http://foo.com:123/resource/%3C%3E%26%27%22',f3c:'http://foo.com:123/resource/<>&\\'\"',f4:'http://foo.com:123/resource/test/uri/b',f5:'http://foo.com:123/resource/f5/x'}",
					"<object f0='http://foo.com:123/resource/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>http://foo.com:123/f1b</f1b><f1c>http://foo.com:123/f1c/x/y</f1c><f1d>http://foo.com:123/resource/f1d</f1d><f1e>http://foo.com:123/resource/f1e/x/y</f1e><f1f>http://foo.com:123/resource</f1f><f1g>http://foo.com:123/resource/f1g/x</f1g><f1h>http://foo.com:123/resource/f1h</f1h><f1i>http://foo.com:123/resource</f1i><f1j>http://foo.com:123</f1j><f1k>http://foo.com:123/f1j/x</f1k><f1l>http://foo.com:123/f1k</f1l><f1m>http://foo.com:123</f1m><f1n>http://foo.com:123/..</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>http://foo.com:123/f2b</f2b><f2c>http://foo.com:123/f2c/x/y</f2c><f2d>http://foo.com:123/resource/f2d</f2d><f2e>http://foo.com:123/resource/f2e/x/y</f2e><f2f>http://foo.com:123/resource</f2f><f2g>http://foo.com:123/resource/f2g/x</f2g><f2h>http://foo.com:123/resource/f2h</f2h><f2i>http://foo.com:123/resource</f2i><f2j>http://foo.com:123</f2j><f2k>http://foo.com:123/f2j/x</f2k><f2l>http://foo.com:123/f2k</f2l><f2m>http://foo.com:123</f2m><f2n>http://foo.com:123/..</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>http://foo.com:123/resource/%3C%3E%26%27%22</f3b><f3c>http://foo.com:123/resource/&lt;&gt;&amp;'\"</f3c><f4>http://foo.com:123/resource/test/uri/b</f4><f5>http://foo.com:123/resource/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='http://foo.com:123/resource/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='http://foo.com:123/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='http://foo.com:123/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='http://foo.com:123/resource/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='http://foo.com:123/resource/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='http://foo.com:123/resource'>resource</a></td></tr><tr><td>f1g</td><td><a href='http://foo.com:123/resource/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='http://foo.com:123/resource/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='http://foo.com:123/resource'>resource</a></td></tr><tr><td>f1j</td><td><a href='http://foo.com:123'>foo.com:123</a></td></tr><tr><td>f1k</td><td><a href='http://foo.com:123/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='http://foo.com:123/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='http://foo.com:123'>foo.com:123</a></td></tr><tr><td>f1n</td><td><a href='http://foo.com:123/..'>..</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='http://foo.com:123/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='http://foo.com:123/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='http://foo.com:123/resource/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='http://foo.com:123/resource/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='http://foo.com:123/resource'>resource</a></td></tr><tr><td>f2g</td><td><a href='http://foo.com:123/resource/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='http://foo.com:123/resource/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='http://foo.com:123/resource'>resource</a></td></tr><tr><td>f2j</td><td><a href='http://foo.com:123'>foo.com:123</a></td></tr><tr><td>f2k</td><td><a href='http://foo.com:123/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='http://foo.com:123/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='http://foo.com:123'>foo.com:123</a></td></tr><tr><td>f2n</td><td><a href='http://foo.com:123/..'>..</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='http://foo.com:123/resource/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='http://foo.com:123/resource/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='http://foo.com:123/resource/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='http://foo.com:123/resource/f5/x'>x</a></td></tr></table>",
					"(f0=http://foo.com:123/resource/f0/x0,f1a=http://www.apache.org/f1a,f1b=http://foo.com:123/f1b,f1c=http://foo.com:123/f1c/x/y,f1d=http://foo.com:123/resource/f1d,f1e=http://foo.com:123/resource/f1e/x/y,f1f=http://foo.com:123/resource,f1g=http://foo.com:123/resource/f1g/x,f1h=http://foo.com:123/resource/f1h,f1i=http://foo.com:123/resource,f1j=http://foo.com:123,f1k=http://foo.com:123/f1j/x,f1l=http://foo.com:123/f1k,f1m=http://foo.com:123,f1n=http://foo.com:123/..,f2a=http://www.apache.org/f2a,f2b=http://foo.com:123/f2b,f2c=http://foo.com:123/f2c/x/y,f2d=http://foo.com:123/resource/f2d,f2e=http://foo.com:123/resource/f2e/x/y,f2f=http://foo.com:123/resource,f2g=http://foo.com:123/resource/f2g/x,f2h=http://foo.com:123/resource/f2h,f2i=http://foo.com:123/resource,f2j=http://foo.com:123,f2k=http://foo.com:123/f2j/x,f2l=http://foo.com:123/f2k,f2m=http://foo.com:123,f2n=http://foo.com:123/..,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=http://foo.com:123/resource/%3C%3E%26%27%22,f3c=http://foo.com:123/resource/<>&~'\",f4=http://foo.com:123/resource/test/uri/b,f5=http://foo.com:123/resource/f5/x)",
					"f0=http://foo.com:123/resource/f0/x0&f1a=http://www.apache.org/f1a&f1b=http://foo.com:123/f1b&f1c=http://foo.com:123/f1c/x/y&f1d=http://foo.com:123/resource/f1d&f1e=http://foo.com:123/resource/f1e/x/y&f1f=http://foo.com:123/resource&f1g=http://foo.com:123/resource/f1g/x&f1h=http://foo.com:123/resource/f1h&f1i=http://foo.com:123/resource&f1j=http://foo.com:123&f1k=http://foo.com:123/f1j/x&f1l=http://foo.com:123/f1k&f1m=http://foo.com:123&f1n=http://foo.com:123/..&f2a=http://www.apache.org/f2a&f2b=http://foo.com:123/f2b&f2c=http://foo.com:123/f2c/x/y&f2d=http://foo.com:123/resource/f2d&f2e=http://foo.com:123/resource/f2e/x/y&f2f=http://foo.com:123/resource&f2g=http://foo.com:123/resource/f2g/x&f2h=http://foo.com:123/resource/f2h&f2i=http://foo.com:123/resource&f2j=http://foo.com:123&f2k=http://foo.com:123/f2j/x&f2l=http://foo.com:123/f2k&f2m=http://foo.com:123&f2n=http://foo.com:123/..&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=http://foo.com:123/resource/%253C%253E%2526%2527%2522&f3c=http://foo.com:123/resource/%3C%3E%26~'%22&f4=http://foo.com:123/resource/test/uri/b&f5=http://foo.com:123/resource/f5/x",
					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
					"<rdf:RDF>\n<rdf:Description rdf:about='http://foo.com:123/resource/f0/x0'>\n<jp:f2i rdf:resource='http://foo.com:123/resource'/>\n<jp:f2j rdf:resource='http://foo.com:123'/>\n<jp:f2h rdf:resource='http://foo.com:123/resource/f2h'/>\n<jp:f1g rdf:resource='http://foo.com:123/resource/f1g/x'/>\n<jp:f1n rdf:resource='http://foo.com:123/..'/>\n<jp:f2b rdf:resource='http://foo.com:123/f2b'/>\n<jp:f1f rdf:resource='http://foo.com:123/resource'/>\n<jp:f2d rdf:resource='http://foo.com:123/resource/f2d'/>\n<jp:f1l rdf:resource='http://foo.com:123/f1k'/>\n<jp:f3b rdf:resource='http://foo.com:123/resource/%3C%3E%26%27%22'/>\n<jp:f4 rdf:resource='http://foo.com:123/resource/test/uri/b'/>\n<jp:f2l rdf:resource='http://foo.com:123/f2k'/>\n<jp:f1d rdf:resource='http://foo.com:123/resource/f1d'/>\n<jp:f1e rdf:resource='http://foo.com:123/resource/f1e/x/y'/>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f1j rdf:resource='http://foo.com:123'/>\n<jp:f1k rdf:resource='http://foo.com:123/f1j/x'/>\n<jp:f2m rdf:resource='http://foo.com:123'/>\n<jp:f1i rdf:resource='http://foo.com:123/resource'/>\n<jp:f5 rdf:resource='http://foo.com:123/resource/f5/x'/>\n<jp:f2f rdf:resource='http://foo.com:123/resource'/>\n<jp:f2n rdf:resource='http://foo.com:123/..'/>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f1h rdf:resource='http://foo.com:123/resource/f1h'/>\n<jp:f1m rdf:resource='http://foo.com:123'/>\n<jp:f2c rdf:resource='http://foo.com:123/f2c/x/y'/>\n<jp:f2g rdf:resource='http://foo.com:123/resource/f2g/x'/>\n<jp:f2k rdf:resource='http://foo.com:123/f2j/x'/>\n<jp:f1b rdf:resource='http://foo.com:123/f1b'/>\n<jp:f1c rdf:resource='http://foo.com:123/f1c/x/y'/>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f3c rdf:resource='http://foo.com:123/resource/&lt;&gt;&amp;&apos;&quot;'/>\n<jp:f2e rdf:resource='http://foo.com:123/resource/f2e/x/y'/>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 10 */
				"NoContext-1-ROOT_RELATIVE",
				input(
					UriResolution.ROOT_RELATIVE, UriRelativity.RESOURCE, "http://foo.com:123","","/resource","/path"
				),
				results(
					"{f0:'/resource/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'/resource/f1d',f1e:'/resource/f1e/x/y',f1f:'/resource',f1g:'/resource/f1g/x',f1h:'/resource/f1h',f1i:'/resource',f1j:'/',f1k:'/f1j/x',f1l:'/f1k',f1m:'/',f1n:'/..',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'/resource/f2d',f2e:'/resource/f2e/x/y',f2f:'/resource',f2g:'/resource/f2g/x',f2h:'/resource/f2h',f2i:'/resource',f2j:'/',f2k:'/f2j/x',f2l:'/f2k',f2m:'/',f2n:'/..',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'/resource/%3C%3E%26%27%22',f3c:'/resource/<>&\\'\"',f4:'/resource/test/uri/b',f5:'/resource/f5/x'}",
					"<object f0='/resource/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>/resource/f1d</f1d><f1e>/resource/f1e/x/y</f1e><f1f>/resource</f1f><f1g>/resource/f1g/x</f1g><f1h>/resource/f1h</f1h><f1i>/resource</f1i><f1j>/</f1j><f1k>/f1j/x</f1k><f1l>/f1k</f1l><f1m>/</f1m><f1n>/..</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>/resource/f2d</f2d><f2e>/resource/f2e/x/y</f2e><f2f>/resource</f2f><f2g>/resource/f2g/x</f2g><f2h>/resource/f2h</f2h><f2i>/resource</f2i><f2j>/</f2j><f2k>/f2j/x</f2k><f2l>/f2k</f2l><f2m>/</f2m><f2n>/..</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>/resource/%3C%3E%26%27%22</f3b><f3c>/resource/&lt;&gt;&amp;'\"</f3c><f4>/resource/test/uri/b</f4><f5>/resource/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='/resource/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='/resource/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='/resource/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='/resource'>resource</a></td></tr><tr><td>f1g</td><td><a href='/resource/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/resource/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/resource'>resource</a></td></tr><tr><td>f1j</td><td><a href='/'>/</a></td></tr><tr><td>f1k</td><td><a href='/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/'>/</a></td></tr><tr><td>f1n</td><td><a href='/..'>..</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='/resource/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='/resource/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='/resource'>resource</a></td></tr><tr><td>f2g</td><td><a href='/resource/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/resource/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/resource'>resource</a></td></tr><tr><td>f2j</td><td><a href='/'>/</a></td></tr><tr><td>f2k</td><td><a href='/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/'>/</a></td></tr><tr><td>f2n</td><td><a href='/..'>..</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='/resource/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='/resource/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='/resource/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='/resource/f5/x'>x</a></td></tr></table>",
					"(f0=/resource/f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=/resource/f1d,f1e=/resource/f1e/x/y,f1f=/resource,f1g=/resource/f1g/x,f1h=/resource/f1h,f1i=/resource,f1j=/,f1k=/f1j/x,f1l=/f1k,f1m=/,f1n=/..,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=/resource/f2d,f2e=/resource/f2e/x/y,f2f=/resource,f2g=/resource/f2g/x,f2h=/resource/f2h,f2i=/resource,f2j=/,f2k=/f2j/x,f2l=/f2k,f2m=/,f2n=/..,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=/resource/%3C%3E%26%27%22,f3c=/resource/<>&~'\",f4=/resource/test/uri/b,f5=/resource/f5/x)",
					"f0=/resource/f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=/resource/f1d&f1e=/resource/f1e/x/y&f1f=/resource&f1g=/resource/f1g/x&f1h=/resource/f1h&f1i=/resource&f1j=/&f1k=/f1j/x&f1l=/f1k&f1m=/&f1n=/..&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=/resource/f2d&f2e=/resource/f2e/x/y&f2f=/resource&f2g=/resource/f2g/x&f2h=/resource/f2h&f2i=/resource&f2j=/&f2k=/f2j/x&f2l=/f2k&f2m=/&f2n=/..&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=/resource/%253C%253E%2526%2527%2522&f3c=/resource/%3C%3E%26~'%22&f4=/resource/test/uri/b&f5=/resource/f5/x",
					"DE0022A26630AF2F7265736F757263652F66302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164AD2F7265736F757263652F663164A3663165B12F7265736F757263652F6631652F782F79A3663166A92F7265736F75726365A3663167AF2F7265736F757263652F6631672F78A3663168AD2F7265736F757263652F663168A3663169A92F7265736F75726365A366316AA12FA366316BA62F66316A2F78A366316CA42F66316BA366316DA12FA366316EA32F2E2EA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264AD2F7265736F757263652F663264A3663265B12F7265736F757263652F6632652F782F79A3663266A92F7265736F75726365A3663267AF2F7265736F757263652F6632672F78A3663268AD2F7265736F757263652F663268A3663269A92F7265736F75726365A366326AA12FA366326BA62F66326A2F78A366326CA42F66326BA366326DA12FA366326EA32F2E2EA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362B92F7265736F757263652F253343253345253236253237253232A3663363AF2F7265736F757263652F3C3E262722A26634B42F7265736F757263652F746573742F7572692F62A26635AE2F7265736F757263652F66352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='/resource/f0/x0'>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f2e>/resource/f2e/x/y</jp:f2e>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f2n>/..</jp:f2n>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f3c>/resource/&lt;&gt;&amp;''</jp:f3c>\n<jp:f2h>/resource/f2h</jp:f2h>\n<jp:f1i>/resource</jp:f1i>\n<jp:f1n>/..</jp:f1n>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1e>/resource/f1e/x/y</jp:f1e>\n<jp:f5>/resource/f5/x</jp:f5>\n<jp:f2l>/f2k</jp:f2l>\n<jp:f2d>/resource/f2d</jp:f2d>\n<jp:f2j>/</jp:f2j>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f3b>/resource/%3C%3E%26%27%22</jp:f3b>\n<jp:f1j>/</jp:f1j>\n<jp:f1d>/resource/f1d</jp:f1d>\n<jp:f2m>/</jp:f2m>\n<jp:f1f>/resource</jp:f1f>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f4>/resource/test/uri/b</jp:f4>\n<jp:f2i>/resource</jp:f2i>\n<jp:f1h>/resource/f1h</jp:f1h>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f1k>/f1j/x</jp:f1k>\n<jp:f1l>/f1k</jp:f1l>\n<jp:f2g>/resource/f2g/x</jp:f2g>\n<jp:f2k>/f2j/x</jp:f2k>\n<jp:f2f>/resource</jp:f2f>\n<jp:f1g>/resource/f1g/x</jp:f1g>\n<jp:f1m>/</jp:f1m>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 11 */
				"NoContext-1-NONE",
				input(
					UriResolution.NONE, UriRelativity.RESOURCE, "http://foo.com:123","","/resource","/path"
				),
				results(
					"{f0:'f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'f1d',f1e:'f1e/x/y',f1f:'',f1g:'/resource/f1g/x',f1h:'/resource/f1h',f1i:'/resource',f1j:'/',f1k:'/f1j/x',f1l:'/f1k',f1m:'/',f1n:'/..',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'f2d',f2e:'f2e/x/y',f2f:'',f2g:'/resource/f2g/x',f2h:'/resource/f2h',f2i:'/resource',f2j:'/',f2k:'/f2j/x',f2l:'/f2k',f2m:'/',f2n:'/..',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'%3C%3E%26%27%22',f3c:'<>&\\'\"',f4:'test/uri/b',f5:'f5/x'}",
					"<object f0='f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>f1d</f1d><f1e>f1e/x/y</f1e><f1f>_xE000_</f1f><f1g>/resource/f1g/x</f1g><f1h>/resource/f1h</f1h><f1i>/resource</f1i><f1j>/</f1j><f1k>/f1j/x</f1k><f1l>/f1k</f1l><f1m>/</f1m><f1n>/..</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>f2d</f2d><f2e>f2e/x/y</f2e><f2f>_xE000_</f2f><f2g>/resource/f2g/x</f2g><f2h>/resource/f2h</f2h><f2i>/resource</f2i><f2j>/</f2j><f2k>/f2j/x</f2k><f2l>/f2k</f2l><f2m>/</f2m><f2n>/..</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>%3C%3E%26%27%22</f3b><f3c>&lt;&gt;&amp;'\"</f3c><f4>test/uri/b</f4><f5>f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href=''>/</a></td></tr><tr><td>f1g</td><td><a href='/resource/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/resource/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/resource'>resource</a></td></tr><tr><td>f1j</td><td><a href='/'>/</a></td></tr><tr><td>f1k</td><td><a href='/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/'>/</a></td></tr><tr><td>f1n</td><td><a href='/..'>..</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href=''>/</a></td></tr><tr><td>f2g</td><td><a href='/resource/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/resource/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/resource'>resource</a></td></tr><tr><td>f2j</td><td><a href='/'>/</a></td></tr><tr><td>f2k</td><td><a href='/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/'>/</a></td></tr><tr><td>f2n</td><td><a href='/..'>..</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='f5/x'>x</a></td></tr></table>",
					"(f0=f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=f1d,f1e=f1e/x/y,f1f='',f1g=/resource/f1g/x,f1h=/resource/f1h,f1i=/resource,f1j=/,f1k=/f1j/x,f1l=/f1k,f1m=/,f1n=/..,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=f2d,f2e=f2e/x/y,f2f='',f2g=/resource/f2g/x,f2h=/resource/f2h,f2i=/resource,f2j=/,f2k=/f2j/x,f2l=/f2k,f2m=/,f2n=/..,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=%3C%3E%26%27%22,f3c=<>&~'\",f4=test/uri/b,f5=f5/x)",
					"f0=f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=f1d&f1e=f1e/x/y&f1f=''&f1g=/resource/f1g/x&f1h=/resource/f1h&f1i=/resource&f1j=/&f1k=/f1j/x&f1l=/f1k&f1m=/&f1n=/..&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=f2d&f2e=f2e/x/y&f2f=''&f2g=/resource/f2g/x&f2h=/resource/f2h&f2i=/resource&f2j=/&f2k=/f2j/x&f2l=/f2k&f2m=/&f2n=/..&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=%253C%253E%2526%2527%2522&f3c=%3C%3E%26~'%22&f4=test/uri/b&f5=f5/x",
					"DE0022A26630A566302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164A3663164A3663165A76631652F782F79A3663166A0A3663167AF2F7265736F757263652F6631672F78A3663168AD2F7265736F757263652F663168A3663169A92F7265736F75726365A366316AA12FA366316BA62F66316A2F78A366316CA42F66316BA366316DA12FA366316EA32F2E2EA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264A3663264A3663265A76632652F782F79A3663266A0A3663267AF2F7265736F757263652F6632672F78A3663268AD2F7265736F757263652F663268A3663269A92F7265736F75726365A366326AA12FA366326BA62F66326A2F78A366326CA42F66326BA366326DA12FA366326EA32F2E2EA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362AF253343253345253236253237253232A3663363A53C3E262722A26634AA746573742F7572692F62A26635A466352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='f0/x0'>\n<jp:f1n>/..</jp:f1n>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f2h>/resource/f2h</jp:f2h>\n<jp:f2n>/..</jp:f2n>\n<jp:f1h>/resource/f1h</jp:f1h>\n<jp:f1i>/resource</jp:f1i>\n<jp:f1g>/resource/f1g/x</jp:f1g>\n<jp:f2f></jp:f2f>\n<jp:f2k>/f2j/x</jp:f2k>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f3b>%3C%3E%26%27%22</jp:f3b>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f2i>/resource</jp:f2i>\n<jp:f5>f5/x</jp:f5>\n<jp:f1m>/</jp:f1m>\n<jp:f2l>/f2k</jp:f2l>\n<jp:f2g>/resource/f2g/x</jp:f2g>\n<jp:f1k>/f1j/x</jp:f1k>\n<jp:f1e>f1e/x/y</jp:f1e>\n<jp:f2j>/</jp:f2j>\n<jp:f2d>f2d</jp:f2d>\n<jp:f3c>&lt;&gt;&amp;''</jp:f3c>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f1l>/f1k</jp:f1l>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f1d>f1d</jp:f1d>\n<jp:f1j>/</jp:f1j>\n<jp:f2e>f2e/x/y</jp:f2e>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1f></jp:f1f>\n<jp:f4>test/uri/b</jp:f4>\n<jp:f2m>/</jp:f2m>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 12 */
				"NoResource-1-ABSOLUTE",
				input(
					UriResolution.ABSOLUTE, UriRelativity.RESOURCE, "http://foo.com:123","/context","","/path"
				),
				results(
					"{f0:'http://foo.com:123/context/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'http://foo.com:123/f1b',f1c:'http://foo.com:123/f1c/x/y',f1d:'http://foo.com:123/context/f1d',f1e:'http://foo.com:123/context/f1e/x/y',f1f:'http://foo.com:123/context',f1g:'http://foo.com:123/context/f1g/x',f1h:'http://foo.com:123/context/f1h',f1i:'http://foo.com:123/context',f1j:'http://foo.com:123',f1k:'http://foo.com:123/context/f1j/x',f1l:'http://foo.com:123/context/f1k',f1m:'http://foo.com:123/context',f1n:'http://foo.com:123',f2a:'http://www.apache.org/f2a',f2b:'http://foo.com:123/f2b',f2c:'http://foo.com:123/f2c/x/y',f2d:'http://foo.com:123/context/f2d',f2e:'http://foo.com:123/context/f2e/x/y',f2f:'http://foo.com:123/context',f2g:'http://foo.com:123/context/f2g/x',f2h:'http://foo.com:123/context/f2h',f2i:'http://foo.com:123/context',f2j:'http://foo.com:123',f2k:'http://foo.com:123/context/f2j/x',f2l:'http://foo.com:123/context/f2k',f2m:'http://foo.com:123/context',f2n:'http://foo.com:123',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'http://foo.com:123/context/%3C%3E%26%27%22',f3c:'http://foo.com:123/context/<>&\\'\"',f4:'http://foo.com:123/context/test/uri/b',f5:'http://foo.com:123/context/f5/x'}",
					"<object f0='http://foo.com:123/context/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>http://foo.com:123/f1b</f1b><f1c>http://foo.com:123/f1c/x/y</f1c><f1d>http://foo.com:123/context/f1d</f1d><f1e>http://foo.com:123/context/f1e/x/y</f1e><f1f>http://foo.com:123/context</f1f><f1g>http://foo.com:123/context/f1g/x</f1g><f1h>http://foo.com:123/context/f1h</f1h><f1i>http://foo.com:123/context</f1i><f1j>http://foo.com:123</f1j><f1k>http://foo.com:123/context/f1j/x</f1k><f1l>http://foo.com:123/context/f1k</f1l><f1m>http://foo.com:123/context</f1m><f1n>http://foo.com:123</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>http://foo.com:123/f2b</f2b><f2c>http://foo.com:123/f2c/x/y</f2c><f2d>http://foo.com:123/context/f2d</f2d><f2e>http://foo.com:123/context/f2e/x/y</f2e><f2f>http://foo.com:123/context</f2f><f2g>http://foo.com:123/context/f2g/x</f2g><f2h>http://foo.com:123/context/f2h</f2h><f2i>http://foo.com:123/context</f2i><f2j>http://foo.com:123</f2j><f2k>http://foo.com:123/context/f2j/x</f2k><f2l>http://foo.com:123/context/f2k</f2l><f2m>http://foo.com:123/context</f2m><f2n>http://foo.com:123</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>http://foo.com:123/context/%3C%3E%26%27%22</f3b><f3c>http://foo.com:123/context/&lt;&gt;&amp;'\"</f3c><f4>http://foo.com:123/context/test/uri/b</f4><f5>http://foo.com:123/context/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='http://foo.com:123/context/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='http://foo.com:123/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='http://foo.com:123/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='http://foo.com:123/context/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='http://foo.com:123/context/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='http://foo.com:123/context'>context</a></td></tr><tr><td>f1g</td><td><a href='http://foo.com:123/context/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='http://foo.com:123/context/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='http://foo.com:123/context'>context</a></td></tr><tr><td>f1j</td><td><a href='http://foo.com:123'>foo.com:123</a></td></tr><tr><td>f1k</td><td><a href='http://foo.com:123/context/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='http://foo.com:123/context/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='http://foo.com:123/context'>context</a></td></tr><tr><td>f1n</td><td><a href='http://foo.com:123'>foo.com:123</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='http://foo.com:123/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='http://foo.com:123/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='http://foo.com:123/context/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='http://foo.com:123/context/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='http://foo.com:123/context'>context</a></td></tr><tr><td>f2g</td><td><a href='http://foo.com:123/context/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='http://foo.com:123/context/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='http://foo.com:123/context'>context</a></td></tr><tr><td>f2j</td><td><a href='http://foo.com:123'>foo.com:123</a></td></tr><tr><td>f2k</td><td><a href='http://foo.com:123/context/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='http://foo.com:123/context/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='http://foo.com:123/context'>context</a></td></tr><tr><td>f2n</td><td><a href='http://foo.com:123'>foo.com:123</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='http://foo.com:123/context/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='http://foo.com:123/context/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='http://foo.com:123/context/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='http://foo.com:123/context/f5/x'>x</a></td></tr></table>",
					"(f0=http://foo.com:123/context/f0/x0,f1a=http://www.apache.org/f1a,f1b=http://foo.com:123/f1b,f1c=http://foo.com:123/f1c/x/y,f1d=http://foo.com:123/context/f1d,f1e=http://foo.com:123/context/f1e/x/y,f1f=http://foo.com:123/context,f1g=http://foo.com:123/context/f1g/x,f1h=http://foo.com:123/context/f1h,f1i=http://foo.com:123/context,f1j=http://foo.com:123,f1k=http://foo.com:123/context/f1j/x,f1l=http://foo.com:123/context/f1k,f1m=http://foo.com:123/context,f1n=http://foo.com:123,f2a=http://www.apache.org/f2a,f2b=http://foo.com:123/f2b,f2c=http://foo.com:123/f2c/x/y,f2d=http://foo.com:123/context/f2d,f2e=http://foo.com:123/context/f2e/x/y,f2f=http://foo.com:123/context,f2g=http://foo.com:123/context/f2g/x,f2h=http://foo.com:123/context/f2h,f2i=http://foo.com:123/context,f2j=http://foo.com:123,f2k=http://foo.com:123/context/f2j/x,f2l=http://foo.com:123/context/f2k,f2m=http://foo.com:123/context,f2n=http://foo.com:123,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=http://foo.com:123/context/%3C%3E%26%27%22,f3c=http://foo.com:123/context/<>&~'\",f4=http://foo.com:123/context/test/uri/b,f5=http://foo.com:123/context/f5/x)",
					"f0=http://foo.com:123/context/f0/x0&f1a=http://www.apache.org/f1a&f1b=http://foo.com:123/f1b&f1c=http://foo.com:123/f1c/x/y&f1d=http://foo.com:123/context/f1d&f1e=http://foo.com:123/context/f1e/x/y&f1f=http://foo.com:123/context&f1g=http://foo.com:123/context/f1g/x&f1h=http://foo.com:123/context/f1h&f1i=http://foo.com:123/context&f1j=http://foo.com:123&f1k=http://foo.com:123/context/f1j/x&f1l=http://foo.com:123/context/f1k&f1m=http://foo.com:123/context&f1n=http://foo.com:123&f2a=http://www.apache.org/f2a&f2b=http://foo.com:123/f2b&f2c=http://foo.com:123/f2c/x/y&f2d=http://foo.com:123/context/f2d&f2e=http://foo.com:123/context/f2e/x/y&f2f=http://foo.com:123/context&f2g=http://foo.com:123/context/f2g/x&f2h=http://foo.com:123/context/f2h&f2i=http://foo.com:123/context&f2j=http://foo.com:123&f2k=http://foo.com:123/context/f2j/x&f2l=http://foo.com:123/context/f2k&f2m=http://foo.com:123/context&f2n=http://foo.com:123&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=http://foo.com:123/context/%253C%253E%2526%2527%2522&f3c=http://foo.com:123/context/%3C%3E%26~'%22&f4=http://foo.com:123/context/test/uri/b&f5=http://foo.com:123/context/f5/x",
					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
					"<rdf:RDF>\n<rdf:Description rdf:about='http://foo.com:123/context/f0/x0'>\n<jp:f1f rdf:resource='http://foo.com:123/context'/>\n<jp:f2c rdf:resource='http://foo.com:123/f2c/x/y'/>\n<jp:f3b rdf:resource='http://foo.com:123/context/%3C%3E%26%27%22'/>\n<jp:f2d rdf:resource='http://foo.com:123/context/f2d'/>\n<jp:f2l rdf:resource='http://foo.com:123/context/f2k'/>\n<jp:f1b rdf:resource='http://foo.com:123/f1b'/>\n<jp:f1c rdf:resource='http://foo.com:123/f1c/x/y'/>\n<jp:f1j rdf:resource='http://foo.com:123'/>\n<jp:f1k rdf:resource='http://foo.com:123/context/f1j/x'/>\n<jp:f2n rdf:resource='http://foo.com:123'/>\n<jp:f2k rdf:resource='http://foo.com:123/context/f2j/x'/>\n<jp:f1g rdf:resource='http://foo.com:123/context/f1g/x'/>\n<jp:f2h rdf:resource='http://foo.com:123/context/f2h'/>\n<jp:f1i rdf:resource='http://foo.com:123/context'/>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f2m rdf:resource='http://foo.com:123/context'/>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f1d rdf:resource='http://foo.com:123/context/f1d'/>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f2j rdf:resource='http://foo.com:123'/>\n<jp:f2i rdf:resource='http://foo.com:123/context'/>\n<jp:f1h rdf:resource='http://foo.com:123/context/f1h'/>\n<jp:f1n rdf:resource='http://foo.com:123'/>\n<jp:f4 rdf:resource='http://foo.com:123/context/test/uri/b'/>\n<jp:f1e rdf:resource='http://foo.com:123/context/f1e/x/y'/>\n<jp:f3c rdf:resource='http://foo.com:123/context/&lt;&gt;&amp;&apos;&quot;'/>\n<jp:f2b rdf:resource='http://foo.com:123/f2b'/>\n<jp:f2g rdf:resource='http://foo.com:123/context/f2g/x'/>\n<jp:f1l rdf:resource='http://foo.com:123/context/f1k'/>\n<jp:f1m rdf:resource='http://foo.com:123/context'/>\n<jp:f2f rdf:resource='http://foo.com:123/context'/>\n<jp:f5 rdf:resource='http://foo.com:123/context/f5/x'/>\n<jp:f2e rdf:resource='http://foo.com:123/context/f2e/x/y'/>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 13 */
				"NoResource-1-ROOT_RELATIVE",
				input(
					UriResolution.ROOT_RELATIVE, UriRelativity.RESOURCE, "http://foo.com:123","/context","","/path"
				),
				results(
					"{f0:'/context/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'/context/f1d',f1e:'/context/f1e/x/y',f1f:'/context',f1g:'/context/f1g/x',f1h:'/context/f1h',f1i:'/context',f1j:'/',f1k:'/context/f1j/x',f1l:'/context/f1k',f1m:'/context',f1n:'/',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'/context/f2d',f2e:'/context/f2e/x/y',f2f:'/context',f2g:'/context/f2g/x',f2h:'/context/f2h',f2i:'/context',f2j:'/',f2k:'/context/f2j/x',f2l:'/context/f2k',f2m:'/context',f2n:'/',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'/context/%3C%3E%26%27%22',f3c:'/context/<>&\\'\"',f4:'/context/test/uri/b',f5:'/context/f5/x'}",
					"<object f0='/context/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>/context/f1d</f1d><f1e>/context/f1e/x/y</f1e><f1f>/context</f1f><f1g>/context/f1g/x</f1g><f1h>/context/f1h</f1h><f1i>/context</f1i><f1j>/</f1j><f1k>/context/f1j/x</f1k><f1l>/context/f1k</f1l><f1m>/context</f1m><f1n>/</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>/context/f2d</f2d><f2e>/context/f2e/x/y</f2e><f2f>/context</f2f><f2g>/context/f2g/x</f2g><f2h>/context/f2h</f2h><f2i>/context</f2i><f2j>/</f2j><f2k>/context/f2j/x</f2k><f2l>/context/f2k</f2l><f2m>/context</f2m><f2n>/</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>/context/%3C%3E%26%27%22</f3b><f3c>/context/&lt;&gt;&amp;'\"</f3c><f4>/context/test/uri/b</f4><f5>/context/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='/context/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='/context/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='/context/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='/context'>context</a></td></tr><tr><td>f1g</td><td><a href='/context/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/context/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/context'>context</a></td></tr><tr><td>f1j</td><td><a href='/'>/</a></td></tr><tr><td>f1k</td><td><a href='/context/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/context/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/context'>context</a></td></tr><tr><td>f1n</td><td><a href='/'>/</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='/context/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='/context/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='/context'>context</a></td></tr><tr><td>f2g</td><td><a href='/context/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/context/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/context'>context</a></td></tr><tr><td>f2j</td><td><a href='/'>/</a></td></tr><tr><td>f2k</td><td><a href='/context/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/context/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/context'>context</a></td></tr><tr><td>f2n</td><td><a href='/'>/</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='/context/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='/context/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='/context/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='/context/f5/x'>x</a></td></tr></table>",
					"(f0=/context/f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=/context/f1d,f1e=/context/f1e/x/y,f1f=/context,f1g=/context/f1g/x,f1h=/context/f1h,f1i=/context,f1j=/,f1k=/context/f1j/x,f1l=/context/f1k,f1m=/context,f1n=/,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=/context/f2d,f2e=/context/f2e/x/y,f2f=/context,f2g=/context/f2g/x,f2h=/context/f2h,f2i=/context,f2j=/,f2k=/context/f2j/x,f2l=/context/f2k,f2m=/context,f2n=/,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=/context/%3C%3E%26%27%22,f3c=/context/<>&~'\",f4=/context/test/uri/b,f5=/context/f5/x)",
					"f0=/context/f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=/context/f1d&f1e=/context/f1e/x/y&f1f=/context&f1g=/context/f1g/x&f1h=/context/f1h&f1i=/context&f1j=/&f1k=/context/f1j/x&f1l=/context/f1k&f1m=/context&f1n=/&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=/context/f2d&f2e=/context/f2e/x/y&f2f=/context&f2g=/context/f2g/x&f2h=/context/f2h&f2i=/context&f2j=/&f2k=/context/f2j/x&f2l=/context/f2k&f2m=/context&f2n=/&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=/context/%253C%253E%2526%2527%2522&f3c=/context/%3C%3E%26~'%22&f4=/context/test/uri/b&f5=/context/f5/x",
					"DE0022A26630AE2F636F6E746578742F66302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164AC2F636F6E746578742F663164A3663165B02F636F6E746578742F6631652F782F79A3663166A82F636F6E74657874A3663167AE2F636F6E746578742F6631672F78A3663168AC2F636F6E746578742F663168A3663169A82F636F6E74657874A366316AA12FA366316BAE2F636F6E746578742F66316A2F78A366316CAC2F636F6E746578742F66316BA366316DA82F636F6E74657874A366316EA12FA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264AC2F636F6E746578742F663264A3663265B02F636F6E746578742F6632652F782F79A3663266A82F636F6E74657874A3663267AE2F636F6E746578742F6632672F78A3663268AC2F636F6E746578742F663268A3663269A82F636F6E74657874A366326AA12FA366326BAE2F636F6E746578742F66326A2F78A366326CAC2F636F6E746578742F66326BA366326DA82F636F6E74657874A366326EA12FA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362B82F636F6E746578742F253343253345253236253237253232A3663363AE2F636F6E746578742F3C3E262722A26634B32F636F6E746578742F746573742F7572692F62A26635AD2F636F6E746578742F66352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='/context/f0/x0'>\n<jp:f2k>/context/f2j/x</jp:f2k>\n<jp:f1f>/context</jp:f1f>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f1e>/context/f1e/x/y</jp:f1e>\n<jp:f2i>/context</jp:f2i>\n<jp:f1j>/</jp:f1j>\n<jp:f3b>/context/%3C%3E%26%27%22</jp:f3b>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f1h>/context/f1h</jp:f1h>\n<jp:f1l>/context/f1k</jp:f1l>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1i>/context</jp:f1i>\n<jp:f2j>/</jp:f2j>\n<jp:f2l>/context/f2k</jp:f2l>\n<jp:f2g>/context/f2g/x</jp:f2g>\n<jp:f3c>/context/&lt;&gt;&amp;''</jp:f3c>\n<jp:f1k>/context/f1j/x</jp:f1k>\n<jp:f2f>/context</jp:f2f>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f1m>/context</jp:f1m>\n<jp:f1d>/context/f1d</jp:f1d>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f2m>/context</jp:f2m>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f1n>/</jp:f1n>\n<jp:f4>/context/test/uri/b</jp:f4>\n<jp:f1g>/context/f1g/x</jp:f1g>\n<jp:f5>/context/f5/x</jp:f5>\n<jp:f2d>/context/f2d</jp:f2d>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f2e>/context/f2e/x/y</jp:f2e>\n<jp:f2n>/</jp:f2n>\n<jp:f2h>/context/f2h</jp:f2h>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 14 */
				"NoResource-1-NONE",
				input(
					UriResolution.NONE, UriRelativity.RESOURCE, "http://foo.com:123","/context","","/path"
				),
				results(
					"{f0:'f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'f1d',f1e:'f1e/x/y',f1f:'',f1g:'/context/f1g/x',f1h:'/context/f1h',f1i:'/context',f1j:'/',f1k:'/context/f1j/x',f1l:'/context/f1k',f1m:'/context',f1n:'/',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'f2d',f2e:'f2e/x/y',f2f:'',f2g:'/context/f2g/x',f2h:'/context/f2h',f2i:'/context',f2j:'/',f2k:'/context/f2j/x',f2l:'/context/f2k',f2m:'/context',f2n:'/',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'%3C%3E%26%27%22',f3c:'<>&\\'\"',f4:'test/uri/b',f5:'f5/x'}",
					"<object f0='f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>f1d</f1d><f1e>f1e/x/y</f1e><f1f>_xE000_</f1f><f1g>/context/f1g/x</f1g><f1h>/context/f1h</f1h><f1i>/context</f1i><f1j>/</f1j><f1k>/context/f1j/x</f1k><f1l>/context/f1k</f1l><f1m>/context</f1m><f1n>/</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>f2d</f2d><f2e>f2e/x/y</f2e><f2f>_xE000_</f2f><f2g>/context/f2g/x</f2g><f2h>/context/f2h</f2h><f2i>/context</f2i><f2j>/</f2j><f2k>/context/f2j/x</f2k><f2l>/context/f2k</f2l><f2m>/context</f2m><f2n>/</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>%3C%3E%26%27%22</f3b><f3c>&lt;&gt;&amp;'\"</f3c><f4>test/uri/b</f4><f5>f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href=''>/</a></td></tr><tr><td>f1g</td><td><a href='/context/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/context/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/context'>context</a></td></tr><tr><td>f1j</td><td><a href='/'>/</a></td></tr><tr><td>f1k</td><td><a href='/context/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/context/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/context'>context</a></td></tr><tr><td>f1n</td><td><a href='/'>/</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href=''>/</a></td></tr><tr><td>f2g</td><td><a href='/context/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/context/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/context'>context</a></td></tr><tr><td>f2j</td><td><a href='/'>/</a></td></tr><tr><td>f2k</td><td><a href='/context/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/context/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/context'>context</a></td></tr><tr><td>f2n</td><td><a href='/'>/</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='f5/x'>x</a></td></tr></table>",
					"(f0=f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=f1d,f1e=f1e/x/y,f1f='',f1g=/context/f1g/x,f1h=/context/f1h,f1i=/context,f1j=/,f1k=/context/f1j/x,f1l=/context/f1k,f1m=/context,f1n=/,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=f2d,f2e=f2e/x/y,f2f='',f2g=/context/f2g/x,f2h=/context/f2h,f2i=/context,f2j=/,f2k=/context/f2j/x,f2l=/context/f2k,f2m=/context,f2n=/,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=%3C%3E%26%27%22,f3c=<>&~'\",f4=test/uri/b,f5=f5/x)",
					"f0=f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=f1d&f1e=f1e/x/y&f1f=''&f1g=/context/f1g/x&f1h=/context/f1h&f1i=/context&f1j=/&f1k=/context/f1j/x&f1l=/context/f1k&f1m=/context&f1n=/&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=f2d&f2e=f2e/x/y&f2f=''&f2g=/context/f2g/x&f2h=/context/f2h&f2i=/context&f2j=/&f2k=/context/f2j/x&f2l=/context/f2k&f2m=/context&f2n=/&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=%253C%253E%2526%2527%2522&f3c=%3C%3E%26~'%22&f4=test/uri/b&f5=f5/x",
					"DE0022A26630A566302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164A3663164A3663165A76631652F782F79A3663166A0A3663167AE2F636F6E746578742F6631672F78A3663168AC2F636F6E746578742F663168A3663169A82F636F6E74657874A366316AA12FA366316BAE2F636F6E746578742F66316A2F78A366316CAC2F636F6E746578742F66316BA366316DA82F636F6E74657874A366316EA12FA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264A3663264A3663265A76632652F782F79A3663266A0A3663267AE2F636F6E746578742F6632672F78A3663268AC2F636F6E746578742F663268A3663269A82F636F6E74657874A366326AA12FA366326BAE2F636F6E746578742F66326A2F78A366326CAC2F636F6E746578742F66326BA366326DA82F636F6E74657874A366326EA12FA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362AF253343253345253236253237253232A3663363A53C3E262722A26634AA746573742F7572692F62A26635A466352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='f0/x0'>\n<jp:f2l>/context/f2k</jp:f2l>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f2i>/context</jp:f2i>\n<jp:f1i>/context</jp:f1i>\n<jp:f2f></jp:f2f>\n<jp:f1g>/context/f1g/x</jp:f1g>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f3b>%3C%3E%26%27%22</jp:f3b>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f5>f5/x</jp:f5>\n<jp:f2k>/context/f2j/x</jp:f2k>\n<jp:f1e>f1e/x/y</jp:f1e>\n<jp:f2n>/</jp:f2n>\n<jp:f2m>/context</jp:f2m>\n<jp:f2j>/</jp:f2j>\n<jp:f2h>/context/f2h</jp:f2h>\n<jp:f1m>/context</jp:f1m>\n<jp:f2d>f2d</jp:f2d>\n<jp:f1l>/context/f1k</jp:f1l>\n<jp:f3c>&lt;&gt;&amp;''</jp:f3c>\n<jp:f1h>/context/f1h</jp:f1h>\n<jp:f1n>/</jp:f1n>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f1k>/context/f1j/x</jp:f1k>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f1d>f1d</jp:f1d>\n<jp:f1j>/</jp:f1j>\n<jp:f2e>f2e/x/y</jp:f2e>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1f></jp:f1f>\n<jp:f4>test/uri/b</jp:f4>\n<jp:f2g>/context/f2g/x</jp:f2g>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 15 */
				"NoPath-1-ABSOLUTE",
				input(
					UriResolution.ABSOLUTE, UriRelativity.RESOURCE, "http://foo.com:123","/context","/resource",""
				),
				results(
					"{f0:'http://foo.com:123/context/resource/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'http://foo.com:123/f1b',f1c:'http://foo.com:123/f1c/x/y',f1d:'http://foo.com:123/context/resource/f1d',f1e:'http://foo.com:123/context/resource/f1e/x/y',f1f:'http://foo.com:123/context/resource',f1g:'http://foo.com:123/context/resource/f1g/x',f1h:'http://foo.com:123/context/resource/f1h',f1i:'http://foo.com:123/context/resource',f1j:'http://foo.com:123/context',f1k:'http://foo.com:123/context/f1j/x',f1l:'http://foo.com:123/context/f1k',f1m:'http://foo.com:123/context',f1n:'http://foo.com:123',f2a:'http://www.apache.org/f2a',f2b:'http://foo.com:123/f2b',f2c:'http://foo.com:123/f2c/x/y',f2d:'http://foo.com:123/context/resource/f2d',f2e:'http://foo.com:123/context/resource/f2e/x/y',f2f:'http://foo.com:123/context/resource',f2g:'http://foo.com:123/context/resource/f2g/x',f2h:'http://foo.com:123/context/resource/f2h',f2i:'http://foo.com:123/context/resource',f2j:'http://foo.com:123/context',f2k:'http://foo.com:123/context/f2j/x',f2l:'http://foo.com:123/context/f2k',f2m:'http://foo.com:123/context',f2n:'http://foo.com:123',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'http://foo.com:123/context/resource/%3C%3E%26%27%22',f3c:'http://foo.com:123/context/resource/<>&\\'\"',f4:'http://foo.com:123/context/resource/test/uri/b',f5:'http://foo.com:123/context/resource/f5/x'}",
					"<object f0='http://foo.com:123/context/resource/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>http://foo.com:123/f1b</f1b><f1c>http://foo.com:123/f1c/x/y</f1c><f1d>http://foo.com:123/context/resource/f1d</f1d><f1e>http://foo.com:123/context/resource/f1e/x/y</f1e><f1f>http://foo.com:123/context/resource</f1f><f1g>http://foo.com:123/context/resource/f1g/x</f1g><f1h>http://foo.com:123/context/resource/f1h</f1h><f1i>http://foo.com:123/context/resource</f1i><f1j>http://foo.com:123/context</f1j><f1k>http://foo.com:123/context/f1j/x</f1k><f1l>http://foo.com:123/context/f1k</f1l><f1m>http://foo.com:123/context</f1m><f1n>http://foo.com:123</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>http://foo.com:123/f2b</f2b><f2c>http://foo.com:123/f2c/x/y</f2c><f2d>http://foo.com:123/context/resource/f2d</f2d><f2e>http://foo.com:123/context/resource/f2e/x/y</f2e><f2f>http://foo.com:123/context/resource</f2f><f2g>http://foo.com:123/context/resource/f2g/x</f2g><f2h>http://foo.com:123/context/resource/f2h</f2h><f2i>http://foo.com:123/context/resource</f2i><f2j>http://foo.com:123/context</f2j><f2k>http://foo.com:123/context/f2j/x</f2k><f2l>http://foo.com:123/context/f2k</f2l><f2m>http://foo.com:123/context</f2m><f2n>http://foo.com:123</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>http://foo.com:123/context/resource/%3C%3E%26%27%22</f3b><f3c>http://foo.com:123/context/resource/&lt;&gt;&amp;'\"</f3c><f4>http://foo.com:123/context/resource/test/uri/b</f4><f5>http://foo.com:123/context/resource/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='http://foo.com:123/context/resource/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='http://foo.com:123/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='http://foo.com:123/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='http://foo.com:123/context/resource/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='http://foo.com:123/context/resource/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='http://foo.com:123/context/resource'>resource</a></td></tr><tr><td>f1g</td><td><a href='http://foo.com:123/context/resource/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='http://foo.com:123/context/resource/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='http://foo.com:123/context/resource'>resource</a></td></tr><tr><td>f1j</td><td><a href='http://foo.com:123/context'>context</a></td></tr><tr><td>f1k</td><td><a href='http://foo.com:123/context/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='http://foo.com:123/context/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='http://foo.com:123/context'>context</a></td></tr><tr><td>f1n</td><td><a href='http://foo.com:123'>foo.com:123</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='http://foo.com:123/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='http://foo.com:123/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='http://foo.com:123/context/resource/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='http://foo.com:123/context/resource/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='http://foo.com:123/context/resource'>resource</a></td></tr><tr><td>f2g</td><td><a href='http://foo.com:123/context/resource/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='http://foo.com:123/context/resource/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='http://foo.com:123/context/resource'>resource</a></td></tr><tr><td>f2j</td><td><a href='http://foo.com:123/context'>context</a></td></tr><tr><td>f2k</td><td><a href='http://foo.com:123/context/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='http://foo.com:123/context/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='http://foo.com:123/context'>context</a></td></tr><tr><td>f2n</td><td><a href='http://foo.com:123'>foo.com:123</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='http://foo.com:123/context/resource/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='http://foo.com:123/context/resource/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='http://foo.com:123/context/resource/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='http://foo.com:123/context/resource/f5/x'>x</a></td></tr></table>",
					"(f0=http://foo.com:123/context/resource/f0/x0,f1a=http://www.apache.org/f1a,f1b=http://foo.com:123/f1b,f1c=http://foo.com:123/f1c/x/y,f1d=http://foo.com:123/context/resource/f1d,f1e=http://foo.com:123/context/resource/f1e/x/y,f1f=http://foo.com:123/context/resource,f1g=http://foo.com:123/context/resource/f1g/x,f1h=http://foo.com:123/context/resource/f1h,f1i=http://foo.com:123/context/resource,f1j=http://foo.com:123/context,f1k=http://foo.com:123/context/f1j/x,f1l=http://foo.com:123/context/f1k,f1m=http://foo.com:123/context,f1n=http://foo.com:123,f2a=http://www.apache.org/f2a,f2b=http://foo.com:123/f2b,f2c=http://foo.com:123/f2c/x/y,f2d=http://foo.com:123/context/resource/f2d,f2e=http://foo.com:123/context/resource/f2e/x/y,f2f=http://foo.com:123/context/resource,f2g=http://foo.com:123/context/resource/f2g/x,f2h=http://foo.com:123/context/resource/f2h,f2i=http://foo.com:123/context/resource,f2j=http://foo.com:123/context,f2k=http://foo.com:123/context/f2j/x,f2l=http://foo.com:123/context/f2k,f2m=http://foo.com:123/context,f2n=http://foo.com:123,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=http://foo.com:123/context/resource/%3C%3E%26%27%22,f3c=http://foo.com:123/context/resource/<>&~'\",f4=http://foo.com:123/context/resource/test/uri/b,f5=http://foo.com:123/context/resource/f5/x)",
					"f0=http://foo.com:123/context/resource/f0/x0&f1a=http://www.apache.org/f1a&f1b=http://foo.com:123/f1b&f1c=http://foo.com:123/f1c/x/y&f1d=http://foo.com:123/context/resource/f1d&f1e=http://foo.com:123/context/resource/f1e/x/y&f1f=http://foo.com:123/context/resource&f1g=http://foo.com:123/context/resource/f1g/x&f1h=http://foo.com:123/context/resource/f1h&f1i=http://foo.com:123/context/resource&f1j=http://foo.com:123/context&f1k=http://foo.com:123/context/f1j/x&f1l=http://foo.com:123/context/f1k&f1m=http://foo.com:123/context&f1n=http://foo.com:123&f2a=http://www.apache.org/f2a&f2b=http://foo.com:123/f2b&f2c=http://foo.com:123/f2c/x/y&f2d=http://foo.com:123/context/resource/f2d&f2e=http://foo.com:123/context/resource/f2e/x/y&f2f=http://foo.com:123/context/resource&f2g=http://foo.com:123/context/resource/f2g/x&f2h=http://foo.com:123/context/resource/f2h&f2i=http://foo.com:123/context/resource&f2j=http://foo.com:123/context&f2k=http://foo.com:123/context/f2j/x&f2l=http://foo.com:123/context/f2k&f2m=http://foo.com:123/context&f2n=http://foo.com:123&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=http://foo.com:123/context/resource/%253C%253E%2526%2527%2522&f3c=http://foo.com:123/context/resource/%3C%3E%26~'%22&f4=http://foo.com:123/context/resource/test/uri/b&f5=http://foo.com:123/context/resource/f5/x",
					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
					"<rdf:RDF>\n<rdf:Description rdf:about='http://foo.com:123/context/resource/f0/x0'>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f1g rdf:resource='http://foo.com:123/context/resource/f1g/x'/>\n<jp:f2m rdf:resource='http://foo.com:123/context'/>\n<jp:f2i rdf:resource='http://foo.com:123/context/resource'/>\n<jp:f5 rdf:resource='http://foo.com:123/context/resource/f5/x'/>\n<jp:f1k rdf:resource='http://foo.com:123/context/f1j/x'/>\n<jp:f2f rdf:resource='http://foo.com:123/context/resource'/>\n<jp:f4 rdf:resource='http://foo.com:123/context/resource/test/uri/b'/>\n<jp:f1f rdf:resource='http://foo.com:123/context/resource'/>\n<jp:f1b rdf:resource='http://foo.com:123/f1b'/>\n<jp:f1c rdf:resource='http://foo.com:123/f1c/x/y'/>\n<jp:f1l rdf:resource='http://foo.com:123/context/f1k'/>\n<jp:f2d rdf:resource='http://foo.com:123/context/resource/f2d'/>\n<jp:f1e rdf:resource='http://foo.com:123/context/resource/f1e/x/y'/>\n<jp:f2k rdf:resource='http://foo.com:123/context/f2j/x'/>\n<jp:f2e rdf:resource='http://foo.com:123/context/resource/f2e/x/y'/>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f2j rdf:resource='http://foo.com:123/context'/>\n<jp:f2n rdf:resource='http://foo.com:123'/>\n<jp:f2b rdf:resource='http://foo.com:123/f2b'/>\n<jp:f2h rdf:resource='http://foo.com:123/context/resource/f2h'/>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f1h rdf:resource='http://foo.com:123/context/resource/f1h'/>\n<jp:f1i rdf:resource='http://foo.com:123/context/resource'/>\n<jp:f1j rdf:resource='http://foo.com:123/context'/>\n<jp:f2g rdf:resource='http://foo.com:123/context/resource/f2g/x'/>\n<jp:f1n rdf:resource='http://foo.com:123'/>\n<jp:f1d rdf:resource='http://foo.com:123/context/resource/f1d'/>\n<jp:f3c rdf:resource='http://foo.com:123/context/resource/&lt;&gt;&amp;&apos;&quot;'/>\n<jp:f2l rdf:resource='http://foo.com:123/context/f2k'/>\n<jp:f2c rdf:resource='http://foo.com:123/f2c/x/y'/>\n<jp:f3b rdf:resource='http://foo.com:123/context/resource/%3C%3E%26%27%22'/>\n<jp:f1m rdf:resource='http://foo.com:123/context'/>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 16 */
				"NoPath-1-ROOT_RELATIVE",
				input(
					UriResolution.ROOT_RELATIVE, UriRelativity.RESOURCE, "http://foo.com:123","/context","/resource",""
				),
				results(
					"{f0:'/context/resource/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'/context/resource/f1d',f1e:'/context/resource/f1e/x/y',f1f:'/context/resource',f1g:'/context/resource/f1g/x',f1h:'/context/resource/f1h',f1i:'/context/resource',f1j:'/context',f1k:'/context/f1j/x',f1l:'/context/f1k',f1m:'/context',f1n:'/',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'/context/resource/f2d',f2e:'/context/resource/f2e/x/y',f2f:'/context/resource',f2g:'/context/resource/f2g/x',f2h:'/context/resource/f2h',f2i:'/context/resource',f2j:'/context',f2k:'/context/f2j/x',f2l:'/context/f2k',f2m:'/context',f2n:'/',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'/context/resource/%3C%3E%26%27%22',f3c:'/context/resource/<>&\\'\"',f4:'/context/resource/test/uri/b',f5:'/context/resource/f5/x'}",
					"<object f0='/context/resource/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>/context/resource/f1d</f1d><f1e>/context/resource/f1e/x/y</f1e><f1f>/context/resource</f1f><f1g>/context/resource/f1g/x</f1g><f1h>/context/resource/f1h</f1h><f1i>/context/resource</f1i><f1j>/context</f1j><f1k>/context/f1j/x</f1k><f1l>/context/f1k</f1l><f1m>/context</f1m><f1n>/</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>/context/resource/f2d</f2d><f2e>/context/resource/f2e/x/y</f2e><f2f>/context/resource</f2f><f2g>/context/resource/f2g/x</f2g><f2h>/context/resource/f2h</f2h><f2i>/context/resource</f2i><f2j>/context</f2j><f2k>/context/f2j/x</f2k><f2l>/context/f2k</f2l><f2m>/context</f2m><f2n>/</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>/context/resource/%3C%3E%26%27%22</f3b><f3c>/context/resource/&lt;&gt;&amp;'\"</f3c><f4>/context/resource/test/uri/b</f4><f5>/context/resource/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='/context/resource/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='/context/resource/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='/context/resource/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f1g</td><td><a href='/context/resource/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/context/resource/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f1j</td><td><a href='/context'>context</a></td></tr><tr><td>f1k</td><td><a href='/context/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/context/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/context'>context</a></td></tr><tr><td>f1n</td><td><a href='/'>/</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='/context/resource/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='/context/resource/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f2g</td><td><a href='/context/resource/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/context/resource/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f2j</td><td><a href='/context'>context</a></td></tr><tr><td>f2k</td><td><a href='/context/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/context/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/context'>context</a></td></tr><tr><td>f2n</td><td><a href='/'>/</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='/context/resource/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='/context/resource/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='/context/resource/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='/context/resource/f5/x'>x</a></td></tr></table>",
					"(f0=/context/resource/f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=/context/resource/f1d,f1e=/context/resource/f1e/x/y,f1f=/context/resource,f1g=/context/resource/f1g/x,f1h=/context/resource/f1h,f1i=/context/resource,f1j=/context,f1k=/context/f1j/x,f1l=/context/f1k,f1m=/context,f1n=/,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=/context/resource/f2d,f2e=/context/resource/f2e/x/y,f2f=/context/resource,f2g=/context/resource/f2g/x,f2h=/context/resource/f2h,f2i=/context/resource,f2j=/context,f2k=/context/f2j/x,f2l=/context/f2k,f2m=/context,f2n=/,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=/context/resource/%3C%3E%26%27%22,f3c=/context/resource/<>&~'\",f4=/context/resource/test/uri/b,f5=/context/resource/f5/x)",
					"f0=/context/resource/f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=/context/resource/f1d&f1e=/context/resource/f1e/x/y&f1f=/context/resource&f1g=/context/resource/f1g/x&f1h=/context/resource/f1h&f1i=/context/resource&f1j=/context&f1k=/context/f1j/x&f1l=/context/f1k&f1m=/context&f1n=/&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=/context/resource/f2d&f2e=/context/resource/f2e/x/y&f2f=/context/resource&f2g=/context/resource/f2g/x&f2h=/context/resource/f2h&f2i=/context/resource&f2j=/context&f2k=/context/f2j/x&f2l=/context/f2k&f2m=/context&f2n=/&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=/context/resource/%253C%253E%2526%2527%2522&f3c=/context/resource/%3C%3E%26~'%22&f4=/context/resource/test/uri/b&f5=/context/resource/f5/x",
					"DE0022A26630B72F636F6E746578742F7265736F757263652F66302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164B52F636F6E746578742F7265736F757263652F663164A3663165B92F636F6E746578742F7265736F757263652F6631652F782F79A3663166B12F636F6E746578742F7265736F75726365A3663167B72F636F6E746578742F7265736F757263652F6631672F78A3663168B52F636F6E746578742F7265736F757263652F663168A3663169B12F636F6E746578742F7265736F75726365A366316AA82F636F6E74657874A366316BAE2F636F6E746578742F66316A2F78A366316CAC2F636F6E746578742F66316BA366316DA82F636F6E74657874A366316EA12FA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264B52F636F6E746578742F7265736F757263652F663264A3663265B92F636F6E746578742F7265736F757263652F6632652F782F79A3663266B12F636F6E746578742F7265736F75726365A3663267B72F636F6E746578742F7265736F757263652F6632672F78A3663268B52F636F6E746578742F7265736F757263652F663268A3663269B12F636F6E746578742F7265736F75726365A366326AA82F636F6E74657874A366326BAE2F636F6E746578742F66326A2F78A366326CAC2F636F6E746578742F66326BA366326DA82F636F6E74657874A366326EA12FA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362D9212F636F6E746578742F7265736F757263652F253343253345253236253237253232A3663363B72F636F6E746578742F7265736F757263652F3C3E262722A26634BC2F636F6E746578742F7265736F757263652F746573742F7572692F62A26635B62F636F6E746578742F7265736F757263652F66352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='/context/resource/f0/x0'>\n<jp:f2h>/context/resource/f2h</jp:f2h>\n<jp:f2f>/context/resource</jp:f2f>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1h>/context/resource/f1h</jp:f1h>\n<jp:f5>/context/resource/f5/x</jp:f5>\n<jp:f1e>/context/resource/f1e/x/y</jp:f1e>\n<jp:f3c>/context/resource/&lt;&gt;&amp;''</jp:f3c>\n<jp:f2k>/context/f2j/x</jp:f2k>\n<jp:f1j>/context</jp:f1j>\n<jp:f2m>/context</jp:f2m>\n<jp:f2i>/context/resource</jp:f2i>\n<jp:f1d>/context/resource/f1d</jp:f1d>\n<jp:f4>/context/resource/test/uri/b</jp:f4>\n<jp:f1n>/</jp:f1n>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f1k>/context/f1j/x</jp:f1k>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f2e>/context/resource/f2e/x/y</jp:f2e>\n<jp:f3b>/context/resource/%3C%3E%26%27%22</jp:f3b>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f1i>/context/resource</jp:f1i>\n<jp:f1g>/context/resource/f1g/x</jp:f1g>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f1m>/context</jp:f1m>\n<jp:f2d>/context/resource/f2d</jp:f2d>\n<jp:f2g>/context/resource/f2g/x</jp:f2g>\n<jp:f2n>/</jp:f2n>\n<jp:f2j>/context</jp:f2j>\n<jp:f1l>/context/f1k</jp:f1l>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f1f>/context/resource</jp:f1f>\n<jp:f2l>/context/f2k</jp:f2l>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 17 */
				"NoPath-1-NONE",
				input(
					UriResolution.NONE, UriRelativity.RESOURCE, "http://foo.com:123","/context","/resource",""
				),
				results(
					"{f0:'f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'f1d',f1e:'f1e/x/y',f1f:'',f1g:'/context/resource/f1g/x',f1h:'/context/resource/f1h',f1i:'/context/resource',f1j:'/context',f1k:'/context/f1j/x',f1l:'/context/f1k',f1m:'/context',f1n:'/',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'f2d',f2e:'f2e/x/y',f2f:'',f2g:'/context/resource/f2g/x',f2h:'/context/resource/f2h',f2i:'/context/resource',f2j:'/context',f2k:'/context/f2j/x',f2l:'/context/f2k',f2m:'/context',f2n:'/',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'%3C%3E%26%27%22',f3c:'<>&\\'\"',f4:'test/uri/b',f5:'f5/x'}",
					"<object f0='f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>f1d</f1d><f1e>f1e/x/y</f1e><f1f>_xE000_</f1f><f1g>/context/resource/f1g/x</f1g><f1h>/context/resource/f1h</f1h><f1i>/context/resource</f1i><f1j>/context</f1j><f1k>/context/f1j/x</f1k><f1l>/context/f1k</f1l><f1m>/context</f1m><f1n>/</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>f2d</f2d><f2e>f2e/x/y</f2e><f2f>_xE000_</f2f><f2g>/context/resource/f2g/x</f2g><f2h>/context/resource/f2h</f2h><f2i>/context/resource</f2i><f2j>/context</f2j><f2k>/context/f2j/x</f2k><f2l>/context/f2k</f2l><f2m>/context</f2m><f2n>/</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>%3C%3E%26%27%22</f3b><f3c>&lt;&gt;&amp;'\"</f3c><f4>test/uri/b</f4><f5>f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href=''>/</a></td></tr><tr><td>f1g</td><td><a href='/context/resource/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/context/resource/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f1j</td><td><a href='/context'>context</a></td></tr><tr><td>f1k</td><td><a href='/context/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/context/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/context'>context</a></td></tr><tr><td>f1n</td><td><a href='/'>/</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href=''>/</a></td></tr><tr><td>f2g</td><td><a href='/context/resource/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/context/resource/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/context/resource'>resource</a></td></tr><tr><td>f2j</td><td><a href='/context'>context</a></td></tr><tr><td>f2k</td><td><a href='/context/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/context/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/context'>context</a></td></tr><tr><td>f2n</td><td><a href='/'>/</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='f5/x'>x</a></td></tr></table>",
					"(f0=f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=f1d,f1e=f1e/x/y,f1f='',f1g=/context/resource/f1g/x,f1h=/context/resource/f1h,f1i=/context/resource,f1j=/context,f1k=/context/f1j/x,f1l=/context/f1k,f1m=/context,f1n=/,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=f2d,f2e=f2e/x/y,f2f='',f2g=/context/resource/f2g/x,f2h=/context/resource/f2h,f2i=/context/resource,f2j=/context,f2k=/context/f2j/x,f2l=/context/f2k,f2m=/context,f2n=/,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=%3C%3E%26%27%22,f3c=<>&~'\",f4=test/uri/b,f5=f5/x)",
					"f0=f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=f1d&f1e=f1e/x/y&f1f=''&f1g=/context/resource/f1g/x&f1h=/context/resource/f1h&f1i=/context/resource&f1j=/context&f1k=/context/f1j/x&f1l=/context/f1k&f1m=/context&f1n=/&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=f2d&f2e=f2e/x/y&f2f=''&f2g=/context/resource/f2g/x&f2h=/context/resource/f2h&f2i=/context/resource&f2j=/context&f2k=/context/f2j/x&f2l=/context/f2k&f2m=/context&f2n=/&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=%253C%253E%2526%2527%2522&f3c=%3C%3E%26~'%22&f4=test/uri/b&f5=f5/x",
					"DE0022A26630A566302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164A3663164A3663165A76631652F782F79A3663166A0A3663167B72F636F6E746578742F7265736F757263652F6631672F78A3663168B52F636F6E746578742F7265736F757263652F663168A3663169B12F636F6E746578742F7265736F75726365A366316AA82F636F6E74657874A366316BAE2F636F6E746578742F66316A2F78A366316CAC2F636F6E746578742F66316BA366316DA82F636F6E74657874A366316EA12FA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264A3663264A3663265A76632652F782F79A3663266A0A3663267B72F636F6E746578742F7265736F757263652F6632672F78A3663268B52F636F6E746578742F7265736F757263652F663268A3663269B12F636F6E746578742F7265736F75726365A366326AA82F636F6E74657874A366326BAE2F636F6E746578742F66326A2F78A366326CAC2F636F6E746578742F66326BA366326DA82F636F6E74657874A366326EA12FA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362AF253343253345253236253237253232A3663363A53C3E262722A26634AA746573742F7572692F62A26635A466352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='f0/x0'>\n<jp:f2l>/context/f2k</jp:f2l>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f2g>/context/resource/f2g/x</jp:f2g>\n<jp:f2f></jp:f2f>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f3b>%3C%3E%26%27%22</jp:f3b>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f5>f5/x</jp:f5>\n<jp:f2k>/context/f2j/x</jp:f2k>\n<jp:f1e>f1e/x/y</jp:f1e>\n<jp:f2i>/context/resource</jp:f2i>\n<jp:f2n>/</jp:f2n>\n<jp:f2m>/context</jp:f2m>\n<jp:f1m>/context</jp:f1m>\n<jp:f2d>f2d</jp:f2d>\n<jp:f1l>/context/f1k</jp:f1l>\n<jp:f3c>&lt;&gt;&amp;''</jp:f3c>\n<jp:f1n>/</jp:f1n>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f1k>/context/f1j/x</jp:f1k>\n<jp:f1h>/context/resource/f1h</jp:f1h>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f2h>/context/resource/f2h</jp:f2h>\n<jp:f1d>f1d</jp:f1d>\n<jp:f1i>/context/resource</jp:f1i>\n<jp:f2e>f2e/x/y</jp:f2e>\n<jp:f2j>/context</jp:f2j>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1j>/context</jp:f1j>\n<jp:f1f></jp:f1f>\n<jp:f1g>/context/resource/f1g/x</jp:f1g>\n<jp:f4>test/uri/b</jp:f4>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 18 */
				"NoAuthorityNoContext-1-ABSOLUTE",
				input(
					UriResolution.ABSOLUTE, UriRelativity.RESOURCE, "","","/resource","/path"
				),
				results(
					"{f0:'/resource/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'/resource/f1d',f1e:'/resource/f1e/x/y',f1f:'/resource',f1g:'/resource/f1g/x',f1h:'/resource/f1h',f1i:'/resource',f1j:'/',f1k:'/f1j/x',f1l:'/f1k',f1m:'/',f1n:'/..',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'/resource/f2d',f2e:'/resource/f2e/x/y',f2f:'/resource',f2g:'/resource/f2g/x',f2h:'/resource/f2h',f2i:'/resource',f2j:'/',f2k:'/f2j/x',f2l:'/f2k',f2m:'/',f2n:'/..',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'/resource/%3C%3E%26%27%22',f3c:'/resource/<>&\\'\"',f4:'/resource/test/uri/b',f5:'/resource/f5/x'}",
					"<object f0='/resource/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>/resource/f1d</f1d><f1e>/resource/f1e/x/y</f1e><f1f>/resource</f1f><f1g>/resource/f1g/x</f1g><f1h>/resource/f1h</f1h><f1i>/resource</f1i><f1j>/</f1j><f1k>/f1j/x</f1k><f1l>/f1k</f1l><f1m>/</f1m><f1n>/..</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>/resource/f2d</f2d><f2e>/resource/f2e/x/y</f2e><f2f>/resource</f2f><f2g>/resource/f2g/x</f2g><f2h>/resource/f2h</f2h><f2i>/resource</f2i><f2j>/</f2j><f2k>/f2j/x</f2k><f2l>/f2k</f2l><f2m>/</f2m><f2n>/..</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>/resource/%3C%3E%26%27%22</f3b><f3c>/resource/&lt;&gt;&amp;'\"</f3c><f4>/resource/test/uri/b</f4><f5>/resource/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='/resource/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='/resource/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='/resource/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='/resource'>resource</a></td></tr><tr><td>f1g</td><td><a href='/resource/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/resource/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/resource'>resource</a></td></tr><tr><td>f1j</td><td><a href='/'>/</a></td></tr><tr><td>f1k</td><td><a href='/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/'>/</a></td></tr><tr><td>f1n</td><td><a href='/..'>..</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='/resource/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='/resource/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='/resource'>resource</a></td></tr><tr><td>f2g</td><td><a href='/resource/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/resource/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/resource'>resource</a></td></tr><tr><td>f2j</td><td><a href='/'>/</a></td></tr><tr><td>f2k</td><td><a href='/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/'>/</a></td></tr><tr><td>f2n</td><td><a href='/..'>..</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='/resource/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='/resource/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='/resource/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='/resource/f5/x'>x</a></td></tr></table>",
					"(f0=/resource/f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=/resource/f1d,f1e=/resource/f1e/x/y,f1f=/resource,f1g=/resource/f1g/x,f1h=/resource/f1h,f1i=/resource,f1j=/,f1k=/f1j/x,f1l=/f1k,f1m=/,f1n=/..,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=/resource/f2d,f2e=/resource/f2e/x/y,f2f=/resource,f2g=/resource/f2g/x,f2h=/resource/f2h,f2i=/resource,f2j=/,f2k=/f2j/x,f2l=/f2k,f2m=/,f2n=/..,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=/resource/%3C%3E%26%27%22,f3c=/resource/<>&~'\",f4=/resource/test/uri/b,f5=/resource/f5/x)",
					"f0=/resource/f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=/resource/f1d&f1e=/resource/f1e/x/y&f1f=/resource&f1g=/resource/f1g/x&f1h=/resource/f1h&f1i=/resource&f1j=/&f1k=/f1j/x&f1l=/f1k&f1m=/&f1n=/..&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=/resource/f2d&f2e=/resource/f2e/x/y&f2f=/resource&f2g=/resource/f2g/x&f2h=/resource/f2h&f2i=/resource&f2j=/&f2k=/f2j/x&f2l=/f2k&f2m=/&f2n=/..&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=/resource/%253C%253E%2526%2527%2522&f3c=/resource/%3C%3E%26~'%22&f4=/resource/test/uri/b&f5=/resource/f5/x",
					"DE0022A26630AF2F7265736F757263652F66302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164AD2F7265736F757263652F663164A3663165B12F7265736F757263652F6631652F782F79A3663166A92F7265736F75726365A3663167AF2F7265736F757263652F6631672F78A3663168AD2F7265736F757263652F663168A3663169A92F7265736F75726365A366316AA12FA366316BA62F66316A2F78A366316CA42F66316BA366316DA12FA366316EA32F2E2EA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264AD2F7265736F757263652F663264A3663265B12F7265736F757263652F6632652F782F79A3663266A92F7265736F75726365A3663267AF2F7265736F757263652F6632672F78A3663268AD2F7265736F757263652F663268A3663269A92F7265736F75726365A366326AA12FA366326BA62F66326A2F78A366326CA42F66326BA366326DA12FA366326EA32F2E2EA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362B92F7265736F757263652F253343253345253236253237253232A3663363AF2F7265736F757263652F3C3E262722A26634B42F7265736F757263652F746573742F7572692F62A26635AE2F7265736F757263652F66352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='/resource/f0/x0'>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f2e>/resource/f2e/x/y</jp:f2e>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f2n>/..</jp:f2n>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f3c>/resource/&lt;&gt;&amp;''</jp:f3c>\n<jp:f2h>/resource/f2h</jp:f2h>\n<jp:f1i>/resource</jp:f1i>\n<jp:f1n>/..</jp:f1n>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1e>/resource/f1e/x/y</jp:f1e>\n<jp:f5>/resource/f5/x</jp:f5>\n<jp:f2l>/f2k</jp:f2l>\n<jp:f2d>/resource/f2d</jp:f2d>\n<jp:f2j>/</jp:f2j>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f3b>/resource/%3C%3E%26%27%22</jp:f3b>\n<jp:f1j>/</jp:f1j>\n<jp:f1d>/resource/f1d</jp:f1d>\n<jp:f2m>/</jp:f2m>\n<jp:f1f>/resource</jp:f1f>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f4>/resource/test/uri/b</jp:f4>\n<jp:f2i>/resource</jp:f2i>\n<jp:f1h>/resource/f1h</jp:f1h>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f1k>/f1j/x</jp:f1k>\n<jp:f1l>/f1k</jp:f1l>\n<jp:f2g>/resource/f2g/x</jp:f2g>\n<jp:f2k>/f2j/x</jp:f2k>\n<jp:f2f>/resource</jp:f2f>\n<jp:f1g>/resource/f1g/x</jp:f1g>\n<jp:f1m>/</jp:f1m>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 19 */
				"NoAuthorityNoContext-1-ROOT_RELATIVE",
				input(
					UriResolution.ROOT_RELATIVE, UriRelativity.RESOURCE, "","","/resource","/path"
				),
				results(
					"{f0:'/resource/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'/resource/f1d',f1e:'/resource/f1e/x/y',f1f:'/resource',f1g:'/resource/f1g/x',f1h:'/resource/f1h',f1i:'/resource',f1j:'/',f1k:'/f1j/x',f1l:'/f1k',f1m:'/',f1n:'/..',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'/resource/f2d',f2e:'/resource/f2e/x/y',f2f:'/resource',f2g:'/resource/f2g/x',f2h:'/resource/f2h',f2i:'/resource',f2j:'/',f2k:'/f2j/x',f2l:'/f2k',f2m:'/',f2n:'/..',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'/resource/%3C%3E%26%27%22',f3c:'/resource/<>&\\'\"',f4:'/resource/test/uri/b',f5:'/resource/f5/x'}",
					"<object f0='/resource/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>/resource/f1d</f1d><f1e>/resource/f1e/x/y</f1e><f1f>/resource</f1f><f1g>/resource/f1g/x</f1g><f1h>/resource/f1h</f1h><f1i>/resource</f1i><f1j>/</f1j><f1k>/f1j/x</f1k><f1l>/f1k</f1l><f1m>/</f1m><f1n>/..</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>/resource/f2d</f2d><f2e>/resource/f2e/x/y</f2e><f2f>/resource</f2f><f2g>/resource/f2g/x</f2g><f2h>/resource/f2h</f2h><f2i>/resource</f2i><f2j>/</f2j><f2k>/f2j/x</f2k><f2l>/f2k</f2l><f2m>/</f2m><f2n>/..</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>/resource/%3C%3E%26%27%22</f3b><f3c>/resource/&lt;&gt;&amp;'\"</f3c><f4>/resource/test/uri/b</f4><f5>/resource/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='/resource/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='/resource/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='/resource/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='/resource'>resource</a></td></tr><tr><td>f1g</td><td><a href='/resource/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/resource/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/resource'>resource</a></td></tr><tr><td>f1j</td><td><a href='/'>/</a></td></tr><tr><td>f1k</td><td><a href='/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/'>/</a></td></tr><tr><td>f1n</td><td><a href='/..'>..</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='/resource/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='/resource/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='/resource'>resource</a></td></tr><tr><td>f2g</td><td><a href='/resource/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/resource/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/resource'>resource</a></td></tr><tr><td>f2j</td><td><a href='/'>/</a></td></tr><tr><td>f2k</td><td><a href='/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/'>/</a></td></tr><tr><td>f2n</td><td><a href='/..'>..</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='/resource/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='/resource/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='/resource/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='/resource/f5/x'>x</a></td></tr></table>",
					"(f0=/resource/f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=/resource/f1d,f1e=/resource/f1e/x/y,f1f=/resource,f1g=/resource/f1g/x,f1h=/resource/f1h,f1i=/resource,f1j=/,f1k=/f1j/x,f1l=/f1k,f1m=/,f1n=/..,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=/resource/f2d,f2e=/resource/f2e/x/y,f2f=/resource,f2g=/resource/f2g/x,f2h=/resource/f2h,f2i=/resource,f2j=/,f2k=/f2j/x,f2l=/f2k,f2m=/,f2n=/..,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=/resource/%3C%3E%26%27%22,f3c=/resource/<>&~'\",f4=/resource/test/uri/b,f5=/resource/f5/x)",
					"f0=/resource/f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=/resource/f1d&f1e=/resource/f1e/x/y&f1f=/resource&f1g=/resource/f1g/x&f1h=/resource/f1h&f1i=/resource&f1j=/&f1k=/f1j/x&f1l=/f1k&f1m=/&f1n=/..&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=/resource/f2d&f2e=/resource/f2e/x/y&f2f=/resource&f2g=/resource/f2g/x&f2h=/resource/f2h&f2i=/resource&f2j=/&f2k=/f2j/x&f2l=/f2k&f2m=/&f2n=/..&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=/resource/%253C%253E%2526%2527%2522&f3c=/resource/%3C%3E%26~'%22&f4=/resource/test/uri/b&f5=/resource/f5/x",
					"DE0022A26630AF2F7265736F757263652F66302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164AD2F7265736F757263652F663164A3663165B12F7265736F757263652F6631652F782F79A3663166A92F7265736F75726365A3663167AF2F7265736F757263652F6631672F78A3663168AD2F7265736F757263652F663168A3663169A92F7265736F75726365A366316AA12FA366316BA62F66316A2F78A366316CA42F66316BA366316DA12FA366316EA32F2E2EA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264AD2F7265736F757263652F663264A3663265B12F7265736F757263652F6632652F782F79A3663266A92F7265736F75726365A3663267AF2F7265736F757263652F6632672F78A3663268AD2F7265736F757263652F663268A3663269A92F7265736F75726365A366326AA12FA366326BA62F66326A2F78A366326CA42F66326BA366326DA12FA366326EA32F2E2EA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362B92F7265736F757263652F253343253345253236253237253232A3663363AF2F7265736F757263652F3C3E262722A26634B42F7265736F757263652F746573742F7572692F62A26635AE2F7265736F757263652F66352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='/resource/f0/x0'>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f2e>/resource/f2e/x/y</jp:f2e>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f2n>/..</jp:f2n>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f3c>/resource/&lt;&gt;&amp;''</jp:f3c>\n<jp:f2h>/resource/f2h</jp:f2h>\n<jp:f1i>/resource</jp:f1i>\n<jp:f1n>/..</jp:f1n>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1e>/resource/f1e/x/y</jp:f1e>\n<jp:f5>/resource/f5/x</jp:f5>\n<jp:f2l>/f2k</jp:f2l>\n<jp:f2d>/resource/f2d</jp:f2d>\n<jp:f2j>/</jp:f2j>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f3b>/resource/%3C%3E%26%27%22</jp:f3b>\n<jp:f1j>/</jp:f1j>\n<jp:f1d>/resource/f1d</jp:f1d>\n<jp:f2m>/</jp:f2m>\n<jp:f1f>/resource</jp:f1f>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f4>/resource/test/uri/b</jp:f4>\n<jp:f2i>/resource</jp:f2i>\n<jp:f1h>/resource/f1h</jp:f1h>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f1k>/f1j/x</jp:f1k>\n<jp:f1l>/f1k</jp:f1l>\n<jp:f2g>/resource/f2g/x</jp:f2g>\n<jp:f2k>/f2j/x</jp:f2k>\n<jp:f2f>/resource</jp:f2f>\n<jp:f1g>/resource/f1g/x</jp:f1g>\n<jp:f1m>/</jp:f1m>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 20 */
				"NoAuthorityNoContext-1-NONE",
				input(
					UriResolution.NONE, UriRelativity.RESOURCE, "","","/resource","/path"
				),
				results(
					"{f0:'f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'f1d',f1e:'f1e/x/y',f1f:'',f1g:'/resource/f1g/x',f1h:'/resource/f1h',f1i:'/resource',f1j:'/',f1k:'/f1j/x',f1l:'/f1k',f1m:'/',f1n:'/..',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'f2d',f2e:'f2e/x/y',f2f:'',f2g:'/resource/f2g/x',f2h:'/resource/f2h',f2i:'/resource',f2j:'/',f2k:'/f2j/x',f2l:'/f2k',f2m:'/',f2n:'/..',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'%3C%3E%26%27%22',f3c:'<>&\\'\"',f4:'test/uri/b',f5:'f5/x'}",
					"<object f0='f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>f1d</f1d><f1e>f1e/x/y</f1e><f1f>_xE000_</f1f><f1g>/resource/f1g/x</f1g><f1h>/resource/f1h</f1h><f1i>/resource</f1i><f1j>/</f1j><f1k>/f1j/x</f1k><f1l>/f1k</f1l><f1m>/</f1m><f1n>/..</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>f2d</f2d><f2e>f2e/x/y</f2e><f2f>_xE000_</f2f><f2g>/resource/f2g/x</f2g><f2h>/resource/f2h</f2h><f2i>/resource</f2i><f2j>/</f2j><f2k>/f2j/x</f2k><f2l>/f2k</f2l><f2m>/</f2m><f2n>/..</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>%3C%3E%26%27%22</f3b><f3c>&lt;&gt;&amp;'\"</f3c><f4>test/uri/b</f4><f5>f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href=''>/</a></td></tr><tr><td>f1g</td><td><a href='/resource/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/resource/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/resource'>resource</a></td></tr><tr><td>f1j</td><td><a href='/'>/</a></td></tr><tr><td>f1k</td><td><a href='/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/'>/</a></td></tr><tr><td>f1n</td><td><a href='/..'>..</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href=''>/</a></td></tr><tr><td>f2g</td><td><a href='/resource/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/resource/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/resource'>resource</a></td></tr><tr><td>f2j</td><td><a href='/'>/</a></td></tr><tr><td>f2k</td><td><a href='/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/'>/</a></td></tr><tr><td>f2n</td><td><a href='/..'>..</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='f5/x'>x</a></td></tr></table>",
					"(f0=f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=f1d,f1e=f1e/x/y,f1f='',f1g=/resource/f1g/x,f1h=/resource/f1h,f1i=/resource,f1j=/,f1k=/f1j/x,f1l=/f1k,f1m=/,f1n=/..,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=f2d,f2e=f2e/x/y,f2f='',f2g=/resource/f2g/x,f2h=/resource/f2h,f2i=/resource,f2j=/,f2k=/f2j/x,f2l=/f2k,f2m=/,f2n=/..,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=%3C%3E%26%27%22,f3c=<>&~'\",f4=test/uri/b,f5=f5/x)",
					"f0=f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=f1d&f1e=f1e/x/y&f1f=''&f1g=/resource/f1g/x&f1h=/resource/f1h&f1i=/resource&f1j=/&f1k=/f1j/x&f1l=/f1k&f1m=/&f1n=/..&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=f2d&f2e=f2e/x/y&f2f=''&f2g=/resource/f2g/x&f2h=/resource/f2h&f2i=/resource&f2j=/&f2k=/f2j/x&f2l=/f2k&f2m=/&f2n=/..&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=%253C%253E%2526%2527%2522&f3c=%3C%3E%26~'%22&f4=test/uri/b&f5=f5/x",
					"DE0022A26630A566302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164A3663164A3663165A76631652F782F79A3663166A0A3663167AF2F7265736F757263652F6631672F78A3663168AD2F7265736F757263652F663168A3663169A92F7265736F75726365A366316AA12FA366316BA62F66316A2F78A366316CA42F66316BA366316DA12FA366316EA32F2E2EA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264A3663264A3663265A76632652F782F79A3663266A0A3663267AF2F7265736F757263652F6632672F78A3663268AD2F7265736F757263652F663268A3663269A92F7265736F75726365A366326AA12FA366326BA62F66326A2F78A366326CA42F66326BA366326DA12FA366326EA32F2E2EA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362AF253343253345253236253237253232A3663363A53C3E262722A26634AA746573742F7572692F62A26635A466352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='f0/x0'>\n<jp:f1n>/..</jp:f1n>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f2h>/resource/f2h</jp:f2h>\n<jp:f2n>/..</jp:f2n>\n<jp:f1h>/resource/f1h</jp:f1h>\n<jp:f1i>/resource</jp:f1i>\n<jp:f1g>/resource/f1g/x</jp:f1g>\n<jp:f2f></jp:f2f>\n<jp:f2k>/f2j/x</jp:f2k>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f3b>%3C%3E%26%27%22</jp:f3b>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f2i>/resource</jp:f2i>\n<jp:f5>f5/x</jp:f5>\n<jp:f1m>/</jp:f1m>\n<jp:f2l>/f2k</jp:f2l>\n<jp:f2g>/resource/f2g/x</jp:f2g>\n<jp:f1k>/f1j/x</jp:f1k>\n<jp:f1e>f1e/x/y</jp:f1e>\n<jp:f2j>/</jp:f2j>\n<jp:f2d>f2d</jp:f2d>\n<jp:f3c>&lt;&gt;&amp;''</jp:f3c>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f1l>/f1k</jp:f1l>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f1d>f1d</jp:f1d>\n<jp:f1j>/</jp:f1j>\n<jp:f2e>f2e/x/y</jp:f2e>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1f></jp:f1f>\n<jp:f4>test/uri/b</jp:f4>\n<jp:f2m>/</jp:f2m>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 21 */
				"NoContextNoResource-1-ABSOLUTE",
				input(
					UriResolution.ABSOLUTE, UriRelativity.RESOURCE, "http://foo.com:123","","","/path"
				),
				results(
					"{f0:'http://foo.com:123/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'http://foo.com:123/f1b',f1c:'http://foo.com:123/f1c/x/y',f1d:'http://foo.com:123/f1d',f1e:'http://foo.com:123/f1e/x/y',f1f:'http://foo.com:123',f1g:'http://foo.com:123/f1g/x',f1h:'http://foo.com:123/f1h',f1i:'http://foo.com:123',f1j:'http://foo.com:123/..',f1k:'http://foo.com:123/f1j/x',f1l:'http://foo.com:123/f1k',f1m:'http://foo.com:123',f1n:'http://foo.com:123/..',f2a:'http://www.apache.org/f2a',f2b:'http://foo.com:123/f2b',f2c:'http://foo.com:123/f2c/x/y',f2d:'http://foo.com:123/f2d',f2e:'http://foo.com:123/f2e/x/y',f2f:'http://foo.com:123',f2g:'http://foo.com:123/f2g/x',f2h:'http://foo.com:123/f2h',f2i:'http://foo.com:123',f2j:'http://foo.com:123/..',f2k:'http://foo.com:123/f2j/x',f2l:'http://foo.com:123/f2k',f2m:'http://foo.com:123',f2n:'http://foo.com:123/..',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'http://foo.com:123/%3C%3E%26%27%22',f3c:'http://foo.com:123/<>&\\'\"',f4:'http://foo.com:123/test/uri/b',f5:'http://foo.com:123/f5/x'}",
					"<object f0='http://foo.com:123/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>http://foo.com:123/f1b</f1b><f1c>http://foo.com:123/f1c/x/y</f1c><f1d>http://foo.com:123/f1d</f1d><f1e>http://foo.com:123/f1e/x/y</f1e><f1f>http://foo.com:123</f1f><f1g>http://foo.com:123/f1g/x</f1g><f1h>http://foo.com:123/f1h</f1h><f1i>http://foo.com:123</f1i><f1j>http://foo.com:123/..</f1j><f1k>http://foo.com:123/f1j/x</f1k><f1l>http://foo.com:123/f1k</f1l><f1m>http://foo.com:123</f1m><f1n>http://foo.com:123/..</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>http://foo.com:123/f2b</f2b><f2c>http://foo.com:123/f2c/x/y</f2c><f2d>http://foo.com:123/f2d</f2d><f2e>http://foo.com:123/f2e/x/y</f2e><f2f>http://foo.com:123</f2f><f2g>http://foo.com:123/f2g/x</f2g><f2h>http://foo.com:123/f2h</f2h><f2i>http://foo.com:123</f2i><f2j>http://foo.com:123/..</f2j><f2k>http://foo.com:123/f2j/x</f2k><f2l>http://foo.com:123/f2k</f2l><f2m>http://foo.com:123</f2m><f2n>http://foo.com:123/..</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>http://foo.com:123/%3C%3E%26%27%22</f3b><f3c>http://foo.com:123/&lt;&gt;&amp;'\"</f3c><f4>http://foo.com:123/test/uri/b</f4><f5>http://foo.com:123/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='http://foo.com:123/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='http://foo.com:123/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='http://foo.com:123/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='http://foo.com:123/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='http://foo.com:123/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='http://foo.com:123'>foo.com:123</a></td></tr><tr><td>f1g</td><td><a href='http://foo.com:123/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='http://foo.com:123/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='http://foo.com:123'>foo.com:123</a></td></tr><tr><td>f1j</td><td><a href='http://foo.com:123/..'>..</a></td></tr><tr><td>f1k</td><td><a href='http://foo.com:123/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='http://foo.com:123/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='http://foo.com:123'>foo.com:123</a></td></tr><tr><td>f1n</td><td><a href='http://foo.com:123/..'>..</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='http://foo.com:123/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='http://foo.com:123/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='http://foo.com:123/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='http://foo.com:123/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='http://foo.com:123'>foo.com:123</a></td></tr><tr><td>f2g</td><td><a href='http://foo.com:123/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='http://foo.com:123/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='http://foo.com:123'>foo.com:123</a></td></tr><tr><td>f2j</td><td><a href='http://foo.com:123/..'>..</a></td></tr><tr><td>f2k</td><td><a href='http://foo.com:123/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='http://foo.com:123/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='http://foo.com:123'>foo.com:123</a></td></tr><tr><td>f2n</td><td><a href='http://foo.com:123/..'>..</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='http://foo.com:123/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='http://foo.com:123/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='http://foo.com:123/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='http://foo.com:123/f5/x'>x</a></td></tr></table>",
					"(f0=http://foo.com:123/f0/x0,f1a=http://www.apache.org/f1a,f1b=http://foo.com:123/f1b,f1c=http://foo.com:123/f1c/x/y,f1d=http://foo.com:123/f1d,f1e=http://foo.com:123/f1e/x/y,f1f=http://foo.com:123,f1g=http://foo.com:123/f1g/x,f1h=http://foo.com:123/f1h,f1i=http://foo.com:123,f1j=http://foo.com:123/..,f1k=http://foo.com:123/f1j/x,f1l=http://foo.com:123/f1k,f1m=http://foo.com:123,f1n=http://foo.com:123/..,f2a=http://www.apache.org/f2a,f2b=http://foo.com:123/f2b,f2c=http://foo.com:123/f2c/x/y,f2d=http://foo.com:123/f2d,f2e=http://foo.com:123/f2e/x/y,f2f=http://foo.com:123,f2g=http://foo.com:123/f2g/x,f2h=http://foo.com:123/f2h,f2i=http://foo.com:123,f2j=http://foo.com:123/..,f2k=http://foo.com:123/f2j/x,f2l=http://foo.com:123/f2k,f2m=http://foo.com:123,f2n=http://foo.com:123/..,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=http://foo.com:123/%3C%3E%26%27%22,f3c=http://foo.com:123/<>&~'\",f4=http://foo.com:123/test/uri/b,f5=http://foo.com:123/f5/x)",
					"f0=http://foo.com:123/f0/x0&f1a=http://www.apache.org/f1a&f1b=http://foo.com:123/f1b&f1c=http://foo.com:123/f1c/x/y&f1d=http://foo.com:123/f1d&f1e=http://foo.com:123/f1e/x/y&f1f=http://foo.com:123&f1g=http://foo.com:123/f1g/x&f1h=http://foo.com:123/f1h&f1i=http://foo.com:123&f1j=http://foo.com:123/..&f1k=http://foo.com:123/f1j/x&f1l=http://foo.com:123/f1k&f1m=http://foo.com:123&f1n=http://foo.com:123/..&f2a=http://www.apache.org/f2a&f2b=http://foo.com:123/f2b&f2c=http://foo.com:123/f2c/x/y&f2d=http://foo.com:123/f2d&f2e=http://foo.com:123/f2e/x/y&f2f=http://foo.com:123&f2g=http://foo.com:123/f2g/x&f2h=http://foo.com:123/f2h&f2i=http://foo.com:123&f2j=http://foo.com:123/..&f2k=http://foo.com:123/f2j/x&f2l=http://foo.com:123/f2k&f2m=http://foo.com:123&f2n=http://foo.com:123/..&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=http://foo.com:123/%253C%253E%2526%2527%2522&f3c=http://foo.com:123/%3C%3E%26~'%22&f4=http://foo.com:123/test/uri/b&f5=http://foo.com:123/f5/x",
					"DE0022A26630B8687474703A2F2F666F6F2E636F6D3A3132332F66302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162B6687474703A2F2F666F6F2E636F6D3A3132332F663162A3663163BA687474703A2F2F666F6F2E636F6D3A3132332F6631632F782F79A3663164B6687474703A2F2F666F6F2E636F6D3A3132332F663164A3663165BA687474703A2F2F666F6F2E636F6D3A3132332F6631652F782F79A3663166B2687474703A2F2F666F6F2E636F6D3A313233A3663167B8687474703A2F2F666F6F2E636F6D3A3132332F6631672F78A3663168B6687474703A2F2F666F6F2E636F6D3A3132332F663168A3663169B2687474703A2F2F666F6F2E636F6D3A313233A366316AB5687474703A2F2F666F6F2E636F6D3A3132332F2E2EA366316BB8687474703A2F2F666F6F2E636F6D3A3132332F66316A2F78A366316CB6687474703A2F2F666F6F2E636F6D3A3132332F66316BA366316DB2687474703A2F2F666F6F2E636F6D3A313233A366316EB5687474703A2F2F666F6F2E636F6D3A3132332F2E2EA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262B6687474703A2F2F666F6F2E636F6D3A3132332F663262A3663263BA687474703A2F2F666F6F2E636F6D3A3132332F6632632F782F79A3663264B6687474703A2F2F666F6F2E636F6D3A3132332F663264A3663265BA687474703A2F2F666F6F2E636F6D3A3132332F6632652F782F79A3663266B2687474703A2F2F666F6F2E636F6D3A313233A3663267B8687474703A2F2F666F6F2E636F6D3A3132332F6632672F78A3663268B6687474703A2F2F666F6F2E636F6D3A3132332F663268A3663269B2687474703A2F2F666F6F2E636F6D3A313233A366326AB5687474703A2F2F666F6F2E636F6D3A3132332F2E2EA366326BB8687474703A2F2F666F6F2E636F6D3A3132332F66326A2F78A366326CB6687474703A2F2F666F6F2E636F6D3A3132332F66326BA366326DB2687474703A2F2F666F6F2E636F6D3A313233A366326EB5687474703A2F2F666F6F2E636F6D3A3132332F2E2EA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362D922687474703A2F2F666F6F2E636F6D3A3132332F253343253345253236253237253232A3663363B8687474703A2F2F666F6F2E636F6D3A3132332F3C3E262722A26634BD687474703A2F2F666F6F2E636F6D3A3132332F746573742F7572692F62A26635B7687474703A2F2F666F6F2E636F6D3A3132332F66352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='http://foo.com:123/f0/x0'>\n<jp:f2f rdf:resource='http://foo.com:123'/>\n<jp:f4 rdf:resource='http://foo.com:123/test/uri/b'/>\n<jp:f1g rdf:resource='http://foo.com:123/f1g/x'/>\n<jp:f1l rdf:resource='http://foo.com:123/f1k'/>\n<jp:f1c rdf:resource='http://foo.com:123/f1c/x/y'/>\n<jp:f1j rdf:resource='http://foo.com:123/..'/>\n<jp:f1n rdf:resource='http://foo.com:123/..'/>\n<jp:f2c rdf:resource='http://foo.com:123/f2c/x/y'/>\n<jp:f2h rdf:resource='http://foo.com:123/f2h'/>\n<jp:f1f rdf:resource='http://foo.com:123'/>\n<jp:f2m rdf:resource='http://foo.com:123'/>\n<jp:f5 rdf:resource='http://foo.com:123/f5/x'/>\n<jp:f1m rdf:resource='http://foo.com:123'/>\n<jp:f2i rdf:resource='http://foo.com:123'/>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f1k rdf:resource='http://foo.com:123/f1j/x'/>\n<jp:f2e rdf:resource='http://foo.com:123/f2e/x/y'/>\n<jp:f2d rdf:resource='http://foo.com:123/f2d'/>\n<jp:f2n rdf:resource='http://foo.com:123/..'/>\n<jp:f2j rdf:resource='http://foo.com:123/..'/>\n<jp:f2k rdf:resource='http://foo.com:123/f2j/x'/>\n<jp:f1d rdf:resource='http://foo.com:123/f1d'/>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f1b rdf:resource='http://foo.com:123/f1b'/>\n<jp:f1e rdf:resource='http://foo.com:123/f1e/x/y'/>\n<jp:f2l rdf:resource='http://foo.com:123/f2k'/>\n<jp:f1h rdf:resource='http://foo.com:123/f1h'/>\n<jp:f3b rdf:resource='http://foo.com:123/%3C%3E%26%27%22'/>\n<jp:f3c rdf:resource='http://foo.com:123/&lt;&gt;&amp;&apos;&quot;'/>\n<jp:f1i rdf:resource='http://foo.com:123'/>\n<jp:f2b rdf:resource='http://foo.com:123/f2b'/>\n<jp:f2g rdf:resource='http://foo.com:123/f2g/x'/>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 22 */
				"NoContextNoResource-1-ROOT_RELATIVE",
				input(
					UriResolution.ROOT_RELATIVE, UriRelativity.RESOURCE, "http://foo.com:123","","","/path"
				),
				results(
					"{f0:'/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'/f1d',f1e:'/f1e/x/y',f1f:'/',f1g:'/f1g/x',f1h:'/f1h',f1i:'/',f1j:'/..',f1k:'/f1j/x',f1l:'/f1k',f1m:'/',f1n:'/..',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'/f2d',f2e:'/f2e/x/y',f2f:'/',f2g:'/f2g/x',f2h:'/f2h',f2i:'/',f2j:'/..',f2k:'/f2j/x',f2l:'/f2k',f2m:'/',f2n:'/..',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'/%3C%3E%26%27%22',f3c:'/<>&\\'\"',f4:'/test/uri/b',f5:'/f5/x'}",
					"<object f0='/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>/f1d</f1d><f1e>/f1e/x/y</f1e><f1f>/</f1f><f1g>/f1g/x</f1g><f1h>/f1h</f1h><f1i>/</f1i><f1j>/..</f1j><f1k>/f1j/x</f1k><f1l>/f1k</f1l><f1m>/</f1m><f1n>/..</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>/f2d</f2d><f2e>/f2e/x/y</f2e><f2f>/</f2f><f2g>/f2g/x</f2g><f2h>/f2h</f2h><f2i>/</f2i><f2j>/..</f2j><f2k>/f2j/x</f2k><f2l>/f2k</f2l><f2m>/</f2m><f2n>/..</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>/%3C%3E%26%27%22</f3b><f3c>/&lt;&gt;&amp;'\"</f3c><f4>/test/uri/b</f4><f5>/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='/'>/</a></td></tr><tr><td>f1g</td><td><a href='/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/'>/</a></td></tr><tr><td>f1j</td><td><a href='/..'>..</a></td></tr><tr><td>f1k</td><td><a href='/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/'>/</a></td></tr><tr><td>f1n</td><td><a href='/..'>..</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='/'>/</a></td></tr><tr><td>f2g</td><td><a href='/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/'>/</a></td></tr><tr><td>f2j</td><td><a href='/..'>..</a></td></tr><tr><td>f2k</td><td><a href='/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/'>/</a></td></tr><tr><td>f2n</td><td><a href='/..'>..</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='/f5/x'>x</a></td></tr></table>",
					"(f0=/f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=/f1d,f1e=/f1e/x/y,f1f=/,f1g=/f1g/x,f1h=/f1h,f1i=/,f1j=/..,f1k=/f1j/x,f1l=/f1k,f1m=/,f1n=/..,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=/f2d,f2e=/f2e/x/y,f2f=/,f2g=/f2g/x,f2h=/f2h,f2i=/,f2j=/..,f2k=/f2j/x,f2l=/f2k,f2m=/,f2n=/..,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=/%3C%3E%26%27%22,f3c=/<>&~'\",f4=/test/uri/b,f5=/f5/x)",
					"f0=/f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=/f1d&f1e=/f1e/x/y&f1f=/&f1g=/f1g/x&f1h=/f1h&f1i=/&f1j=/..&f1k=/f1j/x&f1l=/f1k&f1m=/&f1n=/..&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=/f2d&f2e=/f2e/x/y&f2f=/&f2g=/f2g/x&f2h=/f2h&f2i=/&f2j=/..&f2k=/f2j/x&f2l=/f2k&f2m=/&f2n=/..&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=/%253C%253E%2526%2527%2522&f3c=/%3C%3E%26~'%22&f4=/test/uri/b&f5=/f5/x",
					"DE0022A26630A62F66302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164A42F663164A3663165A82F6631652F782F79A3663166A12FA3663167A62F6631672F78A3663168A42F663168A3663169A12FA366316AA32F2E2EA366316BA62F66316A2F78A366316CA42F66316BA366316DA12FA366316EA32F2E2EA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264A42F663264A3663265A82F6632652F782F79A3663266A12FA3663267A62F6632672F78A3663268A42F663268A3663269A12FA366326AA32F2E2EA366326BA62F66326A2F78A366326CA42F66326BA366326DA12FA366326EA32F2E2EA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362B02F253343253345253236253237253232A3663363A62F3C3E262722A26634AB2F746573742F7572692F62A26635A52F66352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='/f0/x0'>\n<jp:f2n>/..</jp:f2n>\n<jp:f2h>/f2h</jp:f2h>\n<jp:f1j>/..</jp:f1j>\n<jp:f1n>/..</jp:f1n>\n<jp:f1i>/</jp:f1i>\n<jp:f5>/f5/x</jp:f5>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1m>/</jp:f1m>\n<jp:f1g>/f1g/x</jp:f1g>\n<jp:f1h>/f1h</jp:f1h>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f2f>/</jp:f2f>\n<jp:f1l>/f1k</jp:f1l>\n<jp:f2i>/</jp:f2i>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f1k>/f1j/x</jp:f1k>\n<jp:f4>/test/uri/b</jp:f4>\n<jp:f1f>/</jp:f1f>\n<jp:f3c>/&lt;&gt;&amp;''</jp:f3c>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f2k>/f2j/x</jp:f2k>\n<jp:f2l>/f2k</jp:f2l>\n<jp:f2e>/f2e/x/y</jp:f2e>\n<jp:f2m>/</jp:f2m>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f1e>/f1e/x/y</jp:f1e>\n<jp:f3b>/%3C%3E%26%27%22</jp:f3b>\n<jp:f1d>/f1d</jp:f1d>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f2j>/..</jp:f2j>\n<jp:f2d>/f2d</jp:f2d>\n<jp:f2g>/f2g/x</jp:f2g>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 23 */
				"NoContextNoResource-1-NONE",
				input(
					UriResolution.NONE, UriRelativity.RESOURCE, "http://foo.com:123","","","/path"
				),
				results(
					"{f0:'f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'f1d',f1e:'f1e/x/y',f1f:'',f1g:'/f1g/x',f1h:'/f1h',f1i:'/',f1j:'/..',f1k:'/f1j/x',f1l:'/f1k',f1m:'/',f1n:'/..',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'f2d',f2e:'f2e/x/y',f2f:'',f2g:'/f2g/x',f2h:'/f2h',f2i:'/',f2j:'/..',f2k:'/f2j/x',f2l:'/f2k',f2m:'/',f2n:'/..',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'%3C%3E%26%27%22',f3c:'<>&\\'\"',f4:'test/uri/b',f5:'f5/x'}",
					"<object f0='f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>f1d</f1d><f1e>f1e/x/y</f1e><f1f>_xE000_</f1f><f1g>/f1g/x</f1g><f1h>/f1h</f1h><f1i>/</f1i><f1j>/..</f1j><f1k>/f1j/x</f1k><f1l>/f1k</f1l><f1m>/</f1m><f1n>/..</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>f2d</f2d><f2e>f2e/x/y</f2e><f2f>_xE000_</f2f><f2g>/f2g/x</f2g><f2h>/f2h</f2h><f2i>/</f2i><f2j>/..</f2j><f2k>/f2j/x</f2k><f2l>/f2k</f2l><f2m>/</f2m><f2n>/..</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>%3C%3E%26%27%22</f3b><f3c>&lt;&gt;&amp;'\"</f3c><f4>test/uri/b</f4><f5>f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href=''>/</a></td></tr><tr><td>f1g</td><td><a href='/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/'>/</a></td></tr><tr><td>f1j</td><td><a href='/..'>..</a></td></tr><tr><td>f1k</td><td><a href='/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/'>/</a></td></tr><tr><td>f1n</td><td><a href='/..'>..</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href=''>/</a></td></tr><tr><td>f2g</td><td><a href='/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/'>/</a></td></tr><tr><td>f2j</td><td><a href='/..'>..</a></td></tr><tr><td>f2k</td><td><a href='/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/'>/</a></td></tr><tr><td>f2n</td><td><a href='/..'>..</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='f5/x'>x</a></td></tr></table>",
					"(f0=f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=f1d,f1e=f1e/x/y,f1f='',f1g=/f1g/x,f1h=/f1h,f1i=/,f1j=/..,f1k=/f1j/x,f1l=/f1k,f1m=/,f1n=/..,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=f2d,f2e=f2e/x/y,f2f='',f2g=/f2g/x,f2h=/f2h,f2i=/,f2j=/..,f2k=/f2j/x,f2l=/f2k,f2m=/,f2n=/..,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=%3C%3E%26%27%22,f3c=<>&~'\",f4=test/uri/b,f5=f5/x)",
					"f0=f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=f1d&f1e=f1e/x/y&f1f=''&f1g=/f1g/x&f1h=/f1h&f1i=/&f1j=/..&f1k=/f1j/x&f1l=/f1k&f1m=/&f1n=/..&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=f2d&f2e=f2e/x/y&f2f=''&f2g=/f2g/x&f2h=/f2h&f2i=/&f2j=/..&f2k=/f2j/x&f2l=/f2k&f2m=/&f2n=/..&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=%253C%253E%2526%2527%2522&f3c=%3C%3E%26~'%22&f4=test/uri/b&f5=f5/x",
					"DE0022A26630A566302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164A3663164A3663165A76631652F782F79A3663166A0A3663167A62F6631672F78A3663168A42F663168A3663169A12FA366316AA32F2E2EA366316BA62F66316A2F78A366316CA42F66316BA366316DA12FA366316EA32F2E2EA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264A3663264A3663265A76632652F782F79A3663266A0A3663267A62F6632672F78A3663268A42F663268A3663269A12FA366326AA32F2E2EA366326BA62F66326A2F78A366326CA42F66326BA366326DA12FA366326EA32F2E2EA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362AF253343253345253236253237253232A3663363A53C3E262722A26634AA746573742F7572692F62A26635A466352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='f0/x0'>\n<jp:f2i>/</jp:f2i>\n<jp:f1n>/..</jp:f1n>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f1h>/f1h</jp:f1h>\n<jp:f2n>/..</jp:f2n>\n<jp:f2j>/..</jp:f2j>\n<jp:f2f></jp:f2f>\n<jp:f2k>/f2j/x</jp:f2k>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f3b>%3C%3E%26%27%22</jp:f3b>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f1g>/f1g/x</jp:f1g>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f1j>/..</jp:f1j>\n<jp:f5>f5/x</jp:f5>\n<jp:f1m>/</jp:f1m>\n<jp:f2l>/f2k</jp:f2l>\n<jp:f1i>/</jp:f1i>\n<jp:f1k>/f1j/x</jp:f1k>\n<jp:f1e>f1e/x/y</jp:f1e>\n<jp:f2g>/f2g/x</jp:f2g>\n<jp:f2d>f2d</jp:f2d>\n<jp:f3c>&lt;&gt;&amp;''</jp:f3c>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f2h>/f2h</jp:f2h>\n<jp:f1l>/f1k</jp:f1l>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f1d>f1d</jp:f1d>\n<jp:f2e>f2e/x/y</jp:f2e>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1f></jp:f1f>\n<jp:f4>test/uri/b</jp:f4>\n<jp:f2m>/</jp:f2m>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 24 */
				"NoAuthorityNoContextNoResource-1-ABSOLUTE",
				input(
					UriResolution.ABSOLUTE, UriRelativity.RESOURCE, "","","","/path"
				),
				results(
					"{f0:'/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'/f1d',f1e:'/f1e/x/y',f1f:'/',f1g:'/f1g/x',f1h:'/f1h',f1i:'/',f1j:'/..',f1k:'/f1j/x',f1l:'/f1k',f1m:'/',f1n:'/..',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'/f2d',f2e:'/f2e/x/y',f2f:'/',f2g:'/f2g/x',f2h:'/f2h',f2i:'/',f2j:'/..',f2k:'/f2j/x',f2l:'/f2k',f2m:'/',f2n:'/..',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'/%3C%3E%26%27%22',f3c:'/<>&\\'\"',f4:'/test/uri/b',f5:'/f5/x'}",
					"<object f0='/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>/f1d</f1d><f1e>/f1e/x/y</f1e><f1f>/</f1f><f1g>/f1g/x</f1g><f1h>/f1h</f1h><f1i>/</f1i><f1j>/..</f1j><f1k>/f1j/x</f1k><f1l>/f1k</f1l><f1m>/</f1m><f1n>/..</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>/f2d</f2d><f2e>/f2e/x/y</f2e><f2f>/</f2f><f2g>/f2g/x</f2g><f2h>/f2h</f2h><f2i>/</f2i><f2j>/..</f2j><f2k>/f2j/x</f2k><f2l>/f2k</f2l><f2m>/</f2m><f2n>/..</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>/%3C%3E%26%27%22</f3b><f3c>/&lt;&gt;&amp;'\"</f3c><f4>/test/uri/b</f4><f5>/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='/'>/</a></td></tr><tr><td>f1g</td><td><a href='/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/'>/</a></td></tr><tr><td>f1j</td><td><a href='/..'>..</a></td></tr><tr><td>f1k</td><td><a href='/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/'>/</a></td></tr><tr><td>f1n</td><td><a href='/..'>..</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='/'>/</a></td></tr><tr><td>f2g</td><td><a href='/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/'>/</a></td></tr><tr><td>f2j</td><td><a href='/..'>..</a></td></tr><tr><td>f2k</td><td><a href='/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/'>/</a></td></tr><tr><td>f2n</td><td><a href='/..'>..</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='/f5/x'>x</a></td></tr></table>",
					"(f0=/f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=/f1d,f1e=/f1e/x/y,f1f=/,f1g=/f1g/x,f1h=/f1h,f1i=/,f1j=/..,f1k=/f1j/x,f1l=/f1k,f1m=/,f1n=/..,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=/f2d,f2e=/f2e/x/y,f2f=/,f2g=/f2g/x,f2h=/f2h,f2i=/,f2j=/..,f2k=/f2j/x,f2l=/f2k,f2m=/,f2n=/..,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=/%3C%3E%26%27%22,f3c=/<>&~'\",f4=/test/uri/b,f5=/f5/x)",
					"f0=/f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=/f1d&f1e=/f1e/x/y&f1f=/&f1g=/f1g/x&f1h=/f1h&f1i=/&f1j=/..&f1k=/f1j/x&f1l=/f1k&f1m=/&f1n=/..&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=/f2d&f2e=/f2e/x/y&f2f=/&f2g=/f2g/x&f2h=/f2h&f2i=/&f2j=/..&f2k=/f2j/x&f2l=/f2k&f2m=/&f2n=/..&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=/%253C%253E%2526%2527%2522&f3c=/%3C%3E%26~'%22&f4=/test/uri/b&f5=/f5/x",
					"DE0022A26630A62F66302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164A42F663164A3663165A82F6631652F782F79A3663166A12FA3663167A62F6631672F78A3663168A42F663168A3663169A12FA366316AA32F2E2EA366316BA62F66316A2F78A366316CA42F66316BA366316DA12FA366316EA32F2E2EA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264A42F663264A3663265A82F6632652F782F79A3663266A12FA3663267A62F6632672F78A3663268A42F663268A3663269A12FA366326AA32F2E2EA366326BA62F66326A2F78A366326CA42F66326BA366326DA12FA366326EA32F2E2EA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362B02F253343253345253236253237253232A3663363A62F3C3E262722A26634AB2F746573742F7572692F62A26635A52F66352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='/f0/x0'>\n<jp:f2n>/..</jp:f2n>\n<jp:f2h>/f2h</jp:f2h>\n<jp:f1j>/..</jp:f1j>\n<jp:f1n>/..</jp:f1n>\n<jp:f1i>/</jp:f1i>\n<jp:f5>/f5/x</jp:f5>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1m>/</jp:f1m>\n<jp:f1g>/f1g/x</jp:f1g>\n<jp:f1h>/f1h</jp:f1h>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f2f>/</jp:f2f>\n<jp:f1l>/f1k</jp:f1l>\n<jp:f2i>/</jp:f2i>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f1k>/f1j/x</jp:f1k>\n<jp:f4>/test/uri/b</jp:f4>\n<jp:f1f>/</jp:f1f>\n<jp:f3c>/&lt;&gt;&amp;''</jp:f3c>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f2k>/f2j/x</jp:f2k>\n<jp:f2l>/f2k</jp:f2l>\n<jp:f2e>/f2e/x/y</jp:f2e>\n<jp:f2m>/</jp:f2m>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f1e>/f1e/x/y</jp:f1e>\n<jp:f3b>/%3C%3E%26%27%22</jp:f3b>\n<jp:f1d>/f1d</jp:f1d>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f2j>/..</jp:f2j>\n<jp:f2d>/f2d</jp:f2d>\n<jp:f2g>/f2g/x</jp:f2g>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 25 */
				"NoAuthorityNoContextNoResource-1-ROOT_RELATIVE",
				input(
					UriResolution.ROOT_RELATIVE, UriRelativity.RESOURCE, "","","","/path"
				),
				results(
					"{f0:'/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'/f1d',f1e:'/f1e/x/y',f1f:'/',f1g:'/f1g/x',f1h:'/f1h',f1i:'/',f1j:'/..',f1k:'/f1j/x',f1l:'/f1k',f1m:'/',f1n:'/..',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'/f2d',f2e:'/f2e/x/y',f2f:'/',f2g:'/f2g/x',f2h:'/f2h',f2i:'/',f2j:'/..',f2k:'/f2j/x',f2l:'/f2k',f2m:'/',f2n:'/..',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'/%3C%3E%26%27%22',f3c:'/<>&\\'\"',f4:'/test/uri/b',f5:'/f5/x'}",
					"<object f0='/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>/f1d</f1d><f1e>/f1e/x/y</f1e><f1f>/</f1f><f1g>/f1g/x</f1g><f1h>/f1h</f1h><f1i>/</f1i><f1j>/..</f1j><f1k>/f1j/x</f1k><f1l>/f1k</f1l><f1m>/</f1m><f1n>/..</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>/f2d</f2d><f2e>/f2e/x/y</f2e><f2f>/</f2f><f2g>/f2g/x</f2g><f2h>/f2h</f2h><f2i>/</f2i><f2j>/..</f2j><f2k>/f2j/x</f2k><f2l>/f2k</f2l><f2m>/</f2m><f2n>/..</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>/%3C%3E%26%27%22</f3b><f3c>/&lt;&gt;&amp;'\"</f3c><f4>/test/uri/b</f4><f5>/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='/'>/</a></td></tr><tr><td>f1g</td><td><a href='/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/'>/</a></td></tr><tr><td>f1j</td><td><a href='/..'>..</a></td></tr><tr><td>f1k</td><td><a href='/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/'>/</a></td></tr><tr><td>f1n</td><td><a href='/..'>..</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='/'>/</a></td></tr><tr><td>f2g</td><td><a href='/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/'>/</a></td></tr><tr><td>f2j</td><td><a href='/..'>..</a></td></tr><tr><td>f2k</td><td><a href='/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/'>/</a></td></tr><tr><td>f2n</td><td><a href='/..'>..</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='/f5/x'>x</a></td></tr></table>",
					"(f0=/f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=/f1d,f1e=/f1e/x/y,f1f=/,f1g=/f1g/x,f1h=/f1h,f1i=/,f1j=/..,f1k=/f1j/x,f1l=/f1k,f1m=/,f1n=/..,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=/f2d,f2e=/f2e/x/y,f2f=/,f2g=/f2g/x,f2h=/f2h,f2i=/,f2j=/..,f2k=/f2j/x,f2l=/f2k,f2m=/,f2n=/..,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=/%3C%3E%26%27%22,f3c=/<>&~'\",f4=/test/uri/b,f5=/f5/x)",
					"f0=/f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=/f1d&f1e=/f1e/x/y&f1f=/&f1g=/f1g/x&f1h=/f1h&f1i=/&f1j=/..&f1k=/f1j/x&f1l=/f1k&f1m=/&f1n=/..&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=/f2d&f2e=/f2e/x/y&f2f=/&f2g=/f2g/x&f2h=/f2h&f2i=/&f2j=/..&f2k=/f2j/x&f2l=/f2k&f2m=/&f2n=/..&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=/%253C%253E%2526%2527%2522&f3c=/%3C%3E%26~'%22&f4=/test/uri/b&f5=/f5/x",
					"DE0022A26630A62F66302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164A42F663164A3663165A82F6631652F782F79A3663166A12FA3663167A62F6631672F78A3663168A42F663168A3663169A12FA366316AA32F2E2EA366316BA62F66316A2F78A366316CA42F66316BA366316DA12FA366316EA32F2E2EA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264A42F663264A3663265A82F6632652F782F79A3663266A12FA3663267A62F6632672F78A3663268A42F663268A3663269A12FA366326AA32F2E2EA366326BA62F66326A2F78A366326CA42F66326BA366326DA12FA366326EA32F2E2EA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362B02F253343253345253236253237253232A3663363A62F3C3E262722A26634AB2F746573742F7572692F62A26635A52F66352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='/f0/x0'>\n<jp:f2n>/..</jp:f2n>\n<jp:f2h>/f2h</jp:f2h>\n<jp:f1j>/..</jp:f1j>\n<jp:f1n>/..</jp:f1n>\n<jp:f1i>/</jp:f1i>\n<jp:f5>/f5/x</jp:f5>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1m>/</jp:f1m>\n<jp:f1g>/f1g/x</jp:f1g>\n<jp:f1h>/f1h</jp:f1h>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f2f>/</jp:f2f>\n<jp:f1l>/f1k</jp:f1l>\n<jp:f2i>/</jp:f2i>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f1k>/f1j/x</jp:f1k>\n<jp:f4>/test/uri/b</jp:f4>\n<jp:f1f>/</jp:f1f>\n<jp:f3c>/&lt;&gt;&amp;''</jp:f3c>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f2k>/f2j/x</jp:f2k>\n<jp:f2l>/f2k</jp:f2l>\n<jp:f2e>/f2e/x/y</jp:f2e>\n<jp:f2m>/</jp:f2m>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f1e>/f1e/x/y</jp:f1e>\n<jp:f3b>/%3C%3E%26%27%22</jp:f3b>\n<jp:f1d>/f1d</jp:f1d>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f2j>/..</jp:f2j>\n<jp:f2d>/f2d</jp:f2d>\n<jp:f2g>/f2g/x</jp:f2g>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 26 */
				"NoAuthorityNoContextNoResource-1-NONE",
				input(
					UriResolution.NONE, UriRelativity.RESOURCE, "","","","/path"
				),
				results(
					"{f0:'f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'f1d',f1e:'f1e/x/y',f1f:'',f1g:'/f1g/x',f1h:'/f1h',f1i:'/',f1j:'/..',f1k:'/f1j/x',f1l:'/f1k',f1m:'/',f1n:'/..',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'f2d',f2e:'f2e/x/y',f2f:'',f2g:'/f2g/x',f2h:'/f2h',f2i:'/',f2j:'/..',f2k:'/f2j/x',f2l:'/f2k',f2m:'/',f2n:'/..',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'%3C%3E%26%27%22',f3c:'<>&\\'\"',f4:'test/uri/b',f5:'f5/x'}",
					"<object f0='f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>f1d</f1d><f1e>f1e/x/y</f1e><f1f>_xE000_</f1f><f1g>/f1g/x</f1g><f1h>/f1h</f1h><f1i>/</f1i><f1j>/..</f1j><f1k>/f1j/x</f1k><f1l>/f1k</f1l><f1m>/</f1m><f1n>/..</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>f2d</f2d><f2e>f2e/x/y</f2e><f2f>_xE000_</f2f><f2g>/f2g/x</f2g><f2h>/f2h</f2h><f2i>/</f2i><f2j>/..</f2j><f2k>/f2j/x</f2k><f2l>/f2k</f2l><f2m>/</f2m><f2n>/..</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>%3C%3E%26%27%22</f3b><f3c>&lt;&gt;&amp;'\"</f3c><f4>test/uri/b</f4><f5>f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href=''>/</a></td></tr><tr><td>f1g</td><td><a href='/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/'>/</a></td></tr><tr><td>f1j</td><td><a href='/..'>..</a></td></tr><tr><td>f1k</td><td><a href='/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/'>/</a></td></tr><tr><td>f1n</td><td><a href='/..'>..</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href=''>/</a></td></tr><tr><td>f2g</td><td><a href='/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/'>/</a></td></tr><tr><td>f2j</td><td><a href='/..'>..</a></td></tr><tr><td>f2k</td><td><a href='/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/'>/</a></td></tr><tr><td>f2n</td><td><a href='/..'>..</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='f5/x'>x</a></td></tr></table>",
					"(f0=f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=f1d,f1e=f1e/x/y,f1f='',f1g=/f1g/x,f1h=/f1h,f1i=/,f1j=/..,f1k=/f1j/x,f1l=/f1k,f1m=/,f1n=/..,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=f2d,f2e=f2e/x/y,f2f='',f2g=/f2g/x,f2h=/f2h,f2i=/,f2j=/..,f2k=/f2j/x,f2l=/f2k,f2m=/,f2n=/..,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=%3C%3E%26%27%22,f3c=<>&~'\",f4=test/uri/b,f5=f5/x)",
					"f0=f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=f1d&f1e=f1e/x/y&f1f=''&f1g=/f1g/x&f1h=/f1h&f1i=/&f1j=/..&f1k=/f1j/x&f1l=/f1k&f1m=/&f1n=/..&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=f2d&f2e=f2e/x/y&f2f=''&f2g=/f2g/x&f2h=/f2h&f2i=/&f2j=/..&f2k=/f2j/x&f2l=/f2k&f2m=/&f2n=/..&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=%253C%253E%2526%2527%2522&f3c=%3C%3E%26~'%22&f4=test/uri/b&f5=f5/x",
					"DE0022A26630A566302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164A3663164A3663165A76631652F782F79A3663166A0A3663167A62F6631672F78A3663168A42F663168A3663169A12FA366316AA32F2E2EA366316BA62F66316A2F78A366316CA42F66316BA366316DA12FA366316EA32F2E2EA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264A3663264A3663265A76632652F782F79A3663266A0A3663267A62F6632672F78A3663268A42F663268A3663269A12FA366326AA32F2E2EA366326BA62F66326A2F78A366326CA42F66326BA366326DA12FA366326EA32F2E2EA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362AF253343253345253236253237253232A3663363A53C3E262722A26634AA746573742F7572692F62A26635A466352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='f0/x0'>\n<jp:f2i>/</jp:f2i>\n<jp:f1n>/..</jp:f1n>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f1h>/f1h</jp:f1h>\n<jp:f2n>/..</jp:f2n>\n<jp:f2j>/..</jp:f2j>\n<jp:f2f></jp:f2f>\n<jp:f2k>/f2j/x</jp:f2k>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f3b>%3C%3E%26%27%22</jp:f3b>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f1g>/f1g/x</jp:f1g>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f1j>/..</jp:f1j>\n<jp:f5>f5/x</jp:f5>\n<jp:f1m>/</jp:f1m>\n<jp:f2l>/f2k</jp:f2l>\n<jp:f1i>/</jp:f1i>\n<jp:f1k>/f1j/x</jp:f1k>\n<jp:f1e>f1e/x/y</jp:f1e>\n<jp:f2g>/f2g/x</jp:f2g>\n<jp:f2d>f2d</jp:f2d>\n<jp:f3c>&lt;&gt;&amp;''</jp:f3c>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f2h>/f2h</jp:f2h>\n<jp:f1l>/f1k</jp:f1l>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f1d>f1d</jp:f1d>\n<jp:f2e>f2e/x/y</jp:f2e>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1f></jp:f1f>\n<jp:f4>test/uri/b</jp:f4>\n<jp:f2m>/</jp:f2m>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 27 */
				"Nothing-1-ABSOLUTE",
				input(
					UriResolution.ABSOLUTE, UriRelativity.RESOURCE, "","","",""
				),
				results(
					"{f0:'/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'/f1d',f1e:'/f1e/x/y',f1f:'/',f1g:'/f1g/x',f1h:'/f1h',f1i:'/',f1j:'/..',f1k:'/f1j/x',f1l:'/f1k',f1m:'/',f1n:'/..',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'/f2d',f2e:'/f2e/x/y',f2f:'/',f2g:'/f2g/x',f2h:'/f2h',f2i:'/',f2j:'/..',f2k:'/f2j/x',f2l:'/f2k',f2m:'/',f2n:'/..',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'/%3C%3E%26%27%22',f3c:'/<>&\\'\"',f4:'/test/uri/b',f5:'/f5/x'}",
					"<object f0='/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>/f1d</f1d><f1e>/f1e/x/y</f1e><f1f>/</f1f><f1g>/f1g/x</f1g><f1h>/f1h</f1h><f1i>/</f1i><f1j>/..</f1j><f1k>/f1j/x</f1k><f1l>/f1k</f1l><f1m>/</f1m><f1n>/..</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>/f2d</f2d><f2e>/f2e/x/y</f2e><f2f>/</f2f><f2g>/f2g/x</f2g><f2h>/f2h</f2h><f2i>/</f2i><f2j>/..</f2j><f2k>/f2j/x</f2k><f2l>/f2k</f2l><f2m>/</f2m><f2n>/..</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>/%3C%3E%26%27%22</f3b><f3c>/&lt;&gt;&amp;'\"</f3c><f4>/test/uri/b</f4><f5>/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='/'>/</a></td></tr><tr><td>f1g</td><td><a href='/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/'>/</a></td></tr><tr><td>f1j</td><td><a href='/..'>..</a></td></tr><tr><td>f1k</td><td><a href='/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/'>/</a></td></tr><tr><td>f1n</td><td><a href='/..'>..</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='/'>/</a></td></tr><tr><td>f2g</td><td><a href='/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/'>/</a></td></tr><tr><td>f2j</td><td><a href='/..'>..</a></td></tr><tr><td>f2k</td><td><a href='/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/'>/</a></td></tr><tr><td>f2n</td><td><a href='/..'>..</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='/f5/x'>x</a></td></tr></table>",
					"(f0=/f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=/f1d,f1e=/f1e/x/y,f1f=/,f1g=/f1g/x,f1h=/f1h,f1i=/,f1j=/..,f1k=/f1j/x,f1l=/f1k,f1m=/,f1n=/..,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=/f2d,f2e=/f2e/x/y,f2f=/,f2g=/f2g/x,f2h=/f2h,f2i=/,f2j=/..,f2k=/f2j/x,f2l=/f2k,f2m=/,f2n=/..,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=/%3C%3E%26%27%22,f3c=/<>&~'\",f4=/test/uri/b,f5=/f5/x)",
					"f0=/f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=/f1d&f1e=/f1e/x/y&f1f=/&f1g=/f1g/x&f1h=/f1h&f1i=/&f1j=/..&f1k=/f1j/x&f1l=/f1k&f1m=/&f1n=/..&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=/f2d&f2e=/f2e/x/y&f2f=/&f2g=/f2g/x&f2h=/f2h&f2i=/&f2j=/..&f2k=/f2j/x&f2l=/f2k&f2m=/&f2n=/..&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=/%253C%253E%2526%2527%2522&f3c=/%3C%3E%26~'%22&f4=/test/uri/b&f5=/f5/x",
					"DE0022A26630A62F66302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164A42F663164A3663165A82F6631652F782F79A3663166A12FA3663167A62F6631672F78A3663168A42F663168A3663169A12FA366316AA32F2E2EA366316BA62F66316A2F78A366316CA42F66316BA366316DA12FA366316EA32F2E2EA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264A42F663264A3663265A82F6632652F782F79A3663266A12FA3663267A62F6632672F78A3663268A42F663268A3663269A12FA366326AA32F2E2EA366326BA62F66326A2F78A366326CA42F66326BA366326DA12FA366326EA32F2E2EA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362B02F253343253345253236253237253232A3663363A62F3C3E262722A26634AB2F746573742F7572692F62A26635A52F66352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='/f0/x0'>\n<jp:f2n>/..</jp:f2n>\n<jp:f2h>/f2h</jp:f2h>\n<jp:f1j>/..</jp:f1j>\n<jp:f1n>/..</jp:f1n>\n<jp:f1i>/</jp:f1i>\n<jp:f5>/f5/x</jp:f5>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1m>/</jp:f1m>\n<jp:f1g>/f1g/x</jp:f1g>\n<jp:f1h>/f1h</jp:f1h>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f2f>/</jp:f2f>\n<jp:f1l>/f1k</jp:f1l>\n<jp:f2i>/</jp:f2i>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f1k>/f1j/x</jp:f1k>\n<jp:f4>/test/uri/b</jp:f4>\n<jp:f1f>/</jp:f1f>\n<jp:f3c>/&lt;&gt;&amp;''</jp:f3c>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f2k>/f2j/x</jp:f2k>\n<jp:f2l>/f2k</jp:f2l>\n<jp:f2e>/f2e/x/y</jp:f2e>\n<jp:f2m>/</jp:f2m>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f1e>/f1e/x/y</jp:f1e>\n<jp:f3b>/%3C%3E%26%27%22</jp:f3b>\n<jp:f1d>/f1d</jp:f1d>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f2j>/..</jp:f2j>\n<jp:f2d>/f2d</jp:f2d>\n<jp:f2g>/f2g/x</jp:f2g>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 28 */
				"Nothing-1-ROOT_RELATIVE",
				input(
					UriResolution.ROOT_RELATIVE, UriRelativity.RESOURCE, "","","",""
				),
				results(
					"{f0:'/f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'/f1d',f1e:'/f1e/x/y',f1f:'/',f1g:'/f1g/x',f1h:'/f1h',f1i:'/',f1j:'/..',f1k:'/f1j/x',f1l:'/f1k',f1m:'/',f1n:'/..',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'/f2d',f2e:'/f2e/x/y',f2f:'/',f2g:'/f2g/x',f2h:'/f2h',f2i:'/',f2j:'/..',f2k:'/f2j/x',f2l:'/f2k',f2m:'/',f2n:'/..',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'/%3C%3E%26%27%22',f3c:'/<>&\\'\"',f4:'/test/uri/b',f5:'/f5/x'}",
					"<object f0='/f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>/f1d</f1d><f1e>/f1e/x/y</f1e><f1f>/</f1f><f1g>/f1g/x</f1g><f1h>/f1h</f1h><f1i>/</f1i><f1j>/..</f1j><f1k>/f1j/x</f1k><f1l>/f1k</f1l><f1m>/</f1m><f1n>/..</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>/f2d</f2d><f2e>/f2e/x/y</f2e><f2f>/</f2f><f2g>/f2g/x</f2g><f2h>/f2h</f2h><f2i>/</f2i><f2j>/..</f2j><f2k>/f2j/x</f2k><f2l>/f2k</f2l><f2m>/</f2m><f2n>/..</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>/%3C%3E%26%27%22</f3b><f3c>/&lt;&gt;&amp;'\"</f3c><f4>/test/uri/b</f4><f5>/f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='/f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='/f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='/f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href='/'>/</a></td></tr><tr><td>f1g</td><td><a href='/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/'>/</a></td></tr><tr><td>f1j</td><td><a href='/..'>..</a></td></tr><tr><td>f1k</td><td><a href='/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/'>/</a></td></tr><tr><td>f1n</td><td><a href='/..'>..</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='/f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='/f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href='/'>/</a></td></tr><tr><td>f2g</td><td><a href='/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/'>/</a></td></tr><tr><td>f2j</td><td><a href='/..'>..</a></td></tr><tr><td>f2k</td><td><a href='/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/'>/</a></td></tr><tr><td>f2n</td><td><a href='/..'>..</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='/%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='/&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='/test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='/f5/x'>x</a></td></tr></table>",
					"(f0=/f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=/f1d,f1e=/f1e/x/y,f1f=/,f1g=/f1g/x,f1h=/f1h,f1i=/,f1j=/..,f1k=/f1j/x,f1l=/f1k,f1m=/,f1n=/..,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=/f2d,f2e=/f2e/x/y,f2f=/,f2g=/f2g/x,f2h=/f2h,f2i=/,f2j=/..,f2k=/f2j/x,f2l=/f2k,f2m=/,f2n=/..,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=/%3C%3E%26%27%22,f3c=/<>&~'\",f4=/test/uri/b,f5=/f5/x)",
					"f0=/f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=/f1d&f1e=/f1e/x/y&f1f=/&f1g=/f1g/x&f1h=/f1h&f1i=/&f1j=/..&f1k=/f1j/x&f1l=/f1k&f1m=/&f1n=/..&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=/f2d&f2e=/f2e/x/y&f2f=/&f2g=/f2g/x&f2h=/f2h&f2i=/&f2j=/..&f2k=/f2j/x&f2l=/f2k&f2m=/&f2n=/..&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=/%253C%253E%2526%2527%2522&f3c=/%3C%3E%26~'%22&f4=/test/uri/b&f5=/f5/x",
					"DE0022A26630A62F66302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164A42F663164A3663165A82F6631652F782F79A3663166A12FA3663167A62F6631672F78A3663168A42F663168A3663169A12FA366316AA32F2E2EA366316BA62F66316A2F78A366316CA42F66316BA366316DA12FA366316EA32F2E2EA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264A42F663264A3663265A82F6632652F782F79A3663266A12FA3663267A62F6632672F78A3663268A42F663268A3663269A12FA366326AA32F2E2EA366326BA62F66326A2F78A366326CA42F66326BA366326DA12FA366326EA32F2E2EA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362B02F253343253345253236253237253232A3663363A62F3C3E262722A26634AB2F746573742F7572692F62A26635A52F66352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='/f0/x0'>\n<jp:f2n>/..</jp:f2n>\n<jp:f2h>/f2h</jp:f2h>\n<jp:f1j>/..</jp:f1j>\n<jp:f1n>/..</jp:f1n>\n<jp:f1i>/</jp:f1i>\n<jp:f5>/f5/x</jp:f5>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1m>/</jp:f1m>\n<jp:f1g>/f1g/x</jp:f1g>\n<jp:f1h>/f1h</jp:f1h>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f2f>/</jp:f2f>\n<jp:f1l>/f1k</jp:f1l>\n<jp:f2i>/</jp:f2i>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f1k>/f1j/x</jp:f1k>\n<jp:f4>/test/uri/b</jp:f4>\n<jp:f1f>/</jp:f1f>\n<jp:f3c>/&lt;&gt;&amp;''</jp:f3c>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f2k>/f2j/x</jp:f2k>\n<jp:f2l>/f2k</jp:f2l>\n<jp:f2e>/f2e/x/y</jp:f2e>\n<jp:f2m>/</jp:f2m>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f1e>/f1e/x/y</jp:f1e>\n<jp:f3b>/%3C%3E%26%27%22</jp:f3b>\n<jp:f1d>/f1d</jp:f1d>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f2j>/..</jp:f2j>\n<jp:f2d>/f2d</jp:f2d>\n<jp:f2g>/f2g/x</jp:f2g>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
			{	/* 29 */
				"Nothing-1-NONE",
				input(
					UriResolution.NONE, UriRelativity.RESOURCE, "","","",""
				),
				results(
					"{f0:'f0/x0',f1a:'http://www.apache.org/f1a',f1b:'/f1b',f1c:'/f1c/x/y',f1d:'f1d',f1e:'f1e/x/y',f1f:'',f1g:'/f1g/x',f1h:'/f1h',f1i:'/',f1j:'/..',f1k:'/f1j/x',f1l:'/f1k',f1m:'/',f1n:'/..',f2a:'http://www.apache.org/f2a',f2b:'/f2b',f2c:'/f2c/x/y',f2d:'f2d',f2e:'f2e/x/y',f2f:'',f2g:'/f2g/x',f2h:'/f2h',f2i:'/',f2j:'/..',f2k:'/f2j/x',f2l:'/f2k',f2m:'/',f2n:'/..',f3a:'http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b:'%3C%3E%26%27%22',f3c:'<>&\\'\"',f4:'test/uri/b',f5:'f5/x'}",
					"<object f0='f0/x0'><f1a>http://www.apache.org/f1a</f1a><f1b>/f1b</f1b><f1c>/f1c/x/y</f1c><f1d>f1d</f1d><f1e>f1e/x/y</f1e><f1f>_xE000_</f1f><f1g>/f1g/x</f1g><f1h>/f1h</f1h><f1i>/</f1i><f1j>/..</f1j><f1k>/f1j/x</f1k><f1l>/f1k</f1l><f1m>/</f1m><f1n>/..</f1n><f2a>http://www.apache.org/f2a</f2a><f2b>/f2b</f2b><f2c>/f2c/x/y</f2c><f2d>f2d</f2d><f2e>f2e/x/y</f2e><f2f>_xE000_</f2f><f2g>/f2g/x</f2g><f2h>/f2h</f2h><f2i>/</f2i><f2j>/..</f2j><f2k>/f2j/x</f2k><f2l>/f2k</f2l><f2m>/</f2m><f2n>/..</f2n><f3a>http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar</f3a><f3b>%3C%3E%26%27%22</f3b><f3c>&lt;&gt;&amp;'\"</f3c><f4>test/uri/b</f4><f5>f5/x</f5></object>",
					"<table><tr><td>f0</td><td><a href='f0/x0'>x0</a></td></tr><tr><td>f1a</td><td><a href='http://www.apache.org/f1a'>f1a</a></td></tr><tr><td>f1b</td><td><a href='/f1b'>f1b</a></td></tr><tr><td>f1c</td><td><a href='/f1c/x/y'>y</a></td></tr><tr><td>f1d</td><td><a href='f1d'>f1d</a></td></tr><tr><td>f1e</td><td><a href='f1e/x/y'>y</a></td></tr><tr><td>f1f</td><td><a href=''>/</a></td></tr><tr><td>f1g</td><td><a href='/f1g/x'>x</a></td></tr><tr><td>f1h</td><td><a href='/f1h'>f1h</a></td></tr><tr><td>f1i</td><td><a href='/'>/</a></td></tr><tr><td>f1j</td><td><a href='/..'>..</a></td></tr><tr><td>f1k</td><td><a href='/f1j/x'>x</a></td></tr><tr><td>f1l</td><td><a href='/f1k'>f1k</a></td></tr><tr><td>f1m</td><td><a href='/'>/</a></td></tr><tr><td>f1n</td><td><a href='/..'>..</a></td></tr><tr><td>f2a</td><td><a href='http://www.apache.org/f2a'>f2a</a></td></tr><tr><td>f2b</td><td><a href='/f2b'>f2b</a></td></tr><tr><td>f2c</td><td><a href='/f2c/x/y'>y</a></td></tr><tr><td>f2d</td><td><a href='f2d'>f2d</a></td></tr><tr><td>f2e</td><td><a href='f2e/x/y'>y</a></td></tr><tr><td>f2f</td><td><a href=''>/</a></td></tr><tr><td>f2g</td><td><a href='/f2g/x'>x</a></td></tr><tr><td>f2h</td><td><a href='/f2h'>f2h</a></td></tr><tr><td>f2i</td><td><a href='/'>/</a></td></tr><tr><td>f2j</td><td><a href='/..'>..</a></td></tr><tr><td>f2k</td><td><a href='/f2j/x'>x</a></td></tr><tr><td>f2l</td><td><a href='/f2k'>f2k</a></td></tr><tr><td>f2m</td><td><a href='/'>/</a></td></tr><tr><td>f2n</td><td><a href='/..'>..</a></td></tr><tr><td>f3a</td><td><a href='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'>MY_LABEL</a></td></tr><tr><td>f3b</td><td><a href='%3C%3E%26%27%22'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f3c</td><td><a href='&lt;&gt;&amp;&apos;&quot;'>&lt;&gt;&amp;'\"</a></td></tr><tr><td>f4</td><td><a href='test/uri/b'>b</a></td></tr><tr><td>f5</td><td><a href='f5/x'>x</a></td></tr></table>",
					"(f0=f0/x0,f1a=http://www.apache.org/f1a,f1b=/f1b,f1c=/f1c/x/y,f1d=f1d,f1e=f1e/x/y,f1f='',f1g=/f1g/x,f1h=/f1h,f1i=/,f1j=/..,f1k=/f1j/x,f1l=/f1k,f1m=/,f1n=/..,f2a=http://www.apache.org/f2a,f2b=/f2b,f2c=/f2c/x/y,f2d=f2d,f2e=f2e/x/y,f2f='',f2g=/f2g/x,f2h=/f2h,f2i=/,f2j=/..,f2k=/f2j/x,f2l=/f2k,f2m=/,f2n=/..,f3a='http://www.apache.org/f3a/x?label=MY_LABEL&foo=bar',f3b=%3C%3E%26%27%22,f3c=<>&~'\",f4=test/uri/b,f5=f5/x)",
					"f0=f0/x0&f1a=http://www.apache.org/f1a&f1b=/f1b&f1c=/f1c/x/y&f1d=f1d&f1e=f1e/x/y&f1f=''&f1g=/f1g/x&f1h=/f1h&f1i=/&f1j=/..&f1k=/f1j/x&f1l=/f1k&f1m=/&f1n=/..&f2a=http://www.apache.org/f2a&f2b=/f2b&f2c=/f2c/x/y&f2d=f2d&f2e=f2e/x/y&f2f=''&f2g=/f2g/x&f2h=/f2h&f2i=/&f2j=/..&f2k=/f2j/x&f2l=/f2k&f2m=/&f2n=/..&f3a='http://www.apache.org/f3a/x?label=MY_LABEL%26foo=bar'&f3b=%253C%253E%2526%2527%2522&f3c=%3C%3E%26~'%22&f4=test/uri/b&f5=f5/x",
					"DE0022A26630A566302F7830A3663161B9687474703A2F2F7777772E6170616368652E6F72672F663161A3663162A42F663162A3663163A82F6631632F782F79A3663164A3663164A3663165A76631652F782F79A3663166A0A3663167A62F6631672F78A3663168A42F663168A3663169A12FA366316AA32F2E2EA366316BA62F66316A2F78A366316CA42F66316BA366316DA12FA366316EA32F2E2EA3663261B9687474703A2F2F7777772E6170616368652E6F72672F663261A3663262A42F663262A3663263A82F6632632F782F79A3663264A3663264A3663265A76632652F782F79A3663266A0A3663267A62F6632672F78A3663268A42F663268A3663269A12FA366326AA32F2E2EA366326BA62F66326A2F78A366326CA42F66326BA366326DA12FA366326EA32F2E2EA3663361D932687474703A2F2F7777772E6170616368652E6F72672F6633612F783F6C6162656C3D4D595F4C4142454C26666F6F3D626172A3663362AF253343253345253236253237253232A3663363A53C3E262722A26634AA746573742F7572692F62A26635A466352F78",
					"<rdf:RDF>\n<rdf:Description rdf:about='f0/x0'>\n<jp:f2i>/</jp:f2i>\n<jp:f1n>/..</jp:f1n>\n<jp:f1a rdf:resource='http://www.apache.org/f1a'/>\n<jp:f1h>/f1h</jp:f1h>\n<jp:f2n>/..</jp:f2n>\n<jp:f2j>/..</jp:f2j>\n<jp:f2f></jp:f2f>\n<jp:f2k>/f2j/x</jp:f2k>\n<jp:f3a rdf:resource='http://www.apache.org/f3a/x?label=MY_LABEL&amp;foo=bar'/>\n<jp:f3b>%3C%3E%26%27%22</jp:f3b>\n<jp:f2a rdf:resource='http://www.apache.org/f2a'/>\n<jp:f1g>/f1g/x</jp:f1g>\n<jp:f2c>/f2c/x/y</jp:f2c>\n<jp:f1j>/..</jp:f1j>\n<jp:f5>f5/x</jp:f5>\n<jp:f1m>/</jp:f1m>\n<jp:f2l>/f2k</jp:f2l>\n<jp:f1i>/</jp:f1i>\n<jp:f1k>/f1j/x</jp:f1k>\n<jp:f1e>f1e/x/y</jp:f1e>\n<jp:f2g>/f2g/x</jp:f2g>\n<jp:f2d>f2d</jp:f2d>\n<jp:f3c>&lt;&gt;&amp;''</jp:f3c>\n<jp:f1b>/f1b</jp:f1b>\n<jp:f2h>/f2h</jp:f2h>\n<jp:f1l>/f1k</jp:f1l>\n<jp:f2b>/f2b</jp:f2b>\n<jp:f1d>f1d</jp:f1d>\n<jp:f2e>f2e/x/y</jp:f2e>\n<jp:f1c>/f1c/x/y</jp:f1c>\n<jp:f1f></jp:f1f>\n<jp:f4>test/uri/b</jp:f4>\n<jp:f2m>/</jp:f2m>\n</rdf:Description>\n</rdf:RDF>\n"
				)
			},
		});
	};

	private void testSerialize(Serializer s, String expected, Object testBean) throws Exception {
		try {
			String r = s.serializeToString(testBean);

			// Specifying "xxx" in the expected results will spit out what we should populate the field with.
			if (expected.equals("xxx")) {
				System.out.println(label + "/" + s.getClass().getSimpleName() + "=\n" + r.replaceAll("\t", "\\\\t").replaceAll("\\\\", "\\\\\\\\").replaceAll("\\\"", "\\\\\\\"").replaceAll("\n", "\\\\n")); // NOT DEBUG
				System.out.println(r);
			}

			assertString(r).setMsg("{0}/{1} serialize-normal failed", label, s.getClass().getSimpleName()).is(expected);

		} catch (AssertionError e) {
			throw e;
		} catch (Exception e) {
			e.printStackTrace();
			throw new AssertionError(label + "/" + s.getClass().getSimpleName() + " failed.  exception=" + e.getLocalizedMessage());
		}
	}

	private void testParse(Serializer s, Parser p, Object testBean) throws Exception {
		try {
			String r = s.serializeToString(testBean);

			TreeMap<String,String> m = p.parse(r, TreeMap.class, String.class, String.class);

			String r2 = Json5Serializer.DEFAULT.toString(m);
			assertString(results.json).setMsg("{0}/{1} parse failed", label, s.getClass().getSimpleName()).is(r2);

		} catch (AssertionError e) {
			throw e;
		} catch (Exception e) {
			e.printStackTrace();
			throw new AssertionError(label + "/" + s.getClass().getSimpleName() + " failed.  exception=" + e.getLocalizedMessage());
		}
	}

	@Test
	public void a1_testJsonSerialize() throws Exception {
		Serializer s = JsonSerializer.create().json5().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).build();
		testSerialize(s, results.json, new TestURI());
	}

	@Test
	public void a1c_testJsonSerialize_usingConfig() throws Exception {
		Serializer s = JsonSerializer.create().json5().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).applyAnnotations(TestURIc.Config.class).build();
		testSerialize(s, results.json, new TestURIc());
	}

	@Test
	public void a2_testJsonParse() throws Exception {
		Serializer s = JsonSerializer.create().json5().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).build();
		testParse(s, JsonParser.DEFAULT, new TestURI());
	}

	@Test
	public void a2c_testJsonParse_usingConfig() throws Exception {
		Serializer s = JsonSerializer.create().json5().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).applyAnnotations(TestURIc.Config.class).build();
		testParse(s, JsonParser.DEFAULT.copy().applyAnnotations(TestURIc.class).build(), new TestURIc());
	}

	@Test
	public void b1_testXmlSerialize() throws Exception {
		Serializer s = XmlSerializer.create().sq().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).build();
		testSerialize(s, results.xml, new TestURI());
	}

	@Test
	public void b1c_testXmlSerialize_usingConfig() throws Exception {
		Serializer s = XmlSerializer.create().sq().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).applyAnnotations(TestURIc.Config.class).build();
		testSerialize(s, results.xml, new TestURIc());
	}

	@Test
	public void b2_testXmlParse() throws Exception {
		Serializer s = XmlSerializer.create().sq().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).build();
		testParse(s, XmlParser.DEFAULT, new TestURI());
	}

	@Test
	public void b2c_testXmlParse_usingConfig() throws Exception {
		Serializer s = XmlSerializer.create().sq().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).applyAnnotations(TestURIc.Config.class).build();
		testParse(s, XmlParser.DEFAULT.copy().applyAnnotations(TestURIc.Config.class).build(), new TestURIc());
	}

	@Test
	public void c1_testHtmlSerialize() throws Exception {
		Serializer s = HtmlSerializer.create().sq().uriAnchorText(AnchorText.LAST_TOKEN).uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).build();
		testSerialize(s, results.html, new TestURI());
	}

	@Test
	public void c1c_testHtmlSerialize_usingConfig() throws Exception {
		Serializer s = HtmlSerializer.create().sq().uriAnchorText(AnchorText.LAST_TOKEN).uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).applyAnnotations(TestURIc.Config.class).build();
		testSerialize(s, results.html, new TestURIc());
	}

	@Test
	public void c2_testHtmlParse() throws Exception {
		Serializer s = HtmlSerializer.create().sq().uriAnchorText(AnchorText.LAST_TOKEN).uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).build();
		testParse(s, HtmlParser.DEFAULT, new TestURI());
	}

	@Test
	public void c2c_testHtmlParse_usingConfig() throws Exception {
		Serializer s = HtmlSerializer.create().sq().uriAnchorText(AnchorText.LAST_TOKEN).uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).applyAnnotations(TestURIc.Config.class).build();
		testParse(s, HtmlParser.DEFAULT.copy().applyAnnotations(TestURIc.Config.class).build(), new TestURIc());
	}

	@Test
	public void d1_testUonSerialize() throws Exception {
		Serializer s = UonSerializer.create().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).build();
		testSerialize(s, results.uon, new TestURI());
	}

	@Test
	public void d1c_testUonSerialize_usingConfig() throws Exception {
		Serializer s = UonSerializer.create().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).applyAnnotations(TestURIc.Config.class).build();
		testSerialize(s, results.uon, new TestURIc());
	}

	@Test
	public void d2_testUonParse() throws Exception {
		Serializer s = UonSerializer.create().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).build();
		testParse(s, UonParser.DEFAULT, new TestURI());
	}

	@Test
	public void d2c_testUonParse_usingConfig() throws Exception {
		Serializer s = UonSerializer.create().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).applyAnnotations(TestURIc.Config.class).build();
		testParse(s, UonParser.DEFAULT.copy().applyAnnotations(TestURIc.Config.class).build(), new TestURIc());
	}

	@Test
	public void e1_testUrlEncodingSerialize() throws Exception {
		Serializer s = UrlEncodingSerializer.create().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).build();
		testSerialize(s, results.urlEncoding, new TestURI());
	}

	@Test
	public void e1c_testUrlEncodingSerialize_usingConfig() throws Exception {
		Serializer s = UrlEncodingSerializer.create().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).applyAnnotations(TestURIc.Config.class).build();
		testSerialize(s, results.urlEncoding, new TestURIc());
	}

	@Test
	public void e2_testUrlEncodingParse() throws Exception {
		Serializer s = UrlEncodingSerializer.create().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).build();
		testParse(s, UrlEncodingParser.DEFAULT, new TestURI());
	}

	@Test
	public void e2c_testUrlEncodingParse_usingConfig() throws Exception {
		Serializer s = UrlEncodingSerializer.create().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).applyAnnotations(TestURIc.Config.class).build();
		testParse(s, UrlEncodingParser.DEFAULT.copy().applyAnnotations(TestURIc.Config.class).build(), new TestURIc());
	}

	@Test
	public void f1_testMsgPackSerialize() throws Exception {
		Serializer s = MsgPackSerializer.create().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).build();
		testSerialize(s, results.msgPack, new TestURI());
	}

	@Test
	public void f1c_testMsgPackSerialize_usingConfig() throws Exception {
		Serializer s = MsgPackSerializer.create().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).applyAnnotations(TestURIc.Config.class).build();
		testSerialize(s, results.msgPack, new TestURIc());
	}

	@Test
	public void f2_testMsgPackParse() throws Exception {
		Serializer s = MsgPackSerializer.create().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).build();
		testParse(s, MsgPackParser.DEFAULT, new TestURI());
	}

	@Test
	public void f2c_testMsgPackParse_usingConfig() throws Exception {
		Serializer s = MsgPackSerializer.create().uriContext(input.context).uriResolution(input.resolution).uriRelativity(input.relativity).applyAnnotations(TestURIc.Config.class).build();
		testParse(s, MsgPackParser.DEFAULT.copy().applyAnnotations(TestURIc.Config.class).build(), new TestURIc());
	}
}
