/*
 * Decompiled with CFR 0.152.
 */
package org.locationtech.jts.io.twkb;

import java.io.ByteArrayOutputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Objects;
import org.locationtech.jts.geom.CoordinateSequence;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.MultiPoint;
import org.locationtech.jts.geom.MultiPolygon;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.io.twkb.BoundsExtractor;
import org.locationtech.jts.io.twkb.TWKBHeader;
import org.locationtech.jts.io.twkb.Varint;

public class TWKBWriter {
    private TWKBHeader paramsHeader = new TWKBHeader().setXyPrecision(7).setZPrecision(0).setMPrecision(0).setHasBBOX(false).setHasSize(false);

    public TWKBWriter setXYPrecision(int xyprecision) {
        if (xyprecision < -7 || xyprecision > 7) {
            throw new IllegalArgumentException("X/Z precision cannot be greater than 7 or less than -7");
        }
        this.paramsHeader = this.paramsHeader.setXyPrecision(xyprecision);
        return this;
    }

    public TWKBWriter setEncodeZ(boolean includeZDimension) {
        this.paramsHeader = this.paramsHeader.setHasZ(includeZDimension);
        return this;
    }

    public TWKBWriter setEncodeM(boolean includeMDimension) {
        this.paramsHeader = this.paramsHeader.setHasM(includeMDimension);
        return this;
    }

    public TWKBWriter setZPrecision(int zprecision) {
        if (zprecision < 0 || zprecision > 7) {
            throw new IllegalArgumentException("Z precision cannot be negative or greater than 7");
        }
        this.paramsHeader = this.paramsHeader.setZPrecision(zprecision);
        return this;
    }

    public TWKBWriter setMPrecision(int mprecision) {
        if (mprecision < 0 || mprecision > 7) {
            throw new IllegalArgumentException("M precision cannot be negative or greater than 7");
        }
        this.paramsHeader = this.paramsHeader.setMPrecision(mprecision);
        return this;
    }

    public TWKBWriter setIncludeSize(boolean includeSize) {
        this.paramsHeader = this.paramsHeader.setHasSize(includeSize);
        return this;
    }

    public TWKBWriter setIncludeBbox(boolean includeBbox) {
        this.paramsHeader = this.paramsHeader.setHasBBOX(includeBbox);
        return this;
    }

    public byte[] write(Geometry geom) {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        try {
            this.write(geom, out);
        }
        catch (IOException ex) {
            throw new RuntimeException("Unexpected IOException caught: " + ex.getMessage(), ex);
        }
        return out.toByteArray();
    }

    public void write(Geometry geom, OutputStream out) throws IOException {
        this.write(geom, new DataOutputStream(out));
    }

    public void write(Geometry geom, DataOutput out) throws IOException {
        Objects.requireNonNull(geom, "geometry is null");
        this.write(geom, out, this.paramsHeader, false);
    }

    private TWKBHeader write(Geometry geometry, DataOutput out, TWKBHeader params, boolean forcePreserveHeaderDimensions) throws IOException {
        Objects.requireNonNull(geometry, "Geometry is null");
        Objects.requireNonNull(out, "DataOutput is null");
        Objects.requireNonNull(params, "TWKBHeader is null");
        TWKBHeader header = this.prepareHeader(geometry, new TWKBHeader(params), forcePreserveHeaderDimensions);
        if (header.hasSize()) {
            BufferedDataOutput bufferedBody = new BufferedDataOutput();
            this.writeGeometryBody(geometry, bufferedBody, header);
            int bodySize = bufferedBody.size();
            header = header.setGeometryBodySize(bodySize);
            TWKBWriter.writeHeaderTo(header, out);
            out.write(bufferedBody.content());
        } else {
            TWKBWriter.writeHeaderTo(header, out);
            this.writeGeometryBody(geometry, out, header);
        }
        return header;
    }

    private static void writeHeaderTo(TWKBHeader header, DataOutput out) throws IOException {
        Objects.requireNonNull(out);
        int geometryType = header.geometryType().getValue();
        int precisionHeader = Varint.zigZagEncode(header.xyPrecision()) << 4;
        int typeAndPrecisionHeader = precisionHeader | geometryType;
        int metadataHeader = (header.hasBBOX() ? 1 : 0) | (header.hasSize() ? 2 : 0) | (header.hasIdList() ? 4 : 0) | (header.hasExtendedPrecision() ? 8 : 0) | (header.isEmpty() ? 16 : 0);
        out.writeByte(typeAndPrecisionHeader);
        out.writeByte(metadataHeader);
        if (header.hasExtendedPrecision()) {
            int extendedDimsHeader = (header.hasZ() ? 1 : 0) | (header.hasM() ? 2 : 0);
            extendedDimsHeader |= header.zPrecision() << 2;
            out.writeByte(extendedDimsHeader |= header.mPrecision() << 5);
        }
        if (header.hasSize()) {
            Varint.writeUnsignedVarInt(header.geometryBodySize(), out);
        }
    }

    private TWKBHeader prepareHeader(Geometry geometry, TWKBHeader params, boolean forcePreserveHeaderDimensions) {
        boolean isEmpty = geometry.isEmpty();
        TWKBHeader.GeometryType geometryType = TWKBHeader.GeometryType.valueOf(geometry.getClass());
        TWKBHeader header = forcePreserveHeaderDimensions ? params : TWKBWriter.setDimensions(geometry, params);
        header.setEmpty(isEmpty);
        header.setGeometryType(geometryType);
        if (isEmpty && header.hasBBOX()) {
            header = header.setHasBBOX(false);
        }
        return header;
    }

    private void writeGeometryBody(Geometry geom, DataOutput out, TWKBHeader header) throws IOException {
        if (header.isEmpty()) {
            return;
        }
        if (header.hasBBOX()) {
            this.writeBbox(geom, out, header);
        }
        TWKBHeader.GeometryType geometryType = TWKBHeader.GeometryType.valueOf(geom.getClass());
        switch (geometryType) {
            case POINT: {
                this.writePoint((Point)geom, out, header);
                return;
            }
            case LINESTRING: {
                this.writeLineString((LineString)geom, out, header, new long[header.getDimensions()]);
                return;
            }
            case POLYGON: {
                this.writePolygon((Polygon)geom, out, header, new long[header.getDimensions()]);
                return;
            }
            case MULTIPOINT: {
                this.writeMultiPoint((MultiPoint)geom, out, header);
                return;
            }
            case MULTILINESTRING: {
                this.writeMultiLineString((MultiLineString)geom, out, header);
                return;
            }
            case MULTIPOLYGON: {
                this.writeMultiPolygon((MultiPolygon)geom, out, header);
                return;
            }
            case GEOMETRYCOLLECTION: {
                this.writeGeometryCollection((GeometryCollection)geom, out, header);
                return;
            }
        }
    }

    private void writePoint(Point geom, DataOutput out, TWKBHeader header) throws IOException {
        assert (!geom.isEmpty());
        CoordinateSequence seq2 = geom.getCoordinateSequence();
        int dimensions = header.getDimensions();
        for (int d = 0; d < dimensions; ++d) {
            this.writeOrdinate(seq2.getOrdinate(0, d), 0L, header.getPrecision(d), out);
        }
    }

    private void writeCoordinateSequence(CoordinateSequence coordinateSequence, DataOutput out, TWKBHeader header, long[] prev, int minNPoints) throws IOException {
        int dimensions = header.getDimensions();
        long[] delta = new long[dimensions];
        int nPoints = 0;
        int nPointsRemaining = coordinateSequence.size();
        BufferedDataOutput bufferedOut = new BufferedDataOutput();
        for (int coordIndex = 0; coordIndex < coordinateSequence.size(); ++coordIndex) {
            int ordinateIndex;
            long diff = 0L;
            --nPointsRemaining;
            for (ordinateIndex = 0; ordinateIndex < dimensions; ++ordinateIndex) {
                int precision = header.getPrecision(ordinateIndex);
                double ordinate = coordinateSequence.getOrdinate(coordIndex, ordinateIndex);
                long preciseOrdinate = this.makePrecise(ordinate, precision);
                delta[ordinateIndex] = preciseOrdinate - prev[ordinateIndex];
                prev[ordinateIndex] = preciseOrdinate;
                diff += Math.abs(delta[ordinateIndex]);
            }
            if (coordIndex != 0 && diff == 0L && nPoints + nPointsRemaining > minNPoints) continue;
            for (ordinateIndex = 0; ordinateIndex < header.getDimensions(); ++ordinateIndex) {
                Varint.writeSignedVarLong(delta[ordinateIndex], bufferedOut);
            }
            ++nPoints;
        }
        Varint.writeUnsignedVarInt(nPoints, out);
        out.write(bufferedOut.content());
    }

    private long writeOrdinate(double ordinate, long previousOrdinateValue, int precision, DataOutput out) throws IOException {
        long preciseOrdinate = this.makePrecise(ordinate, precision);
        long delta = preciseOrdinate - previousOrdinateValue;
        Varint.writeSignedVarLong(delta, out);
        return preciseOrdinate;
    }

    private long makePrecise(double value, int precision) {
        return Math.round(value * Math.pow(10.0, precision));
    }

    private void writeLineString(LineString geom, DataOutput out, TWKBHeader header, long[] prev) throws IOException {
        this.writeCoordinateSequence(geom.getCoordinateSequence(), out, header, prev, 3);
    }

    private void writePolygon(Polygon geom, DataOutput out, TWKBHeader header, long[] prev) throws IOException {
        if (geom.isEmpty()) {
            Varint.writeUnsignedVarInt(0, out);
            return;
        }
        int numInteriorRing = geom.getNumInteriorRing();
        int nrings = 1 + numInteriorRing;
        Varint.writeUnsignedVarInt(nrings, out);
        this.writeLinearRing(geom.getExteriorRing(), out, header, prev);
        for (int r = 0; r < numInteriorRing; ++r) {
            this.writeLinearRing(geom.getInteriorRingN(r), out, header, prev);
        }
    }

    private void writeLinearRing(LinearRing geom, DataOutput out, TWKBHeader header, long[] prev) throws IOException {
        if (geom.isEmpty()) {
            Varint.writeUnsignedVarInt(0, out);
            return;
        }
        this.writeCoordinateSequence(geom.getCoordinateSequence(), out, header, prev, 3);
    }

    private void writeMultiPoint(MultiPoint geom, DataOutput out, TWKBHeader header) throws IOException {
        assert (!geom.isEmpty());
        CoordinateSequence seq2 = geom.getFactory().getCoordinateSequenceFactory().create(geom.getCoordinates());
        this.writeCoordinateSequence(seq2, out, header, new long[header.getDimensions()], 2);
    }

    private void writeMultiLineString(MultiLineString geom, DataOutput out, TWKBHeader header) throws IOException {
        int size = this.writeNumGeometries(geom, out);
        long[] prev = new long[header.getDimensions()];
        for (int i = 0; i < size; ++i) {
            this.writeLineString((LineString)geom.getGeometryN(i), out, header, prev);
        }
    }

    private void writeMultiPolygon(MultiPolygon geom, DataOutput out, TWKBHeader header) throws IOException {
        int size = this.writeNumGeometries(geom, out);
        long[] prev = new long[header.getDimensions()];
        for (int i = 0; i < size; ++i) {
            this.writePolygon((Polygon)geom.getGeometryN(i), out, header, prev);
        }
    }

    private void writeGeometryCollection(GeometryCollection geom, DataOutput out, TWKBHeader header) throws IOException {
        int size = this.writeNumGeometries(geom, out);
        for (int i = 0; i < size; ++i) {
            Geometry geometryN = geom.getGeometryN(i);
            boolean forcePreserveDimensions = geometryN.isEmpty();
            this.write(geometryN, out, header, forcePreserveDimensions);
        }
    }

    private int writeNumGeometries(GeometryCollection geom, DataOutput out) throws IOException {
        int size = geom.getNumGeometries();
        Varint.writeUnsignedVarInt(size, out);
        return size;
    }

    private void writeBbox(Geometry geom, DataOutput out, TWKBHeader header) throws IOException {
        int dimensions = header.getDimensions();
        double[] boundsCoordinates = TWKBWriter.computeEnvelope(geom, dimensions);
        for (int d = 0; d < dimensions; ++d) {
            int precision = header.getPrecision(d);
            double min2 = boundsCoordinates[2 * d];
            double max2 = boundsCoordinates[2 * d + 1];
            long preciseMin = this.writeOrdinate(min2, 0L, precision, out);
            this.writeOrdinate(max2, preciseMin, precision, out);
        }
    }

    private static double[] computeEnvelope(Geometry geom, int dimensions) {
        BoundsExtractor extractor = new BoundsExtractor(dimensions);
        geom.apply(extractor);
        return extractor.ordinates;
    }

    private static TWKBHeader setDimensions(Geometry g, TWKBHeader header) {
        if (g.isEmpty()) {
            return header.setHasZ(false).setHasM(false);
        }
        if (g instanceof Point) {
            return TWKBWriter.setDimensions(((Point)g).getCoordinateSequence(), header);
        }
        if (g instanceof LineString) {
            return TWKBWriter.setDimensions(((LineString)g).getCoordinateSequence(), header);
        }
        if (g instanceof Polygon) {
            return TWKBWriter.setDimensions(((Polygon)g).getExteriorRing().getCoordinateSequence(), header);
        }
        return TWKBWriter.setDimensions(g.getGeometryN(0), header);
    }

    private static TWKBHeader setDimensions(CoordinateSequence seq2, TWKBHeader header) {
        boolean hasZ = seq2.hasZ();
        boolean hasM = seq2.hasM();
        return header.setHasZ(hasZ).setHasM(hasM);
    }

    private static class BufferedDataOutput
    extends DataOutputStream {
        public BufferedDataOutput() {
            super(new ByteArrayOutputStream());
        }

        public byte[] content() {
            return ((ByteArrayOutputStream)this.out).toByteArray();
        }
    }
}

