/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.io;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.reflect.RecordComponent;
import java.nio.ByteBuffer;
import java.time.Instant;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.IntUnaryOperator;
import java.util.function.ToLongFunction;
import java.util.stream.Collectors;
import org.eclipse.jetty.io.AbstractRetainableByteBuffer;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.io.RetainableByteBuffer;
import org.eclipse.jetty.io.internal.CompoundPool;
import org.eclipse.jetty.io.internal.QueuedPool;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.ConcurrentPool;
import org.eclipse.jetty.util.Pool;
import org.eclipse.jetty.util.annotation.ManagedAttribute;
import org.eclipse.jetty.util.annotation.ManagedObject;
import org.eclipse.jetty.util.annotation.ManagedOperation;
import org.eclipse.jetty.util.component.Dumpable;
import org.eclipse.jetty.util.component.DumpableCollection;
import org.eclipse.jetty.util.component.DumpableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ManagedObject
public class ArrayByteBufferPool
implements ByteBufferPool,
Dumpable {
    static final int DEFAULT_FACTOR = 4096;
    static final int DEFAULT_MAX_CAPACITY_BY_FACTOR = 16;
    private final RetainedBucket[] _direct;
    private final RetainedBucket[] _indirect;
    private final int _minCapacity;
    private final int _maxCapacity;
    private final long _maxHeapMemory;
    private final long _maxDirectMemory;
    private final IntUnaryOperator _bucketIndexFor;
    private final IntUnaryOperator _bucketCapacity;
    private final AtomicBoolean _evictor = new AtomicBoolean(false);
    private final ConcurrentMap<Integer, Long> _noBucketDirectAcquires = new ConcurrentHashMap<Integer, Long>();
    private final ConcurrentMap<Integer, Long> _noBucketIndirectAcquires = new ConcurrentHashMap<Integer, Long>();
    private boolean _statisticsEnabled;

    public ArrayByteBufferPool() {
        this(0, -1, -1);
    }

    public ArrayByteBufferPool(int minCapacity, int factor, int maxCapacity) {
        this(minCapacity, factor, maxCapacity, Integer.MAX_VALUE);
    }

    public ArrayByteBufferPool(int minCapacity, int factor, int maxCapacity, int maxBucketSize) {
        this(minCapacity, factor, maxCapacity, maxBucketSize, 0L, 0L);
    }

    public ArrayByteBufferPool(int minCapacity, int factor, int maxCapacity, int maxBucketSize, long maxHeapMemory, long maxDirectMemory) {
        this(minCapacity, factor, maxCapacity, maxBucketSize, maxHeapMemory, maxDirectMemory, null, null);
    }

    protected ArrayByteBufferPool(int minCapacity, int factor, int maxCapacity, int maxBucketSize, long maxHeapMemory, long maxDirectMemory, IntUnaryOperator bucketIndexFor, IntUnaryOperator bucketCapacity) {
        if (minCapacity <= 0) {
            minCapacity = 0;
        }
        int n = factor = factor <= 0 ? 4096 : factor;
        if (maxCapacity <= 0) {
            maxCapacity = 16 * factor;
        }
        if (maxCapacity % factor != 0 || factor >= maxCapacity) {
            throw new IllegalArgumentException(String.format("The capacity factor(%d) must be a divisor of maxCapacity(%d)", factor, maxCapacity));
        }
        int f = factor;
        if (bucketIndexFor == null) {
            bucketIndexFor = c -> (c - 1) / f;
        }
        if (bucketCapacity == null) {
            bucketCapacity = i -> (i + 1) * f;
        }
        int length = bucketIndexFor.applyAsInt(maxCapacity) + 1;
        RetainedBucket[] directArray = new RetainedBucket[length];
        RetainedBucket[] indirectArray = new RetainedBucket[length];
        for (int i2 = 0; i2 < directArray.length; ++i2) {
            int capacity = Math.min(bucketCapacity.applyAsInt(i2), maxCapacity);
            directArray[i2] = new RetainedBucket(capacity, maxBucketSize);
            indirectArray[i2] = new RetainedBucket(capacity, maxBucketSize);
        }
        this._minCapacity = minCapacity;
        this._maxCapacity = maxCapacity;
        this._direct = directArray;
        this._indirect = indirectArray;
        this._maxHeapMemory = this.maxMemory(maxHeapMemory);
        this._maxDirectMemory = this.maxMemory(maxDirectMemory);
        this._bucketIndexFor = bucketIndexFor;
        this._bucketCapacity = bucketCapacity;
    }

    private long maxMemory(long maxMemory) {
        if (maxMemory < 0L) {
            return -1L;
        }
        if (maxMemory == 0L) {
            return Runtime.getRuntime().maxMemory() / 8L;
        }
        return maxMemory;
    }

    @ManagedAttribute(value="Whether statistics are enabled")
    public boolean isStatisticsEnabled() {
        return this._statisticsEnabled;
    }

    public void setStatisticsEnabled(boolean enabled) {
        this._statisticsEnabled = enabled;
    }

    @ManagedAttribute(value="The minimum pooled buffer capacity")
    public int getMinCapacity() {
        return this._minCapacity;
    }

    @ManagedAttribute(value="The maximum pooled buffer capacity")
    public int getMaxCapacity() {
        return this._maxCapacity;
    }

    @Override
    public RetainableByteBuffer acquire(int size, boolean direct) {
        RetainedBucket bucket = this.bucketFor(size, direct);
        if (bucket == null) {
            this.recordNoBucketAcquire(size, direct);
            return RetainableByteBuffer.wrap(BufferUtil.allocate(size, direct));
        }
        bucket.recordAcquire();
        Pool.Entry<RetainableByteBuffer> entry = bucket.getPool().acquire();
        if (entry == null) {
            ByteBuffer buffer = BufferUtil.allocate(bucket.getCapacity(), direct);
            return new ReservedBuffer(buffer, bucket);
        }
        bucket.recordPooled();
        RetainableByteBuffer buffer = entry.getPooled();
        ((Buffer)buffer).acquire();
        return buffer;
    }

    private void recordNoBucketAcquire(int size, boolean direct) {
        if (this.isStatisticsEnabled()) {
            ConcurrentMap<Integer, Long> map = direct ? this._noBucketDirectAcquires : this._noBucketIndirectAcquires;
            int idx = this._bucketIndexFor.applyAsInt(size);
            int key = this._bucketCapacity.applyAsInt(idx);
            map.compute(key, (k, v) -> {
                if (v == null) {
                    return 1L;
                }
                return v + 1L;
            });
        }
    }

    @Override
    public boolean removeAndRelease(RetainableByteBuffer buffer) {
        RetainableByteBuffer actual = buffer;
        while (actual instanceof RetainableByteBuffer.Wrapper) {
            RetainableByteBuffer.Wrapper wrapper = (RetainableByteBuffer.Wrapper)actual;
            actual = wrapper.getWrapped();
        }
        if (actual instanceof ReservedBuffer) {
            ReservedBuffer reservedBuffer = (ReservedBuffer)actual;
            reservedBuffer.remove();
            return buffer.release();
        }
        if (actual instanceof Buffer) {
            Buffer poolBuffer = (Buffer)actual;
            poolBuffer.remove();
            return buffer.release();
        }
        return ByteBufferPool.super.removeAndRelease(buffer);
    }

    private void reserve(RetainedBucket bucket, ByteBuffer byteBuffer) {
        bucket.recordRelease();
        Pool.Entry<RetainableByteBuffer> entry = bucket.getPool().reserve();
        if (entry == null) {
            bucket.recordNonPooled();
            return;
        }
        BufferUtil.reset(byteBuffer);
        Buffer pooledBuffer = new Buffer(byteBuffer, bucket, entry);
        if (entry.enable(pooledBuffer, false)) {
            this.checkMaxMemory(bucket, byteBuffer.isDirect());
            return;
        }
        bucket.recordNonPooled();
        entry.remove();
    }

    private void release(RetainedBucket bucket, Pool.Entry<RetainableByteBuffer> entry) {
        bucket.recordRelease();
        if (entry.isTerminated()) {
            return;
        }
        RetainableByteBuffer buffer = entry.getPooled();
        BufferUtil.reset(buffer.getByteBuffer());
        int used = ((Buffer)buffer).use();
        if (entry.release()) {
            if (used % 100 == 0) {
                this.checkMaxMemory(bucket, buffer.isDirect());
            }
            return;
        }
        bucket.recordRemove();
        entry.remove();
    }

    private boolean remove(RetainedBucket bucket, Pool.Entry<RetainableByteBuffer> entry) {
        bucket.recordRemove();
        return entry.remove();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void checkMaxMemory(RetainedBucket bucket, boolean direct) {
        long max2;
        long l = max2 = direct ? this._maxDirectMemory : this._maxHeapMemory;
        if (max2 <= 0L || !this._evictor.compareAndSet(false, true)) {
            return;
        }
        try {
            long memory = this.getTotalMemory(direct);
            long excess = memory - max2;
            if (excess > 0L) {
                bucket.recordEvict();
                this.evict(excess, direct);
            }
        }
        finally {
            this._evictor.set(false);
        }
    }

    private void evict(long excessMemory, boolean direct) {
        RetainedBucket[] buckets = direct ? this._direct : this._indirect;
        int length = buckets.length;
        int index = ThreadLocalRandom.current().nextInt(length);
        for (int c = 0; c < length; ++c) {
            int evicted;
            RetainedBucket bucket = buckets[index++];
            if (index == length) {
                index = 0;
            }
            if ((excessMemory -= (long)(evicted = bucket.evict())) > 0L) continue;
            return;
        }
    }

    public Pool<RetainableByteBuffer> poolFor(int capacity, boolean direct) {
        RetainedBucket bucket = this.bucketFor(capacity, direct);
        return bucket == null ? null : bucket.getPool();
    }

    private RetainedBucket bucketFor(int capacity, boolean direct) {
        RetainedBucket[] buckets;
        if (capacity < this.getMinCapacity()) {
            return null;
        }
        int idx = this._bucketIndexFor.applyAsInt(capacity);
        RetainedBucket[] retainedBucketArray = buckets = direct ? this._direct : this._indirect;
        if (idx >= buckets.length) {
            return null;
        }
        return buckets[idx];
    }

    @ManagedAttribute(value="The number of pooled direct ByteBuffers")
    public long getDirectByteBufferCount() {
        return this.getByteBufferCount(true);
    }

    @ManagedAttribute(value="The number of pooled heap ByteBuffers")
    public long getHeapByteBufferCount() {
        return this.getByteBufferCount(false);
    }

    private long getByteBufferCount(boolean direct) {
        RetainedBucket[] buckets = direct ? this._direct : this._indirect;
        return Arrays.stream(buckets).mapToLong(bucket -> bucket.getPool().size()).sum();
    }

    @ManagedAttribute(value="The number of pooled direct ByteBuffers that are available")
    public long getAvailableDirectByteBufferCount() {
        return this.getAvailableByteBufferCount(true);
    }

    @ManagedAttribute(value="The number of pooled heap ByteBuffers that are available")
    public long getAvailableHeapByteBufferCount() {
        return this.getAvailableByteBufferCount(false);
    }

    private long getAvailableByteBufferCount(boolean direct) {
        RetainedBucket[] buckets = direct ? this._direct : this._indirect;
        return Arrays.stream(buckets).mapToLong(bucket -> bucket.getPool().getIdleCount()).sum();
    }

    @ManagedAttribute(value="The total bytes retained by direct ByteBuffers")
    public long getDirectMemory() {
        return this.getTotalMemory(true);
    }

    @ManagedAttribute(value="The total bytes retained by heap ByteBuffers")
    public long getHeapMemory() {
        return this.getTotalMemory(false);
    }

    private long getTotalMemory(boolean direct) {
        return this.getMemory(direct, bucket -> bucket.getPool().size());
    }

    private long getMemory(boolean direct, ToLongFunction<RetainedBucket> count) {
        long size = 0L;
        for (RetainedBucket bucket : direct ? this._direct : this._indirect) {
            size += count.applyAsLong(bucket) * (long)bucket.getCapacity();
        }
        return size;
    }

    @ManagedAttribute(value="The available bytes retained by direct ByteBuffers")
    public long getAvailableDirectMemory() {
        return this.getAvailableMemory(true);
    }

    @ManagedAttribute(value="The available bytes retained by heap ByteBuffers")
    public long getAvailableHeapMemory() {
        return this.getAvailableMemory(false);
    }

    private long getAvailableMemory(boolean direct) {
        return this.getMemory(direct, bucket -> bucket.getPool().getIdleCount());
    }

    @ManagedAttribute(value="The heap buckets statistics")
    public List<Map<String, Object>> getHeapBucketsStatistics() {
        return this.getBucketsStatistics(false);
    }

    @ManagedAttribute(value="The direct buckets statistics")
    public List<Map<String, Object>> getDirectBucketsStatistics() {
        return this.getBucketsStatistics(true);
    }

    private List<Map<String, Object>> getBucketsStatistics(boolean direct) {
        RetainedBucket[] buckets = direct ? this._direct : this._indirect;
        return Arrays.stream(buckets).map(b -> b.getStatistics().toMap()).toList();
    }

    @ManagedAttribute(value="The acquires for direct non-pooled bucket capacities")
    public Map<Integer, Long> getNoBucketDirectAcquires() {
        return this.getNoBucketAcquires(true);
    }

    @ManagedAttribute(value="The acquires for heap non-pooled bucket capacities")
    public Map<Integer, Long> getNoBucketHeapAcquires() {
        return this.getNoBucketAcquires(false);
    }

    private Map<Integer, Long> getNoBucketAcquires(boolean direct) {
        return new HashMap<Integer, Long>(direct ? this._noBucketDirectAcquires : this._noBucketIndirectAcquires);
    }

    @Override
    @ManagedOperation(value="Clears this ByteBufferPool", impact="ACTION")
    public void clear() {
        this.clearBuckets(this._direct);
        this._noBucketDirectAcquires.clear();
        this.clearBuckets(this._indirect);
        this._noBucketIndirectAcquires.clear();
    }

    private void clearBuckets(RetainedBucket[] buckets) {
        for (RetainedBucket bucket : buckets) {
            bucket.clear();
        }
    }

    @Override
    public void dump(Appendable out, String indent) throws IOException {
        Dumpable.dumpObjects(out, indent, this, DumpableCollection.fromArray("direct", this._direct), new DumpableMap("direct non-pooled acquisitions", this._noBucketDirectAcquires), DumpableCollection.fromArray("indirect", this._indirect), new DumpableMap("heap non-pooled acquisitions", this._noBucketIndirectAcquires));
    }

    public String toString() {
        return String.format("%s{min=%d,max=%d,buckets=%d,heap=%d/%d,direct=%d/%d}", super.toString(), this._minCapacity, this._maxCapacity, this._direct.length, this.getHeapMemory(), this._maxHeapMemory, this.getDirectMemory(), this._maxDirectMemory);
    }

    private class RetainedBucket {
        private final LongAdder _acquires = new LongAdder();
        private final LongAdder _pooled = new LongAdder();
        private final LongAdder _nonPooled = new LongAdder();
        private final LongAdder _evicts = new LongAdder();
        private final LongAdder _removes = new LongAdder();
        private final LongAdder _releases = new LongAdder();
        private final Pool<RetainableByteBuffer> _pool;
        private final int _capacity;

        private RetainedBucket(int capacity, int poolSize) {
            this._pool = poolSize <= 256 ? new ConcurrentPool<RetainableByteBuffer>(ConcurrentPool.StrategyType.THREAD_ID, poolSize, e2 -> 1) : new BucketCompoundPool(new ConcurrentPool<RetainableByteBuffer>(ConcurrentPool.StrategyType.THREAD_ID, 256, e2 -> 1), new QueuedPool<RetainableByteBuffer>(poolSize - 256));
            this._capacity = capacity;
        }

        public void recordAcquire() {
            if (ArrayByteBufferPool.this.isStatisticsEnabled()) {
                this._acquires.increment();
            }
        }

        public void recordEvict() {
            if (ArrayByteBufferPool.this.isStatisticsEnabled()) {
                this._evicts.increment();
            }
        }

        public void recordNonPooled() {
            if (ArrayByteBufferPool.this.isStatisticsEnabled()) {
                this._nonPooled.increment();
            }
        }

        public void recordPooled() {
            if (ArrayByteBufferPool.this.isStatisticsEnabled()) {
                this._pooled.increment();
            }
        }

        public void recordRelease() {
            if (ArrayByteBufferPool.this.isStatisticsEnabled()) {
                this._releases.increment();
            }
        }

        public void recordRemove() {
            if (ArrayByteBufferPool.this.isStatisticsEnabled()) {
                this._removes.increment();
            }
        }

        private int getCapacity() {
            return this._capacity;
        }

        private Pool<RetainableByteBuffer> getPool() {
            return this._pool;
        }

        private int evict() {
            Pool.Entry<RetainableByteBuffer> entry;
            Pool<RetainableByteBuffer> pool = this._pool;
            if (pool instanceof BucketCompoundPool) {
                BucketCompoundPool compound = (BucketCompoundPool)pool;
                entry = compound.evict();
            } else {
                entry = this._pool.acquire();
            }
            if (entry == null) {
                return 0;
            }
            this.recordRemove();
            entry.remove();
            return this.getCapacity();
        }

        private Statistics getStatistics() {
            long pooled = this._pooled.longValue();
            long acquires = this._acquires.longValue();
            float hitRatio = acquires == 0L ? Float.NaN : (float)pooled * 100.0f / (float)acquires;
            return new Statistics(this.getCapacity(), this.getPool().getInUseCount(), this.getPool().size(), pooled, acquires, this._releases.longValue(), hitRatio, this._nonPooled.longValue(), this._evicts.longValue(), this._removes.longValue());
        }

        public void clear() {
            this._acquires.reset();
            this._pooled.reset();
            this._nonPooled.reset();
            this._evicts.reset();
            this._removes.reset();
            this._releases.reset();
            this.getPool().stream().forEach(Pool.Entry::remove);
        }

        public String toString() {
            return String.format("%s[%s]", super.toString(), this.getStatistics());
        }

        private static class BucketCompoundPool
        extends CompoundPool<RetainableByteBuffer> {
            private BucketCompoundPool(ConcurrentPool<RetainableByteBuffer> concurrentBucket, QueuedPool<RetainableByteBuffer> queuedBucket) {
                super(concurrentBucket, queuedBucket);
            }

            private Pool.Entry<RetainableByteBuffer> evict() {
                Pool.Entry<RetainableByteBuffer> entry = this.getSecondaryPool().acquire();
                if (entry == null) {
                    entry = this.getPrimaryPool().acquire();
                }
                return entry;
            }
        }

        private record Statistics(int capacity, int inUseEntries, int totalEntries, long pooled, long acquires, long releases, float hitRatio, long nonPooled, long evicts, long removes) {
            private Map<String, Object> toMap() {
                try {
                    HashMap<String, Object> statistics = new HashMap<String, Object>();
                    for (RecordComponent c : this.getClass().getRecordComponents()) {
                        statistics.put(c.getName(), c.getAccessor().invoke((Object)this, new Object[0]));
                    }
                    return statistics;
                }
                catch (Throwable x) {
                    return Map.of();
                }
            }

            @Override
            public String toString() {
                return "capacity=%d,in-use=%d/%d,pooled/acquires/releases=%d/%d/%d(%.3f%%),non-pooled/evicts/removes=%d/%d/%d".formatted(this.capacity, this.inUseEntries, this.totalEntries, this.pooled, this.acquires, this.releases, Float.valueOf(this.hitRatio), this.nonPooled, this.evicts, this.removes);
            }
        }
    }

    private class ReservedBuffer
    extends AbstractRetainableByteBuffer {
        private final RetainedBucket _bucket;
        private final AtomicBoolean _removed;

        private ReservedBuffer(ByteBuffer buffer, RetainedBucket bucket) {
            super(buffer);
            this._removed = new AtomicBoolean();
            this._bucket = Objects.requireNonNull(bucket);
            this.acquire();
        }

        @Override
        public boolean release() {
            boolean released = super.release();
            if (released && this._removed.compareAndSet(false, true)) {
                ArrayByteBufferPool.this.reserve(this._bucket, this.getByteBuffer());
            }
            return released;
        }

        boolean remove() {
            return this._removed.compareAndSet(false, true);
        }
    }

    private class Buffer
    extends AbstractRetainableByteBuffer {
        private final RetainedBucket _bucket;
        private final Pool.Entry<RetainableByteBuffer> _entry;
        private int _usages;

        private Buffer(ByteBuffer buffer, RetainedBucket bucket, Pool.Entry<RetainableByteBuffer> entry) {
            super(buffer);
            this._bucket = Objects.requireNonNull(bucket);
            this._entry = Objects.requireNonNull(entry);
        }

        @Override
        public boolean release() {
            boolean released = super.release();
            if (released) {
                ArrayByteBufferPool.this.release(this._bucket, this._entry);
            }
            return released;
        }

        boolean remove() {
            return ArrayByteBufferPool.this.remove(this._bucket, this._entry);
        }

        private int use() {
            if (++this._usages < 0) {
                this._usages = 0;
            }
            return this._usages;
        }
    }

    public static class Tracking
    extends ArrayByteBufferPool {
        private static final Logger LOG = LoggerFactory.getLogger(Tracking.class);
        private final Set<Buffer> buffers = ConcurrentHashMap.newKeySet();

        public Tracking() {
        }

        public Tracking(int minCapacity, int maxCapacity, int maxBucketSize) {
            super(minCapacity, maxCapacity, maxBucketSize);
        }

        public Tracking(int minCapacity, int maxCapacity, int maxBucketSize, long maxHeapMemory, long maxDirectMemory) {
            super(minCapacity, -1, maxCapacity, maxBucketSize, maxHeapMemory, maxDirectMemory);
        }

        @Override
        public RetainableByteBuffer acquire(int size, boolean direct) {
            RetainableByteBuffer buffer = super.acquire(size, direct);
            Buffer wrapper = new Buffer(buffer, size);
            if (LOG.isDebugEnabled()) {
                LOG.debug("acquired {}", (Object)wrapper);
            }
            this.buffers.add(wrapper);
            return wrapper;
        }

        public Set<Buffer> getLeaks() {
            return this.buffers;
        }

        public String dumpLeaks() {
            return this.getLeaks().stream().map(Buffer::dump).collect(Collectors.joining(System.lineSeparator()));
        }

        public class Buffer
        extends RetainableByteBuffer.Wrapper {
            private final int size;
            private final Instant acquireInstant;
            private final Throwable acquireStack;
            private final List<Throwable> retainStacks;
            private final List<Throwable> releaseStacks;
            private final List<Throwable> overReleaseStacks;

            private Buffer(RetainableByteBuffer wrapped, int size) {
                super(wrapped);
                this.retainStacks = new CopyOnWriteArrayList<Throwable>();
                this.releaseStacks = new CopyOnWriteArrayList<Throwable>();
                this.overReleaseStacks = new CopyOnWriteArrayList<Throwable>();
                this.size = size;
                this.acquireInstant = Instant.now();
                this.acquireStack = new Throwable("Acquired by " + Thread.currentThread().getName());
            }

            public int getSize() {
                return this.size;
            }

            public Instant getAcquireInstant() {
                return this.acquireInstant;
            }

            public Throwable getAcquireStack() {
                return this.acquireStack;
            }

            @Override
            public void retain() {
                super.retain();
                this.retainStacks.add(new Throwable("Retained by " + Thread.currentThread().getName()));
            }

            @Override
            public boolean release() {
                try {
                    boolean released = super.release();
                    if (released) {
                        Tracking.this.buffers.remove(this);
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("released {}", (Object)this);
                        }
                    }
                    this.releaseStacks.add(new Throwable("Released by " + Thread.currentThread().getName()));
                    return released;
                }
                catch (IllegalStateException e2) {
                    Tracking.this.buffers.add(this);
                    this.overReleaseStacks.add(new Throwable("Over-released by " + Thread.currentThread().getName()));
                    IllegalStateException ise = new IllegalStateException(Thread.currentThread().getName() + " over-released " + String.valueOf(this));
                    this.releaseStacks.forEach(ise::addSuppressed);
                    throw ise;
                }
            }

            public String dump() {
                StringWriter w = new StringWriter();
                PrintWriter pw = new PrintWriter(w);
                this.getAcquireStack().printStackTrace(pw);
                pw.println("\n" + this.retainStacks.size() + " retain(s)");
                for (Throwable retainStack : this.retainStacks) {
                    retainStack.printStackTrace(pw);
                }
                pw.println("\n" + this.releaseStacks.size() + " release(s)");
                for (Throwable releaseStack : this.releaseStacks) {
                    releaseStack.printStackTrace(pw);
                }
                pw.println("\n" + this.overReleaseStacks.size() + " over-release(s)");
                for (Throwable overReleaseStack : this.overReleaseStacks) {
                    overReleaseStack.printStackTrace(pw);
                }
                String stacks = w.toString();
                return "%s@%x of %d bytes on %s wrapping %s%n %s%n acquired at %s".formatted(this.getClass().getSimpleName(), this.hashCode(), this.getSize(), this.getAcquireInstant(), this.getWrapped(), BufferUtil.toDetailString(this.getByteBuffer()), stacks);
            }
        }
    }

    public static class Quadratic
    extends ArrayByteBufferPool {
        public Quadratic() {
            this(0, -1, Integer.MAX_VALUE);
        }

        public Quadratic(int minCapacity, int maxCapacity, int maxBucketSize) {
            this(minCapacity, maxCapacity, maxBucketSize, -1L, -1L);
        }

        public Quadratic(int minCapacity, int maxCapacity, int maxBucketSize, long maxHeapMemory, long maxDirectMemory) {
            super(minCapacity, -1, maxCapacity, maxBucketSize, maxHeapMemory, maxDirectMemory, c -> 32 - Integer.numberOfLeadingZeros(c - 1), i -> 1 << i);
        }
    }
}

