/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.server.handler;

import java.time.Duration;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.regex.Pattern;
import org.eclipse.jetty.http.HttpFields;
import org.eclipse.jetty.http.HttpHeader;
import org.eclipse.jetty.http.HttpMethod;
import org.eclipse.jetty.http.PreEncodedHttpField;
import org.eclipse.jetty.server.Handler;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.Response;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.annotation.ManagedAttribute;
import org.eclipse.jetty.util.annotation.ManagedObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ManagedObject
public class CrossOriginHandler
extends Handler.Wrapper {
    private static final Logger LOG = LoggerFactory.getLogger(CrossOriginHandler.class);
    private static final PreEncodedHttpField ACCESS_CONTROL_ALLOW_CREDENTIALS_TRUE = new PreEncodedHttpField(HttpHeader.ACCESS_CONTROL_ALLOW_CREDENTIALS, "true");
    private static final PreEncodedHttpField VARY_ORIGIN = new PreEncodedHttpField(HttpHeader.VARY, HttpHeader.ORIGIN.asString());
    private boolean allowCredentials = true;
    private Set<String> allowedHeaders = Set.of("Content-Type");
    private Set<String> allowedMethods = Set.of("GET", "POST", "HEAD");
    private Set<String> allowedOrigins = Set.of("*");
    private Set<String> allowedTimingOrigins = Set.of();
    private boolean deliverPreflight = false;
    private boolean deliverNonAllowedOrigin = true;
    private boolean deliverNonAllowedOriginWebSocketUpgrade = false;
    private Set<String> exposedHeaders = Set.of();
    private Duration preflightMaxAge = Duration.ofSeconds(60L);
    private boolean anyOriginAllowed;
    private final Set<Pattern> allowedOriginPatterns = new LinkedHashSet<Pattern>();
    private boolean anyTimingOriginAllowed;
    private final Set<Pattern> allowedTimingOriginPatterns = new LinkedHashSet<Pattern>();
    private PreEncodedHttpField accessControlAllowMethodsField;
    private PreEncodedHttpField accessControlAllowHeadersField;
    private PreEncodedHttpField accessControlExposeHeadersField;
    private PreEncodedHttpField accessControlMaxAge;

    @ManagedAttribute(value="Whether the server allows cross-origin requests to include credentials (cookies, authentication headers, etc.)")
    public boolean isAllowCredentials() {
        return this.allowCredentials;
    }

    public void setAllowCredentials(boolean allow) {
        this.throwIfStarted();
        this.allowCredentials = allow;
    }

    @ManagedAttribute(value="The set of allowed headers in a cross-origin request")
    public Set<String> getAllowedHeaders() {
        return this.allowedHeaders;
    }

    public void setAllowedHeaders(Set<String> headers) {
        this.throwIfStarted();
        this.allowedHeaders = Set.copyOf(headers);
    }

    @ManagedAttribute(value="The set of allowed methods in a cross-origin request")
    public Set<String> getAllowedMethods() {
        return this.allowedMethods;
    }

    public void setAllowedMethods(Set<String> methods) {
        this.throwIfStarted();
        this.allowedMethods = Set.copyOf(methods);
    }

    @ManagedAttribute(value="The set of allowed origin regex strings in a cross-origin request")
    public Set<String> getAllowedOriginPatterns() {
        return this.allowedOrigins;
    }

    public void setAllowedOriginPatterns(Set<String> origins) {
        this.throwIfStarted();
        this.allowedOrigins = Set.copyOf(origins);
    }

    @ManagedAttribute(value="The set of allowed timing origin regex strings in a cross-origin request")
    public Set<String> getAllowedTimingOriginPatterns() {
        return this.allowedTimingOrigins;
    }

    public void setAllowedTimingOriginPatterns(Set<String> origins) {
        this.throwIfStarted();
        this.allowedTimingOrigins = Set.copyOf(origins);
    }

    @ManagedAttribute(value="whether preflight requests are delivered to the child Handler")
    public boolean isDeliverPreflightRequests() {
        return this.deliverPreflight;
    }

    public void setDeliverPreflightRequests(boolean deliver) {
        this.throwIfStarted();
        this.deliverPreflight = deliver;
    }

    @ManagedAttribute(value="whether requests whose origin is not allowed are delivered to the child Handler")
    public boolean isDeliverNonAllowedOriginRequests() {
        return this.deliverNonAllowedOrigin;
    }

    public void setDeliverNonAllowedOriginRequests(boolean deliverNonAllowedOrigin) {
        this.deliverNonAllowedOrigin = deliverNonAllowedOrigin;
    }

    @ManagedAttribute(value="whether WebSocket upgrade requests whose origin is not allowed are delivered to the child Handler")
    public boolean isDeliverNonAllowedOriginWebSocketUpgradeRequests() {
        return this.deliverNonAllowedOriginWebSocketUpgrade;
    }

    public void setDeliverNonAllowedOriginWebSocketUpgradeRequests(boolean deliverNonAllowedOriginWebSocketUpgrade) {
        this.deliverNonAllowedOriginWebSocketUpgrade = deliverNonAllowedOriginWebSocketUpgrade;
    }

    @ManagedAttribute(value="The set of headers exposed in a cross-origin response")
    public Set<String> getExposedHeaders() {
        return this.exposedHeaders;
    }

    public void setExposedHeaders(Set<String> headers) {
        this.throwIfStarted();
        this.exposedHeaders = Set.copyOf(headers);
    }

    @ManagedAttribute(value="How long the preflight results can be cached by browsers")
    public Duration getPreflightMaxAge() {
        return this.preflightMaxAge;
    }

    public void setPreflightMaxAge(Duration duration) {
        this.throwIfStarted();
        this.preflightMaxAge = duration;
    }

    @Override
    protected void doStart() throws Exception {
        this.resolveAllowedOrigins();
        this.resolveAllowedTimingOrigins();
        this.accessControlAllowMethodsField = new PreEncodedHttpField(HttpHeader.ACCESS_CONTROL_ALLOW_METHODS, String.join((CharSequence)",", this.getAllowedMethods()));
        this.accessControlAllowHeadersField = new PreEncodedHttpField(HttpHeader.ACCESS_CONTROL_ALLOW_HEADERS, String.join((CharSequence)",", this.getAllowedHeaders()));
        this.accessControlExposeHeadersField = new PreEncodedHttpField(HttpHeader.ACCESS_CONTROL_EXPOSE_HEADERS, String.join((CharSequence)",", this.getExposedHeaders()));
        this.accessControlMaxAge = new PreEncodedHttpField(HttpHeader.ACCESS_CONTROL_MAX_AGE, this.getPreflightMaxAge().toSeconds());
        super.doStart();
    }

    @Override
    public boolean handle(Request request, Response response, Callback callback) throws Exception {
        response.getHeaders().ensureField(VARY_ORIGIN);
        String origins = request.getHeaders().get(HttpHeader.ORIGIN);
        if (origins == null) {
            return super.handle(request, response, callback);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("handling cross-origin request {}", (Object)request);
        }
        boolean preflight = this.isPreflight(request);
        if (this.originMatches(origins)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("cross-origin request matches allowed origins: {} {}", (Object)request, (Object)this.getAllowedOriginPatterns());
            }
            if (preflight) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("preflight cross-origin request {}", (Object)request);
                }
                this.handlePreflightResponse(origins, response);
                if (!this.isDeliverPreflightRequests()) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("preflight cross-origin request not delivered to child handler {}", (Object)request);
                    }
                    callback.succeeded();
                    return true;
                }
            } else {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("simple cross-origin request {}", (Object)request);
                }
                this.handleSimpleResponse(origins, response);
            }
            if (this.timingOriginMatches(origins)) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("cross-origin request matches allowed timing origins: {} {}", (Object)request, (Object)this.getAllowedTimingOriginPatterns());
                }
                response.getHeaders().put(HttpHeader.TIMING_ALLOW_ORIGIN, origins);
            }
            return super.handle(request, response, callback);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("cross-origin request does not match allowed origins: {} {}", (Object)request, (Object)this.getAllowedOriginPatterns());
        }
        if (this.isDeliverNonAllowedOriginRequests()) {
            if (preflight) {
                if (!this.isDeliverPreflightRequests()) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("preflight cross-origin request not delivered to child handler {}", (Object)request);
                    }
                    callback.succeeded();
                    return true;
                }
            } else if (this.isWebSocketUpgrade(request) && !this.isDeliverNonAllowedOriginWebSocketUpgradeRequests()) {
                Response.writeError(request, response, callback, 400, "origin not allowed");
                return true;
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("cross-origin request delivered to child handler {}", (Object)request);
            }
            return super.handle(request, response, callback);
        }
        Response.writeError(request, response, callback, 400, "origin not allowed");
        return true;
    }

    private boolean originMatches(String origins) {
        if (this.anyOriginAllowed) {
            return true;
        }
        if (this.allowedOriginPatterns.isEmpty()) {
            return false;
        }
        return this.originMatches(origins, this.allowedOriginPatterns);
    }

    private boolean timingOriginMatches(String origins) {
        if (this.anyTimingOriginAllowed) {
            return true;
        }
        if (this.allowedTimingOriginPatterns.isEmpty()) {
            return false;
        }
        return this.originMatches(origins, this.allowedTimingOriginPatterns);
    }

    private boolean originMatches(String origins, Set<Pattern> allowedOriginPatterns) {
        for (String origin : origins.split(" ")) {
            if ((origin = origin.trim()).isEmpty()) continue;
            for (Pattern pattern : allowedOriginPatterns) {
                if (!pattern.matcher(origin).matches()) continue;
                return true;
            }
        }
        return false;
    }

    private boolean isPreflight(Request request) {
        return HttpMethod.OPTIONS.is(request.getMethod()) && request.getHeaders().contains(HttpHeader.ACCESS_CONTROL_REQUEST_METHOD);
    }

    private boolean isWebSocketUpgrade(Request request) {
        return request.getHeaders().contains(HttpHeader.SEC_WEBSOCKET_VERSION);
    }

    private void handlePreflightResponse(String origins, Response response) {
        long seconds;
        Set<String> allowedHeaders;
        Set<String> allowedMethods;
        HttpFields.Mutable headers = response.getHeaders();
        headers.put(HttpHeader.ACCESS_CONTROL_ALLOW_ORIGIN, origins);
        if (this.isAllowCredentials()) {
            headers.put(ACCESS_CONTROL_ALLOW_CREDENTIALS_TRUE);
        }
        if (!(allowedMethods = this.getAllowedMethods()).isEmpty()) {
            headers.put(this.accessControlAllowMethodsField);
        }
        if (!(allowedHeaders = this.getAllowedHeaders()).isEmpty()) {
            headers.put(this.accessControlAllowHeadersField);
        }
        if ((seconds = this.getPreflightMaxAge().toSeconds()) > 0L) {
            headers.put(this.accessControlMaxAge);
        }
    }

    private void handleSimpleResponse(String origin, Response response) {
        Set<String> exposedHeaders;
        HttpFields.Mutable headers = response.getHeaders();
        headers.put(HttpHeader.ACCESS_CONTROL_ALLOW_ORIGIN, origin);
        if (this.isAllowCredentials()) {
            headers.put(ACCESS_CONTROL_ALLOW_CREDENTIALS_TRUE);
        }
        if (!(exposedHeaders = this.getExposedHeaders()).isEmpty()) {
            headers.put(this.accessControlExposeHeadersField);
        }
    }

    private void resolveAllowedOrigins() {
        for (String allowedOrigin : this.getAllowedOriginPatterns()) {
            if ((allowedOrigin = allowedOrigin.trim()).isEmpty()) continue;
            if ("*".equals(allowedOrigin)) {
                this.anyOriginAllowed = true;
                return;
            }
            this.allowedOriginPatterns.add(Pattern.compile(allowedOrigin, 2));
        }
    }

    private void resolveAllowedTimingOrigins() {
        for (String allowedTimingOrigin : this.getAllowedTimingOriginPatterns()) {
            if ((allowedTimingOrigin = allowedTimingOrigin.trim()).isEmpty()) continue;
            if ("*".equals(allowedTimingOrigin)) {
                this.anyTimingOriginAllowed = true;
                return;
            }
            this.allowedTimingOriginPatterns.add(Pattern.compile(allowedTimingOrigin, 2));
        }
    }

    private void throwIfStarted() {
        if (this.isStarted()) {
            throw new IllegalStateException("Cannot configure after start");
        }
    }
}

