/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.store;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.IntConsumer;
import java.util.function.IntFunction;
import org.apache.lucene.store.ByteBuffersDataInput;
import org.apache.lucene.store.DataInput;
import org.apache.lucene.store.DataOutput;
import org.apache.lucene.util.Accountable;
import org.apache.lucene.util.BitUtil;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.RamUsageEstimator;
import org.apache.lucene.util.UnicodeUtil;

public final class ByteBuffersDataOutput
extends DataOutput
implements Accountable {
    private static final ByteBuffer EMPTY = ByteBuffer.allocate(0).order(ByteOrder.LITTLE_ENDIAN);
    private static final byte[] EMPTY_BYTE_ARRAY = new byte[0];
    public static final IntFunction<ByteBuffer> ALLOCATE_BB_ON_HEAP = ByteBuffer::allocate;
    public static final Consumer<ByteBuffer> NO_REUSE = bb -> {
        throw new RuntimeException("reset() is not allowed on this buffer.");
    };
    public static final int DEFAULT_MIN_BITS_PER_BLOCK = 10;
    public static final int DEFAULT_MAX_BITS_PER_BLOCK = 26;
    public static final int LIMIT_MIN_BITS_PER_BLOCK = 1;
    public static final int LIMIT_MAX_BITS_PER_BLOCK = 31;
    static final int MAX_BLOCKS_BEFORE_BLOCK_EXPANSION = 100;
    private final int maxBitsPerBlock;
    private final IntFunction<ByteBuffer> blockAllocate;
    private final Consumer<ByteBuffer> blockReuse;
    private int blockBits;
    private final ArrayDeque<ByteBuffer> blocks = new ArrayDeque();
    private long ramBytesUsed;
    private ByteBuffer currentBlock = EMPTY;
    private static final long HALF_SHIFT = 10L;
    private static final int SURROGATE_OFFSET = -56613888;

    public ByteBuffersDataOutput(long expectedSize) {
        this(ByteBuffersDataOutput.computeBlockSizeBitsFor(expectedSize), 26, ALLOCATE_BB_ON_HEAP, NO_REUSE);
    }

    public ByteBuffersDataOutput() {
        this(10, 26, ALLOCATE_BB_ON_HEAP, NO_REUSE);
    }

    public ByteBuffersDataOutput(int minBitsPerBlock, int maxBitsPerBlock, IntFunction<ByteBuffer> blockAllocate, Consumer<ByteBuffer> blockReuse) {
        if (minBitsPerBlock < 1) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "minBitsPerBlock (%s) too small, must be at least %s", minBitsPerBlock, 1));
        }
        if (maxBitsPerBlock > 31) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "maxBitsPerBlock (%s) too large, must not exceed %s", maxBitsPerBlock, 31));
        }
        if (minBitsPerBlock > maxBitsPerBlock) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "minBitsPerBlock (%s) cannot exceed maxBitsPerBlock (%s)", minBitsPerBlock, maxBitsPerBlock));
        }
        this.maxBitsPerBlock = maxBitsPerBlock;
        this.blockBits = minBitsPerBlock;
        this.blockAllocate = Objects.requireNonNull(blockAllocate, "Block allocator must not be null.");
        this.blockReuse = Objects.requireNonNull(blockReuse, "Block reuse must not be null.");
    }

    @Override
    public void writeByte(byte b) {
        if (!this.currentBlock.hasRemaining()) {
            this.appendBlock();
        }
        this.currentBlock.put(b);
    }

    @Override
    public void writeBytes(byte[] src, int offset, int length) {
        assert (length >= 0);
        while (length > 0) {
            if (!this.currentBlock.hasRemaining()) {
                this.appendBlock();
            }
            int chunk = Math.min(this.currentBlock.remaining(), length);
            this.currentBlock.put(src, offset, chunk);
            length -= chunk;
            offset += chunk;
        }
    }

    @Override
    public void writeBytes(byte[] b, int length) {
        this.writeBytes(b, 0, length);
    }

    public void writeBytes(byte[] b) {
        this.writeBytes(b, 0, b.length);
    }

    public void writeBytes(ByteBuffer buffer) {
        int chunk;
        buffer = buffer.duplicate();
        for (int length = buffer.remaining(); length > 0; length -= chunk) {
            if (!this.currentBlock.hasRemaining()) {
                this.appendBlock();
            }
            chunk = Math.min(this.currentBlock.remaining(), length);
            buffer.limit(buffer.position() + chunk);
            this.currentBlock.put(buffer);
        }
    }

    public ArrayList<ByteBuffer> toBufferList() {
        ArrayList<ByteBuffer> result = new ArrayList<ByteBuffer>(Math.max(this.blocks.size(), 1));
        if (this.blocks.isEmpty()) {
            result.add(EMPTY);
        } else {
            for (ByteBuffer bb : this.blocks) {
                bb = bb.asReadOnlyBuffer().flip().order(ByteOrder.LITTLE_ENDIAN);
                result.add(bb);
            }
        }
        return result;
    }

    public ArrayList<ByteBuffer> toWriteableBufferList() {
        ArrayList<ByteBuffer> result = new ArrayList<ByteBuffer>(Math.max(this.blocks.size(), 1));
        if (this.blocks.isEmpty()) {
            result.add(EMPTY);
        } else {
            for (ByteBuffer bb : this.blocks) {
                bb = bb.duplicate().flip();
                result.add(bb);
            }
        }
        return result;
    }

    public ByteBuffersDataInput toDataInput() {
        return new ByteBuffersDataInput(this.toBufferList());
    }

    public byte[] toArrayCopy() {
        if (this.blocks.size() == 0) {
            return EMPTY_BYTE_ARRAY;
        }
        long size = this.size();
        if (size > Integer.MAX_VALUE) {
            throw new RuntimeException("Data exceeds maximum size of a single byte array: " + size);
        }
        byte[] arr = new byte[Math.toIntExact(this.size())];
        int offset = 0;
        for (ByteBuffer bb : this.toBufferList()) {
            int len = bb.remaining();
            bb.get(arr, offset, len);
            offset += len;
        }
        return arr;
    }

    @Override
    public void copyBytes(DataInput input, long numBytes) throws IOException {
        int length;
        int chunk;
        assert (numBytes >= 0L) : "numBytes=" + numBytes;
        for (length = (int)numBytes; length > 0; length -= chunk) {
            if (!this.currentBlock.hasRemaining()) {
                this.appendBlock();
            }
            if (!this.currentBlock.hasArray()) break;
            chunk = Math.min(this.currentBlock.remaining(), length);
            int pos = this.currentBlock.position();
            input.readBytes(this.currentBlock.array(), Math.addExact(this.currentBlock.arrayOffset(), pos), chunk);
            this.currentBlock.position(pos + chunk);
        }
        if (length > 0) {
            super.copyBytes(input, length);
        }
    }

    public void copyTo(DataOutput output) throws IOException {
        for (ByteBuffer bb : this.blocks) {
            if (bb.hasArray()) {
                output.writeBytes(bb.array(), bb.arrayOffset(), bb.position());
                continue;
            }
            bb = bb.asReadOnlyBuffer().flip();
            output.copyBytes(new ByteBuffersDataInput(Collections.singletonList(bb)), bb.remaining());
        }
    }

    public long size() {
        long size = 0L;
        int blockCount = this.blocks.size();
        if (blockCount >= 1) {
            long fullBlockSize = ((long)blockCount - 1L) * (long)this.blockSize();
            long lastBlockSize = this.blocks.getLast().position();
            size = fullBlockSize + lastBlockSize;
        }
        return size;
    }

    public String toString() {
        return String.format(Locale.ROOT, "%,d bytes, block size: %,d, blocks: %,d", this.size(), this.blockSize(), this.blocks.size());
    }

    @Override
    public void writeShort(short v) {
        try {
            if (this.currentBlock.remaining() >= 2) {
                this.currentBlock.putShort(v);
            } else {
                super.writeShort(v);
            }
        }
        catch (IOException e2) {
            throw new UncheckedIOException(e2);
        }
    }

    @Override
    public void writeInt(int v) {
        try {
            if (this.currentBlock.remaining() >= 4) {
                this.currentBlock.putInt(v);
            } else {
                super.writeInt(v);
            }
        }
        catch (IOException e2) {
            throw new UncheckedIOException(e2);
        }
    }

    @Override
    public void writeLong(long v) {
        try {
            if (this.currentBlock.remaining() >= 8) {
                this.currentBlock.putLong(v);
            } else {
                super.writeLong(v);
            }
        }
        catch (IOException e2) {
            throw new UncheckedIOException(e2);
        }
    }

    @Override
    public void writeString(String v) {
        try {
            int MAX_CHARS_PER_WINDOW = 1024;
            if (v.length() <= 1024) {
                BytesRef utf82 = new BytesRef(v);
                this.writeVInt(utf82.length);
                this.writeBytes(utf82.bytes, utf82.offset, utf82.length);
            } else {
                this.writeVInt(UnicodeUtil.calcUTF16toUTF8Length(v, 0, v.length()));
                byte[] buf = new byte[3072];
                ByteBuffersDataOutput.UTF16toUTF8(v, 0, v.length(), buf, len -> this.writeBytes(buf, 0, len));
            }
        }
        catch (IOException e2) {
            throw new UncheckedIOException(e2);
        }
    }

    @Override
    public void writeMapOfStrings(Map<String, String> map) {
        try {
            super.writeMapOfStrings(map);
        }
        catch (IOException e2) {
            throw new UncheckedIOException(e2);
        }
    }

    @Override
    public void writeSetOfStrings(Set<String> set) {
        try {
            super.writeSetOfStrings(set);
        }
        catch (IOException e2) {
            throw new UncheckedIOException(e2);
        }
    }

    @Override
    public long ramBytesUsed() {
        assert (this.ramBytesUsed == this.blocks.stream().mapToLong(Buffer::capacity).sum() + (long)(this.blocks.size() * RamUsageEstimator.NUM_BYTES_OBJECT_REF));
        return this.ramBytesUsed;
    }

    public void reset() {
        if (this.blockReuse != NO_REUSE) {
            this.blocks.forEach(this.blockReuse);
        }
        this.blocks.clear();
        this.ramBytesUsed = 0L;
        this.currentBlock = EMPTY;
    }

    public static ByteBuffersDataOutput newResettableInstance() {
        ByteBufferRecycler reuser = new ByteBufferRecycler(ALLOCATE_BB_ON_HEAP);
        return new ByteBuffersDataOutput(10, 26, reuser::allocate, reuser::reuse);
    }

    private int blockSize() {
        return 1 << this.blockBits;
    }

    private void appendBlock() {
        if (this.blocks.size() >= 100 && this.blockBits < this.maxBitsPerBlock) {
            this.rewriteToBlockSize(this.blockBits + 1);
            if (this.blocks.getLast().hasRemaining()) {
                return;
            }
        }
        int requiredBlockSize = 1 << this.blockBits;
        this.currentBlock = this.blockAllocate.apply(requiredBlockSize).order(ByteOrder.LITTLE_ENDIAN);
        assert (this.currentBlock.capacity() == requiredBlockSize);
        this.blocks.add(this.currentBlock);
        this.ramBytesUsed += (long)(RamUsageEstimator.NUM_BYTES_OBJECT_REF + this.currentBlock.capacity());
    }

    private void rewriteToBlockSize(int targetBlockBits) {
        ByteBuffer block;
        assert (targetBlockBits <= this.maxBitsPerBlock);
        ByteBuffersDataOutput cloned = new ByteBuffersDataOutput(targetBlockBits, targetBlockBits, this.blockAllocate, NO_REUSE);
        while ((block = this.blocks.pollFirst()) != null) {
            block.flip();
            cloned.writeBytes(block);
            if (this.blockReuse == NO_REUSE) continue;
            this.blockReuse.accept(block);
        }
        assert (this.blocks.isEmpty());
        this.blockBits = targetBlockBits;
        this.blocks.addAll(cloned.blocks);
        this.ramBytesUsed = cloned.ramBytesUsed;
    }

    private static int computeBlockSizeBitsFor(long bytes) {
        long powerOfTwo = BitUtil.nextHighestPowerOfTwo(bytes / 100L);
        if (powerOfTwo == 0L) {
            return 10;
        }
        int blockBits = Long.numberOfTrailingZeros(powerOfTwo);
        blockBits = Math.min(blockBits, 26);
        blockBits = Math.max(blockBits, 10);
        return blockBits;
    }

    private static int UTF16toUTF8(CharSequence s, int offset, int length, byte[] buf, IntConsumer bufferFlusher) {
        int utf8Len = 0;
        int j = 0;
        int end = offset + length;
        for (int i = offset; i < end; ++i) {
            int utf32;
            char chr = s.charAt(i);
            if (j + 4 >= buf.length) {
                bufferFlusher.accept(j);
                utf8Len += j;
                j = 0;
            }
            if (chr < '\u0080') {
                buf[j++] = (byte)chr;
                continue;
            }
            if (chr < '\u0800') {
                buf[j++] = (byte)(0xC0 | chr >> 6);
                buf[j++] = (byte)(0x80 | chr & 0x3F);
                continue;
            }
            if (chr < '\ud800' || chr > '\udfff') {
                buf[j++] = (byte)(0xE0 | chr >> 12);
                buf[j++] = (byte)(0x80 | chr >> 6 & 0x3F);
                buf[j++] = (byte)(0x80 | chr & 0x3F);
                continue;
            }
            if (chr < '\udc00' && i < end - 1 && (utf32 = s.charAt(i + 1)) >= 56320 && utf32 <= 57343) {
                utf32 = (chr << 10) + utf32 + -56613888;
                ++i;
                buf[j++] = (byte)(0xF0 | utf32 >> 18);
                buf[j++] = (byte)(0x80 | utf32 >> 12 & 0x3F);
                buf[j++] = (byte)(0x80 | utf32 >> 6 & 0x3F);
                buf[j++] = (byte)(0x80 | utf32 & 0x3F);
                continue;
            }
            buf[j++] = -17;
            buf[j++] = -65;
            buf[j++] = -67;
        }
        bufferFlusher.accept(j);
        return utf8Len += j;
    }

    public static final class ByteBufferRecycler {
        private final ArrayDeque<ByteBuffer> reuse = new ArrayDeque();
        private final IntFunction<ByteBuffer> delegate;

        public ByteBufferRecycler(IntFunction<ByteBuffer> delegate) {
            this.delegate = Objects.requireNonNull(delegate);
        }

        public ByteBuffer allocate(int size) {
            while (!this.reuse.isEmpty()) {
                ByteBuffer bb = this.reuse.removeFirst();
                if (bb.remaining() != size) continue;
                return bb;
            }
            return this.delegate.apply(size);
        }

        public void reuse(ByteBuffer buffer) {
            buffer.rewind();
            this.reuse.addLast(buffer);
        }
    }
}

