/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jena.sparql.exec.http;

import java.io.FileNotFoundException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiConsumer;
import org.apache.jena.atlas.web.ContentType;
import org.apache.jena.atlas.web.HttpException;
import org.apache.jena.graph.Graph;
import org.apache.jena.graph.Node;
import org.apache.jena.http.HttpEnv;
import org.apache.jena.http.HttpLib;
import org.apache.jena.http.HttpRDF;
import org.apache.jena.http.Push;
import org.apache.jena.riot.Lang;
import org.apache.jena.riot.RDFFormat;
import org.apache.jena.riot.RDFLanguages;
import org.apache.jena.riot.RDFWriterRegistry;
import org.apache.jena.riot.system.RiotLib;
import org.apache.jena.riot.system.StreamRDFLib;
import org.apache.jena.riot.system.StreamRDFWriter;
import org.apache.jena.shared.NotFoundException;
import org.apache.jena.sparql.ARQException;
import org.apache.jena.sparql.core.DatasetGraph;
import org.apache.jena.sparql.core.DatasetGraphFactory;
import org.apache.jena.sparql.exec.http.UpdateExecHTTP;
import org.apache.jena.sparql.exec.http.UpdateExecHTTPBuilder;
import org.apache.jena.sparql.graph.GraphFactory;

public class GSP {
    private String serviceEndpoint = null;
    private RDFFormat rdfFormat = null;
    private HttpClient httpClient = null;
    private Map<String, String> httpHeaders = new HashMap<String, String>();
    private boolean defaultGraph = false;
    private String graphName = null;
    private boolean datasetGraph = false;

    public static GSP service(String service) {
        return new GSP().endpoint(service);
    }

    public static GSP request() {
        return new GSP();
    }

    protected GSP() {
    }

    public GSP endpoint(String serviceURL) {
        this.serviceEndpoint = Objects.requireNonNull(serviceURL);
        return this;
    }

    public GSP httpClient(HttpClient httpClient) {
        Objects.requireNonNull(httpClient, "HttpClient");
        this.httpClient = httpClient;
        return this;
    }

    public GSP httpHeader(String headerName, String headerValue) {
        Objects.requireNonNull(headerName);
        Objects.requireNonNull(headerValue);
        if (this.httpHeaders == null) {
            this.httpHeaders = new HashMap<String, String>();
        }
        this.httpHeaders.put(headerName, headerValue);
        return this;
    }

    private String httpHeader(String header) {
        Objects.requireNonNull(header);
        if (this.httpHeaders == null) {
            return null;
        }
        return this.httpHeaders.get(header);
    }

    public GSP graphName(String graphName) {
        Objects.requireNonNull(graphName);
        this.graphName = graphName;
        this.defaultGraph = false;
        return this;
    }

    public GSP graphName(Node graphName) {
        Objects.requireNonNull(graphName);
        this.clearOperation();
        if (!graphName.isURI() && !graphName.isBlank()) {
            throw GSP.exception("Not an acceptable graph name: " + this.graphName);
        }
        Node gn = RiotLib.blankNodeToIri(graphName);
        this.graphName = gn.getURI();
        this.defaultGraph = false;
        this.datasetGraph = false;
        return this;
    }

    public GSP defaultGraph() {
        this.clearOperation();
        this.defaultGraph = true;
        return this;
    }

    public GSP dataset() {
        this.clearOperation();
        this.datasetGraph = true;
        return this;
    }

    private void clearOperation() {
        this.defaultGraph = false;
        this.datasetGraph = false;
        this.graphName = null;
    }

    public GSP acceptHeader(String acceptHeader) {
        this.httpHeader("Accept", acceptHeader);
        return this;
    }

    private String acceptHeader() {
        return this.httpHeader("Accept");
    }

    public GSP accept(Lang lang) {
        String acceptHeader = lang != null ? lang.getContentType().getContentTypeStr() : null;
        this.httpHeader("Accept", acceptHeader);
        return this;
    }

    public GSP contentTypeHeader(String contentType) {
        this.httpHeader("Content-Type", contentType);
        return this;
    }

    private String contentType() {
        return this.httpHeader("Content-Type");
    }

    public GSP contentType(RDFFormat rdfFormat) {
        this.rdfFormat = rdfFormat;
        String contentType = rdfFormat.getLang().getContentType().getContentTypeStr();
        this.httpHeader("Content-Type", contentType);
        return this;
    }

    protected final void validateGraphOperation() {
        Objects.requireNonNull(this.serviceEndpoint);
        if (!this.defaultGraph && this.graphName == null) {
            throw GSP.exception("Need either default graph or a graph name");
        }
        if (this.datasetGraph) {
            throw GSP.exception("Dataset request specified for graph operation");
        }
    }

    protected final void internalDataset() {
        this.datasetGraph = true;
    }

    protected final void validateDatasetOperation() {
        Objects.requireNonNull(this.serviceEndpoint);
        if (this.defaultGraph) {
            throw GSP.exception("Default graph specified for dataset operation");
        }
        if (this.graphName != null) {
            throw GSP.exception("A graph name specified for dataset operation");
        }
        if (!this.datasetGraph) {
            throw GSP.exception("Dataset request not specified for dataset operation");
        }
    }

    private HttpClient requestHttpClient(String serviceURL, String requestURL) {
        if (this.httpClient != null) {
            return this.httpClient;
        }
        return HttpEnv.getHttpClient(serviceURL, this.httpClient);
    }

    private static RuntimeException exception(String msg) {
        return new HttpException(msg);
    }

    public Graph GET() {
        this.validateGraphOperation();
        this.ensureAcceptHeader("text/turtle,application/n-triples;q=0.9,application/ld+json;q=0.8,application/rdf+xml;q=0.7,*/*;q=0.3");
        String url = this.graphRequestURL();
        Graph graph = GraphFactory.createDefaultGraph();
        HttpClient hc = this.requestHttpClient(this.serviceEndpoint, url);
        HttpRDF.httpGetToStream(hc, url, this.httpHeaders, StreamRDFLib.graph(graph));
        return graph;
    }

    public void POST(String file) {
        this.validateGraphOperation();
        String url = this.graphRequestURL();
        String fileExtContentType = this.contentTypeFromFilename(file);
        HttpClient hc = this.requestHttpClient(this.serviceEndpoint, url);
        GSP.uploadTriples(hc, url, file, fileExtContentType, this.httpHeaders, Push.POST);
    }

    public void POST(Graph graph) {
        this.validateGraphOperation();
        RDFFormat requestFmt = this.rdfFormat(HttpEnv.defaultTriplesFormat);
        String url = this.graphRequestURL();
        HttpClient hc = this.requestHttpClient(this.serviceEndpoint, url);
        HttpRDF.httpPostGraph(hc, url, graph, requestFmt, this.httpHeaders);
    }

    public void PUT(String file) {
        this.validateGraphOperation();
        String url = this.graphRequestURL();
        String fileExtContentType = this.contentTypeFromFilename(file);
        HttpClient hc = this.requestHttpClient(this.serviceEndpoint, url);
        GSP.uploadTriples(hc, url, file, fileExtContentType, this.httpHeaders, Push.PUT);
    }

    public void PUT(Graph graph) {
        this.validateGraphOperation();
        RDFFormat requestFmt = this.rdfFormat(HttpEnv.defaultTriplesFormat);
        String url = this.graphRequestURL();
        HttpClient hc = this.requestHttpClient(this.serviceEndpoint, url);
        HttpRDF.httpPutGraph(hc, url, graph, requestFmt, this.httpHeaders);
    }

    public void DELETE() {
        this.validateGraphOperation();
        String url = this.graphRequestURL();
        HttpClient hc = this.requestHttpClient(this.serviceEndpoint, url);
        HttpRDF.httpDeleteGraph(hc, url);
    }

    private String graphRequestURL() {
        return HttpLib.requestURL(this.serviceEndpoint, GSP.queryStringForGraph(this.graphName));
    }

    protected static String queryStringForGraph(String graphName) {
        if (graphName == null) {
            return "default";
        }
        switch (graphName) {
            case "default": {
                return "default";
            }
            case "union": {
                return "graph=union";
            }
        }
        return "graph=" + HttpLib.urlEncodeQueryString(graphName);
    }

    protected final String graphName() {
        return this.graphName;
    }

    protected final String service() {
        return this.serviceEndpoint;
    }

    protected final boolean isDefaultGraph() {
        return this.graphName == null;
    }

    protected final boolean isGraphOperation() {
        return this.defaultGraph || this.graphName != null;
    }

    protected final boolean isDatasetOperation() {
        return this.datasetGraph;
    }

    protected final HttpClient httpClient() {
        return this.httpClient;
    }

    protected final void httpHeaders(BiConsumer<String, String> action) {
        this.httpHeaders.forEach(action);
    }

    public DatasetGraph getDataset() {
        this.internalDataset();
        this.validateDatasetOperation();
        this.ensureAcceptHeader("text/turtle,application/n-triples;q=0.9,application/rdf+xml;q=0.7,application/trig,application/n-quads;q=0.9,application/ld+json;q=0.8,*/*;q=0.5");
        DatasetGraph dsg = DatasetGraphFactory.createTxnMem();
        HttpClient hc = this.requestHttpClient(this.serviceEndpoint, this.serviceEndpoint);
        HttpRDF.httpGetToStream(hc, this.serviceEndpoint, this.httpHeaders, StreamRDFLib.dataset(dsg));
        return dsg;
    }

    private void ensureAcceptHeader(String dftAcceptheader) {
        String requestAccept = GSP.header(this.acceptHeader(), "text/turtle,application/n-triples;q=0.9,application/rdf+xml;q=0.7,application/trig,application/n-quads;q=0.9,application/ld+json;q=0.8,*/*;q=0.5");
        this.acceptHeader(requestAccept);
    }

    public void postDataset(String file) {
        this.internalDataset();
        this.validateDatasetOperation();
        String fileExtContentType = this.contentTypeFromFilename(file);
        HttpClient hc = this.requestHttpClient(this.serviceEndpoint, this.serviceEndpoint);
        GSP.uploadQuads(hc, this.serviceEndpoint, file, fileExtContentType, this.httpHeaders, Push.POST);
    }

    public void postDataset(DatasetGraph dataset) {
        this.internalDataset();
        this.validateDatasetOperation();
        RDFFormat requestFmt = this.rdfFormat(HttpEnv.defaultQuadsFormat);
        HttpClient hc = this.requestHttpClient(this.serviceEndpoint, this.serviceEndpoint);
        HttpRDF.httpPostDataset(hc, this.serviceEndpoint, dataset, requestFmt, this.httpHeaders);
    }

    public void putDataset(String file) {
        this.internalDataset();
        this.validateDatasetOperation();
        String fileExtContentType = this.contentTypeFromFilename(file);
        HttpClient hc = this.requestHttpClient(this.serviceEndpoint, this.serviceEndpoint);
        GSP.uploadQuads(hc, this.serviceEndpoint, file, fileExtContentType, this.httpHeaders, Push.PUT);
    }

    public void putDataset(DatasetGraph dataset) {
        this.internalDataset();
        this.validateDatasetOperation();
        RDFFormat requestFmt = this.rdfFormat(HttpEnv.defaultQuadsFormat);
        HttpClient hc = this.requestHttpClient(this.serviceEndpoint, this.serviceEndpoint);
        HttpRDF.httpPutDataset(hc, this.serviceEndpoint, dataset, requestFmt, this.httpHeaders);
    }

    public void clearDataset() {
        this.internalDataset();
        this.validateDatasetOperation();
        ((UpdateExecHTTPBuilder)UpdateExecHTTP.service(this.serviceEndpoint).update("CLEAR ALL")).execute();
    }

    private static void uploadTriples(HttpClient httpClient, String gspUrl, String file, String fileExtContentType, Map<String, String> headers, Push mode) {
        Lang lang = RDFLanguages.contentTypeToLang(fileExtContentType);
        if (lang == null) {
            throw new ARQException("Not a recognized as an RDF format: " + fileExtContentType);
        }
        if (RDFLanguages.isQuads(lang) && !RDFLanguages.isTriples(lang)) {
            throw new ARQException("Can't load quads into a graph");
        }
        if (!RDFLanguages.isTriples(lang)) {
            throw new ARQException("Not an RDF format: " + file + " (lang=" + lang + ")");
        }
        GSP.pushFile(httpClient, gspUrl, file, fileExtContentType, headers, mode);
    }

    private static void uploadQuads(HttpClient httpClient, String endpoint, String file, String fileExtContentType, Map<String, String> headers, Push mode) {
        Lang lang = RDFLanguages.contentTypeToLang(fileExtContentType);
        if (!RDFLanguages.isQuads(lang) && !RDFLanguages.isTriples(lang)) {
            throw new ARQException("Not an RDF format: " + file + " (lang=" + lang + ")");
        }
        GSP.pushFile(httpClient, endpoint, file, fileExtContentType, headers, mode);
    }

    private static String header(String choice, String dftString) {
        return choice != null ? choice : dftString;
    }

    private RDFFormat rdfFormat(RDFFormat dftFormat) {
        if (this.rdfFormat != null) {
            return this.rdfFormat;
        }
        if (this.contentType() == null) {
            return dftFormat;
        }
        Lang lang = RDFLanguages.contentTypeToLang(this.contentType());
        RDFFormat streamFormat = StreamRDFWriter.defaultSerialization(null);
        if (streamFormat != null) {
            return streamFormat;
        }
        return RDFWriterRegistry.defaultSerialization(lang);
    }

    private String contentTypeFromFilename(String filename) {
        String ctx = this.contentType();
        if (ctx != null) {
            return ctx;
        }
        ContentType ct = RDFLanguages.guessContentType(filename);
        return ct == null ? null : ct.getContentTypeStr();
    }

    protected static void pushFile(HttpClient httpClient, String endpoint, String file, String fileContentType, Map<String, String> httpHeaders, Push style) {
        try {
            Path path = Path.of(file, new String[0]);
            if (fileContentType != null) {
                httpHeaders.put("Content-Type", fileContentType);
            }
            HttpRequest.BodyPublisher body = HttpRequest.BodyPublishers.ofFile(path);
            HttpLib.httpPushData(httpClient, style, endpoint, HttpLib.setHeaders(httpHeaders), body);
        }
        catch (FileNotFoundException ex) {
            throw new NotFoundException(file);
        }
    }
}

