/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jclouds.openstack.poppy.v1.domain;

import java.util.List;
import java.util.Set;

import org.jclouds.javax.annotation.Nullable;
import org.jclouds.json.SerializedNames;
import org.jclouds.openstack.v2_0.domain.Link;

import com.google.auto.value.AutoValue;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;

/**
 * Representation of an OpenStack Poppy Service.
 */
@AutoValue
public abstract class Service {

   /**
    * @return Specifies the service ID that represents distributed content. The value is a UUID, such as
    * 96737ae3-cfc1-4c72-be88-5d0e7cc9a3f0, that is generated by the server.
    */
   public abstract String getId();

   /**
    * @return Specifies the name of the service.
    */
   public abstract String getName();

   /**
    * @return Specifies a list of domains used by users to access their website.
    */
   public abstract List<Domain> getDomains();

   /**
    * @return Specifies a list of origin domains or IP addresses where the original assets are stored.
    */
   public abstract List<Origin> getOrigins();

   /**
    * @return Specifies the TTL rules for the assets under this service. Supports wildcards for fine-grained control.
    */
   @Nullable public abstract List<Caching> getCaching();

   /**
    * @return Specifies the restrictions that define who can access assets (content from the CDN cache).
    */
   @Nullable public abstract List<Restriction> getRestrictions();

   /**
    * @return Specifies the CDN provider flavor ID to use. For a list of flavors, see the operation to list the
    * available flavors.
    */
   @Nullable public abstract String getFlavorId();

   /**
    * @return Specifies the current status of the service.
    */
   public abstract ServiceStatus getStatus();

   /**
    * @return Specifies the list of errors that occurred during the previous service action.
    */
   @Nullable public abstract List<Error> getErrors();

   /**
    * @return Specifies the self-navigating JSON document paths.
    */
   public abstract Set<Link> getLinks();

   /**
    * @return Specifies the delivery logging
    */
   public abstract LogDelivery getLogDelivery();

   @SerializedNames({ "id", "name", "domains", "origins", "caching", "restrictions", "flavor_id",
      "status", "errors", "links", "log_delivery" })
   private static Service create(String id, String name, List<Domain> domains,
         List<Origin> origins, List<Caching> caching, List<Restriction> restrictions,
         String flavorId, ServiceStatus status, List<Error> errors, Set<Link> links, LogDelivery logDelivery) {
      return new AutoValue_Service(
            id,
            name,
            ImmutableList.copyOf(domains),
            ImmutableList.copyOf(origins),
            caching != null ? ImmutableList.copyOf(caching) : null,
            restrictions != null ? ImmutableList.copyOf(restrictions) : null,
            flavorId,
            status,
            errors != null ? ImmutableList.copyOf(errors) : null,
            ImmutableSet.copyOf(links),
            logDelivery);
   }

   public UpdateService.Builder toUpdatableService() {
      return UpdateService.builder()
            .name(getName())
            .domains(getDomains())
            .origins(getOrigins())
            .caching(getCaching())
            .restrictions(getRestrictions())
            .flavorId(getFlavorId());
   }
}
