/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jclouds.json;

import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import com.google.common.annotations.Beta;

/**
 * This annotation identifies the canonical factory method on an {@code AutoValue} type used for json.
 * It also dictates the serialized naming convention of the fields. This is required as there's currently
 * no way to add annotations to the fields generated by {@code AutoValue}.
 *
 * <p/>Example:
 * <pre>{@code @AutoValue class Resource {
 *   abstract String id();
 *   @Nullable abstract Map<String, String> metadata();
 *
 *   @SerializedNames({ "Id", "Metadata" }) // Note case format is controlled here!
 *   static Resource create(String id, Map<String, String> metadata) {
 *     return new AutoValue_Resource(id, metadata);
 *   }
 * }}</pre>
 */
@Beta @Target(METHOD) @Retention(RUNTIME)
public @interface SerializedNames {

   /**
    * Ordered values that dictate the naming convention for serialization.
    *
    * <h3>Note</h3>
    * The order of these names must exactly match the factory method parameters and also match the order of the
    * auto-value constructor parameters.
    */
   String[] value();
}
