/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jclouds.ec2.features;

import static org.testng.Assert.assertEquals;
import static org.testng.Assert.assertNotNull;

import java.util.Set;
import java.util.SortedSet;

import org.jclouds.aws.AWSResponseException;
import org.jclouds.compute.internal.BaseComputeServiceContextLiveTest;
import org.jclouds.ec2.EC2Api;
import org.jclouds.ec2.domain.KeyPair;
import org.testng.annotations.AfterClass;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.Sets;

/**
 * Tests behavior of {@code KeyPairApi}
 */
@Test(groups = "live", singleThreaded = true, testName = "KeyPairApiLiveTest")
public class KeyPairApiLiveTest extends BaseComputeServiceContextLiveTest {
   public KeyPairApiLiveTest() {
      provider = "ec2";
   }

   private EC2Api ec2Api;
   private KeyPairApi client;
   
   @Override
   @BeforeClass(groups = { "integration", "live" })
   public void setupContext() {
      super.setupContext();
      ec2Api = view.unwrapApi(EC2Api.class);
      client = ec2Api.getKeyPairApi().get();
   }

   @Test
   void testDescribeKeyPairs() {
      for (String region : ec2Api.getConfiguredRegions()) {
         SortedSet<KeyPair> allResults = Sets.newTreeSet(client.describeKeyPairsInRegion(region));
         assertNotNull(allResults);
         if (!allResults.isEmpty()) {
            KeyPair pair = allResults.last();
            SortedSet<KeyPair> result = Sets.newTreeSet(client.describeKeyPairsInRegion(region, pair.getKeyName()));
            assertNotNull(result);
            KeyPair compare = result.last();
            assertEquals(compare, pair);
         }
      }
   }

   @Test
   void testDescribeKeyPairsWithFilter() {
      for (String region : ec2Api.getConfiguredRegions()) {
         SortedSet<KeyPair> allResults = Sets.newTreeSet(client.describeKeyPairsInRegion(region));
         assertNotNull(allResults);
         if (!allResults.isEmpty()) {
            KeyPair pair = allResults.last();
            SortedSet<KeyPair> result = Sets.newTreeSet(client.describeKeyPairsInRegionWithFilter(region,
                    ImmutableMultimap.<String, String>builder()
                            .put("key-name", pair.getKeyName()).build()));
            assertNotNull(result);
            KeyPair compare = result.last();
            assertEquals(compare, pair);
         }
      }
   }

   @Test(expectedExceptions = AWSResponseException.class)
   void testDescribeKeyPairsWithInvalidFilter() {
      for (String region : ec2Api.getConfiguredRegions()) {
         SortedSet<KeyPair> allResults = Sets.newTreeSet(client.describeKeyPairsInRegion(region));
         assertNotNull(allResults);
         if (!allResults.isEmpty()) {
            KeyPair pair = allResults.last();
            client.describeKeyPairsInRegionWithFilter(region,
                    ImmutableMultimap.<String, String>builder()
                            .put("invalid-filter", pair.getKeyName()).build());
         }
      }
   }

   public static final String PREFIX = System.getProperty("user.name") + "-ec2";

   @Test
   void testCreateKeyPair() {
      String keyName = PREFIX + "1";
      try {
         client.deleteKeyPairInRegion(null, keyName);
      } catch (Exception e) {

      }
      client.deleteKeyPairInRegion(null, keyName);

      KeyPair result = client.createKeyPairInRegion(null, keyName);
      assertNotNull(result);
      assertNotNull(result.getKeyMaterial());
      assertNotNull(result.getSha1OfPrivateKey());
      assertEquals(result.getKeyName(), keyName);

      Set<KeyPair> twoResults = Sets.newLinkedHashSet(client.describeKeyPairsInRegion(null, keyName));
      assertNotNull(twoResults);
      assertEquals(twoResults.size(), 1);
      KeyPair listPair = twoResults.iterator().next();
      assertEquals(listPair.getKeyName(), result.getKeyName());
      assertEquals(listPair.getSha1OfPrivateKey(), result.getSha1OfPrivateKey());
   }

   @AfterClass(groups = { "integration", "live" })
   @Override
   protected void tearDownContext() {
      String keyName = PREFIX + "1";
      try {
         client.deleteKeyPairInRegion(null, keyName);
      } catch (Exception e) {

      }

      super.tearDownContext();
   }

}
