/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jclouds.ec2.features;

import static org.testng.Assert.assertEquals;
import static org.testng.Assert.assertNotNull;

import java.util.SortedSet;

import org.jclouds.compute.internal.BaseComputeServiceContextLiveTest;
import org.jclouds.ec2.EC2Api;
import org.jclouds.ec2.domain.PublicIpInstanceIdPair;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.Sets;

/**
 * Tests behavior of {@code ElasticIPAddressApi}
 */
@Test(groups = "live", singleThreaded = true, testName = "ElasticIPAddressApiLiveTest")
public class ElasticIPAddressApiLiveTest extends BaseComputeServiceContextLiveTest {
   public ElasticIPAddressApiLiveTest() {
      provider = "ec2";
   }

   private EC2Api ec2Api;
   private ElasticIPAddressApi client;

   @Override
   @BeforeClass(groups = { "integration", "live" })
   public void setupContext() {
      super.setupContext();
      ec2Api = view.unwrapApi(EC2Api.class);
      client = ec2Api.getElasticIPAddressApi().get();
   }

   @Test
   void testDescribeAddresses() {
      for (String region : ec2Api.getConfiguredRegions()) {
         SortedSet<PublicIpInstanceIdPair> allResults = Sets.newTreeSet(client.describeAddressesInRegion(region));
         assertNotNull(allResults);
         if (!allResults.isEmpty()) {
            PublicIpInstanceIdPair pair = allResults.last();
            SortedSet<PublicIpInstanceIdPair> result = Sets.newTreeSet(client.describeAddressesInRegion(region, pair
                     .getPublicIp()));
            assertNotNull(result);
            PublicIpInstanceIdPair compare = result.last();
            assertEquals(compare, pair);

            SortedSet<PublicIpInstanceIdPair> filterResult = Sets.newTreeSet(client.describeAddressesInRegionWithFilter(
                    region, ImmutableMultimap.<String, String>builder().put("public-ip", pair.getPublicIp()).build()));
            assertNotNull(filterResult);
            PublicIpInstanceIdPair filterCompare = filterResult.last();
            assertEquals(filterCompare, pair);
         }
      }
   }

}
