/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jclouds.ec2.compute.functions;

import static org.easymock.EasyMock.createMock;
import static org.easymock.EasyMock.expect;
import static org.easymock.EasyMock.replay;
import static org.easymock.EasyMock.verify;
import static org.testng.Assert.assertEquals;

import java.net.UnknownHostException;

import org.jclouds.ec2.EC2ApiMetadata;
import org.jclouds.ec2.EC2Api;
import org.jclouds.ec2.domain.KeyPair;
import org.jclouds.ec2.features.KeyPairApi;
import org.testng.annotations.Test;

import com.google.common.base.Optional;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.inject.AbstractModule;
import com.google.inject.Guice;
import com.google.inject.TypeLiteral;
import com.google.inject.name.Names;

@Test(groups = "unit", testName = "CreateUniqueKeyPairTest")
public class CreateUniqueKeyPairTest {
   
   @Test
   public void testApply() throws UnknownHostException {
      final EC2Api client = createMock(EC2Api.class);
      KeyPairApi keyClient = createMock(KeyPairApi.class);
      KeyPair pair = createMock(KeyPair.class);

      expect(client.getKeyPairApi()).andReturn((Optional) Optional.of(keyClient)).atLeastOnce();

      expect(keyClient.createKeyPairInRegion("region", "jclouds#group#1")).andReturn(pair);

      replay(client);
      replay(keyClient);

      CreateUniqueKeyPair parser = Guice.createInjector(new AbstractModule() {

         @Override
         protected void configure() {
            Names.bindProperties(binder(), new EC2ApiMetadata().getDefaultProperties());
            bind(new TypeLiteral<Supplier<String>>() {
            }).toInstance(Suppliers.ofInstance("1"));
            bind(EC2Api.class).toInstance(client);
         }

      }).getInstance(CreateUniqueKeyPair.class);

      assertEquals(parser.createNewKeyPairInRegion("region", "group"), pair);

      verify(client);
      verify(keyClient);
   }

   @SuppressWarnings( { "unchecked" })
   @Test
   public void testApplyWithIllegalStateException() throws UnknownHostException {
      final EC2Api client = createMock(EC2Api.class);
      KeyPairApi keyClient = createMock(KeyPairApi.class);
      final Supplier<String> uniqueIdSupplier = createMock(Supplier.class);

      KeyPair pair = createMock(KeyPair.class);

      expect(client.getKeyPairApi()).andReturn((Optional) Optional.of(keyClient)).atLeastOnce();

      expect(uniqueIdSupplier.get()).andReturn("1");
      expect(keyClient.createKeyPairInRegion("region", "jclouds#group#1")).andThrow(new IllegalStateException());
      expect(uniqueIdSupplier.get()).andReturn("2");
      expect(keyClient.createKeyPairInRegion("region", "jclouds#group#2")).andReturn(pair);

      replay(client);
      replay(keyClient);
      replay(uniqueIdSupplier);

      CreateUniqueKeyPair parser = Guice.createInjector(new AbstractModule() {

         @Override
         protected void configure() {
            Names.bindProperties(binder(), new EC2ApiMetadata().getDefaultProperties());
            bind(new TypeLiteral<Supplier<String>>() {
            }).toInstance(uniqueIdSupplier);
            bind(EC2Api.class).toInstance(client);
         }

      }).getInstance(CreateUniqueKeyPair.class);

      assertEquals(parser.createNewKeyPairInRegion("region", "group"), pair);

      verify(client);
      verify(keyClient);
      verify(uniqueIdSupplier);
   }
}
