/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.mailbox.cassandra.mail.task;

import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableList;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicLong;
import javax.inject.Inject;
import javax.mail.Flags;
import org.apache.james.backends.cassandra.init.configuration.JamesExecutionProfiles;
import org.apache.james.mailbox.cassandra.ids.CassandraId;
import org.apache.james.mailbox.cassandra.ids.CassandraMessageId;
import org.apache.james.mailbox.cassandra.mail.CassandraMailboxCounterDAO;
import org.apache.james.mailbox.cassandra.mail.CassandraMailboxDAO;
import org.apache.james.mailbox.cassandra.mail.CassandraMessageIdDAO;
import org.apache.james.mailbox.cassandra.mail.CassandraMessageIdToImapUidDAO;
import org.apache.james.mailbox.cassandra.mail.CassandraMessageMetadata;
import org.apache.james.mailbox.model.ComposedMessageIdWithMetaData;
import org.apache.james.mailbox.model.Mailbox;
import org.apache.james.mailbox.model.MailboxCounters;
import org.apache.james.mailbox.model.MailboxId;
import org.apache.james.mailbox.model.MessageRange;
import org.apache.james.task.Task;
import org.apache.james.util.streams.Limit;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public class RecomputeMailboxCountersService {
    private static final Logger LOGGER = LoggerFactory.getLogger(RecomputeMailboxCountersService.class);
    private static final int MAILBOX_CONCURRENCY = 2;
    private static final int MESSAGE_CONCURRENCY = 8;
    private final CassandraMailboxDAO mailboxDAO;
    private final CassandraMessageIdDAO imapUidToMessageIdDAO;
    private final CassandraMessageIdToImapUidDAO messageIdToImapUidDAO;
    private final CassandraMailboxCounterDAO counterDAO;

    @Inject
    RecomputeMailboxCountersService(CassandraMailboxDAO mailboxDAO, CassandraMessageIdDAO imapUidToMessageIdDAO, CassandraMessageIdToImapUidDAO messageIdToImapUidDAO, CassandraMailboxCounterDAO counterDAO) {
        this.mailboxDAO = mailboxDAO;
        this.imapUidToMessageIdDAO = imapUidToMessageIdDAO;
        this.messageIdToImapUidDAO = messageIdToImapUidDAO;
        this.counterDAO = counterDAO;
    }

    Mono<Task.Result> recomputeMailboxCounters(Context context, Options options) {
        return this.mailboxDAO.retrieveAllMailboxes().flatMap(mailbox -> this.recomputeMailboxCounter(context, (Mailbox)mailbox, options), 2).reduce((Object)Task.Result.COMPLETED, Task::combine).onErrorResume(e -> {
            LOGGER.error("Error listing mailboxes", e);
            return Mono.just((Object)Task.Result.PARTIAL);
        });
    }

    public Mono<Task.Result> recomputeMailboxCounter(Context context, Mailbox mailbox, Options options) {
        CassandraId mailboxId = (CassandraId)mailbox.getMailboxId();
        Counter counter = new Counter(mailboxId);
        return this.imapUidToMessageIdDAO.retrieveMessages(mailboxId, MessageRange.all(), Limit.unlimited()).map(CassandraMessageMetadata::getComposedMessageId).flatMap(message -> this.latestMetadata(mailboxId, (ComposedMessageIdWithMetaData)message, options), 8).doOnNext(counter::process).then(Mono.defer(() -> this.counterDAO.resetCounters(counter.snapshot()))).then(Mono.just((Object)Task.Result.COMPLETED)).doOnNext(any -> {
            LOGGER.info("Counters recomputed for {}", (Object)mailboxId.serialize());
            context.incrementProcessed();
        }).onErrorResume(e -> {
            context.addToFailedMailboxes(mailboxId);
            LOGGER.error("Error while recomputing counters for {}", (Object)mailboxId.serialize(), e);
            return Mono.just((Object)Task.Result.PARTIAL);
        });
    }

    private Flux<ComposedMessageIdWithMetaData> latestMetadata(CassandraId mailboxId, ComposedMessageIdWithMetaData message, Options options) {
        if (options.isMessageProjectionTrusted()) {
            return Flux.just((Object)message);
        }
        CassandraMessageId messageId = (CassandraMessageId)message.getComposedMessageId().getMessageId();
        return this.messageIdToImapUidDAO.retrieve(messageId, Optional.of(mailboxId), JamesExecutionProfiles.ConsistencyChoice.STRONG).map(CassandraMessageMetadata::getComposedMessageId).doOnNext(trustedMessage -> {
            if (!trustedMessage.equals((Object)message)) {
                LOGGER.warn("Possible denormalization issue on {}. Mismatch between the two denormalization table. This can also be due to concurrent modifications.", (Object)message.getComposedMessageId());
            }
        }).switchIfEmpty((Publisher)Flux.empty().doOnComplete(() -> LOGGER.warn("Possible denormalization issue on {}. Source of truth do not contain listed entry.This can also be due to concurrent modifications.", (Object)message.getComposedMessageId())));
    }

    public static class Context {
        private final AtomicLong processedMailboxCount = new AtomicLong();
        private final ConcurrentLinkedDeque<CassandraId> failedMailboxes = new ConcurrentLinkedDeque();

        void incrementProcessed() {
            this.processedMailboxCount.incrementAndGet();
        }

        void addToFailedMailboxes(CassandraId cassandraId) {
            this.failedMailboxes.add(cassandraId);
        }

        Snapshot snapshot() {
            return new Snapshot(this.processedMailboxCount.get(), (ImmutableList<CassandraId>)ImmutableList.copyOf(this.failedMailboxes));
        }

        static class Snapshot {
            private final long processedMailboxCount;
            private final ImmutableList<CassandraId> failedMailboxes;

            private Snapshot(long processedMailboxCount, ImmutableList<CassandraId> failedMailboxes) {
                this.processedMailboxCount = processedMailboxCount;
                this.failedMailboxes = failedMailboxes;
            }

            long getProcessedMailboxCount() {
                return this.processedMailboxCount;
            }

            ImmutableList<CassandraId> getFailedMailboxes() {
                return this.failedMailboxes;
            }

            public final boolean equals(Object o) {
                if (o instanceof Snapshot) {
                    Snapshot snapshot = (Snapshot)o;
                    return Objects.equals(this.processedMailboxCount, snapshot.processedMailboxCount) && Objects.equals(this.failedMailboxes, snapshot.failedMailboxes);
                }
                return false;
            }

            public final int hashCode() {
                return Objects.hash(this.processedMailboxCount, this.failedMailboxes);
            }

            public String toString() {
                return MoreObjects.toStringHelper((Object)this).add("processedMailboxCount", this.processedMailboxCount).add("failedMailboxes", this.failedMailboxes).toString();
            }
        }
    }

    public static class Options {
        private final boolean trustMessageProjection;

        public static Options trustMessageProjection() {
            return Options.of(true);
        }

        public static Options recheckMessageProjection() {
            return Options.of(false);
        }

        public static Options of(boolean value) {
            return new Options(value);
        }

        private Options(boolean trustMessageProjection) {
            this.trustMessageProjection = trustMessageProjection;
        }

        public boolean isMessageProjectionTrusted() {
            return this.trustMessageProjection;
        }

        public final boolean equals(Object o) {
            if (o instanceof Options) {
                Options options = (Options)o;
                return Objects.equals(this.trustMessageProjection, options.trustMessageProjection);
            }
            return false;
        }

        public final int hashCode() {
            return Objects.hash(this.trustMessageProjection);
        }
    }

    private static class Counter {
        private final CassandraId mailboxId;
        private final AtomicLong total;
        private final AtomicLong unseen;

        private Counter(CassandraId mailboxId) {
            this.mailboxId = mailboxId;
            this.unseen = new AtomicLong();
            this.total = new AtomicLong();
        }

        void process(ComposedMessageIdWithMetaData metadata) {
            this.total.incrementAndGet();
            if (!metadata.getFlags().contains(Flags.Flag.SEEN)) {
                this.unseen.incrementAndGet();
            }
        }

        MailboxCounters snapshot() {
            return MailboxCounters.builder().mailboxId((MailboxId)this.mailboxId).count(this.total.get()).unseen(this.unseen.get()).build();
        }
    }
}

