/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.transport.mailets.remote.delivery;

import com.github.fge.lambdas.Throwing;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import java.io.Closeable;
import java.time.Duration;
import java.util.Collection;
import java.util.Date;
import java.util.function.Consumer;
import java.util.function.Supplier;
import org.apache.james.dnsservice.api.DNSService;
import org.apache.james.lifecycle.api.LifecycleUtil;
import org.apache.james.metrics.api.Metric;
import org.apache.james.metrics.api.MetricFactory;
import org.apache.james.metrics.api.TimeMetric;
import org.apache.james.queue.api.MailPrioritySupport;
import org.apache.james.queue.api.MailQueue;
import org.apache.james.transport.mailets.remote.delivery.Bouncer;
import org.apache.james.transport.mailets.remote.delivery.Delay;
import org.apache.james.transport.mailets.remote.delivery.DeliveryRetriesHelper;
import org.apache.james.transport.mailets.remote.delivery.ExecutionResult;
import org.apache.james.transport.mailets.remote.delivery.MailDelivrer;
import org.apache.james.transport.mailets.remote.delivery.MailDelivrerToHost;
import org.apache.james.transport.mailets.remote.delivery.RemoteDeliveryConfiguration;
import org.apache.james.util.MDCBuilder;
import org.apache.mailet.Attribute;
import org.apache.mailet.AttributeValue;
import org.apache.mailet.Mail;
import org.apache.mailet.MailetContext;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.Disposable;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Scheduler;
import reactor.core.scheduler.Schedulers;
import reactor.util.concurrent.Queues;

public class DeliveryRunnable
implements Disposable {
    private static final Logger LOGGER = LoggerFactory.getLogger(DeliveryRunnable.class);
    public static final Supplier<Date> CURRENT_DATE_SUPPLIER = Date::new;
    public static final String OUTGOING_MAILS = "outgoingMails";
    public static final String REMOTE_DELIVERY_TRIAL = "RemoteDeliveryTrial";
    private final MailQueue queue;
    private final RemoteDeliveryConfiguration configuration;
    private final Metric outgoingMailsMetric;
    private final MetricFactory metricFactory;
    private final Bouncer bouncer;
    private final MailDelivrer mailDelivrer;
    private final Supplier<Date> dateSupplier;
    private final MailetContext mailetContext;
    private Disposable disposable;
    private Scheduler remoteDeliveryScheduler;

    public DeliveryRunnable(MailQueue queue, RemoteDeliveryConfiguration configuration, DNSService dnsServer, MetricFactory metricFactory, MailetContext mailetContext, Bouncer bouncer) {
        this(queue, configuration, metricFactory, bouncer, new MailDelivrer(configuration, new MailDelivrerToHost(configuration, mailetContext), dnsServer, bouncer, mailetContext), CURRENT_DATE_SUPPLIER, mailetContext);
    }

    @VisibleForTesting
    DeliveryRunnable(MailQueue queue, RemoteDeliveryConfiguration configuration, MetricFactory metricFactory, Bouncer bouncer, MailDelivrer mailDelivrer, Supplier<Date> dateSupplier, MailetContext mailetContext) {
        this.queue = queue;
        this.configuration = configuration;
        this.outgoingMailsMetric = metricFactory.generate(OUTGOING_MAILS);
        this.bouncer = bouncer;
        this.mailDelivrer = mailDelivrer;
        this.dateSupplier = dateSupplier;
        this.metricFactory = metricFactory;
        this.mailetContext = mailetContext;
    }

    public void start() {
        this.remoteDeliveryScheduler = Schedulers.newBoundedElastic((int)Schedulers.DEFAULT_BOUNDED_ELASTIC_SIZE, (int)Schedulers.DEFAULT_BOUNDED_ELASTIC_QUEUESIZE, (String)"RemoteDelivery");
        this.disposable = Flux.from((Publisher)this.queue.deQueue()).flatMap(queueItem -> this.runStep((MailQueue.MailQueueItem)queueItem).subscribeOn(this.remoteDeliveryScheduler), Queues.SMALL_BUFFER_SIZE).onErrorContinue((throwable, nothing) -> LOGGER.error("Exception caught in RemoteDelivery", throwable)).subscribeOn(this.remoteDeliveryScheduler).subscribe();
    }

    private Mono<Void> runStep(MailQueue.MailQueueItem queueItem) {
        TimeMetric timeMetric = this.metricFactory.timer(REMOTE_DELIVERY_TRIAL);
        return this.processMail(queueItem).doOnSuccess(any -> timeMetric.stopAndPublish());
    }

    private Mono<Void> processMail(MailQueue.MailQueueItem queueItem) {
        return Mono.create(sink -> {
            Mail mail = queueItem.getMail();
            try (Closeable closeable = MDCBuilder.create().addToContext("mail", mail.getName()).addToContext("recipients", ImmutableList.copyOf((Collection)mail.getRecipients()).toString()).addToContext("sender", mail.getMaybeSender().asString()).build();){
                LOGGER.debug("will process mail {}", (Object)mail.getName());
                this.attemptDelivery(mail);
                queueItem.done(true);
                sink.success();
            }
            catch (Exception e) {
                try {
                    queueItem.done(false);
                }
                catch (Exception ex) {
                    sink.error((Throwable)ex);
                    return;
                }
                sink.error((Throwable)e);
            }
            finally {
                LifecycleUtil.dispose((Object)mail);
            }
        });
    }

    @VisibleForTesting
    void attemptDelivery(Mail mail) throws MailQueue.MailQueueException {
        ExecutionResult executionResult = this.mailDelivrer.deliver(mail);
        switch (executionResult.getExecutionState()) {
            case SUCCESS: {
                this.outgoingMailsMetric.increment();
                this.configuration.getOnSuccess().ifPresent((Consumer<String>)Throwing.consumer(onSuccess -> this.mailetContext.sendMail(mail, onSuccess)));
                break;
            }
            case TEMPORARY_FAILURE: {
                this.handleTemporaryFailure(mail, executionResult);
                break;
            }
            case PERMANENT_FAILURE: {
                this.handlePermanentFailure(mail, executionResult);
            }
        }
    }

    private void handlePermanentFailure(Mail mail, ExecutionResult executionResult) {
        mail.setAttribute(new Attribute(Bouncer.IS_DELIVERY_PERMANENT_ERROR, AttributeValue.of((Boolean)true)));
        this.bouncer.bounce(mail, executionResult.getException().orElse(null));
    }

    private void handleTemporaryFailure(Mail mail, ExecutionResult executionResult) throws MailQueue.MailQueueException {
        if (!mail.getState().equals("error")) {
            mail.setState("error");
            DeliveryRetriesHelper.initRetries(mail);
            mail.setLastUpdated(this.dateSupplier.get());
        }
        mail.setAttribute(new Attribute(Bouncer.IS_DELIVERY_PERMANENT_ERROR, AttributeValue.of((Boolean)false)));
        int retries = DeliveryRetriesHelper.retrieveRetries(mail);
        if (retries < this.configuration.getMaxRetries()) {
            this.reAttemptDelivery(mail, retries);
        } else {
            LOGGER.debug("Bouncing message {} after {} retries", (Object)mail.getName(), (Object)retries);
            this.bouncer.bounce(mail, new Exception("Too many retries failure. Bouncing after " + retries + " retries.", executionResult.getException().orElse(null)));
        }
    }

    private void reAttemptDelivery(Mail mail, int retries) throws MailQueue.MailQueueException {
        LOGGER.debug("Storing message {} into outgoing after {} retries", (Object)mail.getName(), (Object)retries);
        DeliveryRetriesHelper.incrementRetries(mail);
        mail.setLastUpdated(this.dateSupplier.get());
        Duration delay = this.getNextDelay(DeliveryRetriesHelper.retrieveRetries(mail));
        if (this.configuration.isUsePriority()) {
            mail.setAttribute(MailPrioritySupport.LOW_PRIORITY_ATTRIBUTE);
        }
        this.queue.enQueue(mail, delay);
    }

    private Duration getNextDelay(int retryCount) {
        if (retryCount > this.configuration.getDelayTimes().size()) {
            return Delay.DEFAULT_DELAY_TIME;
        }
        return this.configuration.getDelayTimes().get(retryCount - 1);
    }

    public void dispose() {
        this.disposable.dispose();
        this.remoteDeliveryScheduler.dispose();
    }
}

