/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.jackrabbit.oak.plugins.document;

import org.apache.jackrabbit.oak.api.CommitFailedException;
import org.jetbrains.annotations.NotNull;

import static java.util.Objects.requireNonNull;
import static org.apache.jackrabbit.oak.api.CommitFailedException.MERGE;

import java.util.Collections;
import java.util.Set;

/**
 * An exception with an optional conflict revision. The DocumentNodeStore
 * implementation will throw this exception when a commit or merge fails with a
 * conflict.
 */
class ConflictException extends Exception {

    private static final long serialVersionUID = 1418838561903727231L;

    /**
     * Optional conflict revisions list.
     */
    private final Set<Revision> conflictRevisions;

    /**
     * @param message the exception / conflict message.
     * @param conflictRevision the conflict revision
     */
    ConflictException(@NotNull String message,
                      @NotNull Revision conflictRevision) {
        super(requireNonNull(message));
        this.conflictRevisions = Collections.singleton(requireNonNull(conflictRevision));
    }

    /**
     * @param message the exception / conflict message.
     * @param conflictRevisions the conflict revision list
     */
    ConflictException(@NotNull String message,
                      @NotNull Set<Revision> conflictRevisions) {
        super(requireNonNull(message));
        this.conflictRevisions = requireNonNull(conflictRevisions);
    }

    /**
     * @param message the exception / conflict message.
     */
    ConflictException(@NotNull String message) {
        super(requireNonNull(message));
        this.conflictRevisions = Collections.emptySet();
    }

    /**
     * Convert this exception into a {@link CommitFailedException}. This
     * exception will be set as the cause of the returned exception.
     *
     * @return a {@link CommitFailedException}.
     */
    CommitFailedException asCommitFailedException() {
        if (!conflictRevisions.isEmpty()) {
            return new FailedWithConflictException(conflictRevisions, getMessage(), this);
        } else {
            return new CommitFailedException(MERGE, 1,
                    "Failed to merge changes to the underlying store", this);
        }
    }

    /**
     * List of conflict revisions.
     *
     * @return a list of conflict revisions (may be empty)
     */
    @NotNull
    Iterable<Revision> getConflictRevisions() {
        return conflictRevisions;
    }
}
