/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.commons.binaryallocator;

import java.time.Duration;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.iotdb.commons.binaryallocator.BinaryAllocatorState;
import org.apache.iotdb.commons.binaryallocator.arena.Arena;
import org.apache.iotdb.commons.binaryallocator.arena.ArenaStrategy;
import org.apache.iotdb.commons.binaryallocator.config.AllocatorConfig;
import org.apache.iotdb.commons.binaryallocator.evictor.Evictor;
import org.apache.iotdb.commons.binaryallocator.metric.BinaryAllocatorMetrics;
import org.apache.iotdb.commons.binaryallocator.utils.SizeClasses;
import org.apache.iotdb.commons.concurrent.ThreadName;
import org.apache.iotdb.commons.service.metric.MetricService;
import org.apache.iotdb.commons.utils.TestOnly;
import org.apache.tsfile.utils.PooledBinary;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BinaryAllocator {
    private static final Logger LOGGER = LoggerFactory.getLogger(BinaryAllocator.class);
    private final Arena[] heapArenas;
    private final AllocatorConfig allocatorConfig;
    private final ArenaStrategy arenaStrategy = new LeastUsedArenaStrategy();
    private final AtomicReference<BinaryAllocatorState> state = new AtomicReference<BinaryAllocatorState>(BinaryAllocatorState.UNINITIALIZED);
    private final BinaryAllocatorMetrics metrics;
    private Evictor sampleEvictor;
    private static final ThreadLocal<ThreadArenaRegistry> arenaRegistry = ThreadLocal.withInitial(() -> new ThreadArenaRegistry());
    private static final int WARNING_GC_TIME_PERCENTAGE = 10;
    private static final int HALF_GC_TIME_PERCENTAGE = 20;
    private static final int SHUTDOWN_GC_TIME_PERCENTAGE = 30;
    private static final int RESTART_GC_TIME_PERCENTAGE = 5;

    public BinaryAllocator(AllocatorConfig allocatorConfig) {
        this.allocatorConfig = allocatorConfig;
        this.heapArenas = new Arena[allocatorConfig.arenaNum];
        SizeClasses sizeClasses = new SizeClasses(allocatorConfig);
        for (int i = 0; i < this.heapArenas.length; ++i) {
            Arena arena;
            this.heapArenas[i] = arena = new Arena(this, sizeClasses, i, allocatorConfig);
        }
        this.metrics = new BinaryAllocatorMetrics(this);
        if (allocatorConfig.enableBinaryAllocator) {
            this.start();
        } else {
            this.state.set(BinaryAllocatorState.CLOSE);
        }
    }

    public synchronized void start() {
        if (this.state.get() == BinaryAllocatorState.OPEN) {
            return;
        }
        this.state.set(BinaryAllocatorState.OPEN);
        MetricService.getInstance().addMetricSet(this.metrics);
        this.sampleEvictor = new SampleEvictor(ThreadName.BINARY_ALLOCATOR_SAMPLE_EVICTOR.getName(), this.allocatorConfig.durationEvictorShutdownTimeout);
        this.sampleEvictor.startEvictor(this.allocatorConfig.durationBetweenEvictorRuns);
    }

    public synchronized void close(boolean forceClose) {
        if (forceClose) {
            this.state.set(BinaryAllocatorState.CLOSE);
            MetricService.getInstance().removeMetricSet(this.metrics);
        } else {
            this.state.set(BinaryAllocatorState.PENDING);
        }
        this.sampleEvictor.stopEvictor();
        for (Arena arena : this.heapArenas) {
            arena.close();
        }
    }

    public PooledBinary allocateBinary(int reqCapacity) {
        if (reqCapacity < this.allocatorConfig.minAllocateSize || reqCapacity > this.allocatorConfig.maxAllocateSize || this.state.get() != BinaryAllocatorState.OPEN) {
            return new PooledBinary(new byte[reqCapacity]);
        }
        Arena arena = this.arenaStrategy.choose(this.heapArenas);
        return new PooledBinary(arena.allocate(reqCapacity), reqCapacity, arena.getArenaID());
    }

    public void deallocateBinary(PooledBinary binary) {
        int arenaIndex;
        if (binary != null && binary.getLength() >= this.allocatorConfig.minAllocateSize && binary.getLength() <= this.allocatorConfig.maxAllocateSize && this.state.get() == BinaryAllocatorState.OPEN && (arenaIndex = binary.getArenaIndex()) != -1) {
            Arena arena = this.heapArenas[arenaIndex];
            arena.deallocate(binary.getValues());
        }
    }

    public long getTotalUsedMemory() {
        long totalUsedMemory = 0L;
        for (Arena arena : this.heapArenas) {
            totalUsedMemory += arena.getTotalUsedMemory();
        }
        return totalUsedMemory;
    }

    public long getTotalActiveMemory() {
        long totalActiveMemory = 0L;
        for (Arena arena : this.heapArenas) {
            totalActiveMemory += arena.getActiveMemory();
        }
        return totalActiveMemory;
    }

    @TestOnly
    public void resetArenaBinding() {
        arenaRegistry.get().unbindArena();
    }

    public BinaryAllocatorMetrics getMetrics() {
        return this.metrics;
    }

    private long evict(double ratio) {
        long evictedSize = 0L;
        for (Arena arena : this.heapArenas) {
            evictedSize += arena.evict(ratio);
        }
        return evictedSize;
    }

    public static BinaryAllocator getInstance() {
        return BinaryAllocatorHolder.INSTANCE;
    }

    public void runGcEviction(long curGcTimePercent) {
        if (this.state.get() == BinaryAllocatorState.CLOSE) {
            return;
        }
        LOGGER.debug("Binary allocator running GC eviction");
        if (this.state.get() == BinaryAllocatorState.PENDING) {
            if (curGcTimePercent <= 5L) {
                this.start();
            }
            return;
        }
        long evictedSize = 0L;
        if (curGcTimePercent > 30L) {
            LOGGER.info("Binary allocator is shutting down because of high GC time percentage {}%.", (Object)curGcTimePercent);
            evictedSize = this.evict(1.0);
            this.close(false);
        } else if (curGcTimePercent > 20L) {
            evictedSize = this.evict(0.5);
        } else if (curGcTimePercent > 10L) {
            evictedSize = this.evict(0.2);
        }
        this.metrics.updateGcEvictionCounter(evictedSize);
    }

    private static class LeastUsedArenaStrategy
    implements ArenaStrategy {
        private LeastUsedArenaStrategy() {
        }

        @Override
        public Arena choose(Arena[] arenas) {
            Arena boundArena = ((ThreadArenaRegistry)arenaRegistry.get()).getArena();
            if (boundArena != null) {
                return boundArena;
            }
            Arena minArena = arenas[0];
            for (int i = 1; i < arenas.length; ++i) {
                Arena arena = arenas[i];
                if (arena.getNumRegisteredThread() >= minArena.getNumRegisteredThread()) continue;
                minArena = arena;
            }
            ((ThreadArenaRegistry)arenaRegistry.get()).bindArena(minArena);
            return minArena;
        }
    }

    public class SampleEvictor
    extends Evictor {
        public SampleEvictor(String name, Duration evictorShutdownTimeoutDuration) {
            super(name, evictorShutdownTimeoutDuration);
        }

        @Override
        public void run() {
            long evictedSize = 0L;
            for (Arena arena : BinaryAllocator.this.heapArenas) {
                evictedSize += arena.runSampleEviction();
            }
            BinaryAllocator.this.metrics.updateSampleEvictionCounter(evictedSize);
        }
    }

    private static class ThreadArenaRegistry {
        private Arena threadArenaBinding = null;

        private ThreadArenaRegistry() {
        }

        public Arena getArena() {
            return this.threadArenaBinding;
        }

        public void bindArena(Arena arena) {
            this.threadArenaBinding = arena;
            arena.incRegisteredThread();
        }

        public void unbindArena() {
            Arena arena = this.threadArenaBinding;
            if (arena != null) {
                arena.decRegisteredThread();
                this.threadArenaBinding = null;
            }
        }

        protected void finalize() {
            this.unbindArena();
        }
    }

    private static class BinaryAllocatorHolder {
        private static final BinaryAllocator INSTANCE = new BinaryAllocator(AllocatorConfig.DEFAULT_CONFIG);

        private BinaryAllocatorHolder() {
        }
    }
}

