/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.pipe.resource.memory;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.function.LongUnaryOperator;
import org.apache.iotdb.commons.exception.pipe.PipeRuntimeOutOfMemoryCriticalException;
import org.apache.iotdb.commons.pipe.config.PipeConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.pipe.agent.PipeDataNodeAgent;
import org.apache.iotdb.db.pipe.resource.memory.PipeMemoryBlock;
import org.apache.iotdb.db.pipe.resource.memory.PipeMemoryBlockType;
import org.apache.iotdb.db.pipe.resource.memory.PipeTabletMemoryBlock;
import org.apache.iotdb.db.pipe.resource.memory.PipeTsFileMemoryBlock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PipeMemoryManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(PipeMemoryManager.class);
    private static final boolean PIPE_MEMORY_MANAGEMENT_ENABLED = PipeConfig.getInstance().getPipeMemoryManagementEnabled();
    private static final int MEMORY_ALLOCATE_MAX_RETRIES = PipeConfig.getInstance().getPipeMemoryAllocateMaxRetries();
    private static final long MEMORY_ALLOCATE_RETRY_INTERVAL_IN_MS = PipeConfig.getInstance().getPipeMemoryAllocateRetryIntervalInMs();
    private static final long TOTAL_MEMORY_SIZE_IN_BYTES = IoTDBDescriptor.getInstance().getConfig().getAllocateMemoryForPipe();
    private static final long MEMORY_ALLOCATE_MIN_SIZE_IN_BYTES = PipeConfig.getInstance().getPipeMemoryAllocateMinSizeInBytes();
    private long usedMemorySizeInBytes;
    private static final double TABLET_MEMORY_REJECT_THRESHOLD = PipeConfig.getInstance().getPipeDataStructureTabletMemoryBlockAllocationRejectThreshold();
    private volatile long usedMemorySizeInBytesOfTablets;
    private static final double TS_FILE_MEMORY_REJECT_THRESHOLD = PipeConfig.getInstance().getPipeDataStructureTsFileMemoryBlockAllocationRejectThreshold();
    private volatile long usedMemorySizeInBytesOfTsFiles;
    private final Set<PipeMemoryBlock> allocatedBlocks = new HashSet<PipeMemoryBlock>();

    public PipeMemoryManager() {
        PipeDataNodeAgent.runtime().registerPeriodicalJob("PipeMemoryManager#tryExpandAll()", this::tryExpandAllAndCheckConsistency, PipeConfig.getInstance().getPipeMemoryExpanderIntervalSeconds());
    }

    public boolean isEnough4TabletParsing() {
        return (double)this.usedMemorySizeInBytesOfTablets < 0.95 * TABLET_MEMORY_REJECT_THRESHOLD * (double)TOTAL_MEMORY_SIZE_IN_BYTES;
    }

    public boolean isEnough4TsFileSlicing() {
        return (double)this.usedMemorySizeInBytesOfTsFiles < 0.95 * TS_FILE_MEMORY_REJECT_THRESHOLD * (double)TOTAL_MEMORY_SIZE_IN_BYTES;
    }

    public synchronized PipeMemoryBlock forceAllocate(long sizeInBytes) throws PipeRuntimeOutOfMemoryCriticalException {
        return this.forceAllocate(sizeInBytes, PipeMemoryBlockType.NORMAL);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public PipeTabletMemoryBlock forceAllocateForTabletWithRetry(long tabletSizeInBytes) throws PipeRuntimeOutOfMemoryCriticalException {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED) {
            return new PipeTabletMemoryBlock(0L);
        }
        for (int i = 1; i <= MEMORY_ALLOCATE_MAX_RETRIES && !((double)this.usedMemorySizeInBytesOfTablets / (double)TOTAL_MEMORY_SIZE_IN_BYTES < TABLET_MEMORY_REJECT_THRESHOLD); ++i) {
            try {
                Thread.sleep(MEMORY_ALLOCATE_RETRY_INTERVAL_IN_MS);
                continue;
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                LOGGER.warn("forceAllocateWithRetry: interrupted while waiting for available memory", (Throwable)ex);
            }
        }
        if ((double)this.usedMemorySizeInBytesOfTablets / (double)TOTAL_MEMORY_SIZE_IN_BYTES >= TABLET_MEMORY_REJECT_THRESHOLD) {
            throw new PipeRuntimeOutOfMemoryCriticalException(String.format("forceAllocateForTablet: failed to allocate because there's too much memory for tablets, total memory size %d bytes, used memory for tablet size %d bytes", TOTAL_MEMORY_SIZE_IN_BYTES, this.usedMemorySizeInBytesOfTablets));
        }
        PipeMemoryManager pipeMemoryManager = this;
        synchronized (pipeMemoryManager) {
            PipeTabletMemoryBlock block = (PipeTabletMemoryBlock)this.forceAllocate(tabletSizeInBytes, PipeMemoryBlockType.TABLET);
            this.usedMemorySizeInBytesOfTablets += block.getMemoryUsageInBytes();
            return block;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public PipeTsFileMemoryBlock forceAllocateForTsFileWithRetry(long tsFileSizeInBytes) throws PipeRuntimeOutOfMemoryCriticalException {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED) {
            return new PipeTsFileMemoryBlock(0L);
        }
        for (int i = 1; i <= MEMORY_ALLOCATE_MAX_RETRIES && !((double)this.usedMemorySizeInBytesOfTsFiles / (double)TOTAL_MEMORY_SIZE_IN_BYTES < TS_FILE_MEMORY_REJECT_THRESHOLD); ++i) {
            try {
                Thread.sleep(MEMORY_ALLOCATE_RETRY_INTERVAL_IN_MS);
                continue;
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                LOGGER.warn("forceAllocateWithRetry: interrupted while waiting for available memory", (Throwable)ex);
            }
        }
        if ((double)this.usedMemorySizeInBytesOfTsFiles / (double)TOTAL_MEMORY_SIZE_IN_BYTES >= TS_FILE_MEMORY_REJECT_THRESHOLD) {
            throw new PipeRuntimeOutOfMemoryCriticalException(String.format("forceAllocateForTsFile: failed to allocate because there's too much memory for tsfiles, total memory size %d bytes, used memory for tsfile size %d bytes", TOTAL_MEMORY_SIZE_IN_BYTES, this.usedMemorySizeInBytesOfTsFiles));
        }
        PipeMemoryManager pipeMemoryManager = this;
        synchronized (pipeMemoryManager) {
            PipeTsFileMemoryBlock block = (PipeTsFileMemoryBlock)this.forceAllocate(tsFileSizeInBytes, PipeMemoryBlockType.TS_FILE);
            this.usedMemorySizeInBytesOfTsFiles += block.getMemoryUsageInBytes();
            return block;
        }
    }

    private PipeMemoryBlock forceAllocate(long sizeInBytes, PipeMemoryBlockType type) throws PipeRuntimeOutOfMemoryCriticalException {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED) {
            switch (type) {
                case TABLET: {
                    return new PipeTabletMemoryBlock(sizeInBytes);
                }
                case TS_FILE: {
                    return new PipeTsFileMemoryBlock(sizeInBytes);
                }
            }
            return new PipeMemoryBlock(sizeInBytes);
        }
        for (int i = 1; i <= MEMORY_ALLOCATE_MAX_RETRIES; ++i) {
            if (TOTAL_MEMORY_SIZE_IN_BYTES - this.usedMemorySizeInBytes >= sizeInBytes) {
                return this.registerMemoryBlock(sizeInBytes, type);
            }
            try {
                this.tryShrink4Allocate(sizeInBytes);
                this.wait(MEMORY_ALLOCATE_RETRY_INTERVAL_IN_MS);
                continue;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LOGGER.warn("forceAllocate: interrupted while waiting for available memory", (Throwable)e);
            }
        }
        throw new PipeRuntimeOutOfMemoryCriticalException(String.format("forceAllocate: failed to allocate memory after %d retries, total memory size %d bytes, used memory size %d bytes, requested memory size %d bytes", MEMORY_ALLOCATE_MAX_RETRIES, TOTAL_MEMORY_SIZE_IN_BYTES, this.usedMemorySizeInBytes, sizeInBytes));
    }

    public synchronized void forceResize(PipeMemoryBlock block, long targetSize) {
        if (block == null || block.isReleased()) {
            LOGGER.warn("forceResize: cannot resize a null or released memory block");
            return;
        }
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED) {
            block.setMemoryUsageInBytes(targetSize);
            return;
        }
        long oldSize = block.getMemoryUsageInBytes();
        if (oldSize >= targetSize) {
            this.usedMemorySizeInBytes -= oldSize - targetSize;
            if (block instanceof PipeTabletMemoryBlock) {
                this.usedMemorySizeInBytesOfTablets -= oldSize - targetSize;
            }
            if (block instanceof PipeTsFileMemoryBlock) {
                this.usedMemorySizeInBytesOfTsFiles -= oldSize - targetSize;
            }
            block.setMemoryUsageInBytes(targetSize);
            return;
        }
        long sizeInBytes = targetSize - oldSize;
        for (int i = 1; i <= MEMORY_ALLOCATE_MAX_RETRIES; ++i) {
            if (TOTAL_MEMORY_SIZE_IN_BYTES - this.usedMemorySizeInBytes >= sizeInBytes) {
                this.usedMemorySizeInBytes += sizeInBytes;
                if (block instanceof PipeTabletMemoryBlock) {
                    this.usedMemorySizeInBytesOfTablets += sizeInBytes;
                }
                if (block instanceof PipeTsFileMemoryBlock) {
                    this.usedMemorySizeInBytesOfTsFiles += sizeInBytes;
                }
                block.setMemoryUsageInBytes(targetSize);
                return;
            }
            try {
                this.tryShrink4Allocate(sizeInBytes);
                this.wait(MEMORY_ALLOCATE_RETRY_INTERVAL_IN_MS);
                continue;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LOGGER.warn("forceResize: interrupted while waiting for available memory", (Throwable)e);
            }
        }
        throw new PipeRuntimeOutOfMemoryCriticalException(String.format("forceResize: failed to allocate memory after %d retries, total memory size %d bytes, used memory size %d bytes, requested memory size %d bytes", MEMORY_ALLOCATE_MAX_RETRIES, TOTAL_MEMORY_SIZE_IN_BYTES, this.usedMemorySizeInBytes, sizeInBytes));
    }

    public synchronized PipeMemoryBlock forceAllocateIfSufficient(long sizeInBytes, float usedThreshold) {
        if (usedThreshold < 0.0f || usedThreshold > 1.0f) {
            return null;
        }
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED) {
            return new PipeMemoryBlock(sizeInBytes);
        }
        if (TOTAL_MEMORY_SIZE_IN_BYTES - this.usedMemorySizeInBytes >= sizeInBytes && (float)this.usedMemorySizeInBytes / (float)TOTAL_MEMORY_SIZE_IN_BYTES < usedThreshold) {
            return this.forceAllocate(sizeInBytes);
        }
        long memoryToShrink = Math.max(this.usedMemorySizeInBytes - (long)((float)TOTAL_MEMORY_SIZE_IN_BYTES * usedThreshold), sizeInBytes);
        if (this.tryShrink4Allocate(memoryToShrink)) {
            return this.forceAllocate(sizeInBytes);
        }
        return null;
    }

    public synchronized PipeMemoryBlock tryAllocate(long sizeInBytes) {
        return this.tryAllocate(sizeInBytes, currentSize -> currentSize * 2L / 3L);
    }

    public synchronized PipeMemoryBlock tryAllocate(long sizeInBytes, LongUnaryOperator customAllocateStrategy) {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED) {
            return new PipeMemoryBlock(sizeInBytes);
        }
        if (TOTAL_MEMORY_SIZE_IN_BYTES - this.usedMemorySizeInBytes >= sizeInBytes) {
            return this.registerMemoryBlock(sizeInBytes);
        }
        long sizeToAllocateInBytes = sizeInBytes;
        while (sizeToAllocateInBytes > MEMORY_ALLOCATE_MIN_SIZE_IN_BYTES) {
            if (TOTAL_MEMORY_SIZE_IN_BYTES - this.usedMemorySizeInBytes >= sizeToAllocateInBytes) {
                LOGGER.info("tryAllocate: allocated memory, total memory size {} bytes, used memory size {} bytes, original requested memory size {} bytes, actual requested memory size {} bytes", new Object[]{TOTAL_MEMORY_SIZE_IN_BYTES, this.usedMemorySizeInBytes, sizeInBytes, sizeToAllocateInBytes});
                return this.registerMemoryBlock(sizeToAllocateInBytes);
            }
            sizeToAllocateInBytes = Math.max(customAllocateStrategy.applyAsLong(sizeToAllocateInBytes), MEMORY_ALLOCATE_MIN_SIZE_IN_BYTES);
        }
        if (this.tryShrink4Allocate(sizeToAllocateInBytes)) {
            LOGGER.info("tryAllocate: allocated memory, total memory size {} bytes, used memory size {} bytes, original requested memory size {} bytes, actual requested memory size {} bytes", new Object[]{TOTAL_MEMORY_SIZE_IN_BYTES, this.usedMemorySizeInBytes, sizeInBytes, sizeToAllocateInBytes});
            return this.registerMemoryBlock(sizeToAllocateInBytes);
        }
        LOGGER.warn("tryAllocate: failed to allocate memory, total memory size {} bytes, used memory size {} bytes, requested memory size {} bytes", new Object[]{TOTAL_MEMORY_SIZE_IN_BYTES, this.usedMemorySizeInBytes, sizeInBytes});
        return this.registerMemoryBlock(0L);
    }

    public synchronized boolean tryAllocate(PipeMemoryBlock block, long memoryInBytesNeededToBeAllocated) {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED || block == null || block.isReleased()) {
            return false;
        }
        if (TOTAL_MEMORY_SIZE_IN_BYTES - this.usedMemorySizeInBytes >= memoryInBytesNeededToBeAllocated) {
            this.usedMemorySizeInBytes += memoryInBytesNeededToBeAllocated;
            if (block instanceof PipeTabletMemoryBlock) {
                this.usedMemorySizeInBytesOfTablets += memoryInBytesNeededToBeAllocated;
            }
            if (block instanceof PipeTsFileMemoryBlock) {
                this.usedMemorySizeInBytesOfTsFiles += memoryInBytesNeededToBeAllocated;
            }
            block.setMemoryUsageInBytes(block.getMemoryUsageInBytes() + memoryInBytesNeededToBeAllocated);
            return true;
        }
        return false;
    }

    private PipeMemoryBlock registerMemoryBlock(long sizeInBytes) {
        return this.registerMemoryBlock(sizeInBytes, PipeMemoryBlockType.NORMAL);
    }

    private PipeMemoryBlock registerMemoryBlock(long sizeInBytes, PipeMemoryBlockType type) {
        PipeMemoryBlock returnedMemoryBlock;
        this.usedMemorySizeInBytes += sizeInBytes;
        switch (type) {
            case TABLET: {
                returnedMemoryBlock = new PipeTabletMemoryBlock(sizeInBytes);
                break;
            }
            case TS_FILE: {
                returnedMemoryBlock = new PipeTsFileMemoryBlock(sizeInBytes);
                break;
            }
            default: {
                returnedMemoryBlock = new PipeMemoryBlock(sizeInBytes);
            }
        }
        this.allocatedBlocks.add(returnedMemoryBlock);
        return returnedMemoryBlock;
    }

    private boolean tryShrink4Allocate(long sizeInBytes) {
        boolean hasAtLeastOneBlockShrinkable;
        ArrayList<PipeMemoryBlock> shuffledBlocks = new ArrayList<PipeMemoryBlock>(this.allocatedBlocks);
        Collections.shuffle(shuffledBlocks);
        do {
            hasAtLeastOneBlockShrinkable = false;
            for (PipeMemoryBlock block : shuffledBlocks) {
                if (!block.shrink()) continue;
                hasAtLeastOneBlockShrinkable = true;
                if (TOTAL_MEMORY_SIZE_IN_BYTES - this.usedMemorySizeInBytes < sizeInBytes) continue;
                return true;
            }
        } while (hasAtLeastOneBlockShrinkable);
        return false;
    }

    public synchronized void tryExpandAllAndCheckConsistency() {
        this.allocatedBlocks.forEach(PipeMemoryBlock::expand);
        long blockSum = this.allocatedBlocks.stream().mapToLong(PipeMemoryBlock::getMemoryUsageInBytes).sum();
        if (blockSum != this.usedMemorySizeInBytes) {
            LOGGER.warn("tryExpandAllAndCheckConsistency: memory usage is not consistent with allocated blocks, usedMemorySizeInBytes is {} but sum of all blocks is {}", (Object)this.usedMemorySizeInBytes, (Object)blockSum);
        }
        long tabletBlockSum = this.allocatedBlocks.stream().filter(PipeTabletMemoryBlock.class::isInstance).mapToLong(PipeMemoryBlock::getMemoryUsageInBytes).sum();
        if (tabletBlockSum != this.usedMemorySizeInBytesOfTablets) {
            LOGGER.warn("tryExpandAllAndCheckConsistency: memory usage of tablets is not consistent with allocated blocks, usedMemorySizeInBytesOfTablets is {} but sum of all tablet blocks is {}", (Object)this.usedMemorySizeInBytesOfTablets, (Object)tabletBlockSum);
        }
        long tsFileBlockSum = this.allocatedBlocks.stream().filter(PipeTsFileMemoryBlock.class::isInstance).mapToLong(PipeMemoryBlock::getMemoryUsageInBytes).sum();
        if (tsFileBlockSum != this.usedMemorySizeInBytesOfTsFiles) {
            LOGGER.warn("tryExpandAllAndCheckConsistency: memory usage of tsfiles is not consistent with allocated blocks, usedMemorySizeInBytesOfTsFiles is {} but sum of all tsfile blocks is {}", (Object)this.usedMemorySizeInBytesOfTsFiles, (Object)tsFileBlockSum);
        }
    }

    public synchronized void release(PipeMemoryBlock block) {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED || block == null || block.isReleased()) {
            return;
        }
        this.allocatedBlocks.remove(block);
        this.usedMemorySizeInBytes -= block.getMemoryUsageInBytes();
        if (block instanceof PipeTabletMemoryBlock) {
            this.usedMemorySizeInBytesOfTablets -= block.getMemoryUsageInBytes();
        }
        if (block instanceof PipeTsFileMemoryBlock) {
            this.usedMemorySizeInBytesOfTsFiles -= block.getMemoryUsageInBytes();
        }
        block.markAsReleased();
        this.notifyAll();
    }

    public synchronized boolean release(PipeMemoryBlock block, long sizeInBytes) {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED || block == null || block.isReleased()) {
            return false;
        }
        this.usedMemorySizeInBytes -= sizeInBytes;
        if (block instanceof PipeTabletMemoryBlock) {
            this.usedMemorySizeInBytesOfTablets -= sizeInBytes;
        }
        if (block instanceof PipeTsFileMemoryBlock) {
            this.usedMemorySizeInBytesOfTsFiles -= sizeInBytes;
        }
        block.setMemoryUsageInBytes(block.getMemoryUsageInBytes() - sizeInBytes);
        this.notifyAll();
        return true;
    }

    public long getUsedMemorySizeInBytes() {
        return this.usedMemorySizeInBytes;
    }

    public long getFreeMemorySizeInBytes() {
        return TOTAL_MEMORY_SIZE_IN_BYTES - this.usedMemorySizeInBytes;
    }

    public long getTotalMemorySizeInBytes() {
        return TOTAL_MEMORY_SIZE_IN_BYTES;
    }
}

