/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.commons.utils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.text.StringCharacterIterator;
import java.util.Arrays;
import java.util.Objects;
import org.apache.iotdb.commons.file.SystemFileFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileUtils.class);
    private static final int BUFFER_SIZE = 1024;

    private FileUtils() {
    }

    public static boolean deleteFileIfExist(File file) {
        try {
            Files.deleteIfExists(file.toPath());
            return true;
        }
        catch (IOException e) {
            LOGGER.error(e.getMessage(), e);
            return false;
        }
    }

    public static void deleteFileOrDirectory(File file) {
        if (file.isDirectory()) {
            for (File subfile : file.listFiles()) {
                FileUtils.deleteFileOrDirectory(subfile);
            }
        }
        try {
            Files.delete(file.toPath());
        }
        catch (DirectoryNotEmptyException | NoSuchFileException e) {
            LOGGER.warn("{}: {}", e.getMessage(), Arrays.toString(file.list()), e);
        }
        catch (Exception e) {
            LOGGER.warn("{}: {}", e.getMessage(), file.getName(), e);
        }
    }

    public static void deleteDirectoryAndEmptyParent(File folder) {
        FileUtils.deleteFileOrDirectory(folder);
        File parentFolder = folder.getParentFile();
        if (parentFolder.isDirectory() && Objects.requireNonNull(parentFolder.listFiles()).length == 0 && !parentFolder.delete()) {
            LOGGER.warn("Delete folder failed: {}", (Object)parentFolder.getAbsolutePath());
        }
    }

    public static boolean copyDir(File sourceDir, File targetDir) throws IOException {
        File[] files;
        if (!sourceDir.exists() || !sourceDir.isDirectory()) {
            LOGGER.error("Failed to copy folder, because source folder [{}] doesn't exist.", (Object)sourceDir.getAbsolutePath());
            return false;
        }
        if (!targetDir.exists()) {
            if (!targetDir.mkdirs()) {
                LOGGER.error("Failed to copy folder, because failed to create target folder[{}].", (Object)targetDir.getAbsolutePath());
                return false;
            }
        } else if (!targetDir.isDirectory()) {
            LOGGER.error("Failed to copy folder, because target folder [{}] already exist.", (Object)targetDir.getAbsolutePath());
            return false;
        }
        if ((files = sourceDir.listFiles()) == null || files.length == 0) {
            return true;
        }
        boolean result = true;
        for (File file : files) {
            if (!file.exists()) continue;
            File targetFile = new File(targetDir, file.getName());
            if (file.isDirectory()) {
                result &= FileUtils.copyDir(file.getAbsoluteFile(), targetFile);
                continue;
            }
            try (BufferedInputStream in = new BufferedInputStream(new FileInputStream(file));
                 FileOutputStream fileOutputStream = new FileOutputStream(targetFile);
                 BufferedOutputStream out = new BufferedOutputStream(fileOutputStream);){
                int size;
                byte[] bytes = new byte[1024];
                while ((size = in.read(bytes)) > 0) {
                    out.write(bytes, 0, size);
                }
                out.flush();
                fileOutputStream.getFD().sync();
            }
            catch (IOException e) {
                LOGGER.warn("get ioexception on file {}", (Object)file.getAbsolutePath(), (Object)e);
                throw e;
            }
        }
        return result;
    }

    public static long getDirSize(String path) {
        long sum = 0L;
        File file = SystemFileFactory.INSTANCE.getFile(path);
        if (file.isDirectory()) {
            String[] list;
            for (String item : list = file.list()) {
                String subPath = path + File.separator + item;
                sum += FileUtils.getDirSize(subPath);
            }
        } else {
            sum += file.length();
        }
        return sum;
    }

    public static void recursivelyDeleteFolder(String path) throws IOException {
        File file = new File(path);
        if (file.isDirectory()) {
            File[] files = file.listFiles();
            if (files == null || files.length == 0) {
                org.apache.commons.io.FileUtils.deleteDirectory(file);
            } else {
                for (File f : files) {
                    FileUtils.recursivelyDeleteFolder(f.getAbsolutePath());
                }
                org.apache.commons.io.FileUtils.deleteDirectory(file);
            }
        } else {
            org.apache.commons.io.FileUtils.delete(file);
        }
    }

    public static String addPrefix2FilePath(String prefix, String file) {
        if (!new File(file).isAbsolute() && prefix != null && prefix.length() > 0) {
            file = !prefix.endsWith(File.separator) ? prefix + File.separatorChar + file : prefix + file;
        }
        return file;
    }

    public static boolean moveFileSafe(File source, File target) {
        if (target.exists()) {
            LOGGER.info("won't move file again because target file already exists: {}", (Object)target.getAbsolutePath());
            LOGGER.info("you may manually delete source file if necessary: {}", (Object)source.getAbsolutePath());
            return true;
        }
        String fromTo = String.format("from %s to %s", source.getAbsolutePath(), target.getAbsolutePath());
        LOGGER.info("start to move file, {}", (Object)fromTo);
        File unfinishedTarget = new File(target.getAbsolutePath() + ".unfinished");
        try {
            if (unfinishedTarget.exists()) {
                if (unfinishedTarget.isFile()) {
                    org.apache.commons.io.FileUtils.delete(unfinishedTarget);
                } else {
                    FileUtils.recursivelyDeleteFolder(unfinishedTarget.getAbsolutePath());
                }
            }
        }
        catch (IOException e) {
            LOGGER.error("delete unfinished target file failed: {}", (Object)unfinishedTarget.getAbsolutePath(), (Object)e);
            return false;
        }
        LOGGER.info("unfinished target file which was created last time has been deleted: {}", (Object)unfinishedTarget.getAbsolutePath());
        try {
            if (source.isDirectory()) {
                if (!FileUtils.copyDir(source, unfinishedTarget)) {
                    LOGGER.error("file copy fail");
                    return false;
                }
            } else {
                org.apache.commons.io.FileUtils.copyFile(source, unfinishedTarget);
            }
        }
        catch (IOException e) {
            LOGGER.error("file copy fail", e);
            return false;
        }
        if (!unfinishedTarget.renameTo(target)) {
            LOGGER.error("file rename fail");
            return false;
        }
        try {
            if (source.isDirectory()) {
                FileUtils.recursivelyDeleteFolder(source.getAbsolutePath());
            } else {
                org.apache.commons.io.FileUtils.delete(source);
            }
        }
        catch (IOException e) {
            LOGGER.error("delete source file fail: {}", (Object)source.getAbsolutePath(), (Object)e);
        }
        LOGGER.info("move file success, {}", (Object)fromTo);
        return true;
    }

    public static File createHardLink(File sourceFile, File hardlink) throws IOException {
        if (!hardlink.getParentFile().exists() && !hardlink.getParentFile().mkdirs()) {
            throw new IOException(String.format("failed to create hardlink %s for file %s: failed to create parent dir %s", hardlink.getPath(), sourceFile.getPath(), hardlink.getParentFile().getPath()));
        }
        Path sourcePath = FileSystems.getDefault().getPath(sourceFile.getAbsolutePath(), new String[0]);
        Path linkPath = FileSystems.getDefault().getPath(hardlink.getAbsolutePath(), new String[0]);
        Files.createLink(linkPath, sourcePath);
        return hardlink;
    }

    public static File copyFile(File sourceFile, File targetFile) throws IOException {
        if (!targetFile.getParentFile().exists() && !targetFile.getParentFile().mkdirs()) {
            throw new IOException(String.format("failed to copy file %s to %s: failed to create parent dir %s", sourceFile.getPath(), targetFile.getPath(), targetFile.getParentFile().getPath()));
        }
        Files.copy(sourceFile.toPath(), targetFile.toPath(), new CopyOption[0]);
        return targetFile;
    }

    public static String humanReadableByteCountSI(long bytes) {
        if (-1000L < bytes && bytes < 1000L) {
            return bytes + " B";
        }
        StringCharacterIterator ci = new StringCharacterIterator("KMGTPE");
        while (bytes <= -999950L || bytes >= 999950L) {
            bytes /= 1000L;
            ci.next();
        }
        return String.format("%.2f %cB", (double)bytes / 1000.0, Character.valueOf(ci.current()));
    }
}

