/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.commons.udf.service;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.iotdb.commons.udf.UDFInformation;
import org.apache.iotdb.commons.udf.UDFTable;
import org.apache.iotdb.commons.udf.builtin.BuiltinAggregationFunction;
import org.apache.iotdb.commons.udf.service.UDFClassLoader;
import org.apache.iotdb.commons.udf.service.UDFClassLoaderManager;
import org.apache.iotdb.commons.udf.service.UDFExecutableManager;
import org.apache.iotdb.udf.api.UDAF;
import org.apache.iotdb.udf.api.UDF;
import org.apache.iotdb.udf.api.UDTF;
import org.apache.iotdb.udf.api.exception.UDFManagementException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UDFManagementService {
    private static final Logger LOGGER = LoggerFactory.getLogger(UDFManagementService.class);
    private final ReentrantLock lock = new ReentrantLock();
    private final UDFTable udfTable = new UDFTable();

    private UDFManagementService() {
    }

    public void acquireLock() {
        this.lock.lock();
    }

    public void releaseLock() {
        this.lock.unlock();
    }

    public void validate(UDFInformation udfInformation) {
        try {
            this.acquireLock();
            this.checkIfRegistered(udfInformation);
        }
        finally {
            this.releaseLock();
        }
    }

    public void register(UDFInformation udfInformation, ByteBuffer jarFile) throws Exception {
        try {
            this.acquireLock();
            this.checkIfRegistered(udfInformation);
            this.saveJarFile(udfInformation.getJarName(), jarFile);
            this.doRegister(udfInformation);
        }
        finally {
            this.releaseLock();
        }
    }

    public void register(UDFInformation udfInformation) throws Exception {
        try {
            this.acquireLock();
            this.checkIfRegistered(udfInformation);
            this.doRegister(udfInformation);
        }
        finally {
            this.releaseLock();
        }
    }

    private void checkIsBuiltInAggregationFunctionName(UDFInformation udfInformation) throws UDFManagementException {
        String functionName = udfInformation.getFunctionName();
        String className = udfInformation.getClassName();
        if (!BuiltinAggregationFunction.getNativeFunctionNames().contains(functionName.toLowerCase())) {
            return;
        }
        String errorMessage = String.format("Failed to register UDF %s(%s), because the given function name conflicts with the built-in function name", functionName, className);
        LOGGER.warn(errorMessage);
        throw new UDFManagementException(errorMessage);
    }

    private void checkIfRegistered(UDFInformation udfInformation) throws UDFManagementException {
        this.checkIsBuiltInAggregationFunctionName(udfInformation);
        String functionName = udfInformation.getFunctionName();
        String className = udfInformation.getClassName();
        UDFInformation information = this.udfTable.getUDFInformation(functionName);
        if (information == null) {
            return;
        }
        if (information.isBuiltin()) {
            String errorMessage = String.format("Failed to register UDF %s(%s), because the given function name is the same as a built-in UDF function name.", functionName, className);
            LOGGER.warn(errorMessage);
            throw new UDFManagementException(errorMessage);
        }
        if (UDFExecutableManager.getInstance().hasFileUnderInstallDir(udfInformation.getJarName()) && this.isLocalJarConflicted(udfInformation)) {
            String errorMessage = String.format("Failed to register function %s, because existed md5 of jar file for function %s is different from the new jar file. ", functionName, functionName);
            LOGGER.warn(errorMessage);
            throw new UDFManagementException(errorMessage);
        }
    }

    public boolean isLocalJarConflicted(UDFInformation udfInformation) throws UDFManagementException {
        String functionName = udfInformation.getFunctionName();
        String existedMd5 = "";
        String md5FilePath = functionName + ".txt";
        boolean hasComputed = false;
        if (UDFExecutableManager.getInstance().hasFileUnderTemporaryRoot(md5FilePath)) {
            try {
                existedMd5 = UDFExecutableManager.getInstance().readTextFromFileUnderTemporaryRoot(md5FilePath);
                hasComputed = true;
            }
            catch (IOException e) {
                LOGGER.warn("Error occurred when trying to read md5 of {}", (Object)md5FilePath);
            }
        }
        if (!hasComputed) {
            try {
                existedMd5 = DigestUtils.md5Hex(Files.newInputStream(Paths.get(UDFExecutableManager.getInstance().getInstallDir() + File.separator + udfInformation.getJarName(), new String[0]), new OpenOption[0]));
                UDFExecutableManager.getInstance().saveTextAsFileUnderTemporaryRoot(existedMd5, md5FilePath);
            }
            catch (IOException e) {
                String errorMessage = String.format("Failed to registered function %s, because error occurred when trying to compute md5 of jar file for function %s ", functionName, functionName);
                LOGGER.warn(errorMessage, e);
                throw new UDFManagementException(errorMessage);
            }
        }
        return !existedMd5.equals(udfInformation.getJarMD5());
    }

    private void saveJarFile(String jarName, ByteBuffer byteBuffer) throws IOException {
        if (byteBuffer != null) {
            UDFExecutableManager.getInstance().saveToInstallDir(byteBuffer, jarName);
        }
    }

    public void doRegister(UDFInformation udfInformation) throws UDFManagementException {
        String functionName = udfInformation.getFunctionName();
        String className = udfInformation.getClassName();
        try {
            UDFClassLoader currentActiveClassLoader = UDFClassLoaderManager.getInstance().updateAndGetActiveClassLoader();
            this.updateAllRegisteredClasses(currentActiveClassLoader);
            Class<?> functionClass = Class.forName(className, true, currentActiveClassLoader);
            UDF udf = (UDF)functionClass.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
            this.udfTable.addUDFInformation(functionName, udfInformation);
            this.udfTable.addFunctionAndClass(functionName, functionClass);
        }
        catch (IOException | ClassCastException | ClassNotFoundException | IllegalAccessException | InstantiationException | NoSuchMethodException | InvocationTargetException e) {
            String errorMessage = String.format("Failed to register UDF %s(%s), because its instance can not be constructed successfully. Exception: %s", functionName.toUpperCase(), className, e);
            LOGGER.warn(errorMessage, e);
            throw new UDFManagementException(errorMessage);
        }
    }

    private void updateAllRegisteredClasses(UDFClassLoader activeClassLoader) throws ClassNotFoundException {
        for (UDFInformation information : this.getAllUDFInformation()) {
            if (information.isBuiltin()) continue;
            this.udfTable.updateFunctionClass(information, activeClassLoader);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deregister(String functionName, boolean needToDeleteJar) throws Exception {
        try {
            this.acquireLock();
            UDFInformation information = this.udfTable.getUDFInformation(functionName);
            if (information == null) {
                return;
            }
            if (information.isBuiltin()) {
                String errorMessage = String.format("Built-in function %s can not be deregistered.", functionName.toUpperCase());
                LOGGER.warn(errorMessage);
                throw new UDFManagementException(errorMessage);
            }
            this.udfTable.removeUDFInformation(functionName);
            this.udfTable.removeFunctionClass(functionName);
            if (needToDeleteJar) {
                UDFExecutableManager.getInstance().removeFileUnderLibRoot(information.getJarName());
                UDFExecutableManager.getInstance().removeFileUnderTemporaryRoot(functionName.toUpperCase() + ".txt");
            }
        }
        finally {
            this.releaseLock();
        }
    }

    public UDF reflect(String functionName) {
        UDFInformation information = this.udfTable.getUDFInformation(functionName);
        if (information == null) {
            String errorMessage = String.format("Failed to reflect UDF instance, because UDF %s has not been registered.", functionName.toUpperCase());
            LOGGER.warn(errorMessage);
            throw new RuntimeException(errorMessage);
        }
        try {
            return (UDF)this.udfTable.getFunctionClass(functionName).getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
        }
        catch (IllegalAccessException | InstantiationException | NoSuchMethodException | InvocationTargetException e) {
            String errorMessage = String.format("Failed to reflect UDF %s(%s) instance, because %s", functionName, information.getClassName(), e);
            LOGGER.warn(errorMessage, e);
            throw new RuntimeException(errorMessage);
        }
    }

    public UDFInformation[] getAllUDFInformation() {
        return this.udfTable.getAllUDFInformation();
    }

    public List<UDFInformation> getAllBuiltInTimeSeriesGeneratingInformation() {
        return Arrays.stream(this.getAllUDFInformation()).filter(UDFInformation::isBuiltin).collect(Collectors.toList());
    }

    public boolean isUDTF(String functionName) {
        Class<?> udfClass = this.udfTable.getFunctionClass(functionName);
        UDFInformation information = this.udfTable.getUDFInformation(functionName);
        if (udfClass != null) {
            try {
                return udfClass.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]) instanceof UDTF;
            }
            catch (IllegalAccessException | InstantiationException | NoSuchMethodException | InvocationTargetException e) {
                String errorMessage = String.format("Failed to reflect UDTF %s(%s) instance, because %s", functionName, information.getClassName(), e);
                LOGGER.warn(errorMessage, e);
                throw new RuntimeException(errorMessage);
            }
        }
        return false;
    }

    public boolean isUDAF(String functionName) {
        Class<?> udfClass = this.udfTable.getFunctionClass(functionName);
        UDFInformation information = this.udfTable.getUDFInformation(functionName);
        if (udfClass != null) {
            try {
                return udfClass.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]) instanceof UDAF;
            }
            catch (IllegalAccessException | InstantiationException | NoSuchMethodException | InvocationTargetException e) {
                String errorMessage = String.format("Failed to reflect UDAF %s(%s) instance, because %s", functionName, information.getClassName(), e);
                LOGGER.warn(errorMessage, e);
                throw new RuntimeException(errorMessage);
            }
        }
        return false;
    }

    public void deregisterAll() throws UDFManagementException {
        for (UDFInformation information : this.getAllUDFInformation()) {
            if (information.isBuiltin()) continue;
            try {
                this.deregister(information.getFunctionName(), false);
            }
            catch (Exception e) {
                throw new UDFManagementException(e.getMessage());
            }
        }
    }

    public static UDFManagementService getInstance() {
        return UDFManagementServiceHolder.INSTANCE;
    }

    private static class UDFManagementServiceHolder {
        private static final UDFManagementService INSTANCE = new UDFManagementService();

        private UDFManagementServiceHolder() {
        }
    }
}

