/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.pipe.resource.memory;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.LongUnaryOperator;
import org.apache.iotdb.commons.exception.pipe.PipeRuntimeOutOfMemoryCriticalException;
import org.apache.iotdb.commons.pipe.config.PipeConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.pipe.agent.PipeAgent;
import org.apache.iotdb.db.pipe.resource.memory.PipeMemoryBlock;
import org.apache.iotdb.db.pipe.resource.memory.PipeTabletMemoryBlock;
import org.apache.iotdb.tsfile.file.metadata.enums.TSDataType;
import org.apache.iotdb.tsfile.utils.Binary;
import org.apache.iotdb.tsfile.write.record.Tablet;
import org.apache.iotdb.tsfile.write.schema.MeasurementSchema;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PipeMemoryManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(PipeMemoryManager.class);
    private static final boolean PIPE_MEMORY_MANAGEMENT_ENABLED = PipeConfig.getInstance().getPipeMemoryManagementEnabled();
    private static final int MEMORY_ALLOCATE_MAX_RETRIES = PipeConfig.getInstance().getPipeMemoryAllocateMaxRetries();
    private static final long MEMORY_ALLOCATE_RETRY_INTERVAL_IN_MS = PipeConfig.getInstance().getPipeMemoryAllocateRetryIntervalInMs();
    private static final long TOTAL_MEMORY_SIZE_IN_BYTES = IoTDBDescriptor.getInstance().getConfig().getAllocateMemoryForPipe();
    private static final long MEMORY_ALLOCATE_MIN_SIZE_IN_BYTES = PipeConfig.getInstance().getPipeMemoryAllocateMinSizeInBytes();
    private long usedMemorySizeInBytes;
    private static final double TABLET_MEMORY_REJECT_THRESHOLD = PipeConfig.getInstance().getPipeDataStructureTabletMemoryBlockAllocationRejectThreshold();
    private volatile long usedMemorySizeInBytesOfTablets;
    private final Set<PipeMemoryBlock> allocatedBlocks = new HashSet<PipeMemoryBlock>();

    public PipeMemoryManager() {
        PipeAgent.runtime().registerPeriodicalJob("PipeMemoryManager#tryExpandAll()", this::tryExpandAll, PipeConfig.getInstance().getPipeMemoryExpanderIntervalSeconds());
    }

    public synchronized PipeMemoryBlock forceAllocate(long sizeInBytes) throws PipeRuntimeOutOfMemoryCriticalException {
        return this.forceAllocate(sizeInBytes, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public PipeTabletMemoryBlock forceAllocateWithRetry(Tablet tablet) throws PipeRuntimeOutOfMemoryCriticalException {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED) {
            return new PipeTabletMemoryBlock(0L);
        }
        for (int i = 1; i <= MEMORY_ALLOCATE_MAX_RETRIES && !((double)this.usedMemorySizeInBytesOfTablets / (double)TOTAL_MEMORY_SIZE_IN_BYTES < TABLET_MEMORY_REJECT_THRESHOLD); ++i) {
            try {
                Thread.sleep(MEMORY_ALLOCATE_RETRY_INTERVAL_IN_MS);
                continue;
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                LOGGER.warn("forceAllocateWithRetry: interrupted while waiting for available memory", (Throwable)ex);
            }
        }
        if ((double)this.usedMemorySizeInBytesOfTablets / (double)TOTAL_MEMORY_SIZE_IN_BYTES >= TABLET_MEMORY_REJECT_THRESHOLD) {
            throw new PipeRuntimeOutOfMemoryCriticalException(String.format("forceAllocateForTablet: failed to allocate because there's too much memory for tablets, total memory size %d bytes, used memory for tablet size %d bytes,", TOTAL_MEMORY_SIZE_IN_BYTES, this.usedMemorySizeInBytesOfTablets));
        }
        PipeMemoryManager pipeMemoryManager = this;
        synchronized (pipeMemoryManager) {
            PipeTabletMemoryBlock block = (PipeTabletMemoryBlock)this.forceAllocate(PipeMemoryManager.calculateTabletSizeInBytes(tablet), true);
            this.usedMemorySizeInBytesOfTablets += block.getMemoryUsageInBytes();
            return block;
        }
    }

    private PipeMemoryBlock forceAllocate(long sizeInBytes, boolean isForTablet) throws PipeRuntimeOutOfMemoryCriticalException {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED) {
            return isForTablet ? new PipeTabletMemoryBlock(sizeInBytes) : new PipeMemoryBlock(sizeInBytes);
        }
        for (int i = 1; i <= MEMORY_ALLOCATE_MAX_RETRIES; ++i) {
            if (TOTAL_MEMORY_SIZE_IN_BYTES - this.usedMemorySizeInBytes >= sizeInBytes) {
                return this.registerMemoryBlock(sizeInBytes, isForTablet);
            }
            try {
                this.tryShrink4Allocate(sizeInBytes);
                this.wait(MEMORY_ALLOCATE_RETRY_INTERVAL_IN_MS);
                continue;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LOGGER.warn("forceAllocate: interrupted while waiting for available memory", (Throwable)e);
            }
        }
        throw new PipeRuntimeOutOfMemoryCriticalException(String.format("forceAllocate: failed to allocate memory after %d retries, total memory size %d bytes, used memory size %d bytes, requested memory size %d bytes", MEMORY_ALLOCATE_MAX_RETRIES, TOTAL_MEMORY_SIZE_IN_BYTES, this.usedMemorySizeInBytes, sizeInBytes));
    }

    public synchronized PipeMemoryBlock forceAllocateIfSufficient(long sizeInBytes, float usedThreshold) {
        if (usedThreshold < 0.0f || usedThreshold > 1.0f) {
            return null;
        }
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED) {
            return new PipeMemoryBlock(sizeInBytes);
        }
        if (TOTAL_MEMORY_SIZE_IN_BYTES - this.usedMemorySizeInBytes >= sizeInBytes && (float)this.usedMemorySizeInBytes / (float)TOTAL_MEMORY_SIZE_IN_BYTES < usedThreshold) {
            return this.forceAllocate(sizeInBytes);
        }
        long memoryToShrink = Math.max(this.usedMemorySizeInBytes - (long)((float)TOTAL_MEMORY_SIZE_IN_BYTES * usedThreshold), sizeInBytes);
        if (this.tryShrink4Allocate(memoryToShrink)) {
            return this.forceAllocate(sizeInBytes);
        }
        return null;
    }

    public static long calculateTabletSizeInBytes(Tablet tablet) {
        List timeseries;
        long totalSizeInBytes = 0L;
        if (tablet == null) {
            return totalSizeInBytes;
        }
        if (tablet.timestamps != null) {
            totalSizeInBytes += (long)tablet.timestamps.length * 8L;
        }
        if ((timeseries = tablet.getSchemas()) != null) {
            for (int column = 0; column < timeseries.size(); ++column) {
                TSDataType tsDataType;
                MeasurementSchema measurementSchema = (MeasurementSchema)timeseries.get(column);
                if (measurementSchema == null || (tsDataType = measurementSchema.getType()) == null) continue;
                if (tsDataType == TSDataType.TEXT) {
                    Binary[] values;
                    if (tablet.values == null || tablet.values.length <= column || (values = (Binary[])tablet.values[column]) == null) continue;
                    for (Binary value : values) {
                        totalSizeInBytes += value == null ? 0L : (long)(value.getLength() == -1 ? 0 : value.getLength());
                    }
                    continue;
                }
                totalSizeInBytes += (long)tablet.timestamps.length * (long)tsDataType.getDataTypeSize();
            }
        }
        if (tablet.bitMaps != null) {
            for (int i = 0; i < tablet.bitMaps.length; ++i) {
                totalSizeInBytes += tablet.bitMaps[i] == null ? 0L : (long)tablet.bitMaps[i].getSize();
            }
        }
        return totalSizeInBytes += 100L;
    }

    public synchronized PipeMemoryBlock tryAllocate(long sizeInBytes) {
        return this.tryAllocate(sizeInBytes, currentSize -> currentSize * 2L / 3L);
    }

    public synchronized PipeMemoryBlock tryAllocate(long sizeInBytes, LongUnaryOperator customAllocateStrategy) {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED) {
            return new PipeMemoryBlock(sizeInBytes);
        }
        if (TOTAL_MEMORY_SIZE_IN_BYTES - this.usedMemorySizeInBytes >= sizeInBytes) {
            return this.registerMemoryBlock(sizeInBytes);
        }
        long sizeToAllocateInBytes = sizeInBytes;
        while (sizeToAllocateInBytes > MEMORY_ALLOCATE_MIN_SIZE_IN_BYTES) {
            if (TOTAL_MEMORY_SIZE_IN_BYTES - this.usedMemorySizeInBytes >= sizeToAllocateInBytes) {
                LOGGER.info("tryAllocate: allocated memory, total memory size {} bytes, used memory size {} bytes, original requested memory size {} bytes,actual requested memory size {} bytes", new Object[]{TOTAL_MEMORY_SIZE_IN_BYTES, this.usedMemorySizeInBytes, sizeInBytes, sizeToAllocateInBytes});
                return this.registerMemoryBlock(sizeToAllocateInBytes);
            }
            sizeToAllocateInBytes = Math.max(customAllocateStrategy.applyAsLong(sizeToAllocateInBytes), MEMORY_ALLOCATE_MIN_SIZE_IN_BYTES);
        }
        if (this.tryShrink4Allocate(sizeToAllocateInBytes)) {
            LOGGER.info("tryAllocate: allocated memory, total memory size {} bytes, used memory size {} bytes, original requested memory size {} bytes,actual requested memory size {} bytes", new Object[]{TOTAL_MEMORY_SIZE_IN_BYTES, this.usedMemorySizeInBytes, sizeInBytes, sizeToAllocateInBytes});
            return this.registerMemoryBlock(sizeToAllocateInBytes);
        }
        LOGGER.warn("tryAllocate: failed to allocate memory, total memory size {} bytes, used memory size {} bytes, requested memory size {} bytes", new Object[]{TOTAL_MEMORY_SIZE_IN_BYTES, this.usedMemorySizeInBytes, sizeInBytes});
        return this.registerMemoryBlock(0L);
    }

    public synchronized boolean tryAllocate(PipeMemoryBlock block, long memoryInBytesNeededToBeAllocated) {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED || block == null || block.isReleased()) {
            return false;
        }
        if (TOTAL_MEMORY_SIZE_IN_BYTES - this.usedMemorySizeInBytes >= memoryInBytesNeededToBeAllocated) {
            this.usedMemorySizeInBytes += memoryInBytesNeededToBeAllocated;
            if (block instanceof PipeTabletMemoryBlock) {
                this.usedMemorySizeInBytesOfTablets += memoryInBytesNeededToBeAllocated;
            }
            block.setMemoryUsageInBytes(block.getMemoryUsageInBytes() + memoryInBytesNeededToBeAllocated);
            return true;
        }
        return false;
    }

    private PipeMemoryBlock registerMemoryBlock(long sizeInBytes) {
        return this.registerMemoryBlock(sizeInBytes, false);
    }

    private PipeMemoryBlock registerMemoryBlock(long sizeInBytes, boolean isForTablet) {
        this.usedMemorySizeInBytes += sizeInBytes;
        PipeMemoryBlock returnedMemoryBlock = isForTablet ? new PipeTabletMemoryBlock(sizeInBytes) : new PipeMemoryBlock(sizeInBytes);
        this.allocatedBlocks.add(returnedMemoryBlock);
        return returnedMemoryBlock;
    }

    private boolean tryShrink4Allocate(long sizeInBytes) {
        boolean hasAtLeastOneBlockShrinkable;
        ArrayList<PipeMemoryBlock> shuffledBlocks = new ArrayList<PipeMemoryBlock>(this.allocatedBlocks);
        Collections.shuffle(shuffledBlocks);
        do {
            hasAtLeastOneBlockShrinkable = false;
            for (PipeMemoryBlock block : shuffledBlocks) {
                if (!block.shrink()) continue;
                hasAtLeastOneBlockShrinkable = true;
                if (TOTAL_MEMORY_SIZE_IN_BYTES - this.usedMemorySizeInBytes < sizeInBytes) continue;
                return true;
            }
        } while (hasAtLeastOneBlockShrinkable);
        return false;
    }

    public synchronized void tryExpandAll() {
        this.allocatedBlocks.forEach(PipeMemoryBlock::expand);
    }

    public synchronized void release(PipeMemoryBlock block) {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED || block == null || block.isReleased()) {
            return;
        }
        this.allocatedBlocks.remove(block);
        this.usedMemorySizeInBytes -= block.getMemoryUsageInBytes();
        if (block instanceof PipeTabletMemoryBlock) {
            this.usedMemorySizeInBytesOfTablets -= block.getMemoryUsageInBytes();
        }
        block.markAsReleased();
        this.notifyAll();
    }

    public synchronized boolean release(PipeMemoryBlock block, long sizeInBytes) {
        if (!PIPE_MEMORY_MANAGEMENT_ENABLED || block == null || block.isReleased()) {
            return false;
        }
        this.usedMemorySizeInBytes -= sizeInBytes;
        if (block instanceof PipeTabletMemoryBlock) {
            this.usedMemorySizeInBytesOfTablets -= sizeInBytes;
        }
        block.setMemoryUsageInBytes(block.getMemoryUsageInBytes() - sizeInBytes);
        this.notifyAll();
        return true;
    }

    public long getUsedMemorySizeInBytes() {
        return this.usedMemorySizeInBytes;
    }

    public long getTotalMemorySizeInBytes() {
        return TOTAL_MEMORY_SIZE_IN_BYTES;
    }
}

