/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.iotdb.confignode.manager.load.cache;

import org.apache.iotdb.commons.cluster.NodeStatus;
import org.apache.iotdb.confignode.manager.load.cache.node.ConfigNodeHeartbeatCache;
import org.apache.iotdb.confignode.manager.load.cache.node.DataNodeHeartbeatCache;
import org.apache.iotdb.confignode.manager.load.cache.node.NodeHeartbeatSample;
import org.apache.iotdb.mpp.rpc.thrift.TDataNodeHeartbeatResp;

import org.junit.Assert;
import org.junit.Test;

public class NodeCacheTest {

  @Test
  public void forceUpdateTest() {
    DataNodeHeartbeatCache dataNodeHeartbeatCache = new DataNodeHeartbeatCache(1);

    // Test default
    Assert.assertEquals(NodeStatus.Unknown, dataNodeHeartbeatCache.getNodeStatus());
    Assert.assertEquals(Long.MAX_VALUE, dataNodeHeartbeatCache.getLoadScore());

    // Test force update to RunningStatus
    long currentTime = System.currentTimeMillis() - 2000;
    dataNodeHeartbeatCache.forceUpdate(
        new NodeHeartbeatSample(
            new TDataNodeHeartbeatResp(currentTime, NodeStatus.Running.getStatus()), currentTime));
    Assert.assertEquals(NodeStatus.Running, dataNodeHeartbeatCache.getNodeStatus());
    Assert.assertEquals(0, dataNodeHeartbeatCache.getLoadScore());

    // Test force update to ReadOnlyStatus
    currentTime += 2000;
    dataNodeHeartbeatCache.forceUpdate(
        new NodeHeartbeatSample(
            new TDataNodeHeartbeatResp(currentTime, NodeStatus.ReadOnly.getStatus()), currentTime));
    Assert.assertEquals(NodeStatus.ReadOnly, dataNodeHeartbeatCache.getNodeStatus());
    Assert.assertEquals(Long.MAX_VALUE, dataNodeHeartbeatCache.getLoadScore());
  }

  @Test
  public void periodicUpdateTest() {
    // Test DataNode heartbeat cache
    DataNodeHeartbeatCache dataNodeHeartbeatCache = new DataNodeHeartbeatCache(1);
    long currentTime = System.currentTimeMillis();
    dataNodeHeartbeatCache.cacheHeartbeatSample(
        new NodeHeartbeatSample(
            new TDataNodeHeartbeatResp(currentTime, NodeStatus.Running.getStatus()), currentTime));
    Assert.assertTrue(dataNodeHeartbeatCache.periodicUpdate());
    Assert.assertEquals(NodeStatus.Running, dataNodeHeartbeatCache.getNodeStatus());
    Assert.assertEquals(0, dataNodeHeartbeatCache.getLoadScore());

    // Test ConfigNode heartbeat cache
    ConfigNodeHeartbeatCache configNodeHeartbeatCache = new ConfigNodeHeartbeatCache(2);
    currentTime = System.currentTimeMillis();
    configNodeHeartbeatCache.cacheHeartbeatSample(
        new NodeHeartbeatSample(currentTime, currentTime));
    Assert.assertTrue(configNodeHeartbeatCache.periodicUpdate());
    Assert.assertEquals(NodeStatus.Running, configNodeHeartbeatCache.getNodeStatus());
    Assert.assertEquals(0, configNodeHeartbeatCache.getLoadScore());
  }
}
