/*
 * Decompiled with CFR 0.152.
 */
package io.moquette.broker;

import io.moquette.broker.Authorizator;
import io.moquette.broker.ClientDescriptor;
import io.moquette.broker.IQueueRepository;
import io.moquette.broker.ISessionsRepository;
import io.moquette.broker.InMemoryQueue;
import io.moquette.broker.Session;
import io.moquette.broker.SessionCorruptedException;
import io.moquette.broker.SessionEventLoopGroup;
import io.moquette.broker.SessionMessageQueue;
import io.moquette.broker.subscriptions.ISubscriptionsDirectory;
import io.moquette.broker.subscriptions.Subscription;
import io.moquette.broker.subscriptions.Topic;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.handler.codec.mqtt.MqttConnectMessage;
import io.netty.handler.codec.mqtt.MqttQoS;
import io.netty.handler.codec.mqtt.MqttVersion;
import java.net.InetSocketAddress;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SessionRegistry {
    private final ScheduledFuture<?> scheduledExpiredSessions;
    private int globalExpirySeconds;
    private final SessionEventLoopGroup loopsGroup;
    static final Duration EXPIRED_SESSION_CLEANER_TASK_INTERVAL = Duration.ofSeconds(1L);
    private static final Logger LOG = LoggerFactory.getLogger(SessionRegistry.class);
    private final ConcurrentMap<String, Session> pool = new ConcurrentHashMap<String, Session>();
    private final ISubscriptionsDirectory subscriptionsDirectory;
    private final ISessionsRepository sessionsRepository;
    private final IQueueRepository queueRepository;
    private final Authorizator authorizator;
    private final DelayQueue<ISessionsRepository.SessionData> removableSessions = new DelayQueue();
    private final Clock clock;

    SessionRegistry(ISubscriptionsDirectory subscriptionsDirectory, ISessionsRepository sessionsRepository, IQueueRepository queueRepository, Authorizator authorizator, ScheduledExecutorService scheduler, SessionEventLoopGroup loopsGroup) {
        this(subscriptionsDirectory, sessionsRepository, queueRepository, authorizator, scheduler, Clock.systemDefaultZone(), Integer.MAX_VALUE, loopsGroup);
    }

    SessionRegistry(ISubscriptionsDirectory subscriptionsDirectory, ISessionsRepository sessionsRepository, IQueueRepository queueRepository, Authorizator authorizator, ScheduledExecutorService scheduler, Clock clock, int globalExpirySeconds, SessionEventLoopGroup loopsGroup) {
        this.subscriptionsDirectory = subscriptionsDirectory;
        this.sessionsRepository = sessionsRepository;
        this.queueRepository = queueRepository;
        this.authorizator = authorizator;
        this.scheduledExpiredSessions = scheduler.scheduleWithFixedDelay(this::checkExpiredSessions, EXPIRED_SESSION_CLEANER_TASK_INTERVAL.getSeconds(), EXPIRED_SESSION_CLEANER_TASK_INTERVAL.getSeconds(), TimeUnit.SECONDS);
        this.clock = clock;
        this.globalExpirySeconds = globalExpirySeconds;
        this.loopsGroup = loopsGroup;
        this.recreateSessionPool();
    }

    private void checkExpiredSessions() {
        ArrayList expiredSessions = new ArrayList();
        int drainedSessions = this.removableSessions.drainTo(expiredSessions);
        LOG.debug("Retrieved {} expired sessions or {}", (Object)drainedSessions, (Object)this.removableSessions.size());
        for (ISessionsRepository.SessionData expiredSession : expiredSessions) {
            String expiredAt = expiredSession.expireAt().map(Instant::toString).orElse("UNDEFINED");
            LOG.debug("Removing session {}, expired on {}", (Object)expiredSession.clientId(), (Object)expiredAt);
            this.remove(expiredSession.clientId());
            this.sessionsRepository.delete(expiredSession);
        }
    }

    private void trackForRemovalOnExpiration(ISessionsRepository.SessionData session) {
        if (!session.expireAt().isPresent()) {
            throw new RuntimeException("Can't track for expiration a session without expiry instant, client_id: " + session.clientId());
        }
        LOG.debug("start tracking the session {} for removal", (Object)session.clientId());
        this.removableSessions.add(session);
    }

    private void untrackFromRemovalOnExpiration(ISessionsRepository.SessionData session) {
        this.removableSessions.remove(session);
    }

    private void recreateSessionPool() {
        Set<String> queues = this.queueRepository.listQueueNames();
        for (ISessionsRepository.SessionData session : this.sessionsRepository.list()) {
            if (!this.queueRepository.containsQueue(session.clientId())) continue;
            SessionMessageQueue<EnqueuedMessage> persistentQueue = this.queueRepository.getOrCreateQueue(session.clientId());
            queues.remove(session.clientId());
            Session rehydrated = new Session(session, false, persistentQueue);
            this.pool.put(session.clientId(), rehydrated);
            this.trackForRemovalOnExpiration(session);
        }
        if (!queues.isEmpty()) {
            LOG.error("Recreating sessions left {} unused queues. This is probably a bug. Session IDs: {}", (Object)queues.size(), (Object)Arrays.toString(queues.toArray()));
        }
    }

    SessionCreationResult createOrReopenSession(MqttConnectMessage msg, String clientId, String username) {
        SessionCreationResult postConnectAction;
        Session oldSession = this.retrieve(clientId);
        if (oldSession == null) {
            Session newSession = this.createNewSession(msg, clientId);
            postConnectAction = new SessionCreationResult(newSession, CreationModeEnum.CREATED_CLEAN_NEW, false);
            Session previous = this.pool.put(clientId, newSession);
            if (previous != null) {
                LOG.error("Another thread added a Session for our clientId {}, this is a bug!", (Object)clientId);
            }
            LOG.trace("case 1, not existing session with CId {}", (Object)clientId);
        } else {
            postConnectAction = this.reopenExistingSession(msg, clientId, oldSession, username);
        }
        return postConnectAction;
    }

    private SessionCreationResult reopenExistingSession(MqttConnectMessage msg, String clientId, Session oldSession, String username) {
        SessionCreationResult creationResult;
        boolean newIsClean = msg.variableHeader().isCleanSession();
        if (!oldSession.disconnected()) {
            oldSession.closeImmediately();
        }
        if (newIsClean) {
            this.purgeSessionState(oldSession);
            Session newSession = this.createNewSession(msg, clientId);
            this.pool.put(clientId, newSession);
            LOG.trace("case 2, oldSession with same CId {} disconnected", (Object)clientId);
            creationResult = new SessionCreationResult(newSession, CreationModeEnum.CREATED_CLEAN_NEW, true);
        } else {
            boolean connecting = oldSession.assignState(Session.SessionStatus.DISCONNECTED, Session.SessionStatus.CONNECTING);
            if (!connecting) {
                throw new SessionCorruptedException("old session moved in connected state by other thread");
            }
            this.copySessionConfig(msg, oldSession);
            this.reactivateSubscriptions(oldSession, username);
            LOG.trace("case 3, oldSession with same CId {} disconnected", (Object)clientId);
            creationResult = new SessionCreationResult(oldSession, CreationModeEnum.REOPEN_EXISTING, true);
        }
        this.untrackFromRemovalOnExpiration(creationResult.session.getSessionData());
        return creationResult;
    }

    private void reactivateSubscriptions(Session session, String username) {
        for (Subscription existingSub : session.getSubscriptions()) {
            boolean topicReadable = this.authorizator.canRead(existingSub.getTopicFilter(), username, session.getClientID());
            if (topicReadable) continue;
            this.subscriptionsDirectory.removeSubscription(existingSub.getTopicFilter(), session.getClientID());
        }
    }

    private void unsubscribe(Session session) {
        for (Subscription existingSub : session.getSubscriptions()) {
            this.subscriptionsDirectory.removeSubscription(existingSub.getTopicFilter(), session.getClientID());
        }
    }

    private Session createNewSession(MqttConnectMessage msg, String clientId) {
        Session newSession;
        boolean clean = msg.variableHeader().isCleanSession();
        SessionMessageQueue<EnqueuedMessage> queue = !clean ? this.queueRepository.getOrCreateQueue(clientId) : new InMemoryQueue();
        int expiryInterval = clean ? 0 : this.globalExpirySeconds;
        ISessionsRepository.SessionData sessionData = new ISessionsRepository.SessionData(clientId, MqttVersion.MQTT_3_1_1, expiryInterval, this.clock);
        if (msg.variableHeader().isWillFlag()) {
            Session.Will will = this.createWill(msg);
            newSession = new Session(sessionData, clean, will, queue);
        } else {
            newSession = new Session(sessionData, clean, queue);
        }
        newSession.markConnecting();
        this.sessionsRepository.saveSession(sessionData);
        return newSession;
    }

    private void copySessionConfig(MqttConnectMessage msg, Session session) {
        boolean clean = msg.variableHeader().isCleanSession();
        Session.Will will = msg.variableHeader().isWillFlag() ? this.createWill(msg) : null;
        session.update(clean, will);
    }

    private Session.Will createWill(MqttConnectMessage msg) {
        ByteBuf willPayload = Unpooled.copiedBuffer((byte[])msg.payload().willMessageInBytes());
        String willTopic = msg.payload().willTopic();
        boolean retained = msg.variableHeader().isWillRetain();
        MqttQoS qos = MqttQoS.valueOf((int)msg.variableHeader().willQos());
        return new Session.Will(willTopic, willPayload, qos, retained);
    }

    Session retrieve(String clientID) {
        return (Session)this.pool.get(clientID);
    }

    void connectionClosed(Session session) {
        session.disconnect();
        if (session.expireImmediately()) {
            this.purgeSessionState(session);
        } else {
            this.trackForRemovalOnExpiration(session.getSessionData().withExpirationComputed());
        }
    }

    private void purgeSessionState(Session session) {
        LOG.debug("Remove session state for client {}", (Object)session.getClientID());
        boolean result = session.assignState(Session.SessionStatus.DISCONNECTED, Session.SessionStatus.DESTROYED);
        if (!result) {
            throw new SessionCorruptedException("Session has already changed state: " + session);
        }
        this.unsubscribe(session);
        this.remove(session.getClientID());
    }

    void remove(String clientID) {
        Session old = (Session)this.pool.remove(clientID);
        if (old != null) {
            this.removableSessions.remove(old.getSessionData());
            this.loopsGroup.routeCommand(clientID, "Clean up removed session", () -> {
                old.cleanUp();
                return null;
            });
        }
    }

    Collection<ClientDescriptor> listConnectedClients() {
        return this.pool.values().stream().filter(Session::connected).map(this::createClientDescriptor).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
    }

    boolean dropSession(String clientId, boolean removeSessionState) {
        LOG.debug("Disconnecting client: {}", (Object)clientId);
        if (clientId == null) {
            return false;
        }
        Session client = (Session)this.pool.get(clientId);
        if (client == null) {
            LOG.debug("Client {} not found, nothing disconnected", (Object)clientId);
            return false;
        }
        client.closeImmediately();
        if (removeSessionState) {
            this.purgeSessionState(client);
        }
        LOG.debug("Client {} successfully disconnected from broker", (Object)clientId);
        return true;
    }

    private Optional<ClientDescriptor> createClientDescriptor(Session s) {
        String clientID = s.getClientID();
        Optional<InetSocketAddress> remoteAddressOpt = s.remoteAddress();
        return remoteAddressOpt.map(r -> new ClientDescriptor(clientID, r.getHostString(), r.getPort()));
    }

    public void close() {
        if (this.scheduledExpiredSessions.cancel(false)) {
            LOG.info("Successfully cancelled expired sessions task");
        } else {
            LOG.warn("Can't cancel the execution of expired sessions task, was already cancelled? {}, was done? {}", (Object)this.scheduledExpiredSessions.isCancelled(), (Object)this.scheduledExpiredSessions.isDone());
        }
        this.updateNotCleanSessionsWithProperExpire();
        this.queueRepository.close();
    }

    private void updateNotCleanSessionsWithProperExpire() {
        this.pool.values().stream().filter(s -> !s.isClean()).map(Session::getSessionData).filter(s -> !s.expireAt().isPresent()).map(ISessionsRepository.SessionData::withExpirationComputed).forEach(this.sessionsRepository::saveSession);
    }

    public static class SessionCreationResult {
        final Session session;
        final CreationModeEnum mode;
        final boolean alreadyStored;

        public SessionCreationResult(Session session, CreationModeEnum mode, boolean alreadyStored) {
            this.session = session;
            this.mode = mode;
            this.alreadyStored = alreadyStored;
        }
    }

    public static enum CreationModeEnum {
        CREATED_CLEAN_NEW,
        REOPEN_EXISTING,
        DROP_EXISTING;

    }

    public static final class PubRelMarker
    extends EnqueuedMessage {
    }

    public static class PublishedMessage
    extends EnqueuedMessage {
        final Topic topic;
        final MqttQoS publishingQos;
        final ByteBuf payload;
        final boolean retained;

        public PublishedMessage(Topic topic, MqttQoS publishingQos, ByteBuf payload, boolean retained) {
            this.topic = topic;
            this.publishingQos = publishingQos;
            this.payload = payload;
            this.retained = false;
        }

        public Topic getTopic() {
            return this.topic;
        }

        public MqttQoS getPublishingQos() {
            return this.publishingQos;
        }

        public ByteBuf getPayload() {
            return this.payload;
        }

        @Override
        public void release() {
            this.payload.release();
        }

        @Override
        public void retain() {
            this.payload.retain();
        }
    }

    public static abstract class EnqueuedMessage {
        public void release() {
        }

        public void retain() {
        }
    }
}

