/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.tsfile.compress;

import com.github.luben.zstd.Zstd;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import net.jpountz.lz4.LZ4Compressor;
import net.jpountz.lz4.LZ4Factory;
import org.apache.iotdb.tsfile.exception.compress.CompressionTypeNotSupportedException;
import org.apache.iotdb.tsfile.exception.compress.GZIPCompressOverflowException;
import org.apache.iotdb.tsfile.file.metadata.enums.CompressionType;
import org.tukaani.xz.FilterOptions;
import org.tukaani.xz.LZMA2Options;
import org.tukaani.xz.XZInputStream;
import org.tukaani.xz.XZOutputStream;
import org.xerial.snappy.Snappy;

public interface ICompressor
extends Serializable {
    public static ICompressor getCompressor(String name) {
        return ICompressor.getCompressor(CompressionType.valueOf(name));
    }

    public static ICompressor getCompressor(CompressionType name) {
        if (name == null) {
            throw new CompressionTypeNotSupportedException("NULL");
        }
        switch (name) {
            case UNCOMPRESSED: {
                return new NoCompressor();
            }
            case SNAPPY: {
                return new SnappyCompressor();
            }
            case LZ4: {
                return new IOTDBLZ4Compressor();
            }
            case GZIP: {
                return new GZIPCompressor();
            }
            case ZSTD: {
                return new ZstdCompressor();
            }
            case LZMA2: {
                return new LZMA2Compressor();
            }
        }
        throw new CompressionTypeNotSupportedException(name.toString());
    }

    public byte[] compress(byte[] var1) throws IOException;

    public byte[] compress(byte[] var1, int var2, int var3) throws IOException;

    public int compress(byte[] var1, int var2, int var3, byte[] var4) throws IOException;

    public int compress(ByteBuffer var1, ByteBuffer var2) throws IOException;

    public int getMaxBytesForCompression(int var1);

    public CompressionType getType();

    public static class LZMA2Compressor
    implements ICompressor {
        @Override
        public byte[] compress(byte[] data) throws IOException {
            if (null == data) {
                return new byte[0];
            }
            byte[] r = LZMA2Compress.compress(data);
            return r;
        }

        @Override
        public byte[] compress(byte[] data, int offset, int length) throws IOException {
            byte[] dataBefore = new byte[length];
            System.arraycopy(data, offset, dataBefore, 0, length);
            byte[] r = LZMA2Compress.compress(dataBefore);
            return r;
        }

        @Override
        public int compress(byte[] data, int offset, int length, byte[] compressed) throws IOException {
            byte[] dataBefore = new byte[length];
            System.arraycopy(data, offset, dataBefore, 0, length);
            byte[] res = LZMA2Compress.compress(dataBefore);
            System.arraycopy(res, 0, compressed, 0, res.length);
            return res.length;
        }

        @Override
        public int compress(ByteBuffer data, ByteBuffer compressed) throws IOException {
            int length = data.remaining();
            byte[] dataBefore = new byte[length];
            data.get(dataBefore, 0, length);
            byte[] res = LZMA2Compress.compress(dataBefore);
            compressed.put(res);
            return res.length;
        }

        @Override
        public int getMaxBytesForCompression(int uncompressedDataSize) {
            return 100 + uncompressedDataSize;
        }

        @Override
        public CompressionType getType() {
            return CompressionType.LZMA2;
        }
    }

    public static class LZMA2Compress {
        public static byte[] compress(byte[] data) throws IOException {
            LZMA2Options options = new LZMA2Options();
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            XZOutputStream lzma2 = new XZOutputStream((OutputStream)out, (FilterOptions)options);
            lzma2.write(data);
            lzma2.close();
            byte[] r = out.toByteArray();
            return r;
        }

        public static byte[] uncompress(byte[] data) throws IOException {
            int n;
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            ByteArrayInputStream in = new ByteArrayInputStream(data);
            XZInputStream unlzma2 = new XZInputStream((InputStream)in);
            byte[] buffer = new byte[256];
            while ((n = unlzma2.read(buffer)) > 0) {
                out.write(buffer, 0, n);
            }
            in.close();
            byte[] r = out.toByteArray();
            return r;
        }
    }

    public static class ZstdCompressor
    implements ICompressor {
        private int compressionLevel = Zstd.maxCompressionLevel();

        @Override
        public byte[] compress(byte[] data) throws IOException {
            return Zstd.compress((byte[])data, (int)this.compressionLevel);
        }

        @Override
        public byte[] compress(byte[] data, int offset, int length) throws IOException {
            if (data == null) {
                return new byte[0];
            }
            byte[] compressedData = new byte[this.getMaxBytesForCompression(length)];
            int compressedSize = this.compress(data, offset, length, compressedData);
            byte[] result = new byte[compressedSize];
            System.arraycopy(compressedData, 0, result, 0, compressedSize);
            return result;
        }

        @Override
        public int compress(byte[] data, int offset, int length, byte[] compressed) throws IOException {
            return (int)Zstd.compressByteArray((byte[])compressed, (int)0, (int)compressed.length, (byte[])data, (int)offset, (int)length, (int)this.compressionLevel);
        }

        @Override
        public int compress(ByteBuffer data, ByteBuffer compressed) throws IOException {
            return Zstd.compress((ByteBuffer)compressed, (ByteBuffer)data, (int)this.compressionLevel);
        }

        @Override
        public int getMaxBytesForCompression(int uncompressedDataSize) {
            return (int)Zstd.compressBound((long)uncompressedDataSize);
        }

        @Override
        public CompressionType getType() {
            return CompressionType.ZSTD;
        }
    }

    public static class GZIPCompressor
    implements ICompressor {
        @Override
        public byte[] compress(byte[] data) throws IOException {
            if (null == data) {
                return new byte[0];
            }
            return GZIPCompress.compress(data);
        }

        @Override
        public byte[] compress(byte[] data, int offset, int length) throws IOException {
            byte[] dataBefore = new byte[length];
            System.arraycopy(data, offset, dataBefore, 0, length);
            return GZIPCompress.compress(dataBefore);
        }

        @Override
        public int compress(byte[] data, int offset, int length, byte[] compressed) throws IOException {
            byte[] dataBefore = new byte[length];
            System.arraycopy(data, offset, dataBefore, 0, length);
            byte[] res = GZIPCompress.compress(dataBefore);
            if (res.length > compressed.length) {
                throw new GZIPCompressOverflowException();
            }
            System.arraycopy(res, 0, compressed, 0, res.length);
            return res.length;
        }

        @Override
        public int compress(ByteBuffer data, ByteBuffer compressed) throws IOException {
            int length = data.remaining();
            byte[] dataBefore = new byte[length];
            data.get(dataBefore, 0, length);
            byte[] res = GZIPCompress.compress(dataBefore);
            if (res.length > compressed.capacity()) {
                throw new GZIPCompressOverflowException();
            }
            compressed.put(res);
            return res.length;
        }

        @Override
        public int getMaxBytesForCompression(int uncompressedDataSize) {
            return Math.max(40 + uncompressedDataSize / 2, uncompressedDataSize);
        }

        @Override
        public CompressionType getType() {
            return CompressionType.GZIP;
        }
    }

    public static class GZIPCompress {
        public static byte[] compress(byte[] data) throws IOException {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            GZIPOutputStream gzip = new GZIPOutputStream(out);
            gzip.write(data);
            gzip.close();
            return out.toByteArray();
        }

        public static byte[] uncompress(byte[] data) throws IOException {
            int n;
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            ByteArrayInputStream in = new ByteArrayInputStream(data);
            GZIPInputStream ungzip = new GZIPInputStream(in);
            byte[] buffer = new byte[256];
            while ((n = ungzip.read(buffer)) > 0) {
                out.write(buffer, 0, n);
            }
            in.close();
            return out.toByteArray();
        }
    }

    public static class IOTDBLZ4Compressor
    implements ICompressor {
        private LZ4Compressor compressor;

        public IOTDBLZ4Compressor() {
            LZ4Factory factory = LZ4Factory.fastestInstance();
            this.compressor = factory.fastCompressor();
        }

        @Override
        public byte[] compress(byte[] data) {
            if (data == null) {
                return new byte[0];
            }
            return this.compressor.compress(data);
        }

        @Override
        public byte[] compress(byte[] data, int offset, int length) throws IOException {
            return this.compressor.compress(data, offset, length);
        }

        @Override
        public int compress(byte[] data, int offset, int length, byte[] compressed) {
            return this.compressor.compress(data, offset, length, compressed, 0);
        }

        @Override
        public int compress(ByteBuffer data, ByteBuffer compressed) {
            int startPosition = compressed.position();
            this.compressor.compress(data, compressed);
            return compressed.position() - startPosition;
        }

        @Override
        public int getMaxBytesForCompression(int uncompressedDataSize) {
            return this.compressor.maxCompressedLength(uncompressedDataSize);
        }

        @Override
        public CompressionType getType() {
            return CompressionType.LZ4;
        }
    }

    public static class SnappyCompressor
    implements ICompressor {
        @Override
        public byte[] compress(byte[] data) throws IOException {
            if (data == null) {
                return new byte[0];
            }
            return Snappy.compress((byte[])data);
        }

        @Override
        public byte[] compress(byte[] data, int offset, int length) throws IOException {
            byte[] maxCompressed = new byte[this.getMaxBytesForCompression(length)];
            int compressedSize = Snappy.compress((byte[])data, (int)offset, (int)length, (byte[])maxCompressed, (int)0);
            byte[] compressed = null;
            if (compressedSize < maxCompressed.length) {
                compressed = new byte[compressedSize];
                System.arraycopy(maxCompressed, 0, compressed, 0, compressedSize);
            } else {
                compressed = maxCompressed;
            }
            return compressed;
        }

        @Override
        public int compress(byte[] data, int offset, int length, byte[] compressed) throws IOException {
            return Snappy.compress((byte[])data, (int)offset, (int)length, (byte[])compressed, (int)0);
        }

        @Override
        public int compress(ByteBuffer data, ByteBuffer compressed) throws IOException {
            return Snappy.compress((ByteBuffer)data, (ByteBuffer)compressed);
        }

        @Override
        public int getMaxBytesForCompression(int uncompressedDataSize) {
            return Snappy.maxCompressedLength((int)uncompressedDataSize);
        }

        @Override
        public CompressionType getType() {
            return CompressionType.SNAPPY;
        }
    }

    public static class NoCompressor
    implements ICompressor {
        @Override
        public byte[] compress(byte[] data) {
            return data;
        }

        @Override
        public byte[] compress(byte[] data, int offset, int length) throws IOException {
            throw new IOException("No Compressor does not support compression function");
        }

        @Override
        public int compress(byte[] data, int offset, int length, byte[] compressed) throws IOException {
            throw new IOException("No Compressor does not support compression function");
        }

        @Override
        public int compress(ByteBuffer data, ByteBuffer compressed) throws IOException {
            throw new IOException("No Compressor does not support compression function");
        }

        @Override
        public int getMaxBytesForCompression(int uncompressedDataSize) {
            return uncompressedDataSize;
        }

        @Override
        public CompressionType getType() {
            return CompressionType.UNCOMPRESSED;
        }
    }
}

