/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.mpp.execution.driver;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.SettableFuture;
import io.airlift.units.Duration;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;
import javax.annotation.concurrent.GuardedBy;
import org.apache.iotdb.db.mpp.execution.driver.DriverContext;
import org.apache.iotdb.db.mpp.execution.driver.IDriver;
import org.apache.iotdb.db.mpp.execution.exchange.sink.ISink;
import org.apache.iotdb.db.mpp.execution.operator.Operator;
import org.apache.iotdb.db.mpp.execution.operator.OperatorContext;
import org.apache.iotdb.db.mpp.execution.schedule.task.DriverTaskId;
import org.apache.iotdb.db.mpp.metric.QueryMetricsManager;
import org.apache.iotdb.tsfile.read.common.block.TsBlock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class Driver
implements IDriver {
    protected static final Logger LOGGER = LoggerFactory.getLogger(Driver.class);
    protected final DriverContext driverContext;
    protected final Operator root;
    protected final ISink sink;
    protected final AtomicReference<SettableFuture<?>> driverBlockedFuture = new AtomicReference();
    protected final AtomicReference<State> state = new AtomicReference<State>(State.ALIVE);
    protected final DriverLock exclusiveLock = new DriverLock();
    protected final QueryMetricsManager QUERY_METRICS = QueryMetricsManager.getInstance();

    protected Driver(Operator root, DriverContext driverContext) {
        Preconditions.checkNotNull((Object)root, (Object)"root Operator should not be null");
        Preconditions.checkNotNull((Object)driverContext.getSink(), (Object)"Sink should not be null");
        this.driverContext = driverContext;
        this.root = root;
        this.sink = driverContext.getSink();
        SettableFuture future = SettableFuture.create();
        future.set(null);
        this.driverBlockedFuture.set(future);
    }

    @Override
    public boolean isFinished() {
        this.checkLockNotHeld("Cannot check finished status while holding the driver lock");
        Optional<Boolean> result = this.tryWithLockUnInterruptibly(this::isFinishedInternal);
        return result.orElseGet(() -> this.state.get() != State.ALIVE || this.driverContext.isDone());
    }

    @Override
    public DriverContext getDriverContext() {
        return this.driverContext;
    }

    protected abstract boolean init(SettableFuture<?> var1);

    protected abstract void releaseResource();

    public int getDependencyDriverIndex() {
        return this.driverContext.getDependencyDriverIndex();
    }

    @Override
    public ListenableFuture<?> processFor(Duration duration) {
        SettableFuture<?> blockedFuture = this.driverBlockedFuture.get();
        if (!blockedFuture.isDone()) {
            return blockedFuture;
        }
        long maxRuntime = duration.roundTo(TimeUnit.NANOSECONDS);
        Optional<ListenableFuture> result = this.tryWithLock(100L, TimeUnit.MILLISECONDS, false, () -> {
            if (this.state.get() == State.ALIVE) {
                long start = System.nanoTime();
                if (!this.init(blockedFuture)) {
                    return blockedFuture;
                }
                do {
                    ListenableFuture<?> future;
                    if ((future = this.processInternal()).isDone()) continue;
                    return this.updateDriverBlockedFuture(future);
                } while (System.nanoTime() - start < maxRuntime && !this.isFinishedInternal());
            }
            return Operator.NOT_BLOCKED;
        });
        return result.orElse(Operator.NOT_BLOCKED);
    }

    @Override
    public DriverTaskId getDriverTaskId() {
        return this.driverContext.getDriverTaskID();
    }

    @Override
    public void setDriverTaskId(DriverTaskId driverTaskId) {
        this.driverContext.setDriverTaskID(driverTaskId);
    }

    @Override
    public void close() {
        if (!this.state.compareAndSet(State.ALIVE, State.NEED_DESTRUCTION)) {
            return;
        }
        this.exclusiveLock.interruptCurrentOwner();
        this.tryWithLockUnInterruptibly(() -> Boolean.TRUE);
    }

    @Override
    public void failed(Throwable t) {
        this.driverContext.failed(t);
    }

    @Override
    public ISink getSink() {
        return this.sink;
    }

    @GuardedBy(value="exclusiveLock")
    private boolean isFinishedInternal() {
        boolean finished;
        this.checkLockHeld("Lock must be held to call isFinishedInternal");
        try {
            finished = this.state.get() != State.ALIVE || this.driverContext.isDone() || this.root.isFinished() || this.sink.isClosed();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        if (finished) {
            this.state.compareAndSet(State.ALIVE, State.NEED_DESTRUCTION);
        }
        return finished;
    }

    private ListenableFuture<?> processInternal() {
        long startTimeNanos = System.nanoTime();
        try {
            TsBlock tsBlock;
            ListenableFuture<?> blocked = this.root.isBlocked();
            if (!blocked.isDone()) {
                ListenableFuture<?> listenableFuture = blocked;
                return listenableFuture;
            }
            blocked = this.sink.isFull();
            if (!blocked.isDone()) {
                ListenableFuture<?> listenableFuture = blocked;
                return listenableFuture;
            }
            if (this.root.hasNextWithTimer() && (tsBlock = this.root.nextWithTimer()) != null && !tsBlock.isEmpty()) {
                this.sink.send(tsBlock);
            }
            tsBlock = Operator.NOT_BLOCKED;
            return tsBlock;
        }
        catch (Throwable t) {
            List<StackTraceElement> interrupterStack = this.exclusiveLock.getInterrupterStack();
            if (interrupterStack == null) {
                this.driverContext.failed(t);
                throw new RuntimeException(t);
            }
            Exception exception = new Exception("Interrupted By");
            exception.setStackTrace(interrupterStack.toArray(new StackTraceElement[0]));
            RuntimeException newException = new RuntimeException("Driver was interrupted", exception);
            newException.addSuppressed(t);
            this.driverContext.failed(newException);
            throw newException;
        }
        finally {
            this.QUERY_METRICS.recordExecutionCost("driver_internal_process", System.nanoTime() - startTimeNanos);
        }
    }

    private ListenableFuture<?> updateDriverBlockedFuture(ListenableFuture<?> sourceBlockedFuture) {
        SettableFuture newDriverBlockedFuture = SettableFuture.create();
        this.driverBlockedFuture.set(newDriverBlockedFuture);
        sourceBlockedFuture.addListener(() -> newDriverBlockedFuture.set(null), MoreExecutors.directExecutor());
        return newDriverBlockedFuture;
    }

    private synchronized void checkLockNotHeld(String message) {
        Preconditions.checkState((!this.exclusiveLock.isHeldByCurrentThread() ? 1 : 0) != 0, (Object)message);
    }

    @GuardedBy(value="exclusiveLock")
    private synchronized void checkLockHeld(String message) {
        Preconditions.checkState((boolean)this.exclusiveLock.isHeldByCurrentThread(), (Object)message);
    }

    private <T> Optional<T> tryWithLockUnInterruptibly(Supplier<T> task) {
        return this.tryWithLock(0L, TimeUnit.MILLISECONDS, false, task);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <T> Optional<T> tryWithLock(long timeout, TimeUnit unit, boolean interruptOnClose, Supplier<T> task) {
        Optional<T> result;
        this.checkLockNotHeld("Lock cannot be reacquired");
        boolean acquired = false;
        try {
            acquired = this.exclusiveLock.tryLock(timeout, unit, interruptOnClose);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        if (!acquired) {
            return Optional.empty();
        }
        try {
            result = Optional.of(task.get());
        }
        finally {
            try {
                this.destroyIfNecessary();
            }
            finally {
                this.exclusiveLock.unlock();
            }
        }
        if (this.state.get() == State.NEED_DESTRUCTION && this.exclusiveLock.tryLock(interruptOnClose)) {
            try {
                this.destroyIfNecessary();
            }
            finally {
                this.exclusiveLock.unlock();
            }
        }
        return result;
    }

    @GuardedBy(value="exclusiveLock")
    private void destroyIfNecessary() {
        this.checkLockHeld("Lock must be held to call destroyIfNecessary");
        if (!this.state.compareAndSet(State.NEED_DESTRUCTION, State.DESTROYED)) {
            return;
        }
        Throwable inFlightException = null;
        try {
            inFlightException = this.closeAndDestroyOperators();
            this.driverContext.finished();
        }
        catch (Throwable t) {
            inFlightException = Driver.addSuppressedException(inFlightException, t, "Error destroying driver for task %s", this.driverContext.getDriverTaskID());
        }
        finally {
            this.releaseResource();
        }
        if (inFlightException != null) {
            Throwables.throwIfUnchecked((Throwable)inFlightException);
            throw new RuntimeException(inFlightException);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Throwable closeAndDestroyOperators() {
        boolean wasInterrupted = Thread.interrupted();
        Throwable inFlightException = null;
        try {
            this.root.close();
            this.sink.setNoMoreTsBlocks();
            List<OperatorContext> operatorContexts = this.driverContext.getOperatorContexts();
            for (OperatorContext operatorContext : operatorContexts) {
                String operatorType = operatorContext.getOperatorType();
                this.QUERY_METRICS.recordOperatorExecutionCost(operatorType, operatorContext.getTotalExecutionTimeInNanos());
                this.QUERY_METRICS.recordOperatorExecutionCount(operatorType, operatorContext.getNextCalledCount());
            }
        }
        catch (InterruptedException t) {
            wasInterrupted = true;
        }
        catch (Throwable t) {
            inFlightException = Driver.addSuppressedException(inFlightException, t, "Error closing operator {} for driver task {}", this.root.getOperatorContext().getOperatorId(), this.driverContext.getDriverTaskID());
        }
        finally {
            if (wasInterrupted) {
                Thread.currentThread().interrupt();
            }
        }
        return inFlightException;
    }

    private static Throwable addSuppressedException(Throwable inFlightException, Throwable newException, String message, Object ... args) {
        if (newException instanceof Error) {
            if (inFlightException == null) {
                inFlightException = newException;
            } else if (inFlightException != newException) {
                inFlightException.addSuppressed(newException);
            }
        } else {
            LOGGER.error(message, (Object)args, (Object)newException);
        }
        return inFlightException;
    }

    private static class DriverLock {
        private final ReentrantLock lock = new ReentrantLock();
        @GuardedBy(value="this")
        private Thread currentOwner;
        @GuardedBy(value="this")
        private boolean currentOwnerInterruptionAllowed;
        @GuardedBy(value="this")
        private List<StackTraceElement> interrupterStack;

        private DriverLock() {
        }

        public boolean isHeldByCurrentThread() {
            return this.lock.isHeldByCurrentThread();
        }

        public boolean tryLock(boolean currentThreadInterruptionAllowed) {
            Preconditions.checkState((!this.lock.isHeldByCurrentThread() ? 1 : 0) != 0, (Object)"Lock is not reentrant");
            boolean acquired = this.lock.tryLock();
            if (acquired) {
                this.setOwner(currentThreadInterruptionAllowed);
            }
            return acquired;
        }

        public boolean tryLock(long timeout, TimeUnit unit, boolean currentThreadInterruptionAllowed) throws InterruptedException {
            Preconditions.checkState((!this.lock.isHeldByCurrentThread() ? 1 : 0) != 0, (Object)"Lock is not reentrant");
            boolean acquired = this.lock.tryLock(timeout, unit);
            if (acquired) {
                this.setOwner(currentThreadInterruptionAllowed);
            }
            return acquired;
        }

        private synchronized void setOwner(boolean interruptionAllowed) {
            Preconditions.checkState((boolean)this.lock.isHeldByCurrentThread(), (Object)"Current thread does not hold lock");
            this.currentOwner = Thread.currentThread();
            this.currentOwnerInterruptionAllowed = interruptionAllowed;
        }

        public synchronized void unlock() {
            Preconditions.checkState((boolean)this.lock.isHeldByCurrentThread(), (Object)"Current thread does not hold lock");
            this.currentOwner = null;
            this.currentOwnerInterruptionAllowed = false;
            this.lock.unlock();
        }

        public synchronized List<StackTraceElement> getInterrupterStack() {
            return this.interrupterStack;
        }

        public synchronized void interruptCurrentOwner() {
            if (!this.currentOwnerInterruptionAllowed) {
                return;
            }
            if (this.interrupterStack == null) {
                this.interrupterStack = ImmutableList.copyOf((Object[])Thread.currentThread().getStackTrace());
            }
            if (this.currentOwner != null) {
                this.currentOwner.interrupt();
            }
        }
    }

    protected static enum State {
        ALIVE,
        NEED_DESTRUCTION,
        DESTROYED;

    }
}

