/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.engine.cache;

import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import java.io.IOException;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.iotdb.db.conf.IoTDBConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.query.control.FileReaderManager;
import org.apache.iotdb.tsfile.read.TsFileSequenceReader;
import org.apache.iotdb.tsfile.utils.BloomFilter;
import org.apache.iotdb.tsfile.utils.FilePathUtils;
import org.apache.iotdb.tsfile.utils.Pair;
import org.apache.iotdb.tsfile.utils.RamUsageEstimator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BloomFilterCache {
    private static final Logger logger = LoggerFactory.getLogger(BloomFilterCache.class);
    private static final Logger DEBUG_LOGGER = LoggerFactory.getLogger((String)"QUERY_DEBUG");
    private static final IoTDBConfig config = IoTDBDescriptor.getInstance().getConfig();
    private static final long MEMORY_THRESHOLD_IN_BLOOM_FILTER_CACHE = config.getAllocateMemoryForBloomFilterCache();
    private static final boolean CACHE_ENABLE = config.isMetaDataCacheEnable();
    private final AtomicLong entryAverageSize = new AtomicLong(0L);
    private final LoadingCache<BloomFilterCacheKey, BloomFilter> lruCache;

    private BloomFilterCache() {
        if (CACHE_ENABLE) {
            logger.info("BloomFilterCache size = {}", (Object)MEMORY_THRESHOLD_IN_BLOOM_FILTER_CACHE);
        }
        this.lruCache = Caffeine.newBuilder().maximumWeight(MEMORY_THRESHOLD_IN_BLOOM_FILTER_CACHE).weigher((key, bloomFilter) -> (int)(RamUsageEstimator.shallowSizeOf((Object)key) + RamUsageEstimator.sizeOf((Object)((BloomFilterCacheKey)key).tsFilePrefixPath) + RamUsageEstimator.sizeOf((Object)bloomFilter))).recordStats().build(key -> {
            try {
                TsFileSequenceReader reader = FileReaderManager.getInstance().get(((BloomFilterCacheKey)key).filePath, true);
                return reader.readBloomFilter();
            }
            catch (IOException e) {
                logger.error("Something wrong happened in reading bloom filter in tsfile {}", (Object)((BloomFilterCacheKey)key).filePath, (Object)e);
                throw e;
            }
        });
    }

    public static BloomFilterCache getInstance() {
        return BloomFilterCacheHolder.INSTANCE;
    }

    public BloomFilter get(BloomFilterCacheKey key) throws IOException {
        return this.get(key, false);
    }

    public BloomFilter get(BloomFilterCacheKey key, boolean debug) throws IOException {
        if (!CACHE_ENABLE) {
            TsFileSequenceReader reader = FileReaderManager.getInstance().get(key.filePath, true);
            return reader.readBloomFilter();
        }
        BloomFilter bloomFilter = (BloomFilter)this.lruCache.get((Object)key);
        if (debug) {
            DEBUG_LOGGER.info("get bloomFilter from cache where filePath is: {}", (Object)key.filePath);
        }
        return bloomFilter;
    }

    public double calculateChunkHitRatio() {
        return this.lruCache.stats().hitRate();
    }

    public long getEvictionCount() {
        return this.lruCache.stats().evictionCount();
    }

    public long getMaxMemory() {
        return MEMORY_THRESHOLD_IN_BLOOM_FILTER_CACHE;
    }

    public double getAverageLoadPenalty() {
        return this.lruCache.stats().averageLoadPenalty();
    }

    public long getAverageSize() {
        return this.entryAverageSize.get();
    }

    public void clear() {
        this.lruCache.invalidateAll();
        this.lruCache.cleanUp();
    }

    public void remove(BloomFilterCacheKey key) {
        this.lruCache.invalidate((Object)key);
    }

    public BloomFilter getIfPresent(BloomFilterCacheKey key) {
        return (BloomFilter)this.lruCache.getIfPresent((Object)key);
    }

    private static class BloomFilterCacheHolder {
        private static final BloomFilterCache INSTANCE = new BloomFilterCache();

        private BloomFilterCacheHolder() {
        }
    }

    public static class BloomFilterCacheKey {
        private final String filePath;
        private final String tsFilePrefixPath;
        private final long tsFileVersion;
        private final long compactionVersion;

        public BloomFilterCacheKey(String filePath) {
            this.filePath = filePath;
            Pair tsFilePrefixPathAndTsFileVersionPair = FilePathUtils.getTsFilePrefixPathAndTsFileVersionPair((String)filePath);
            this.tsFilePrefixPath = (String)tsFilePrefixPathAndTsFileVersionPair.left;
            this.tsFileVersion = ((long[])tsFilePrefixPathAndTsFileVersionPair.right)[0];
            this.compactionVersion = ((long[])tsFilePrefixPathAndTsFileVersionPair.right)[1];
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            BloomFilterCacheKey that = (BloomFilterCacheKey)o;
            return this.tsFileVersion == that.tsFileVersion && this.compactionVersion == that.compactionVersion && this.tsFilePrefixPath.equals(that.tsFilePrefixPath);
        }

        public int hashCode() {
            return Objects.hash(this.tsFilePrefixPath, this.tsFileVersion, this.compactionVersion);
        }
    }
}

