/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.engine;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.commons.io.FileUtils;
import org.apache.iotdb.common.rpc.thrift.TFlushReq;
import org.apache.iotdb.common.rpc.thrift.TSStatus;
import org.apache.iotdb.common.rpc.thrift.TSetTTLReq;
import org.apache.iotdb.commons.concurrent.IoTDBThreadPoolFactory;
import org.apache.iotdb.commons.concurrent.ThreadName;
import org.apache.iotdb.commons.concurrent.threadpool.ScheduledExecutorUtil;
import org.apache.iotdb.commons.conf.CommonDescriptor;
import org.apache.iotdb.commons.consensus.DataRegionId;
import org.apache.iotdb.commons.exception.ShutdownException;
import org.apache.iotdb.commons.file.SystemFileFactory;
import org.apache.iotdb.commons.service.IService;
import org.apache.iotdb.commons.service.ServiceType;
import org.apache.iotdb.db.conf.IoTDBConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.consensus.statemachine.visitor.DataExecutionVisitor;
import org.apache.iotdb.db.engine.cache.BloomFilterCache;
import org.apache.iotdb.db.engine.cache.ChunkCache;
import org.apache.iotdb.db.engine.cache.TimeSeriesMetadataCache;
import org.apache.iotdb.db.engine.flush.CloseFileListener;
import org.apache.iotdb.db.engine.flush.FlushListener;
import org.apache.iotdb.db.engine.flush.TsFileFlushPolicy;
import org.apache.iotdb.db.engine.load.LoadTsFileManager;
import org.apache.iotdb.db.engine.storagegroup.DataRegion;
import org.apache.iotdb.db.engine.storagegroup.TsFileProcessor;
import org.apache.iotdb.db.exception.DataRegionException;
import org.apache.iotdb.db.exception.LoadFileException;
import org.apache.iotdb.db.exception.StorageEngineException;
import org.apache.iotdb.db.exception.TsFileProcessorException;
import org.apache.iotdb.db.exception.WriteProcessRejectException;
import org.apache.iotdb.db.exception.runtime.StorageEngineFailureException;
import org.apache.iotdb.db.mpp.plan.planner.plan.node.PlanNode;
import org.apache.iotdb.db.mpp.plan.planner.plan.node.load.LoadTsFilePieceNode;
import org.apache.iotdb.db.mpp.plan.scheduler.load.LoadTsFileScheduler;
import org.apache.iotdb.db.rescon.SystemInfo;
import org.apache.iotdb.db.sync.SyncService;
import org.apache.iotdb.db.utils.ThreadUtils;
import org.apache.iotdb.db.utils.UpgradeUtils;
import org.apache.iotdb.db.wal.WALManager;
import org.apache.iotdb.db.wal.exception.WALException;
import org.apache.iotdb.db.wal.recover.WALRecoverManager;
import org.apache.iotdb.rpc.RpcUtils;
import org.apache.iotdb.rpc.TSStatusCode;
import org.apache.iotdb.tsfile.exception.write.PageException;
import org.apache.iotdb.tsfile.utils.FilePathUtils;
import org.apache.iotdb.tsfile.utils.ReadWriteIOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StorageEngine
implements IService {
    private static final Logger logger = LoggerFactory.getLogger(StorageEngine.class);
    private static final IoTDBConfig config = IoTDBDescriptor.getInstance().getConfig();
    private static final long TTL_CHECK_INTERVAL = 60000L;
    private static long timePartitionInterval = -1L;
    private final String systemDir = FilePathUtils.regularizePath((String)config.getSystemDir()) + "databases";
    private final ConcurrentHashMap<DataRegionId, DataRegion> dataRegionMap = new ConcurrentHashMap();
    private final ConcurrentHashMap<DataRegionId, DataRegion> deletingDataRegionMap = new ConcurrentHashMap();
    private final Map<String, Long> ttlMapForRecover = new ConcurrentHashMap<String, Long>();
    private AtomicInteger readyDataRegionNum;
    private AtomicBoolean isAllSgReady = new AtomicBoolean(false);
    private ScheduledExecutorService ttlCheckThread;
    private ScheduledExecutorService seqMemtableTimedFlushCheckThread;
    private ScheduledExecutorService unseqMemtableTimedFlushCheckThread;
    private TsFileFlushPolicy fileFlushPolicy = new TsFileFlushPolicy.DirectFlushPolicy();
    private ExecutorService cachedThreadPool;
    private List<CloseFileListener> customCloseFileListeners = new ArrayList<CloseFileListener>();
    private List<FlushListener> customFlushListeners = new ArrayList<FlushListener>();
    private int recoverDataRegionNum = 0;
    private LoadTsFileManager loadTsFileManager = new LoadTsFileManager();

    private StorageEngine() {
    }

    public static StorageEngine getInstance() {
        return InstanceHolder.INSTANCE;
    }

    private static void initTimePartition() {
        timePartitionInterval = IoTDBDescriptor.getInstance().getConfig().getTimePartitionInterval();
    }

    public static long getTimePartitionInterval() {
        if (timePartitionInterval == -1L) {
            StorageEngine.initTimePartition();
        }
        return timePartitionInterval;
    }

    public static long getTimePartition(long time) {
        if (timePartitionInterval == -1L) {
            StorageEngine.initTimePartition();
        }
        return time / timePartitionInterval;
    }

    public static void blockInsertionIfReject(TsFileProcessor tsFileProcessor) throws WriteProcessRejectException {
        long startTime = System.currentTimeMillis();
        while (SystemInfo.getInstance().isRejected() && (tsFileProcessor == null || !tsFileProcessor.shouldFlush())) {
            try {
                TimeUnit.MILLISECONDS.sleep(config.getCheckPeriodWhenInsertBlocked());
                if (System.currentTimeMillis() - startTime <= (long)config.getMaxWaitingTimeWhenInsertBlocked()) continue;
                throw new WriteProcessRejectException("System rejected over " + (System.currentTimeMillis() - startTime) + "ms");
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

    public void updateTTLInfo(byte[] allTTLInformation) {
        if (allTTLInformation == null) {
            return;
        }
        ByteBuffer buffer = ByteBuffer.wrap(allTTLInformation);
        int mapSize = ReadWriteIOUtils.readInt((ByteBuffer)buffer);
        for (int i = 0; i < mapSize; ++i) {
            this.ttlMapForRecover.put(Objects.requireNonNull(ReadWriteIOUtils.readString((ByteBuffer)buffer)), ReadWriteIOUtils.readLong((ByteBuffer)buffer));
        }
    }

    public boolean isAllSgReady() {
        return this.isAllSgReady.get();
    }

    public void setAllSgReady(boolean allSgReady) {
        this.isAllSgReady.set(allSgReady);
    }

    public void recover() {
        this.setAllSgReady(false);
        this.cachedThreadPool = IoTDBThreadPoolFactory.newCachedThreadPool((String)ThreadName.STORAGE_ENGINE_CACHED_SERVICE.getName());
        LinkedList<Future<Void>> futures = new LinkedList<Future<Void>>();
        this.asyncRecover(futures);
        if (!config.isClusterMode() || !config.getDataRegionConsensusProtocolClass().equals("org.apache.iotdb.consensus.ratis.RatisConsensus")) {
            try {
                WALRecoverManager.getInstance().recover();
            }
            catch (WALException e) {
                logger.error("Fail to recover wal.", (Throwable)((Object)e));
            }
        }
        Thread recoverEndTrigger = new Thread(() -> {
            this.checkResults(futures, "StorageEngine failed to recover.");
            this.setAllSgReady(true);
            this.ttlMapForRecover.clear();
        });
        recoverEndTrigger.start();
    }

    private void asyncRecover(List<Future<Void>> futures) {
        Map<String, List<DataRegionId>> localDataRegionInfo = this.getLocalDataRegionInfo();
        localDataRegionInfo.values().forEach(list -> this.recoverDataRegionNum += list.size());
        this.readyDataRegionNum = new AtomicInteger(0);
        WALRecoverManager.getInstance().setAllDataRegionScannedLatch(new CountDownLatch(this.recoverDataRegionNum));
        for (Map.Entry<String, List<DataRegionId>> entry : localDataRegionInfo.entrySet()) {
            String sgName = entry.getKey();
            for (DataRegionId dataRegionId : entry.getValue()) {
                Callable<Void> recoverDataRegionTask = () -> {
                    DataRegion dataRegion = null;
                    try {
                        dataRegion = this.buildNewDataRegion(sgName, dataRegionId, this.ttlMapForRecover.getOrDefault(sgName, Long.MAX_VALUE));
                    }
                    catch (DataRegionException e) {
                        logger.error("Failed to recover data region {}[{}]", new Object[]{sgName, dataRegionId.getId(), e});
                    }
                    this.dataRegionMap.put(dataRegionId, dataRegion);
                    logger.info("Data regions have been recovered {}/{}", (Object)this.readyDataRegionNum.incrementAndGet(), (Object)this.recoverDataRegionNum);
                    return null;
                };
                futures.add(this.cachedThreadPool.submit(recoverDataRegionTask));
            }
        }
    }

    public Map<String, List<DataRegionId>> getLocalDataRegionInfo() {
        File system = SystemFileFactory.INSTANCE.getFile(this.systemDir);
        File[] sgDirs = system.listFiles();
        HashMap<String, List<DataRegionId>> localDataRegionInfo = new HashMap<String, List<DataRegionId>>();
        if (sgDirs == null) {
            return localDataRegionInfo;
        }
        for (File sgDir : sgDirs) {
            if (!sgDir.isDirectory()) continue;
            String sgName = sgDir.getName();
            ArrayList<DataRegionId> dataRegionIdList = new ArrayList<DataRegionId>();
            for (File dataRegionDir : sgDir.listFiles()) {
                if (!dataRegionDir.isDirectory()) continue;
                dataRegionIdList.add(new DataRegionId(Integer.parseInt(dataRegionDir.getName())));
            }
            localDataRegionInfo.put(sgName, dataRegionIdList);
        }
        return localDataRegionInfo;
    }

    public void start() {
        StorageEngine.initTimePartition();
        try {
            FileUtils.forceMkdir((File)SystemFileFactory.INSTANCE.getFile(this.systemDir));
        }
        catch (IOException e) {
            throw new StorageEngineFailureException(e);
        }
        UpgradeUtils.recoverUpgrade();
        this.recover();
        this.ttlCheckThread = IoTDBThreadPoolFactory.newSingleThreadScheduledExecutor((String)"TTL-Check");
        ScheduledExecutorUtil.safelyScheduleAtFixedRate((ScheduledExecutorService)this.ttlCheckThread, this::checkTTL, (long)60000L, (long)60000L, (TimeUnit)TimeUnit.MILLISECONDS);
        logger.info("start ttl check thread successfully.");
        this.startTimedService();
    }

    private void checkTTL() {
        try {
            for (DataRegion dataRegion : this.dataRegionMap.values()) {
                if (dataRegion == null) continue;
                dataRegion.checkFilesTTL();
            }
        }
        catch (ConcurrentModificationException concurrentModificationException) {
        }
        catch (Exception e) {
            logger.error("An error occurred when checking TTL", (Throwable)e);
        }
    }

    private void startTimedService() {
        if (config.isEnableTimedFlushSeqMemtable()) {
            this.seqMemtableTimedFlushCheckThread = IoTDBThreadPoolFactory.newSingleThreadScheduledExecutor((String)ThreadName.TIMED_FLUSH_SEQ_MEMTABLE.getName());
            ScheduledExecutorUtil.safelyScheduleAtFixedRate((ScheduledExecutorService)this.seqMemtableTimedFlushCheckThread, this::timedFlushSeqMemTable, (long)config.getSeqMemtableFlushCheckInterval(), (long)config.getSeqMemtableFlushCheckInterval(), (TimeUnit)TimeUnit.MILLISECONDS);
            logger.info("start sequence memtable timed flush check thread successfully.");
        }
        if (config.isEnableTimedFlushUnseqMemtable()) {
            this.unseqMemtableTimedFlushCheckThread = IoTDBThreadPoolFactory.newSingleThreadScheduledExecutor((String)ThreadName.TIMED_FLUSH_UNSEQ_MEMTABLE.getName());
            ScheduledExecutorUtil.safelyScheduleAtFixedRate((ScheduledExecutorService)this.unseqMemtableTimedFlushCheckThread, this::timedFlushUnseqMemTable, (long)config.getUnseqMemtableFlushCheckInterval(), (long)config.getUnseqMemtableFlushCheckInterval(), (TimeUnit)TimeUnit.MILLISECONDS);
            logger.info("start unsequence memtable timed flush check thread successfully.");
        }
    }

    private void timedFlushSeqMemTable() {
        for (DataRegion dataRegion : this.dataRegionMap.values()) {
            if (dataRegion == null) continue;
            dataRegion.timedFlushSeqMemTable();
        }
    }

    private void timedFlushUnseqMemTable() {
        for (DataRegion dataRegion : this.dataRegionMap.values()) {
            if (dataRegion == null) continue;
            dataRegion.timedFlushUnseqMemTable();
        }
    }

    public void stop() {
        for (DataRegion dataRegion : this.dataRegionMap.values()) {
            if (dataRegion == null) continue;
            ThreadUtils.stopThreadPool(dataRegion.getTimedCompactionScheduleTask(), ThreadName.COMPACTION_SCHEDULE);
        }
        this.syncCloseAllProcessor();
        ThreadUtils.stopThreadPool(this.ttlCheckThread, ThreadName.TTL_CHECK_SERVICE);
        ThreadUtils.stopThreadPool(this.seqMemtableTimedFlushCheckThread, ThreadName.TIMED_FLUSH_SEQ_MEMTABLE);
        ThreadUtils.stopThreadPool(this.unseqMemtableTimedFlushCheckThread, ThreadName.TIMED_FLUSH_UNSEQ_MEMTABLE);
        if (this.cachedThreadPool != null) {
            this.cachedThreadPool.shutdownNow();
        }
        this.dataRegionMap.clear();
    }

    public void shutdown(long milliseconds) throws ShutdownException {
        try {
            for (DataRegion dataRegion : this.dataRegionMap.values()) {
                ThreadUtils.stopThreadPool(dataRegion.getTimedCompactionScheduleTask(), ThreadName.COMPACTION_SCHEDULE);
            }
            this.forceCloseAllProcessor();
        }
        catch (TsFileProcessorException e) {
            throw new ShutdownException((Throwable)((Object)e));
        }
        this.shutdownTimedService(this.ttlCheckThread, "TTlCheckThread");
        this.shutdownTimedService(this.seqMemtableTimedFlushCheckThread, "SeqMemtableTimedFlushCheckThread");
        this.shutdownTimedService(this.unseqMemtableTimedFlushCheckThread, "UnseqMemtableTimedFlushCheckThread");
        this.cachedThreadPool.shutdownNow();
        this.dataRegionMap.clear();
    }

    private void shutdownTimedService(ScheduledExecutorService pool, String poolName) {
        if (pool != null) {
            pool.shutdownNow();
            try {
                pool.awaitTermination(30L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                logger.warn("{} still doesn't exit after 30s", (Object)poolName);
                Thread.currentThread().interrupt();
            }
        }
    }

    public ServiceType getID() {
        return ServiceType.STORAGE_ENGINE_SERVICE;
    }

    public DataRegion buildNewDataRegion(String logicalStorageGroupName, DataRegionId dataRegionId, long ttl) throws DataRegionException {
        logger.info("construct a data region instance, the database is {}, Thread is {}", (Object)logicalStorageGroupName, (Object)Thread.currentThread().getId());
        DataRegion dataRegion = new DataRegion(this.systemDir + File.separator + logicalStorageGroupName, String.valueOf(dataRegionId.getId()), this.fileFlushPolicy, logicalStorageGroupName);
        dataRegion.setDataTTLWithTimePrecisionCheck(ttl);
        dataRegion.setCustomFlushListeners(this.customFlushListeners);
        dataRegion.setCustomCloseFileListeners(this.customCloseFileListeners);
        return dataRegion;
    }

    public TSStatus write(DataRegionId groupId, PlanNode planNode) {
        return planNode.accept(new DataExecutionVisitor(), this.dataRegionMap.get(groupId));
    }

    public synchronized void reset() {
        this.dataRegionMap.clear();
    }

    public void syncCloseAllProcessor() {
        logger.info("Start closing all database processor");
        ArrayList tasks = new ArrayList();
        for (DataRegion dataRegion : this.dataRegionMap.values()) {
            if (dataRegion == null) continue;
            tasks.add(this.cachedThreadPool.submit(() -> {
                dataRegion.syncCloseAllWorkingTsFileProcessors();
                return null;
            }));
        }
        this.checkResults(tasks, "Failed to sync close processor.");
    }

    public void forceCloseAllProcessor() throws TsFileProcessorException {
        logger.info("Start force closing all database processor");
        ArrayList tasks = new ArrayList();
        for (DataRegion dataRegion : this.dataRegionMap.values()) {
            if (dataRegion == null) continue;
            tasks.add(this.cachedThreadPool.submit(() -> {
                dataRegion.forceCloseAllWorkingTsFileProcessors();
                return null;
            }));
        }
        this.checkResults(tasks, "Failed to force close processor.");
    }

    public void closeStorageGroupProcessor(String storageGroupPath, boolean isSeq) {
        ArrayList tasks = new ArrayList();
        for (DataRegion dataRegion : this.dataRegionMap.values()) {
            if (!dataRegion.getDatabaseName().equals(storageGroupPath)) continue;
            if (isSeq) {
                for (TsFileProcessor tsFileProcessor : dataRegion.getWorkSequenceTsFileProcessors()) {
                    tasks.add(this.cachedThreadPool.submit(() -> {
                        dataRegion.syncCloseOneTsFileProcessor(isSeq, tsFileProcessor);
                        return null;
                    }));
                }
                continue;
            }
            for (TsFileProcessor tsFileProcessor : dataRegion.getWorkUnsequenceTsFileProcessors()) {
                tasks.add(this.cachedThreadPool.submit(() -> {
                    dataRegion.syncCloseOneTsFileProcessor(isSeq, tsFileProcessor);
                    return null;
                }));
            }
        }
        this.checkResults(tasks, "Failed to close database processor.");
    }

    private <V> void checkResults(List<Future<V>> tasks, String errorMsg) {
        for (Future<V> task : tasks) {
            try {
                task.get();
            }
            catch (ExecutionException e) {
                throw new StorageEngineFailureException(errorMsg, e);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new StorageEngineFailureException(errorMsg, e);
            }
        }
    }

    public void mergeAll() throws StorageEngineException {
        if (CommonDescriptor.getInstance().getConfig().isReadOnly()) {
            throw new StorageEngineException("Current system mode is read only, does not support merge");
        }
        this.dataRegionMap.values().forEach(DataRegion::compact);
    }

    public void operateFlush(TFlushReq req) {
        if (req.storageGroups == null) {
            StorageEngine.getInstance().syncCloseAllProcessor();
            WALManager.getInstance().deleteOutdatedWALFiles();
        } else {
            for (String storageGroup : req.storageGroups) {
                if (req.isSeq == null) {
                    StorageEngine.getInstance().closeStorageGroupProcessor(storageGroup, true);
                    StorageEngine.getInstance().closeStorageGroupProcessor(storageGroup, false);
                    continue;
                }
                StorageEngine.getInstance().closeStorageGroupProcessor(storageGroup, Boolean.parseBoolean(req.isSeq));
            }
        }
    }

    public void clearCache() {
        ChunkCache.getInstance().clear();
        TimeSeriesMetadataCache.getInstance().clear();
        BloomFilterCache.getInstance().clear();
    }

    public void setTTL(List<DataRegionId> dataRegionIdList, long dataTTL) {
        for (DataRegionId dataRegionId : dataRegionIdList) {
            DataRegion dataRegion = this.dataRegionMap.get(dataRegionId);
            if (dataRegion == null) continue;
            dataRegion.setDataTTLWithTimePrecisionCheck(dataTTL);
        }
    }

    public void registerFlushListener(FlushListener listener) {
        this.customFlushListeners.add(listener);
    }

    public void registerCloseFileListener(CloseFileListener listener) {
        this.customCloseFileListeners.add(listener);
    }

    private void makeSureNoOldRegion(DataRegionId regionId) {
        while (this.deletingDataRegionMap.containsKey(regionId)) {
            DataRegion oldRegion = this.deletingDataRegionMap.get(regionId);
            if (oldRegion == null) continue;
            oldRegion.waitForDeleted();
        }
    }

    public DataRegion createDataRegion(DataRegionId regionId, String sg, long ttl) throws DataRegionException {
        this.makeSureNoOldRegion(regionId);
        AtomicReference<Object> exceptionAtomicReference = new AtomicReference<Object>(null);
        DataRegion dataRegion = this.dataRegionMap.computeIfAbsent(regionId, x -> {
            try {
                return this.buildNewDataRegion(sg, (DataRegionId)x, ttl);
            }
            catch (DataRegionException e) {
                exceptionAtomicReference.set(e);
                return null;
            }
        });
        if (exceptionAtomicReference.get() != null) {
            throw (DataRegionException)exceptionAtomicReference.get();
        }
        return dataRegion;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteDataRegion(DataRegionId regionId) {
        if (!this.dataRegionMap.containsKey(regionId) || this.deletingDataRegionMap.containsKey(regionId)) {
            return;
        }
        DataRegion region = this.deletingDataRegionMap.computeIfAbsent(regionId, k -> this.dataRegionMap.remove(regionId));
        if (region != null) {
            region.markDeleted();
            try {
                region.abortCompaction();
                region.syncDeleteDataFiles();
                region.deleteFolder(this.systemDir);
                if (config.isClusterMode() && config.getDataRegionConsensusProtocolClass().equals("org.apache.iotdb.consensus.iot.IoTConsensus")) {
                    WALManager.getInstance().deleteWALNode(region.getDatabaseName() + "-" + region.getDataRegionId());
                    for (String dataDir : config.getDataDirs()) {
                        File regionSnapshotDir = new File(dataDir + File.separator + "snapshot", region.getDatabaseName() + "-" + regionId.getId());
                        if (!regionSnapshotDir.exists()) continue;
                        try {
                            FileUtils.deleteDirectory((File)regionSnapshotDir);
                        }
                        catch (IOException e) {
                            logger.error("Failed to delete snapshot dir {}", (Object)regionSnapshotDir, (Object)e);
                        }
                    }
                }
                SyncService.getInstance().unregisterDataRegion(region.getDataRegionId());
            }
            catch (Exception e) {
                logger.error("Error occurs when deleting data region {}-{}", new Object[]{region.getDatabaseName(), region.getDataRegionId(), e});
            }
            finally {
                this.deletingDataRegionMap.remove(regionId);
            }
        }
    }

    public DataRegion getDataRegion(DataRegionId regionId) {
        return this.dataRegionMap.get(regionId);
    }

    public List<DataRegion> getAllDataRegions() {
        return new ArrayList<DataRegion>(this.dataRegionMap.values());
    }

    public List<DataRegionId> getAllDataRegionIds() {
        return new ArrayList<DataRegionId>(this.dataRegionMap.keySet());
    }

    public void setDataRegion(DataRegionId regionId, DataRegion newRegion) {
        if (this.dataRegionMap.containsKey(regionId)) {
            DataRegion oldRegion = this.dataRegionMap.get(regionId);
            oldRegion.syncCloseAllWorkingTsFileProcessors();
            oldRegion.abortCompaction();
        }
        this.dataRegionMap.put(regionId, newRegion);
    }

    public TSStatus setTTL(TSetTTLReq req) {
        Map<String, List<DataRegionId>> localDataRegionInfo = StorageEngine.getInstance().getLocalDataRegionInfo();
        ArrayList dataRegionIdList = new ArrayList();
        req.storageGroupPathPattern.forEach(storageGroup -> dataRegionIdList.addAll((Collection)localDataRegionInfo.get(storageGroup)));
        for (DataRegionId dataRegionId : dataRegionIdList) {
            DataRegion dataRegion = this.dataRegionMap.get(dataRegionId);
            if (dataRegion == null) continue;
            dataRegion.setDataTTLWithTimePrecisionCheck(req.TTL);
        }
        return RpcUtils.getStatus((TSStatusCode)TSStatusCode.SUCCESS_STATUS);
    }

    public TsFileFlushPolicy getFileFlushPolicy() {
        return this.fileFlushPolicy;
    }

    public TSStatus writeLoadTsFileNode(DataRegionId dataRegionId, LoadTsFilePieceNode pieceNode, String uuid) {
        TSStatus status = new TSStatus();
        try {
            this.loadTsFileManager.writeToDataRegion(this.getDataRegion(dataRegionId), pieceNode, uuid);
        }
        catch (PageException e) {
            logger.error(String.format("Parse Page error when writing piece node of TsFile %s to DataRegion %s.", pieceNode.getTsFile(), dataRegionId), (Throwable)e);
            status.setCode(TSStatusCode.LOAD_PIECE_OF_TSFILE_ERROR.getStatusCode());
            status.setMessage(e.getMessage());
            return status;
        }
        catch (IOException e) {
            logger.error(String.format("IO error when writing piece node of TsFile %s to DataRegion %s.", pieceNode.getTsFile(), dataRegionId), (Throwable)e);
            status.setCode(TSStatusCode.LOAD_FILE_ERROR.getStatusCode());
            status.setMessage(e.getMessage());
            return status;
        }
        return RpcUtils.SUCCESS_STATUS;
    }

    public TSStatus executeLoadCommand(LoadTsFileScheduler.LoadCommand loadCommand, String uuid) {
        TSStatus status = new TSStatus();
        try {
            switch (loadCommand) {
                case EXECUTE: {
                    if (this.loadTsFileManager.loadAll(uuid)) {
                        status = RpcUtils.SUCCESS_STATUS;
                        break;
                    }
                    status.setCode(TSStatusCode.LOAD_FILE_ERROR.getStatusCode());
                    status.setMessage(String.format("No load TsFile uuid %s recorded for execute load command %s.", new Object[]{uuid, loadCommand}));
                    break;
                }
                case ROLLBACK: {
                    if (this.loadTsFileManager.deleteAll(uuid)) {
                        status = RpcUtils.SUCCESS_STATUS;
                        break;
                    }
                    status.setCode(TSStatusCode.LOAD_FILE_ERROR.getStatusCode());
                    status.setMessage(String.format("No load TsFile uuid %s recorded for execute load command %s.", new Object[]{uuid, loadCommand}));
                    break;
                }
                default: {
                    status.setCode(TSStatusCode.ILLEGAL_PARAMETER.getStatusCode());
                    status.setMessage(String.format("Wrong load command %s.", new Object[]{loadCommand}));
                    break;
                }
            }
        }
        catch (IOException | LoadFileException e) {
            logger.error(String.format("Execute load command %s error.", new Object[]{loadCommand}), (Throwable)e);
            status.setCode(TSStatusCode.LOAD_FILE_ERROR.getStatusCode());
            status.setMessage(((Throwable)e).getMessage());
        }
        return status;
    }

    public void rebootTimedService() throws ShutdownException {
        logger.info("Start rebooting all timed service.");
        this.stopTimedServiceAndThrow(this.seqMemtableTimedFlushCheckThread, "SeqMemtableTimedFlushCheckThread");
        this.stopTimedServiceAndThrow(this.unseqMemtableTimedFlushCheckThread, "UnseqMemtableTimedFlushCheckThread");
        logger.info("Stop all timed service successfully, and now restart them.");
        this.startTimedService();
        logger.info("Reboot all timed service successfully");
    }

    private void stopTimedServiceAndThrow(ScheduledExecutorService pool, String poolName) throws ShutdownException {
        if (pool != null) {
            pool.shutdownNow();
            try {
                pool.awaitTermination(30L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                logger.warn("{} still doesn't exit after 30s", (Object)poolName);
                throw new ShutdownException((Throwable)e);
            }
        }
    }

    static class InstanceHolder {
        private static final StorageEngine INSTANCE = new StorageEngine();

        private InstanceHolder() {
        }
    }
}

