/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.wal.buffer;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.iotdb.commons.cluster.NodeStatus;
import org.apache.iotdb.commons.concurrent.IoTDBThreadPoolFactory;
import org.apache.iotdb.commons.concurrent.ThreadName;
import org.apache.iotdb.commons.conf.CommonDescriptor;
import org.apache.iotdb.db.conf.IoTDBConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.mpp.plan.planner.plan.node.write.DeleteDataNode;
import org.apache.iotdb.db.mpp.plan.planner.plan.node.write.InsertNode;
import org.apache.iotdb.db.utils.MmapUtil;
import org.apache.iotdb.db.wal.buffer.AbstractWALBuffer;
import org.apache.iotdb.db.wal.buffer.IWALByteBufferView;
import org.apache.iotdb.db.wal.buffer.WALEntry;
import org.apache.iotdb.db.wal.buffer.WALEntryType;
import org.apache.iotdb.db.wal.buffer.WALSignalEntry;
import org.apache.iotdb.db.wal.exception.WALNodeClosedException;
import org.apache.iotdb.db.wal.io.WALMetaData;
import org.apache.iotdb.db.wal.utils.WALFileStatus;
import org.apache.iotdb.db.wal.utils.listener.WALFlushListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WALBuffer
extends AbstractWALBuffer {
    private static final Logger logger = LoggerFactory.getLogger(WALBuffer.class);
    private static final IoTDBConfig config = IoTDBDescriptor.getInstance().getConfig();
    private static final int HALF_WAL_BUFFER_SIZE = config.getWalBufferSize() / 2;
    private static final int QUEUE_CAPACITY = config.getWalBufferQueueCapacity();
    private volatile boolean isClosed = false;
    private final BlockingQueue<WALEntry> walEntries = new ArrayBlockingQueue<WALEntry>(QUEUE_CAPACITY);
    private final Lock buffersLock = new ReentrantLock();
    private final Condition idleBufferReadyCondition = this.buffersLock.newCondition();
    private volatile ByteBuffer workingBuffer;
    private volatile ByteBuffer idleBuffer;
    private volatile ByteBuffer syncingBuffer;
    protected volatile WALFileStatus currentFileStatus = WALFileStatus.CONTAINS_NONE_SEARCH_INDEX;
    private final ExecutorService serializeThread;
    private final ExecutorService syncBufferThread;

    public WALBuffer(String identifier, String logDirectory) throws FileNotFoundException {
        this(identifier, logDirectory, 0L, 0L);
    }

    public WALBuffer(String identifier, String logDirectory, long startFileVersion, long startSearchIndex) throws FileNotFoundException {
        super(identifier, logDirectory, startFileVersion, startSearchIndex);
        this.allocateBuffers();
        this.serializeThread = IoTDBThreadPoolFactory.newSingleThreadExecutor((String)(ThreadName.WAL_SERIALIZE.getName() + "(node-" + identifier + ")"));
        this.syncBufferThread = IoTDBThreadPoolFactory.newSingleThreadExecutor((String)(ThreadName.WAL_SYNC.getName() + "(node-" + identifier + ")"));
        this.serializeThread.submit(new SerializeTask());
    }

    private void allocateBuffers() {
        try {
            this.workingBuffer = ByteBuffer.allocateDirect(HALF_WAL_BUFFER_SIZE);
            this.idleBuffer = ByteBuffer.allocateDirect(HALF_WAL_BUFFER_SIZE);
        }
        catch (OutOfMemoryError e) {
            logger.error("Fail to allocate wal node-{}'s buffer because out of memory.", (Object)this.identifier, (Object)e);
            this.close();
            throw e;
        }
    }

    @Override
    public void write(WALEntry walEntry) {
        if (this.isClosed) {
            logger.error("Fail to write WALEntry into wal node-{} because this node is closed.", (Object)this.identifier);
            walEntry.getWalFlushListener().fail((Exception)((Object)new WALNodeClosedException(this.identifier)));
            return;
        }
        try {
            this.walEntries.put(walEntry);
        }
        catch (InterruptedException e) {
            logger.warn("Interrupted when waiting for adding WALEntry to buffer.");
            Thread.currentThread().interrupt();
        }
    }

    private void syncWorkingBuffer(long searchIndex, WALFileStatus fileStatus) {
        this.switchWorkingBufferToFlushing();
        this.syncBufferThread.submit(new SyncBufferTask(searchIndex, fileStatus, false));
        this.currentFileStatus = WALFileStatus.CONTAINS_NONE_SEARCH_INDEX;
    }

    private void fsyncWorkingBuffer(long searchIndex, WALFileStatus fileStatus, SerializeInfo info) {
        this.switchWorkingBufferToFlushing();
        this.syncBufferThread.submit(new SyncBufferTask(searchIndex, fileStatus, true, info));
        this.currentFileStatus = WALFileStatus.CONTAINS_NONE_SEARCH_INDEX;
    }

    private void switchWorkingBufferToFlushing() {
        this.buffersLock.lock();
        try {
            while (this.idleBuffer == null) {
                this.idleBufferReadyCondition.await();
            }
            this.syncingBuffer = this.workingBuffer;
            this.workingBuffer = this.idleBuffer;
            this.workingBuffer.clear();
            this.idleBuffer = null;
        }
        catch (InterruptedException e) {
            logger.warn("Interrupted When waiting for available working buffer.");
            Thread.currentThread().interrupt();
        }
        finally {
            this.buffersLock.unlock();
        }
    }

    private void switchSyncingBufferToIdle() {
        this.buffersLock.lock();
        try {
            this.idleBuffer = this.syncingBuffer;
            this.syncingBuffer = null;
            this.idleBufferReadyCondition.signalAll();
        }
        finally {
            this.buffersLock.unlock();
        }
    }

    @Override
    public void waitForFlush() throws InterruptedException {
        this.buffersLock.lock();
        try {
            this.idleBufferReadyCondition.await();
        }
        finally {
            this.buffersLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean waitForFlush(long time, TimeUnit unit) throws InterruptedException {
        this.buffersLock.lock();
        try {
            boolean bl = this.idleBufferReadyCondition.await(time, unit);
            return bl;
        }
        finally {
            this.buffersLock.unlock();
        }
    }

    @Override
    public void close() {
        this.isClosed = true;
        if (this.serializeThread != null) {
            try {
                this.walEntries.put(new WALSignalEntry(WALEntryType.CLOSE_SIGNAL));
            }
            catch (InterruptedException e) {
                logger.error("Fail to put CLOSE_SIGNAL to walEntries.", (Throwable)e);
            }
            this.shutdownThread(this.serializeThread, ThreadName.WAL_SERIALIZE);
        }
        if (this.syncBufferThread != null) {
            this.shutdownThread(this.syncBufferThread, ThreadName.WAL_SYNC);
        }
        if (this.currentWALFileWriter != null) {
            try {
                this.currentWALFileWriter.close();
            }
            catch (IOException e) {
                logger.error("Fail to close wal node-{}'s log writer.", (Object)this.identifier, (Object)e);
            }
        }
        if (this.workingBuffer != null) {
            MmapUtil.clean((MappedByteBuffer)this.workingBuffer);
        }
        if (this.idleBuffer != null) {
            MmapUtil.clean((MappedByteBuffer)this.workingBuffer);
        }
        if (this.syncingBuffer != null) {
            MmapUtil.clean((MappedByteBuffer)this.syncingBuffer);
        }
    }

    private void shutdownThread(ExecutorService thread, ThreadName threadName) {
        thread.shutdown();
        try {
            if (!thread.awaitTermination(30L, TimeUnit.SECONDS)) {
                logger.warn("Waiting thread {} to be terminated is timeout", (Object)threadName.getName());
            }
        }
        catch (InterruptedException e) {
            logger.warn("Thread {} still doesn't exit after 30s", (Object)threadName.getName());
            Thread.currentThread().interrupt();
        }
    }

    @Override
    public boolean isAllWALEntriesConsumed() {
        return this.walEntries.isEmpty();
    }

    private class SyncBufferTask
    implements Runnable {
        private final long searchIndex;
        private final WALFileStatus fileStatus;
        private final boolean forceFlag;
        private final SerializeInfo info;

        public SyncBufferTask(long searchIndex, WALFileStatus fileStatus, boolean forceFlag) {
            this(searchIndex, fileStatus, forceFlag, null);
        }

        public SyncBufferTask(long searchIndex, WALFileStatus fileStatus, boolean forceFlag, SerializeInfo info) {
            this.searchIndex = searchIndex;
            this.fileStatus = fileStatus;
            this.forceFlag = forceFlag;
            this.info = info == null ? new SerializeInfo() : info;
        }

        @Override
        public void run() {
            WALBuffer.this.currentWALFileWriter.updateFileStatus(this.fileStatus);
            try {
                WALBuffer.this.currentWALFileWriter.write(WALBuffer.this.syncingBuffer, this.info.metaData);
            }
            catch (Throwable e) {
                logger.error("Fail to sync wal node-{}'s buffer, change system mode to error.", (Object)WALBuffer.this.identifier, (Object)e);
                CommonDescriptor.getInstance().getConfig().setNodeStatus(NodeStatus.Error);
            }
            finally {
                WALBuffer.this.switchSyncingBufferToIdle();
            }
            boolean forceSuccess = false;
            if (this.info.rollWALFileWriterListener != null || this.forceFlag && WALBuffer.this.currentWALFileWriter.size() >= config.getWalFileSizeThresholdInByte()) {
                try {
                    WALBuffer.this.rollLogWriter(this.searchIndex, WALBuffer.this.currentWALFileWriter.getWalFileStatus());
                    forceSuccess = true;
                    if (this.info.rollWALFileWriterListener != null) {
                        this.info.rollWALFileWriterListener.succeed();
                    }
                }
                catch (IOException e) {
                    logger.error("Fail to roll wal node-{}'s log writer, change system mode to error.", (Object)WALBuffer.this.identifier, (Object)e);
                    if (this.info.rollWALFileWriterListener != null) {
                        this.info.rollWALFileWriterListener.fail(e);
                    }
                    CommonDescriptor.getInstance().getConfig().setNodeStatus(NodeStatus.Error);
                }
            } else if (this.forceFlag) {
                try {
                    WALBuffer.this.currentWALFileWriter.force();
                    forceSuccess = true;
                }
                catch (IOException e) {
                    logger.error("Fail to fsync wal node-{}'s log writer, change system mode to error.", (Object)WALBuffer.this.identifier, (Object)e);
                    for (WALFlushListener fsyncListener : this.info.fsyncListeners) {
                        fsyncListener.fail(e);
                    }
                    CommonDescriptor.getInstance().getConfig().setNodeStatus(NodeStatus.Error);
                }
            }
            if (forceSuccess) {
                for (WALFlushListener fsyncListener : this.info.fsyncListeners) {
                    fsyncListener.succeed();
                }
            }
        }
    }

    private class ByteBufferView
    implements IWALByteBufferView {
        private int flushedBytesNum = 0;

        private ByteBufferView() {
        }

        private void ensureEnoughSpace(int bytesNum) {
            if (WALBuffer.this.workingBuffer.remaining() < bytesNum) {
                this.rollBuffer();
            }
        }

        private void rollBuffer() {
            this.flushedBytesNum += WALBuffer.this.workingBuffer.position();
            WALBuffer.this.syncWorkingBuffer(WALBuffer.this.currentSearchIndex, WALBuffer.this.currentFileStatus);
        }

        @Override
        public void put(byte b) {
            this.ensureEnoughSpace(1);
            WALBuffer.this.workingBuffer.put(b);
        }

        @Override
        public void put(byte[] src) {
            int needCapacity;
            int offset = 0;
            while (true) {
                int leftCapacity;
                if ((leftCapacity = WALBuffer.this.workingBuffer.remaining()) >= (needCapacity = src.length - offset)) break;
                WALBuffer.this.workingBuffer.put(src, offset, leftCapacity);
                offset += leftCapacity;
                this.rollBuffer();
            }
            WALBuffer.this.workingBuffer.put(src, offset, needCapacity);
        }

        @Override
        public void putChar(char value) {
            this.ensureEnoughSpace(2);
            WALBuffer.this.workingBuffer.putChar(value);
        }

        @Override
        public void putShort(short value) {
            this.ensureEnoughSpace(2);
            WALBuffer.this.workingBuffer.putShort(value);
        }

        @Override
        public void putInt(int value) {
            this.ensureEnoughSpace(4);
            WALBuffer.this.workingBuffer.putInt(value);
        }

        @Override
        public void putLong(long value) {
            this.ensureEnoughSpace(8);
            WALBuffer.this.workingBuffer.putLong(value);
        }

        @Override
        public void putFloat(float value) {
            this.ensureEnoughSpace(4);
            WALBuffer.this.workingBuffer.putFloat(value);
        }

        @Override
        public void putDouble(double value) {
            this.ensureEnoughSpace(8);
            WALBuffer.this.workingBuffer.putDouble(value);
        }

        @Override
        public int position() {
            return this.flushedBytesNum + WALBuffer.this.workingBuffer.position();
        }
    }

    private class SerializeTask
    implements Runnable {
        private final ByteBufferView byteBufferVew;
        private final SerializeInfo info;
        private int batchSize;

        private SerializeTask() {
            this.byteBufferVew = new ByteBufferView();
            this.info = new SerializeInfo();
            this.batchSize = 0;
        }

        @Override
        public void run() {
            try {
                this.serialize();
            }
            finally {
                if (!WALBuffer.this.isClosed) {
                    WALBuffer.this.serializeThread.submit(new SerializeTask());
                }
            }
        }

        private void serialize() {
            try {
                WALEntry firstWALEntry = (WALEntry)WALBuffer.this.walEntries.take();
                boolean returnFlag = this.handleWALEntry(firstWALEntry);
                if (returnFlag) {
                    return;
                }
            }
            catch (InterruptedException e) {
                logger.warn("Interrupted when waiting for taking WALEntry from blocking queue to serialize.");
                Thread.currentThread().interrupt();
            }
            long fsyncDelay = config.getFsyncWalDelayInMs();
            if (fsyncDelay > 0L) {
                try {
                    Thread.sleep(fsyncDelay);
                }
                catch (InterruptedException e) {
                    logger.warn("Interrupted when sleeping a while to enlarge wal write batch.");
                    Thread.currentThread().interrupt();
                }
            }
            while (WALBuffer.this.walEntries.peek() != null && this.batchSize < QUEUE_CAPACITY) {
                WALEntry walEntry = (WALEntry)WALBuffer.this.walEntries.poll();
                boolean returnFlag = this.handleWALEntry(walEntry);
                if (!returnFlag) continue;
                return;
            }
            if (this.batchSize > 0) {
                WALBuffer.this.fsyncWorkingBuffer(WALBuffer.this.currentSearchIndex, WALBuffer.this.currentFileStatus, this.info);
            }
        }

        private boolean handleWALEntry(WALEntry walEntry) {
            if (walEntry.isSignal()) {
                return this.handleSignalEntry((WALSignalEntry)walEntry);
            }
            boolean success = this.handleInfoEntry(walEntry);
            if (success) {
                ++this.batchSize;
                this.info.fsyncListeners.add(walEntry.getWalFlushListener());
            }
            return false;
        }

        private boolean handleInfoEntry(WALEntry walEntry) {
            int size = this.byteBufferVew.position();
            try {
                walEntry.serialize(this.byteBufferVew);
                size = this.byteBufferVew.position() - size;
                logger.debug("wal entry size is: {}", (Object)size);
            }
            catch (Exception e) {
                logger.error("Fail to serialize WALEntry to wal node-{}'s buffer, discard it.", (Object)WALBuffer.this.identifier, (Object)e);
                walEntry.getWalFlushListener().fail(e);
                return false;
            }
            long searchIndex = -1L;
            if (walEntry.getType().needSearch() && (searchIndex = walEntry.getType() == WALEntryType.DELETE_DATA_NODE ? ((DeleteDataNode)walEntry.getValue()).getSearchIndex() : ((InsertNode)((Object)walEntry.getValue())).getSearchIndex()) != -1L) {
                WALBuffer.this.currentSearchIndex = searchIndex;
                WALBuffer.this.currentFileStatus = WALFileStatus.CONTAINS_SEARCH_INDEX;
            }
            this.info.metaData.add(size, searchIndex);
            return true;
        }

        private boolean handleSignalEntry(WALSignalEntry walSignalEntry) {
            switch (walSignalEntry.getType()) {
                case ROLL_WAL_LOG_WRITER_SIGNAL: {
                    logger.debug("Handle roll log writer signal for wal node-{}.", (Object)WALBuffer.this.identifier);
                    this.info.rollWALFileWriterListener = walSignalEntry.getWalFlushListener();
                    WALBuffer.this.fsyncWorkingBuffer(WALBuffer.this.currentSearchIndex, WALBuffer.this.currentFileStatus, this.info);
                    return true;
                }
                case CLOSE_SIGNAL: {
                    boolean dataExists;
                    logger.debug("Handle close signal for wal node-{}, there are {} entries left.", (Object)WALBuffer.this.identifier, (Object)WALBuffer.this.walEntries.size());
                    boolean bl = dataExists = this.batchSize > 0;
                    if (dataExists) {
                        WALBuffer.this.fsyncWorkingBuffer(WALBuffer.this.currentSearchIndex, WALBuffer.this.currentFileStatus, this.info);
                    }
                    return dataExists;
                }
            }
            return false;
        }
    }

    private static class SerializeInfo {
        final WALMetaData metaData = new WALMetaData();
        final List<WALFlushListener> fsyncListeners = new LinkedList<WALFlushListener>();
        WALFlushListener rollWALFileWriterListener = null;

        private SerializeInfo() {
        }
    }
}

