/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.sync.sender.pipe;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.iotdb.commons.exception.MetadataException;
import org.apache.iotdb.commons.path.PartialPath;
import org.apache.iotdb.db.engine.modification.Deletion;
import org.apache.iotdb.db.exception.sync.PipeException;
import org.apache.iotdb.db.qp.physical.PhysicalPlan;
import org.apache.iotdb.db.sync.conf.SyncPathUtil;
import org.apache.iotdb.db.sync.pipedata.DeletionPipeData;
import org.apache.iotdb.db.sync.pipedata.PipeData;
import org.apache.iotdb.db.sync.pipedata.SchemaPipeData;
import org.apache.iotdb.db.sync.pipedata.TsFilePipeData;
import org.apache.iotdb.db.sync.pipedata.queue.BufferedPipeDataQueue;
import org.apache.iotdb.db.sync.sender.manager.SchemaSyncManager;
import org.apache.iotdb.db.sync.sender.manager.TsFileSyncManager;
import org.apache.iotdb.db.sync.sender.pipe.Pipe;
import org.apache.iotdb.db.sync.sender.pipe.PipeSink;
import org.apache.iotdb.db.sync.sender.recovery.TsFilePipeLogger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TsFilePipe
implements Pipe {
    private static final Logger logger = LoggerFactory.getLogger(TsFilePipe.class);
    private final SchemaSyncManager schemaSyncManager = SchemaSyncManager.getInstance();
    private final TsFileSyncManager tsFileSyncManager = TsFileSyncManager.getInstance();
    private final long createTime;
    private final String name;
    private final PipeSink pipeSink;
    private final long dataStartTime;
    private final boolean syncDelOp;
    private final BufferedPipeDataQueue historyQueue;
    private final BufferedPipeDataQueue realTimeQueue;
    private final TsFilePipeLogger pipeLog;
    private final ReentrantLock collectRealTimeDataLock;
    private boolean isCollectingRealTimeData;
    private long maxSerialNumber;
    private boolean disconnected;
    private Pipe.PipeStatus status;

    public TsFilePipe(long createTime, String name, PipeSink pipeSink, long dataStartTime, boolean syncDelOp) {
        this.createTime = createTime;
        this.name = name;
        this.pipeSink = pipeSink;
        this.dataStartTime = dataStartTime;
        this.syncDelOp = syncDelOp;
        this.historyQueue = new BufferedPipeDataQueue(SyncPathUtil.getSenderHistoryPipeLogDir(name, createTime));
        this.realTimeQueue = new BufferedPipeDataQueue(SyncPathUtil.getSenderRealTimePipeLogDir(name, createTime));
        this.pipeLog = new TsFilePipeLogger(this);
        this.collectRealTimeDataLock = new ReentrantLock();
        this.isCollectingRealTimeData = false;
        this.maxSerialNumber = Math.max(0L, this.realTimeQueue.getLastMaxSerialNumber());
        this.status = Pipe.PipeStatus.STOP;
        this.disconnected = false;
    }

    @Override
    public synchronized void start() throws PipeException {
        if (this.status == Pipe.PipeStatus.DROP) {
            throw new PipeException(String.format("Can not start pipe %s, because the pipe has been drop.", this.name));
        }
        if (this.status == Pipe.PipeStatus.RUNNING) {
            return;
        }
        try {
            if (!this.pipeLog.isCollectFinished()) {
                this.pipeLog.clear();
                this.collectData();
                this.pipeLog.finishCollect();
            }
            if (!this.isCollectingRealTimeData) {
                this.registerMetadata();
                this.registerTsFile();
                this.isCollectingRealTimeData = true;
            }
            this.status = Pipe.PipeStatus.RUNNING;
        }
        catch (IOException e) {
            logger.error(String.format("Clear pipe dir %s error.", SyncPathUtil.getSenderPipeDir(this.name, this.createTime)), (Throwable)e);
            throw new PipeException("Start error, can not clear pipe log.");
        }
    }

    private void collectData() {
        long serialNumber;
        int i;
        this.registerMetadata();
        List<PhysicalPlan> historyMetadata = this.collectHistoryMetadata();
        List<File> historyTsFiles = this.registerAndCollectHistoryTsFile();
        this.isCollectingRealTimeData = true;
        int historyMetadataSize = historyMetadata.size();
        int historyTsFilesSize = historyTsFiles.size();
        for (i = 0; i < historyMetadataSize; ++i) {
            serialNumber = 1 - historyTsFilesSize - historyMetadataSize + i;
            this.historyQueue.offer(new SchemaPipeData(historyMetadata.get(i), serialNumber));
        }
        for (i = 0; i < historyTsFilesSize; ++i) {
            serialNumber = 1 - historyTsFilesSize + i;
            File tsFile = historyTsFiles.get(i);
            this.historyQueue.offer(new TsFilePipeData(tsFile.getParent(), tsFile.getName(), serialNumber));
        }
    }

    private void registerMetadata() {
        this.schemaSyncManager.registerSyncTask(this);
    }

    private void deregisterMetadata() {
        this.schemaSyncManager.deregisterSyncTask();
    }

    private List<PhysicalPlan> collectHistoryMetadata() {
        return this.schemaSyncManager.collectHistoryMetadata();
    }

    public void collectRealTimeMetaData(PhysicalPlan plan) {
        this.collectRealTimeDataLock.lock();
        try {
            ++this.maxSerialNumber;
            SchemaPipeData metaData = new SchemaPipeData(plan, this.maxSerialNumber);
            this.realTimeQueue.offer(metaData);
        }
        finally {
            this.collectRealTimeDataLock.unlock();
        }
    }

    private void registerTsFile() {
        this.tsFileSyncManager.registerSyncTask(this);
    }

    private void deregisterTsFile() {
        this.tsFileSyncManager.deregisterSyncTask();
    }

    private List<File> registerAndCollectHistoryTsFile() {
        return this.tsFileSyncManager.registerAndCollectHistoryTsFile(this, this.dataStartTime);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public File createHistoryTsFileHardlink(File tsFile, long modsOffset) {
        this.collectRealTimeDataLock.lock();
        try {
            if (this.pipeLog.isHardlinkExist(tsFile)) {
                File file = null;
                return file;
            }
            File file = this.pipeLog.createTsFileAndModsHardlink(tsFile, modsOffset);
            return file;
        }
        catch (IOException e) {
            logger.error(String.format("Create hardlink for history tsfile %s error.", tsFile.getPath()), (Throwable)e);
            File file = null;
            return file;
        }
        finally {
            this.collectRealTimeDataLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void collectRealTimeDeletion(Deletion deletion) {
        this.collectRealTimeDataLock.lock();
        try {
            if (!this.syncDelOp) {
                return;
            }
            for (PartialPath deletePath : this.schemaSyncManager.splitPathPatternByDevice(deletion.getPath())) {
                Deletion splitDeletion = new Deletion(deletePath, deletion.getFileOffset(), deletion.getStartTime(), deletion.getEndTime());
                ++this.maxSerialNumber;
                DeletionPipeData deletionData = new DeletionPipeData(splitDeletion, this.maxSerialNumber);
                this.realTimeQueue.offer(deletionData);
            }
        }
        catch (MetadataException e) {
            logger.warn(String.format("Collect deletion %s error.", deletion), (Throwable)e);
        }
        finally {
            this.collectRealTimeDataLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void collectRealTimeTsFile(File tsFile) {
        this.collectRealTimeDataLock.lock();
        try {
            if (this.pipeLog.isHardlinkExist(tsFile)) {
                return;
            }
            ++this.maxSerialNumber;
            File hardlink = this.pipeLog.createTsFileHardlink(tsFile);
            TsFilePipeData tsFileData = new TsFilePipeData(hardlink.getParent(), hardlink.getName(), this.maxSerialNumber);
            this.realTimeQueue.offer(tsFileData);
        }
        catch (IOException e) {
            logger.warn(String.format("Create Hardlink tsfile %s on disk error, serial number is %d.", tsFile.getPath(), this.maxSerialNumber), (Throwable)e);
        }
        finally {
            this.collectRealTimeDataLock.unlock();
        }
    }

    public void collectRealTimeResource(File tsFile) {
        try {
            this.pipeLog.createTsFileResourceHardlink(tsFile);
        }
        catch (IOException e) {
            logger.warn(String.format("Record tsfile resource %s on disk error.", tsFile.getPath()), (Throwable)e);
        }
    }

    @Override
    public PipeData take() throws InterruptedException {
        if (!this.historyQueue.isEmpty()) {
            return this.historyQueue.take();
        }
        return this.realTimeQueue.take();
    }

    public List<PipeData> pull(long serialNumber) {
        ArrayList<PipeData> pullPipeData = new ArrayList<PipeData>();
        if (!this.historyQueue.isEmpty()) {
            pullPipeData.addAll(this.historyQueue.pull(serialNumber));
        }
        if (serialNumber > 0L) {
            pullPipeData.addAll(this.realTimeQueue.pull(serialNumber));
        }
        return pullPipeData;
    }

    @Override
    public void commit() {
        if (!this.historyQueue.isEmpty()) {
            this.historyQueue.commit();
        }
        this.realTimeQueue.commit();
    }

    @Override
    public void setDisconnected(boolean disconnected) {
        this.disconnected = disconnected;
    }

    @Override
    public boolean isDisconnected() {
        return this.disconnected;
    }

    public void commit(long serialNumber) {
        if (!this.historyQueue.isEmpty()) {
            this.historyQueue.commit(serialNumber);
        }
        if (serialNumber > 0L) {
            this.realTimeQueue.commit(serialNumber);
        }
    }

    @Override
    public synchronized void stop() throws PipeException {
        if (this.status == Pipe.PipeStatus.DROP) {
            throw new PipeException(String.format("Can not stop pipe %s, because the pipe is drop.", this.name));
        }
        if (!this.isCollectingRealTimeData) {
            this.registerMetadata();
            this.registerTsFile();
            this.isCollectingRealTimeData = true;
        }
        this.status = Pipe.PipeStatus.STOP;
    }

    @Override
    public synchronized void drop() throws PipeException {
        if (this.status == Pipe.PipeStatus.DROP) {
            return;
        }
        this.clear();
        this.status = Pipe.PipeStatus.DROP;
    }

    private void clear() {
        this.deregisterMetadata();
        this.deregisterTsFile();
        this.isCollectingRealTimeData = false;
        try {
            this.historyQueue.clear();
            this.realTimeQueue.clear();
            this.pipeLog.clear();
        }
        catch (IOException e) {
            logger.warn(String.format("Clear pipe %s %d error.", this.name, this.createTime), (Throwable)e);
        }
    }

    @Override
    public void close() throws PipeException {
        if (this.status == Pipe.PipeStatus.DROP) {
            return;
        }
        this.deregisterMetadata();
        this.deregisterTsFile();
        this.isCollectingRealTimeData = false;
        this.historyQueue.close();
        this.realTimeQueue.close();
    }

    @Override
    public String getName() {
        return this.name;
    }

    @Override
    public PipeSink getPipeSink() {
        return this.pipeSink;
    }

    @Override
    public long getCreateTime() {
        return this.createTime;
    }

    @Override
    public synchronized Pipe.PipeStatus getStatus() {
        return this.status;
    }

    public String toString() {
        return "TsFilePipe{createTime=" + this.createTime + ", name='" + this.name + '\'' + ", pipeSink=" + this.pipeSink + ", dataStartTime=" + this.dataStartTime + ", syncDelOp=" + this.syncDelOp + ", pipeLog=" + this.pipeLog + ", isCollectingRealTimeData=" + this.isCollectingRealTimeData + ", maxSerialNumber=" + this.maxSerialNumber + ", status=" + (Object)((Object)this.status) + '}';
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        TsFilePipe that = (TsFilePipe)o;
        return this.createTime == that.createTime && Objects.equals(this.name, that.name) && Objects.equals(this.pipeSink, that.pipeSink);
    }

    public int hashCode() {
        return Objects.hash(this.createTime, this.name, this.pipeSink);
    }
}

