/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.engine;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.apache.commons.io.FileUtils;
import org.apache.iotdb.db.concurrent.IoTDBThreadPoolFactory;
import org.apache.iotdb.db.conf.IoTDBConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.engine.fileSystem.SystemFileFactory;
import org.apache.iotdb.db.engine.flush.CloseFileListener;
import org.apache.iotdb.db.engine.flush.FlushListener;
import org.apache.iotdb.db.engine.flush.TsFileFlushPolicy;
import org.apache.iotdb.db.engine.storagegroup.StorageGroupProcessor;
import org.apache.iotdb.db.engine.storagegroup.TsFileProcessor;
import org.apache.iotdb.db.engine.storagegroup.TsFileResource;
import org.apache.iotdb.db.engine.storagegroup.virtualSg.VirtualStorageGroupManager;
import org.apache.iotdb.db.exception.BatchProcessException;
import org.apache.iotdb.db.exception.LoadFileException;
import org.apache.iotdb.db.exception.ShutdownException;
import org.apache.iotdb.db.exception.StorageEngineException;
import org.apache.iotdb.db.exception.StorageGroupNotReadyException;
import org.apache.iotdb.db.exception.StorageGroupProcessorException;
import org.apache.iotdb.db.exception.TsFileProcessorException;
import org.apache.iotdb.db.exception.WriteProcessException;
import org.apache.iotdb.db.exception.WriteProcessRejectException;
import org.apache.iotdb.db.exception.metadata.IllegalPathException;
import org.apache.iotdb.db.exception.metadata.MetadataException;
import org.apache.iotdb.db.exception.metadata.StorageGroupNotSetException;
import org.apache.iotdb.db.exception.query.QueryProcessException;
import org.apache.iotdb.db.exception.runtime.StorageEngineFailureException;
import org.apache.iotdb.db.metadata.PartialPath;
import org.apache.iotdb.db.metadata.mnode.StorageGroupMNode;
import org.apache.iotdb.db.monitor.StatMonitor;
import org.apache.iotdb.db.qp.physical.crud.InsertPlan;
import org.apache.iotdb.db.qp.physical.crud.InsertRowPlan;
import org.apache.iotdb.db.qp.physical.crud.InsertRowsOfOneDevicePlan;
import org.apache.iotdb.db.qp.physical.crud.InsertTabletPlan;
import org.apache.iotdb.db.rescon.SystemInfo;
import org.apache.iotdb.db.service.IService;
import org.apache.iotdb.db.service.IoTDB;
import org.apache.iotdb.db.service.ServiceType;
import org.apache.iotdb.db.utils.ThreadUtils;
import org.apache.iotdb.db.utils.UpgradeUtils;
import org.apache.iotdb.rpc.RpcUtils;
import org.apache.iotdb.rpc.TSStatusCode;
import org.apache.iotdb.service.rpc.thrift.TSStatus;
import org.apache.iotdb.tsfile.utils.FilePathUtils;
import org.apache.iotdb.tsfile.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StorageEngine
implements IService {
    private static final Logger logger = LoggerFactory.getLogger(StorageEngine.class);
    private static final IoTDBConfig config = IoTDBDescriptor.getInstance().getConfig();
    private static final long TTL_CHECK_INTERVAL = 60000L;
    private static long timePartitionInterval = -1L;
    private static boolean enablePartition = config.isEnablePartition();
    private final boolean enableMemControl = config.isEnableMemControl();
    private final String systemDir;
    private final ConcurrentHashMap<PartialPath, VirtualStorageGroupManager> processorMap = new ConcurrentHashMap();
    private AtomicBoolean isAllSgReady = new AtomicBoolean(false);
    private ExecutorService recoverAllSgThreadPool;
    private ScheduledExecutorService ttlCheckThread;
    private ScheduledExecutorService seqMemtableTimedFlushCheckThread;
    private ScheduledExecutorService unseqMemtableTimedFlushCheckThread;
    private ScheduledExecutorService tsFileTimedCloseCheckThread;
    private TsFileFlushPolicy fileFlushPolicy = new TsFileFlushPolicy.DirectFlushPolicy();
    private ExecutorService recoveryThreadPool;
    private List<CloseFileListener> customCloseFileListeners = new ArrayList<CloseFileListener>();
    private List<FlushListener> customFlushListeners = new ArrayList<FlushListener>();

    private StorageEngine() {
        this.systemDir = FilePathUtils.regularizePath((String)config.getSystemDir()) + "storage_groups";
        if (!enablePartition) {
            timePartitionInterval = Long.MAX_VALUE;
        } else {
            StorageEngine.initTimePartition();
        }
        try {
            FileUtils.forceMkdir((File)SystemFileFactory.INSTANCE.getFile(this.systemDir));
        }
        catch (IOException e) {
            throw new StorageEngineFailureException(e);
        }
        UpgradeUtils.recoverUpgrade();
        this.recover();
    }

    public static StorageEngine getInstance() {
        return InstanceHolder.INSTANCE;
    }

    private static void initTimePartition() {
        timePartitionInterval = StorageEngine.convertMilliWithPrecision(IoTDBDescriptor.getInstance().getConfig().getPartitionInterval() * 1000L);
    }

    public static long convertMilliWithPrecision(long milliTime) {
        String timePrecision;
        long result = milliTime;
        switch (timePrecision = IoTDBDescriptor.getInstance().getConfig().getTimestampPrecision()) {
            case "ns": {
                result = milliTime * 1000000L;
                break;
            }
            case "us": {
                result = milliTime * 1000L;
                break;
            }
        }
        return result;
    }

    public static long getTimePartitionInterval() {
        if (timePartitionInterval == -1L) {
            StorageEngine.initTimePartition();
        }
        return timePartitionInterval;
    }

    public static void setTimePartitionInterval(long timePartitionInterval) {
        StorageEngine.timePartitionInterval = timePartitionInterval;
    }

    public static long getTimePartition(long time) {
        return enablePartition ? time / timePartitionInterval : 0L;
    }

    public static boolean isEnablePartition() {
        return enablePartition;
    }

    public static void setEnablePartition(boolean enablePartition) {
        StorageEngine.enablePartition = enablePartition;
    }

    public static void blockInsertionIfReject(TsFileProcessor tsFileProcessor) throws WriteProcessRejectException {
        long startTime = System.currentTimeMillis();
        while (SystemInfo.getInstance().isRejected() && (tsFileProcessor == null || !tsFileProcessor.shouldFlush())) {
            try {
                TimeUnit.MILLISECONDS.sleep(config.getCheckPeriodWhenInsertBlocked());
                if (System.currentTimeMillis() - startTime <= (long)config.getMaxWaitingTimeWhenInsertBlocked()) continue;
                throw new WriteProcessRejectException("System rejected over " + (System.currentTimeMillis() - startTime) + "ms");
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

    public boolean isAllSgReady() {
        return this.isAllSgReady.get();
    }

    public void setAllSgReady(boolean allSgReady) {
        this.isAllSgReady.set(allSgReady);
    }

    public void recover() {
        this.setAllSgReady(false);
        this.recoveryThreadPool = IoTDBThreadPoolFactory.newFixedThreadPool(Runtime.getRuntime().availableProcessors(), "Recovery-Thread-Pool");
        this.recoverAllSgThreadPool = IoTDBThreadPoolFactory.newSingleThreadExecutor("Begin-Recovery-Pool");
        this.recoverAllSgThreadPool.submit(this::recoverAllSgs);
    }

    private void recoverAllSgs() {
        ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>();
        this.recoverStorageGroupProcessor(futures);
        for (Future future : futures) {
            try {
                future.get();
            }
            catch (ExecutionException e) {
                throw new StorageEngineFailureException("StorageEngine failed to recover.", e);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new StorageEngineFailureException("StorageEngine failed to recover.", e);
            }
        }
        this.recoveryThreadPool.shutdown();
        this.setAllSgReady(true);
    }

    private void recoverStorageGroupProcessor(List<Future<Void>> futures) {
        List<StorageGroupMNode> sgNodes = IoTDB.metaManager.getAllStorageGroupNodes();
        for (StorageGroupMNode storageGroup : sgNodes) {
            futures.add(this.recoveryThreadPool.submit(() -> {
                try {
                    VirtualStorageGroupManager virtualStorageGroupManager = this.processorMap.computeIfAbsent(storageGroup.getPartialPath(), id -> new VirtualStorageGroupManager());
                    virtualStorageGroupManager.recover(storageGroup);
                    logger.info("Storage Group Processor {} is recovered successfully", (Object)storageGroup.getFullPath());
                }
                catch (Exception e) {
                    logger.error("meet error when recovered storage group: {}", (Object)storageGroup.getFullPath(), (Object)e);
                }
                return null;
            }));
        }
    }

    @Override
    public void start() {
        this.ttlCheckThread = IoTDBThreadPoolFactory.newSingleThreadScheduledExecutor("TTL");
        this.ttlCheckThread.scheduleAtFixedRate(this::checkTTL, 60000L, 60000L, TimeUnit.MILLISECONDS);
        logger.info("start ttl check thread successfully.");
        this.startTimedService();
    }

    private void checkTTL() {
        try {
            for (VirtualStorageGroupManager processor : this.processorMap.values()) {
                processor.checkTTL();
            }
        }
        catch (ConcurrentModificationException concurrentModificationException) {
        }
        catch (Exception e) {
            logger.error("An error occurred when checking TTL", (Throwable)e);
        }
        if (this.isAllSgReady.get() && !this.recoverAllSgThreadPool.isShutdown()) {
            this.recoverAllSgThreadPool.shutdownNow();
        }
    }

    private void startTimedService() {
        if (config.isEnableTimedFlushSeqMemtable()) {
            this.seqMemtableTimedFlushCheckThread = IoTDBThreadPoolFactory.newSingleThreadScheduledExecutor("TimedFlushSeqMemtable");
            this.seqMemtableTimedFlushCheckThread.scheduleAtFixedRate(this::timedFlushSeqMemTable, config.getSeqMemtableFlushCheckInterval(), config.getSeqMemtableFlushCheckInterval(), TimeUnit.MILLISECONDS);
            logger.info("start sequence memtable timed flush check thread successfully.");
        }
        if (config.isEnableTimedFlushUnseqMemtable()) {
            this.unseqMemtableTimedFlushCheckThread = IoTDBThreadPoolFactory.newSingleThreadScheduledExecutor("TimedFlushUnSeqMemtable");
            this.unseqMemtableTimedFlushCheckThread.scheduleAtFixedRate(this::timedFlushUnseqMemTable, config.getUnseqMemtableFlushCheckInterval(), config.getUnseqMemtableFlushCheckInterval(), TimeUnit.MILLISECONDS);
            logger.info("start unsequence memtable timed flush check thread successfully.");
        }
        if (config.isEnableTimedCloseTsFile()) {
            this.tsFileTimedCloseCheckThread = IoTDBThreadPoolFactory.newSingleThreadScheduledExecutor("TimedCloseTsFileProcessor");
            this.tsFileTimedCloseCheckThread.scheduleAtFixedRate(this::timedCloseTsFileProcessor, config.getCloseTsFileCheckInterval(), config.getCloseTsFileCheckInterval(), TimeUnit.MILLISECONDS);
            logger.info("start tsfile timed close check thread successfully.");
        }
    }

    private void timedFlushSeqMemTable() {
        try {
            for (VirtualStorageGroupManager processor : this.processorMap.values()) {
                processor.timedFlushSeqMemTable();
            }
        }
        catch (Exception e) {
            logger.error("An error occurred when timed flushing sequence memtables", (Throwable)e);
        }
    }

    private void timedFlushUnseqMemTable() {
        try {
            for (VirtualStorageGroupManager processor : this.processorMap.values()) {
                processor.timedFlushUnseqMemTable();
            }
        }
        catch (Exception e) {
            logger.error("An error occurred when timed flushing unsequence memtables", (Throwable)e);
        }
    }

    private void timedCloseTsFileProcessor() {
        try {
            for (VirtualStorageGroupManager processor : this.processorMap.values()) {
                processor.timedCloseTsFileProcessor();
            }
        }
        catch (Exception e) {
            logger.error("An error occurred when timed closing tsfiles interval", (Throwable)e);
        }
    }

    @Override
    public void stop() {
        for (VirtualStorageGroupManager storageGroupManager : this.processorMap.values()) {
            storageGroupManager.stopSchedulerPool();
        }
        this.syncCloseAllProcessor();
        ThreadUtils.stopThreadPool(this.ttlCheckThread, "TTlCheckThread");
        ThreadUtils.stopThreadPool(this.seqMemtableTimedFlushCheckThread, "SeqMemtableTimedFlushCheckThread");
        ThreadUtils.stopThreadPool(this.unseqMemtableTimedFlushCheckThread, "UnseqMemtableTimedFlushCheckThread");
        ThreadUtils.stopThreadPool(this.tsFileTimedCloseCheckThread, "TsFileTimedCloseCheckThread");
        this.recoveryThreadPool.shutdownNow();
        if (!this.recoverAllSgThreadPool.isShutdown()) {
            this.recoverAllSgThreadPool.shutdownNow();
            try {
                this.recoverAllSgThreadPool.awaitTermination(60L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                logger.warn("recoverAllSgThreadPool thread still doesn't exit after 60s");
                Thread.currentThread().interrupt();
                throw new StorageEngineFailureException("StorageEngine failed to stop because of recoverAllSgThreadPool.", e);
            }
        }
        for (PartialPath storageGroup : IoTDB.metaManager.getAllStorageGroupPaths()) {
            this.releaseWalDirectByteBufferPoolInOneStorageGroup(storageGroup);
        }
        this.reset();
    }

    @Override
    public void shutdown(long milliseconds) throws ShutdownException {
        try {
            for (VirtualStorageGroupManager storageGroupManager : this.processorMap.values()) {
                storageGroupManager.stopSchedulerPool();
            }
            this.forceCloseAllProcessor();
        }
        catch (TsFileProcessorException e) {
            throw new ShutdownException(e);
        }
        this.shutdownTimedService(this.ttlCheckThread, "TTlCheckThread");
        this.shutdownTimedService(this.seqMemtableTimedFlushCheckThread, "SeqMemtableTimedFlushCheckThread");
        this.shutdownTimedService(this.unseqMemtableTimedFlushCheckThread, "UnseqMemtableTimedFlushCheckThread");
        this.shutdownTimedService(this.tsFileTimedCloseCheckThread, "TsFileTimedCloseCheckThread");
        this.recoveryThreadPool.shutdownNow();
        this.reset();
    }

    private void shutdownTimedService(ScheduledExecutorService pool, String poolName) {
        if (pool != null) {
            pool.shutdownNow();
            try {
                pool.awaitTermination(30L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                logger.warn("{} still doesn't exit after 30s", (Object)poolName);
                Thread.currentThread().interrupt();
            }
        }
    }

    public void rebootTimedService() throws ShutdownException {
        logger.info("Start rebooting all timed service.");
        this.stopTimedServiceAndThrow(this.seqMemtableTimedFlushCheckThread, "SeqMemtableTimedFlushCheckThread");
        this.stopTimedServiceAndThrow(this.unseqMemtableTimedFlushCheckThread, "UnseqMemtableTimedFlushCheckThread");
        this.stopTimedServiceAndThrow(this.tsFileTimedCloseCheckThread, "TsFileTimedCloseCheckThread");
        logger.info("Stop all timed service successfully, and now restart them.");
        this.startTimedService();
        logger.info("Reboot all timed service successfully");
    }

    private void stopTimedServiceAndThrow(ScheduledExecutorService pool, String poolName) throws ShutdownException {
        if (pool != null) {
            pool.shutdownNow();
            try {
                pool.awaitTermination(30L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                logger.warn("{} still doesn't exit after 30s", (Object)poolName);
                throw new ShutdownException(e);
            }
        }
    }

    @Override
    public ServiceType getID() {
        return ServiceType.STORAGE_ENGINE_SERVICE;
    }

    public StorageGroupProcessor getProcessorDirectly(PartialPath path) throws StorageEngineException {
        try {
            StorageGroupMNode storageGroupMNode = IoTDB.metaManager.getStorageGroupNodeByPath(path);
            PartialPath storageGroupPath = storageGroupMNode.getPartialPath();
            return this.getStorageGroupProcessorByPath(storageGroupPath, storageGroupMNode);
        }
        catch (StorageGroupProcessorException | MetadataException e) {
            throw new StorageEngineException(e);
        }
    }

    public StorageGroupProcessor getProcessorDirectly(PartialPath path, int virtualStorageGroupId) throws StorageEngineException {
        try {
            StorageGroupMNode storageGroupMNode = IoTDB.metaManager.getStorageGroupNodeByPath(path);
            return this.getStorageGroupProcessorByPath(storageGroupMNode, virtualStorageGroupId);
        }
        catch (StorageGroupProcessorException | MetadataException e) {
            throw new StorageEngineException(e);
        }
    }

    public StorageGroupProcessor getProcessor(PartialPath deviceId) throws StorageEngineException {
        try {
            StorageGroupMNode storageGroupMNode = IoTDB.metaManager.getStorageGroupNodeByPath(deviceId);
            return this.getStorageGroupProcessorByPath(deviceId, storageGroupMNode);
        }
        catch (StorageGroupProcessorException | MetadataException e) {
            throw new StorageEngineException(e);
        }
    }

    public List<String> getLockInfo(List<PartialPath> pathList) throws StorageEngineException {
        try {
            ArrayList<String> lockHolderList = new ArrayList<String>(pathList.size());
            for (PartialPath path : pathList) {
                StorageGroupMNode storageGroupMNode = IoTDB.metaManager.getStorageGroupNodeByPath(path);
                StorageGroupProcessor storageGroupProcessor = this.getStorageGroupProcessorByPath(path, storageGroupMNode);
                lockHolderList.add(storageGroupProcessor.getInsertWriteLockHolder());
            }
            return lockHolderList;
        }
        catch (StorageGroupProcessorException | MetadataException e) {
            throw new StorageEngineException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private StorageGroupProcessor getStorageGroupProcessorByPath(PartialPath devicePath, StorageGroupMNode storageGroupMNode) throws StorageGroupProcessorException, StorageEngineException {
        VirtualStorageGroupManager virtualStorageGroupManager = this.processorMap.get(storageGroupMNode.getPartialPath());
        if (virtualStorageGroupManager == null) {
            if (this.isAllSgReady.get()) {
                StorageEngine storageEngine = this;
                synchronized (storageEngine) {
                    virtualStorageGroupManager = this.processorMap.get(storageGroupMNode.getPartialPath());
                    if (virtualStorageGroupManager == null) {
                        virtualStorageGroupManager = new VirtualStorageGroupManager();
                        this.processorMap.put(storageGroupMNode.getPartialPath(), virtualStorageGroupManager);
                    }
                }
            } else {
                throw new StorageGroupNotReadyException(storageGroupMNode.getFullPath(), TSStatusCode.STORAGE_GROUP_NOT_READY.getStatusCode());
            }
        }
        return virtualStorageGroupManager.getProcessor(devicePath, storageGroupMNode);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private StorageGroupProcessor getStorageGroupProcessorByPath(StorageGroupMNode storageGroupMNode, int virtualStorageGroupId) throws StorageGroupProcessorException, StorageEngineException {
        VirtualStorageGroupManager virtualStorageGroupManager = this.processorMap.get(storageGroupMNode.getPartialPath());
        if (virtualStorageGroupManager == null) {
            if (this.isAllSgReady.get()) {
                StorageEngine storageEngine = this;
                synchronized (storageEngine) {
                    virtualStorageGroupManager = this.processorMap.get(storageGroupMNode.getPartialPath());
                    if (virtualStorageGroupManager == null) {
                        virtualStorageGroupManager = new VirtualStorageGroupManager();
                        this.processorMap.put(storageGroupMNode.getPartialPath(), virtualStorageGroupManager);
                    }
                }
            } else {
                throw new StorageGroupNotReadyException(storageGroupMNode.getFullPath(), TSStatusCode.STORAGE_GROUP_NOT_READY.getStatusCode());
            }
        }
        return virtualStorageGroupManager.getProcessor(virtualStorageGroupId, storageGroupMNode);
    }

    public StorageGroupProcessor buildNewStorageGroupProcessor(PartialPath logicalStorageGroupName, StorageGroupMNode storageGroupMNode, String virtualStorageGroupId) throws StorageGroupProcessorException {
        logger.info("construct a processor instance, the storage group is {}, Thread is {}", (Object)logicalStorageGroupName, (Object)Thread.currentThread().getId());
        StorageGroupProcessor processor = new StorageGroupProcessor(this.systemDir + File.separator + logicalStorageGroupName, virtualStorageGroupId, this.fileFlushPolicy, storageGroupMNode.getFullPath());
        processor.setDataTTL(storageGroupMNode.getDataTTL());
        processor.setCustomFlushListeners(this.customFlushListeners);
        processor.setCustomCloseFileListeners(this.customCloseFileListeners);
        return processor;
    }

    private void waitAllSgReady(PartialPath storageGroupPath) throws StorageEngineException {
        if (this.isAllSgReady.get()) {
            return;
        }
        long waitStart = System.currentTimeMillis();
        long waited = 0L;
        long MAX_WAIT = 5000L;
        while (!this.isAllSgReady.get() && waited < 5000L) {
            try {
                Thread.sleep(10L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new StorageEngineException(e);
            }
            waited = System.currentTimeMillis() - waitStart;
        }
        if (!this.isAllSgReady.get()) {
            throw new StorageEngineException("the sg " + storageGroupPath + " may not ready now, please wait and retry later", TSStatusCode.STORAGE_GROUP_NOT_READY.getStatusCode());
        }
    }

    public synchronized void reset() {
        for (VirtualStorageGroupManager virtualStorageGroupManager : this.processorMap.values()) {
            virtualStorageGroupManager.reset();
        }
    }

    public void insert(InsertRowPlan insertRowPlan) throws StorageEngineException {
        if (this.enableMemControl) {
            try {
                StorageEngine.blockInsertionIfReject(null);
            }
            catch (WriteProcessException e) {
                throw new StorageEngineException(e);
            }
        }
        StorageGroupProcessor storageGroupProcessor = this.getProcessor(insertRowPlan.getDeviceId());
        try {
            storageGroupProcessor.insert(insertRowPlan);
            if (config.isEnableStatMonitor()) {
                try {
                    StorageGroupMNode storageGroupMNode = IoTDB.metaManager.getStorageGroupNodeByPath(insertRowPlan.getDeviceId());
                    this.updateMonitorStatistics(this.processorMap.get(storageGroupMNode.getPartialPath()), insertRowPlan);
                }
                catch (MetadataException e) {
                    logger.error("failed to record status", (Throwable)e);
                }
            }
        }
        catch (WriteProcessException e) {
            throw new StorageEngineException(e);
        }
    }

    public void insert(InsertRowsOfOneDevicePlan insertRowsOfOneDevicePlan) throws StorageEngineException {
        if (this.enableMemControl) {
            try {
                StorageEngine.blockInsertionIfReject(null);
            }
            catch (WriteProcessException e) {
                throw new StorageEngineException(e);
            }
        }
        StorageGroupProcessor storageGroupProcessor = this.getProcessor(insertRowsOfOneDevicePlan.getDeviceId());
        try {
            storageGroupProcessor.insert(insertRowsOfOneDevicePlan);
        }
        catch (WriteProcessException e) {
            throw new StorageEngineException(e);
        }
    }

    public void insertTablet(InsertTabletPlan insertTabletPlan) throws StorageEngineException, BatchProcessException {
        StorageGroupProcessor storageGroupProcessor;
        if (this.enableMemControl) {
            try {
                StorageEngine.blockInsertionIfReject(null);
            }
            catch (WriteProcessRejectException e) {
                Object[] results = new TSStatus[insertTabletPlan.getRowCount()];
                Arrays.fill(results, RpcUtils.getStatus((TSStatusCode)TSStatusCode.WRITE_PROCESS_REJECT));
                throw new BatchProcessException((TSStatus[])results);
            }
        }
        try {
            storageGroupProcessor = this.getProcessor(insertTabletPlan.getDeviceId());
        }
        catch (StorageEngineException e) {
            throw new StorageEngineException(String.format("Get StorageGroupProcessor of device %s failed", insertTabletPlan.getDeviceId()), e);
        }
        storageGroupProcessor.insertTablet(insertTabletPlan);
        if (config.isEnableStatMonitor()) {
            try {
                StorageGroupMNode storageGroupMNode = IoTDB.metaManager.getStorageGroupNodeByPath(insertTabletPlan.getDeviceId());
                this.updateMonitorStatistics(this.processorMap.get(storageGroupMNode.getPartialPath()), insertTabletPlan);
            }
            catch (MetadataException e) {
                logger.error("failed to record status", (Throwable)e);
            }
        }
    }

    private void updateMonitorStatistics(VirtualStorageGroupManager virtualStorageGroupManager, InsertPlan insertPlan) {
        StatMonitor monitor = StatMonitor.getInstance();
        int successPointsNum = insertPlan.getMeasurements().length - insertPlan.getFailedMeasurementNumber();
        virtualStorageGroupManager.updateMonitorSeriesValue(successPointsNum);
        monitor.updateStatGlobalValue(successPointsNum);
    }

    public void syncCloseAllProcessor() {
        logger.info("Start closing all storage group processor");
        for (VirtualStorageGroupManager processor : this.processorMap.values()) {
            processor.syncCloseAllWorkingTsFileProcessors();
        }
    }

    public void forceCloseAllProcessor() throws TsFileProcessorException {
        logger.info("Start force closing all storage group processor");
        for (VirtualStorageGroupManager processor : this.processorMap.values()) {
            processor.forceCloseAllWorkingTsFileProcessors();
        }
    }

    public void closeStorageGroupProcessor(PartialPath storageGroupPath, boolean isSeq, boolean isSync) {
        if (!this.processorMap.containsKey(storageGroupPath)) {
            return;
        }
        VirtualStorageGroupManager virtualStorageGroupManager = this.processorMap.get(storageGroupPath);
        virtualStorageGroupManager.closeStorageGroupProcessor(isSeq, isSync);
    }

    public void closeStorageGroupProcessor(PartialPath storageGroupPath, long partitionId, boolean isSeq, boolean isSync) throws StorageGroupNotSetException {
        if (!this.processorMap.containsKey(storageGroupPath)) {
            throw new StorageGroupNotSetException(storageGroupPath.getFullPath());
        }
        VirtualStorageGroupManager virtualStorageGroupManager = this.processorMap.get(storageGroupPath);
        virtualStorageGroupManager.closeStorageGroupProcessor(partitionId, isSeq, isSync);
    }

    public void delete(PartialPath path, long startTime, long endTime, long planIndex) throws StorageEngineException {
        try {
            List<PartialPath> sgPaths = IoTDB.metaManager.searchAllRelatedStorageGroups(path);
            for (PartialPath storageGroupPath : sgPaths) {
                if (!this.processorMap.containsKey(storageGroupPath)) continue;
                PartialPath newPath = path.alterPrefixPath(storageGroupPath);
                this.processorMap.get(storageGroupPath).delete(newPath, startTime, endTime, planIndex);
            }
        }
        catch (IOException | MetadataException e) {
            throw new StorageEngineException(e.getMessage());
        }
    }

    public void deleteTimeseries(PartialPath path, long planIndex) throws StorageEngineException {
        try {
            List<PartialPath> sgPaths = IoTDB.metaManager.searchAllRelatedStorageGroups(path);
            for (PartialPath storageGroupPath : sgPaths) {
                if (!this.processorMap.containsKey(storageGroupPath)) continue;
                PartialPath newPath = path.alterPrefixPath(storageGroupPath);
                this.processorMap.get(storageGroupPath).delete(newPath, Long.MIN_VALUE, Long.MAX_VALUE, planIndex);
            }
        }
        catch (IOException | MetadataException e) {
            throw new StorageEngineException(e.getMessage());
        }
    }

    public String getStorageGroupPath(PartialPath path) throws StorageEngineException {
        PartialPath deviceId = path.getDevicePath();
        StorageGroupProcessor storageGroupProcessor = this.getProcessor(deviceId);
        return storageGroupProcessor.getLogicalStorageGroupName() + File.separator + storageGroupProcessor.getVirtualStorageGroupId();
    }

    public int countUpgradeFiles() {
        int totalUpgradeFileNum = 0;
        for (VirtualStorageGroupManager virtualStorageGroupManager : this.processorMap.values()) {
            totalUpgradeFileNum += virtualStorageGroupManager.countUpgradeFiles();
        }
        return totalUpgradeFileNum;
    }

    public void upgradeAll() throws StorageEngineException {
        if (IoTDBDescriptor.getInstance().getConfig().isReadOnly()) {
            throw new StorageEngineException("Current system mode is read only, does not support file upgrade");
        }
        for (VirtualStorageGroupManager virtualStorageGroupManager : this.processorMap.values()) {
            virtualStorageGroupManager.upgradeAll();
        }
    }

    public void mergeAll(boolean isFullMerge) throws StorageEngineException {
        if (IoTDBDescriptor.getInstance().getConfig().isReadOnly()) {
            throw new StorageEngineException("Current system mode is read only, does not support merge");
        }
        for (VirtualStorageGroupManager virtualStorageGroupManager : this.processorMap.values()) {
            virtualStorageGroupManager.mergeAll(isFullMerge);
        }
    }

    public void deleteAllDataFilesInOneStorageGroup(PartialPath storageGroupPath) {
        if (this.processorMap.containsKey(storageGroupPath)) {
            this.syncDeleteDataFiles(storageGroupPath);
        }
    }

    private void syncDeleteDataFiles(PartialPath storageGroupPath) {
        logger.info("Force to delete the data in storage group processor {}", (Object)storageGroupPath);
        this.processorMap.get(storageGroupPath).syncDeleteDataFiles();
    }

    public void releaseWalDirectByteBufferPoolInOneStorageGroup(PartialPath storageGroupPath) {
        if (this.processorMap.containsKey(storageGroupPath)) {
            this.processorMap.get(storageGroupPath).releaseWalDirectByteBufferPool();
        }
    }

    public synchronized boolean deleteAll() {
        logger.info("Start deleting all storage groups' timeseries");
        this.syncCloseAllProcessor();
        for (PartialPath storageGroup : IoTDB.metaManager.getAllStorageGroupPaths()) {
            this.deleteAllDataFilesInOneStorageGroup(storageGroup);
        }
        this.processorMap.clear();
        return true;
    }

    public void setTTL(PartialPath storageGroup, long dataTTL) {
        if (!this.processorMap.containsKey(storageGroup)) {
            return;
        }
        this.processorMap.get(storageGroup).setTTL(dataTTL);
    }

    public void deleteStorageGroup(PartialPath storageGroupPath) {
        if (!this.processorMap.containsKey(storageGroupPath)) {
            return;
        }
        this.deleteAllDataFilesInOneStorageGroup(storageGroupPath);
        this.releaseWalDirectByteBufferPoolInOneStorageGroup(storageGroupPath);
        VirtualStorageGroupManager virtualStorageGroupManager = this.processorMap.remove(storageGroupPath);
        virtualStorageGroupManager.deleteStorageGroupSystemFolder(this.systemDir + File.pathSeparator + storageGroupPath);
        virtualStorageGroupManager.stopSchedulerPool();
    }

    public void loadNewTsFileForSync(TsFileResource newTsFileResource) throws StorageEngineException, LoadFileException, IllegalPathException {
        this.getProcessorDirectly(new PartialPath(this.getSgByEngineFile(newTsFileResource.getTsFile()))).loadNewTsFileForSync(newTsFileResource);
    }

    public void loadNewTsFile(TsFileResource newTsFileResource) throws LoadFileException, StorageEngineException, MetadataException {
        Set<String> deviceSet = newTsFileResource.getDevices();
        if (deviceSet == null || deviceSet.isEmpty()) {
            throw new StorageEngineException("The TsFile is empty, cannot be loaded.");
        }
        String device = deviceSet.iterator().next();
        PartialPath devicePath = new PartialPath(device);
        PartialPath storageGroupPath = IoTDB.metaManager.getStorageGroupPath(devicePath);
        this.getProcessorDirectly(storageGroupPath).loadNewTsFile(newTsFileResource);
    }

    public boolean deleteTsfileForSync(File deletedTsfile) throws StorageEngineException, IllegalPathException {
        return this.getProcessorDirectly(new PartialPath(this.getSgByEngineFile(deletedTsfile))).deleteTsfile(deletedTsfile);
    }

    public boolean deleteTsfile(File deletedTsfile) throws StorageEngineException, IllegalPathException {
        return this.getProcessorDirectly(new PartialPath(this.getSgByEngineFile(deletedTsfile)), this.getVirtualSgIdByEngineFile(deletedTsfile)).deleteTsfile(deletedTsfile);
    }

    public boolean moveTsfile(File tsfileToBeMoved, File targetDir) throws StorageEngineException, IllegalPathException {
        return this.getProcessorDirectly(new PartialPath(this.getSgByEngineFile(tsfileToBeMoved)), this.getVirtualSgIdByEngineFile(tsfileToBeMoved)).moveTsfile(tsfileToBeMoved, targetDir);
    }

    public String getSgByEngineFile(File file) {
        return file.getParentFile().getParentFile().getParentFile().getName();
    }

    public int getVirtualSgIdByEngineFile(File file) {
        return Integer.parseInt(file.getParentFile().getParentFile().getName());
    }

    public Map<PartialPath, Map<Long, List<TsFileResource>>> getAllClosedStorageGroupTsFile() {
        HashMap<PartialPath, Map<Long, List<TsFileResource>>> ret = new HashMap<PartialPath, Map<Long, List<TsFileResource>>>();
        for (Map.Entry<PartialPath, VirtualStorageGroupManager> entry : this.processorMap.entrySet()) {
            entry.getValue().getAllClosedStorageGroupTsFile(entry.getKey(), ret);
        }
        return ret;
    }

    public void setFileFlushPolicy(TsFileFlushPolicy fileFlushPolicy) {
        this.fileFlushPolicy = fileFlushPolicy;
    }

    public boolean isFileAlreadyExist(TsFileResource tsFileResource, PartialPath storageGroup, long partitionNum) {
        VirtualStorageGroupManager virtualStorageGroupManager = this.processorMap.get(storageGroup);
        if (virtualStorageGroupManager == null) {
            return false;
        }
        Iterator<String> partialPathIterator = tsFileResource.getDevices().iterator();
        try {
            return this.getProcessor(new PartialPath(partialPathIterator.next())).isFileAlreadyExist(tsFileResource, partitionNum);
        }
        catch (StorageEngineException | IllegalPathException e) {
            logger.error("can't find processor with: " + tsFileResource, (Throwable)e);
            return false;
        }
    }

    public void setPartitionVersionToMax(PartialPath storageGroup, long partitionId, long newMaxVersion) {
        this.processorMap.get(storageGroup).setPartitionVersionToMax(partitionId, newMaxVersion);
    }

    public void removePartitions(PartialPath storageGroupPath, StorageGroupProcessor.TimePartitionFilter filter) {
        this.processorMap.get(storageGroupPath).removePartitions(filter);
    }

    public Map<PartialPath, VirtualStorageGroupManager> getProcessorMap() {
        return this.processorMap;
    }

    public Map<String, List<Pair<Long, Boolean>>> getWorkingStorageGroupPartitions() {
        ConcurrentHashMap<String, List<Pair<Long, Boolean>>> res = new ConcurrentHashMap<String, List<Pair<Long, Boolean>>>();
        for (Map.Entry<PartialPath, VirtualStorageGroupManager> entry : this.processorMap.entrySet()) {
            entry.getValue().getWorkingStorageGroupPartitions(entry.getKey().getFullPath(), res);
        }
        return res;
    }

    public void registerFlushListener(FlushListener listener) {
        this.customFlushListeners.add(listener);
    }

    public void registerCloseFileListener(CloseFileListener listener) {
        this.customCloseFileListeners.add(listener);
    }

    public Pair<List<StorageGroupProcessor>, Map<StorageGroupProcessor, List<PartialPath>>> mergeLock(List<PartialPath> pathList) throws StorageEngineException, QueryProcessException {
        HashMap<StorageGroupProcessor, List> map = new HashMap<StorageGroupProcessor, List>();
        for (PartialPath path : pathList) {
            map.computeIfAbsent(this.getProcessor(path.getDevicePath()), key -> new ArrayList()).add(path);
        }
        List<StorageGroupProcessor> list = map.keySet().stream().sorted(Comparator.comparing(StorageGroupProcessor::getVirtualStorageGroupId)).collect(Collectors.toList());
        list.forEach(StorageGroupProcessor::readLock);
        return new Pair(list, map);
    }

    public void mergeUnLock(List<StorageGroupProcessor> list) {
        list.forEach(StorageGroupProcessor::readUnlock);
    }

    static class InstanceHolder {
        private static final StorageEngine INSTANCE = new StorageEngine();

        private InstanceHolder() {
        }
    }
}

