/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.sync.receiver.transfer;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import org.apache.iotdb.db.concurrent.ThreadName;
import org.apache.iotdb.db.conf.IoTDBConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.conf.directories.DirectoryManager;
import org.apache.iotdb.db.engine.storagegroup.TsFileResource;
import org.apache.iotdb.db.exception.DiskSpaceInsufficientException;
import org.apache.iotdb.db.exception.SyncDeviceOwnerConflictException;
import org.apache.iotdb.db.metadata.logfile.MLogReader;
import org.apache.iotdb.db.qp.logical.Operator;
import org.apache.iotdb.db.qp.physical.PhysicalPlan;
import org.apache.iotdb.db.service.IoTDB;
import org.apache.iotdb.db.sync.receiver.load.FileLoader;
import org.apache.iotdb.db.sync.receiver.load.FileLoaderManager;
import org.apache.iotdb.db.sync.receiver.load.IFileLoader;
import org.apache.iotdb.db.sync.receiver.recover.SyncReceiverLogAnalyzer;
import org.apache.iotdb.db.sync.receiver.recover.SyncReceiverLogger;
import org.apache.iotdb.db.utils.SyncUtils;
import org.apache.iotdb.service.sync.thrift.ConfirmInfo;
import org.apache.iotdb.service.sync.thrift.SyncService;
import org.apache.iotdb.service.sync.thrift.SyncStatus;
import org.apache.iotdb.tsfile.utils.FilePathUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SyncServiceImpl
implements SyncService.Iface {
    private static final Logger logger = LoggerFactory.getLogger(SyncServiceImpl.class);
    private IoTDBConfig config = IoTDBDescriptor.getInstance().getConfig();
    private ThreadLocal<String> syncFolderPath = new ThreadLocal();
    private ThreadLocal<String> currentSG = new ThreadLocal();
    private ThreadLocal<SyncReceiverLogger> syncLog = new ThreadLocal();
    private ThreadLocal<String> senderName = new ThreadLocal();
    private ThreadLocal<File> currentFile = new ThreadLocal();
    private ThreadLocal<FileOutputStream> currentFileWriter = new ThreadLocal();
    private ThreadLocal<MessageDigest> messageDigest = new ThreadLocal();

    public SyncStatus check(ConfirmInfo info) {
        String ipAddress = info.address;
        String uuid = info.uuid;
        Thread.currentThread().setName(ThreadName.SYNC_SERVER.getName());
        if (!this.config.getIoTDBMajorVersion(info.version).equals(this.config.getIoTDBMajorVersion())) {
            return this.getErrorResult(String.format("Version mismatch: the sender <%s>, the receiver <%s>", info.version, this.config.getIoTDBVersion()));
        }
        if (info.partitionInterval != IoTDBDescriptor.getInstance().getConfig().getPartitionInterval()) {
            return this.getErrorResult(String.format("Partition interval mismatch: the sender <%d>, the receiver <%d>", info.partitionInterval, IoTDBDescriptor.getInstance().getConfig().getPartitionInterval()));
        }
        if (SyncUtils.verifyIPSegment(this.config.getIpWhiteList(), ipAddress)) {
            this.senderName.set(ipAddress + "_" + uuid);
            if (this.checkRecovery()) {
                logger.info("Start to sync with sender {}", (Object)this.senderName.get());
                return this.getSuccessResult();
            }
            return this.getErrorResult("Receiver is processing data from previous sync tasks");
        }
        return this.getErrorResult("Sender IP is not in the white list of receiver IP and synchronization tasks are not allowed.");
    }

    private boolean checkRecovery() {
        try {
            if (this.currentFileWriter.get() != null) {
                this.currentFileWriter.get().close();
            }
            if (this.syncLog.get() != null) {
                this.syncLog.get().close();
            }
            return SyncReceiverLogAnalyzer.getInstance().recover(this.senderName.get());
        }
        catch (IOException e) {
            logger.error("Check recovery state fail", (Throwable)e);
            return false;
        }
    }

    public SyncStatus startSync() {
        try {
            this.initPath();
            this.currentSG.remove();
            FileLoader.createFileLoader(this.senderName.get(), this.syncFolderPath.get());
            this.syncLog.set(new SyncReceiverLogger(new File(this.syncFolderPath.get(), "sync.log")));
            return this.getSuccessResult();
        }
        catch (IOException | DiskSpaceInsufficientException e) {
            logger.error("Can not receiver data from sender", (Throwable)e);
            return this.getErrorResult(e.getMessage());
        }
    }

    private void initPath() throws DiskSpaceInsufficientException {
        String dataDir = new File(DirectoryManager.getInstance().getNextFolderForSequenceFile()).getParentFile().getAbsolutePath();
        this.syncFolderPath.set(FilePathUtils.regularizePath((String)dataDir) + "sync-receiver" + File.separatorChar + this.senderName.get());
    }

    public SyncStatus init(String storageGroup) {
        logger.info("Sync process started to receive data of storage group {}", (Object)storageGroup);
        this.currentSG.set(storageGroup);
        try {
            this.syncLog.get().startSyncDeletedFilesName();
        }
        catch (IOException e) {
            logger.error("Can not init sync process", (Throwable)e);
            return this.getErrorResult(e.getMessage());
        }
        return this.getSuccessResult();
    }

    public SyncStatus syncDeletedFileName(String fileInfo) {
        String filePath = this.currentSG.get() + File.separator + this.getFilePathByFileInfo(fileInfo);
        try {
            this.syncLog.get().startSyncDeletedFilesName();
            this.syncLog.get().finishSyncDeletedFileName(new File(this.getSyncDataPath(), filePath));
            FileLoaderManager.getInstance().getFileLoader(this.senderName.get()).addDeletedFileName(new File(this.getSyncDataPath(), filePath));
        }
        catch (IOException e) {
            logger.error("Can not sync deleted file", (Throwable)e);
            return this.getErrorResult(String.format("Can not sync deleted file %s because %s", filePath, e.getMessage()));
        }
        return this.getSuccessResult();
    }

    public String getFilePathByFileInfo(String fileInfo) {
        String filePath = "";
        String[] fileInfos = fileInfo.split("!");
        for (int i = 0; i < fileInfos.length - 1; ++i) {
            filePath = filePath + fileInfos[i] + File.separator;
        }
        return filePath + fileInfos[fileInfos.length - 1];
    }

    public SyncStatus initSyncData(String fileInfo) {
        String filePath = fileInfo;
        try {
            File file;
            if (fileInfo.equals("mlog.bin")) {
                file = new File(this.getSyncDataPath(), filePath);
            } else {
                filePath = this.currentSG.get() + File.separator + this.getFilePathByFileInfo(fileInfo);
                file = new File(this.getSyncDataPath(), filePath);
            }
            file.delete();
            this.currentFile.set(file);
            if (!file.getParentFile().exists()) {
                file.getParentFile().mkdirs();
            }
            if (this.currentFileWriter.get() != null) {
                this.currentFileWriter.get().close();
            }
            this.currentFileWriter.set(new FileOutputStream(file));
            this.syncLog.get().startSyncTsFiles();
            this.messageDigest.set(MessageDigest.getInstance("SHA-256"));
        }
        catch (IOException | NoSuchAlgorithmException e) {
            logger.error("Can not init sync resource for file {}", (Object)filePath, (Object)e);
            return this.getErrorResult(String.format("Can not init sync resource for file %s because %s", filePath, e.getMessage()));
        }
        return this.getSuccessResult();
    }

    public SyncStatus syncData(ByteBuffer buff) {
        try {
            int pos = buff.position();
            this.currentFileWriter.get().getChannel().write(buff);
            buff.position(pos);
            this.messageDigest.get().update(buff);
        }
        catch (IOException e) {
            logger.error("Can not sync data for file {}", (Object)this.currentFile.get().getAbsoluteFile(), (Object)e);
            return this.getErrorResult(String.format("Can not sync data for file %s because %s", this.currentFile.get().getName(), e.getMessage()));
        }
        return this.getSuccessResult();
    }

    public SyncStatus checkDataDigest(String digestOfSender) {
        String digestOfReceiver = new BigInteger(1, this.messageDigest.get().digest()).toString(16);
        try {
            if (this.currentFileWriter.get() != null) {
                this.currentFileWriter.get().close();
            }
            if (!digestOfSender.equals(digestOfReceiver)) {
                this.currentFile.get().delete();
                this.currentFileWriter.set(new FileOutputStream(this.currentFile.get()));
                return this.getErrorResult(String.format("Digest of the sender is differ from digest of the receiver of the file %s.", this.currentFile.get().getAbsolutePath()));
            }
            if (this.currentFile.get().getName().endsWith("mlog.bin")) {
                this.loadMetadata();
            } else if (!this.currentFile.get().getName().endsWith(".resource")) {
                logger.info("Receiver has received {} successfully.", (Object)this.currentFile.get());
                FileLoaderManager.getInstance().checkAndUpdateDeviceOwner(new TsFileResource(new File(this.currentFile.get() + ".resource")));
                this.syncLog.get().finishSyncTsfile(this.currentFile.get());
                FileLoaderManager.getInstance().getFileLoader(this.senderName.get()).addTsfile(this.currentFile.get());
            }
        }
        catch (IOException e) {
            logger.error("Can not check data digest for file {}", (Object)this.currentFile.get().getAbsoluteFile(), (Object)e);
            return this.getErrorResult(String.format("Can not check data digest for file %s because %s", this.currentFile.get().getName(), e.getMessage()));
        }
        catch (SyncDeviceOwnerConflictException e) {
            logger.error("Device owner has conflicts, skip all other tsfiles in the sg {}.", (Object)this.currentSG.get());
            return new SyncStatus(-2, String.format("Device owner has conflicts, skip all other tsfiles in the same sg %s because %s", this.currentSG.get(), e.getMessage()));
        }
        return new SyncStatus(1, digestOfReceiver);
    }

    private void loadMetadata() {
        block12: {
            logger.info("Start to load metadata in sync process.");
            if (this.currentFile.get().exists()) {
                try {
                    MLogReader mLogReader = new MLogReader(this.currentFile.get());
                    block9: while (true) {
                        while (mLogReader.hasNext()) {
                            PhysicalPlan plan = null;
                            try {
                                plan = mLogReader.next();
                                if (plan == null) continue;
                                if (plan.getOperatorType() == Operator.OperatorType.CHANGE_TAG_OFFSET) continue block9;
                                IoTDB.metaManager.operation(plan);
                                continue block9;
                            }
                            catch (Exception e) {
                                logger.error("Can not operate metadata operation {} for err:{}", plan == null ? "" : plan.getOperatorType(), (Object)e);
                            }
                        }
                        break block12;
                        {
                            continue block9;
                            break;
                        }
                        break;
                    }
                    finally {
                        mLogReader.close();
                    }
                }
                catch (IOException e) {
                    logger.error("Cannot read the file {}.", (Object)this.currentFile.get().getAbsoluteFile(), (Object)e);
                }
            }
        }
    }

    public SyncStatus endSync() {
        try {
            IFileLoader loader;
            if (this.syncLog.get() != null) {
                this.syncLog.get().close();
            }
            if ((loader = FileLoaderManager.getInstance().getFileLoader(this.senderName.get())) == null) {
                SyncStatus syncStatus = this.getErrorResult(String.format("File Loader of the storage group %s is null", this.currentSG.get()));
                return syncStatus;
            }
            loader.endSync();
            if (this.currentFileWriter.get() != null) {
                this.currentFileWriter.get().close();
            }
            logger.info("Sync process with sender {} finished.", (Object)this.senderName.get());
        }
        catch (IOException e) {
            logger.error("Can not end sync", (Throwable)e);
            SyncStatus syncStatus = this.getErrorResult(String.format("Can not end sync because %s", e.getMessage()));
            return syncStatus;
        }
        finally {
            this.syncFolderPath.remove();
            this.currentSG.remove();
            this.syncLog.remove();
            this.senderName.remove();
            this.currentFile.remove();
            this.currentFileWriter.remove();
            this.messageDigest.remove();
        }
        return this.getSuccessResult();
    }

    private String getSyncDataPath() {
        return this.syncFolderPath.get() + File.separatorChar + "data";
    }

    private SyncStatus getSuccessResult() {
        return new SyncStatus(1, "");
    }

    private SyncStatus getErrorResult(String errorMsg) {
        return new SyncStatus(-1, errorMsg);
    }

    public void handleClientExit() {
    }
}

