/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.writelog.node;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.BufferOverflowException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.io.FileUtils;
import org.apache.iotdb.db.concurrent.IoTDBThreadPoolFactory;
import org.apache.iotdb.db.conf.IoTDBConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.conf.directories.DirectoryManager;
import org.apache.iotdb.db.engine.fileSystem.SystemFileFactory;
import org.apache.iotdb.db.qp.physical.PhysicalPlan;
import org.apache.iotdb.db.writelog.io.ILogReader;
import org.apache.iotdb.db.writelog.io.ILogWriter;
import org.apache.iotdb.db.writelog.io.LogWriter;
import org.apache.iotdb.db.writelog.io.MultiFileLogReader;
import org.apache.iotdb.db.writelog.node.WriteLogNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExclusiveWriteLogNode
implements WriteLogNode,
Comparable<ExclusiveWriteLogNode> {
    public static final String WAL_FILE_NAME = "wal";
    private static final Logger logger = LoggerFactory.getLogger(ExclusiveWriteLogNode.class);
    private final String identifier;
    private final String logDirectory;
    private ILogWriter currentFileWriter;
    private final IoTDBConfig config = IoTDBDescriptor.getInstance().getConfig();
    private volatile ByteBuffer logBufferWorking;
    private volatile ByteBuffer logBufferIdle;
    private volatile ByteBuffer logBufferFlushing;
    private volatile ByteBuffer[] bufferArray;
    private final Object switchBufferCondition = new Object();
    private final ReentrantLock lock = new ReentrantLock();
    private final ExecutorService FLUSH_BUFFER_THREAD_POOL;
    private long fileId = 0L;
    private long lastFlushedId = 0L;
    private int bufferedLogNum = 0;
    private final AtomicBoolean deleted = new AtomicBoolean(false);
    private int bufferOverflowNum = 0;

    public ExclusiveWriteLogNode(String identifier) {
        this.identifier = identifier;
        this.logDirectory = DirectoryManager.getInstance().getWALFolder() + File.separator + this.identifier;
        if (SystemFileFactory.INSTANCE.getFile(this.logDirectory).mkdirs()) {
            logger.info("create the WAL folder {}.", (Object)this.logDirectory);
        }
        this.FLUSH_BUFFER_THREAD_POOL = IoTDBThreadPoolFactory.newSingleThreadExecutor("Flush-WAL-Thread-" + this.identifier);
    }

    @Override
    public void initBuffer(ByteBuffer[] byteBuffers) {
        this.logBufferWorking = byteBuffers[0];
        this.logBufferIdle = byteBuffers[1];
        this.bufferArray = byteBuffers;
    }

    @Override
    public void write(PhysicalPlan plan) throws IOException {
        if (this.deleted.get()) {
            throw new IOException("WAL node deleted");
        }
        this.lock.lock();
        try {
            this.putLog(plan);
            if (this.bufferedLogNum >= this.config.getFlushWalThreshold()) {
                this.sync();
            }
        }
        catch (BufferOverflowException e) {
            throw new IOException("Log cannot fit into the buffer, please increase wal_buffer_size", e);
        }
        finally {
            this.lock.unlock();
        }
    }

    private void putLog(PhysicalPlan plan) {
        try {
            plan.serialize(this.logBufferWorking);
        }
        catch (BufferOverflowException e) {
            ++this.bufferOverflowNum;
            if (this.bufferOverflowNum > 200) {
                logger.info("WAL bytebuffer overflows too many times. If this occurs frequently, please increase wal_buffer_size.");
                this.bufferOverflowNum = 0;
            }
            this.sync();
            plan.serialize(this.logBufferWorking);
        }
        ++this.bufferedLogNum;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        this.sync();
        this.forceWal();
        this.lock.lock();
        try {
            Object object = this.switchBufferCondition;
            synchronized (object) {
                while (this.logBufferFlushing != null && !this.deleted.get()) {
                    this.switchBufferCondition.wait();
                }
                this.switchBufferCondition.notifyAll();
            }
            if (this.currentFileWriter != null) {
                this.currentFileWriter.close();
                logger.debug("WAL file {} is closed", (Object)this.currentFileWriter);
                this.currentFileWriter = null;
            }
            logger.debug("Log node {} closed successfully", (Object)this.identifier);
        }
        catch (IOException e) {
            logger.warn("Cannot close log node {} because:", (Object)this.identifier, (Object)e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            logger.warn("Waiting for current buffer being flushed interrupted");
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void forceSync() {
        if (this.deleted.get()) {
            return;
        }
        this.sync();
        this.forceWal();
    }

    @Override
    public void notifyStartFlush() throws FileNotFoundException {
        this.lock.lock();
        try {
            this.close();
            this.nextFileWriter();
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void notifyEndFlush() {
        this.lock.lock();
        try {
            File logFile = SystemFileFactory.INSTANCE.getFile(this.logDirectory, WAL_FILE_NAME + ++this.lastFlushedId);
            this.discard(logFile);
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public String getIdentifier() {
        return this.identifier;
    }

    @Override
    public String getLogDirectory() {
        return this.logDirectory;
    }

    @Override
    public ByteBuffer[] delete() throws IOException {
        this.lock.lock();
        try {
            this.close();
            FileUtils.deleteDirectory((File)SystemFileFactory.INSTANCE.getFile(this.logDirectory));
            this.deleted.set(true);
            ByteBuffer[] byteBufferArray = this.bufferArray;
            return byteBufferArray;
        }
        finally {
            this.FLUSH_BUFFER_THREAD_POOL.shutdown();
            this.lock.unlock();
        }
    }

    @Override
    public ILogReader getLogReader() {
        File[] logFiles = SystemFileFactory.INSTANCE.getFile(this.logDirectory).listFiles();
        Arrays.sort(logFiles, Comparator.comparingInt(f -> Integer.parseInt(f.getName().replace(WAL_FILE_NAME, ""))));
        return new MultiFileLogReader(logFiles);
    }

    private void discard(File logFile) {
        if (!logFile.exists()) {
            logger.info("Log file does not exist");
        } else {
            try {
                FileUtils.forceDelete((File)logFile);
                logger.info("Log node {} cleaned old file", (Object)this.identifier);
            }
            catch (IOException e) {
                logger.warn("Old log file {} of {} cannot be deleted", new Object[]{logFile.getName(), this.identifier, e});
            }
        }
    }

    private void forceWal() {
        this.lock.lock();
        try {
            try {
                if (this.currentFileWriter != null) {
                    this.currentFileWriter.force();
                }
            }
            catch (IOException e) {
                logger.warn("Log node {} force failed.", (Object)this.identifier, (Object)e);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private void sync() {
        this.lock.lock();
        try {
            if (this.bufferedLogNum == 0) {
                return;
            }
            this.switchBufferWorkingToFlushing();
            ILogWriter currWriter = this.getCurrentFileWriter();
            this.FLUSH_BUFFER_THREAD_POOL.submit(() -> this.flushBuffer(currWriter));
            this.bufferedLogNum = 0;
            logger.debug("Log node {} ends sync.", (Object)this.identifier);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            logger.warn("Waiting for available buffer interrupted");
        }
        catch (FileNotFoundException e) {
            logger.warn("can not found file {}", (Object)this.identifier, (Object)e);
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void flushBuffer(ILogWriter writer) {
        try {
            writer.write(this.logBufferFlushing);
        }
        catch (ClosedChannelException closedChannelException) {
        }
        catch (IOException e) {
            logger.warn("Log node {} sync failed, change system mode to read-only", (Object)this.identifier, (Object)e);
            IoTDBDescriptor.getInstance().getConfig().setReadOnly(true);
            return;
        }
        Object object = this.switchBufferCondition;
        synchronized (object) {
            this.logBufferIdle = this.logBufferFlushing;
            this.logBufferFlushing = null;
            this.switchBufferCondition.notifyAll();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void switchBufferWorkingToFlushing() throws InterruptedException {
        Object object = this.switchBufferCondition;
        synchronized (object) {
            while (this.logBufferFlushing != null && !this.deleted.get()) {
                this.switchBufferCondition.wait(100L);
            }
            this.logBufferFlushing = this.logBufferWorking;
            this.logBufferWorking = this.logBufferIdle;
            this.logBufferWorking.clear();
            this.logBufferIdle = null;
        }
    }

    private ILogWriter getCurrentFileWriter() throws FileNotFoundException {
        if (this.currentFileWriter == null) {
            this.nextFileWriter();
        }
        return this.currentFileWriter;
    }

    private void nextFileWriter() throws FileNotFoundException {
        ++this.fileId;
        File newFile = SystemFileFactory.INSTANCE.getFile(this.logDirectory, WAL_FILE_NAME + this.fileId);
        if (newFile.getParentFile().mkdirs()) {
            logger.info("create WAL parent folder {}.", (Object)newFile.getParent());
        }
        logger.debug("WAL file {} is opened", (Object)newFile);
        this.currentFileWriter = new LogWriter(newFile, this.config.getForceWalPeriodInMs() == 0L);
    }

    public int hashCode() {
        return this.identifier.hashCode();
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        return this.compareTo((ExclusiveWriteLogNode)obj) == 0;
    }

    public String toString() {
        return "Log node " + this.identifier;
    }

    @Override
    public int compareTo(ExclusiveWriteLogNode o) {
        return this.identifier.compareTo(o.identifier);
    }
}

