/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.cluster.partition.slot;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.iotdb.cluster.config.ClusterDescriptor;
import org.apache.iotdb.cluster.rpc.thrift.Node;
import org.apache.iotdb.cluster.utils.NodeSerializeUtils;
import org.apache.iotdb.db.exception.StorageEngineException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SlotManager {
    private static final Logger logger = LoggerFactory.getLogger(SlotManager.class);
    private static final long SLOT_WAIT_INTERVAL_MS = 10L;
    private static final String SLOT_FILE_NAME = "SLOT_STATUS";
    private String slotFilePath;
    private Map<Integer, SlotDescriptor> idSlotMap;

    public SlotManager(long totalSlotNumber, String memberDir) {
        if (memberDir != null) {
            this.slotFilePath = memberDir + File.separator + SLOT_FILE_NAME;
        }
        if (!this.load()) {
            this.init(totalSlotNumber);
        }
    }

    private void init(long totalSlotNumber) {
        this.idSlotMap = new ConcurrentHashMap<Integer, SlotDescriptor>();
        int i = 0;
        while ((long)i < totalSlotNumber) {
            this.idSlotMap.put(i, new SlotDescriptor(SlotStatus.NULL));
            ++i;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void waitSlot(int slotId) {
        SlotDescriptor slotDescriptor = this.idSlotMap.get(slotId);
        while (true) {
            SlotDescriptor slotDescriptor2 = slotDescriptor;
            synchronized (slotDescriptor2) {
                if (slotDescriptor.slotStatus == SlotStatus.PULLING || slotDescriptor.slotStatus == SlotStatus.PULLING_WRITABLE) {
                    try {
                        slotDescriptor.wait(10L);
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        logger.error("Unexpected interruption when waiting for slot {}", (Object)slotId, (Object)e);
                    }
                } else {
                    return;
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void waitSlotForWrite(int slotId) throws StorageEngineException {
        SlotDescriptor slotDescriptor = this.idSlotMap.get(slotId);
        while (true) {
            SlotDescriptor slotDescriptor2 = slotDescriptor;
            synchronized (slotDescriptor2) {
                if (slotDescriptor.slotStatus == SlotStatus.SENDING || slotDescriptor.slotStatus == SlotStatus.SENT) {
                    throw new StorageEngineException(String.format("Slot %d no longer belongs to the node", slotId));
                }
                if (slotDescriptor.slotStatus != SlotStatus.NULL && slotDescriptor.slotStatus != SlotStatus.PULLING_WRITABLE) {
                    try {
                        slotDescriptor.wait(10L);
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        logger.error("Unexpected interruption when waiting for slot {}", (Object)slotId, (Object)e);
                    }
                } else {
                    return;
                }
            }
        }
    }

    public SlotStatus getStatus(int slotId) {
        return this.idSlotMap.get(slotId).slotStatus;
    }

    public Node getSource(int slotId) {
        return this.idSlotMap.get(slotId).source;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setToPulling(int slotId, Node source) {
        SlotDescriptor slotDescriptor;
        SlotDescriptor slotDescriptor2 = slotDescriptor = this.idSlotMap.get(slotId);
        synchronized (slotDescriptor2) {
            slotDescriptor.slotStatus = SlotStatus.PULLING;
            slotDescriptor.source = source;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setToPullingWritable(int slotId) {
        SlotDescriptor slotDescriptor;
        SlotDescriptor slotDescriptor2 = slotDescriptor = this.idSlotMap.get(slotId);
        synchronized (slotDescriptor2) {
            slotDescriptor.slotStatus = SlotStatus.PULLING_WRITABLE;
            slotDescriptor.notifyAll();
        }
        this.save();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setToNull(int slotId) {
        SlotDescriptor slotDescriptor;
        SlotDescriptor slotDescriptor2 = slotDescriptor = this.idSlotMap.get(slotId);
        synchronized (slotDescriptor2) {
            slotDescriptor.slotStatus = SlotStatus.NULL;
            slotDescriptor.source = null;
            slotDescriptor.notifyAll();
        }
        this.save();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setToSending(int slotId) {
        SlotDescriptor slotDescriptor;
        this.waitSlot(slotId);
        SlotDescriptor slotDescriptor2 = slotDescriptor = this.idSlotMap.get(slotId);
        synchronized (slotDescriptor2) {
            slotDescriptor.slotStatus = SlotStatus.SENDING;
            slotDescriptor.snapshotReceivedCount = 0;
        }
        this.save();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void setToSent(int slotId) {
        SlotDescriptor slotDescriptor;
        SlotDescriptor slotDescriptor2 = slotDescriptor = this.idSlotMap.get(slotId);
        synchronized (slotDescriptor2) {
            slotDescriptor.slotStatus = SlotStatus.SENT;
        }
        this.save();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int sentOneReplication(int slotId) {
        SlotDescriptor slotDescriptor;
        SlotDescriptor slotDescriptor2 = slotDescriptor = this.idSlotMap.get(slotId);
        synchronized (slotDescriptor2) {
            int sentReplicaNum = ++slotDescriptor.snapshotReceivedCount;
            if (sentReplicaNum >= ClusterDescriptor.getInstance().getConfig().getReplicationNum()) {
                this.setToSent(slotId);
            }
            this.save();
            return sentReplicaNum;
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private boolean load() {
        if (this.slotFilePath == null) {
            return false;
        }
        File slotFile = new File(this.slotFilePath);
        if (!slotFile.exists()) {
            return false;
        }
        try (FileInputStream fileInputStream = new FileInputStream(slotFile);){
            boolean bl;
            try (BufferedInputStream bufferedInputStream = new BufferedInputStream(fileInputStream);){
                byte[] bytes = new byte[(int)slotFile.length()];
                int read = bufferedInputStream.read(bytes);
                if ((long)read != slotFile.length() && logger.isWarnEnabled()) {
                    logger.warn("SlotManager in {} read size does not equal to file size: {}/{}", new Object[]{this.slotFilePath, read, slotFile.length()});
                }
                this.deserialize(ByteBuffer.wrap(bytes));
                bl = true;
            }
            return bl;
        }
        catch (Exception e) {
            logger.warn("Cannot deserialize slotManager from {}", (Object)this.slotFilePath, (Object)e);
            return false;
        }
    }

    private synchronized void save() {
        if (this.slotFilePath == null) {
            return;
        }
        File slotFile = new File(this.slotFilePath);
        if (!slotFile.getParentFile().exists() && !slotFile.getParentFile().mkdirs()) {
            logger.warn("Cannot mkdirs for {}", (Object)slotFile);
        }
        try (FileOutputStream outputStream = new FileOutputStream(this.slotFilePath);
             BufferedOutputStream bufferedOutputStream = new BufferedOutputStream(outputStream);
             DataOutputStream dataOutputStream = new DataOutputStream(bufferedOutputStream);){
            this.serialize(dataOutputStream);
        }
        catch (IOException e) {
            logger.warn("SlotManager in {} cannot be saved", (Object)this.slotFilePath, (Object)e);
        }
    }

    private void serialize(DataOutputStream outputStream) throws IOException {
        outputStream.writeInt(this.idSlotMap.size());
        for (Map.Entry<Integer, SlotDescriptor> integerSlotDescriptorEntry : this.idSlotMap.entrySet()) {
            outputStream.writeInt(integerSlotDescriptorEntry.getKey());
            integerSlotDescriptorEntry.getValue().serialize(outputStream);
        }
    }

    private void deserialize(ByteBuffer buffer) {
        int slotNum = buffer.getInt();
        this.idSlotMap = new ConcurrentHashMap<Integer, SlotDescriptor>(slotNum);
        for (int i = 0; i < slotNum; ++i) {
            int slotId = buffer.getInt();
            SlotDescriptor descriptor = SlotDescriptor.deserialize(buffer);
            this.idSlotMap.put(slotId, descriptor);
        }
    }

    private static class SlotDescriptor {
        private SlotStatus slotStatus;
        private Node source;
        private volatile int snapshotReceivedCount;

        SlotDescriptor() {
        }

        SlotDescriptor(SlotStatus slotStatus) {
            this.slotStatus = slotStatus;
        }

        private void serialize(DataOutputStream outputStream) throws IOException {
            outputStream.writeInt(this.slotStatus.ordinal());
            if (this.slotStatus == SlotStatus.PULLING || this.slotStatus == SlotStatus.PULLING_WRITABLE) {
                NodeSerializeUtils.serialize(this.source, outputStream);
            } else if (this.slotStatus == SlotStatus.SENDING) {
                outputStream.writeInt(this.snapshotReceivedCount);
            }
        }

        private static SlotDescriptor deserialize(ByteBuffer buffer) {
            SlotDescriptor descriptor = new SlotDescriptor();
            descriptor.slotStatus = SlotStatus.values()[buffer.getInt()];
            if (descriptor.slotStatus == SlotStatus.PULLING || descriptor.slotStatus == SlotStatus.PULLING_WRITABLE) {
                descriptor.source = new Node();
                NodeSerializeUtils.deserialize(descriptor.source, buffer);
            } else if (descriptor.slotStatus == SlotStatus.SENDING) {
                descriptor.snapshotReceivedCount = buffer.getInt();
            }
            return descriptor;
        }
    }

    public static enum SlotStatus {
        NULL,
        PULLING,
        PULLING_WRITABLE,
        SENDING,
        SENT;

    }
}

