/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.iotdb.db.conf.directories;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.apache.iotdb.db.conf.IoTDBConstant;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.conf.directories.strategy.DirectoryStrategy;
import org.apache.iotdb.db.exception.DiskSpaceInsufficientException;
import org.apache.iotdb.db.exception.LoadConfigurationException;
import org.apache.iotdb.tsfile.fileSystem.FSFactoryProducer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The main class of multiple directories. Used to allocate folders to data files.
 */
public class DirectoryManager {

  private static final Logger logger = LoggerFactory.getLogger(DirectoryManager.class);

  private List<String> sequenceFileFolders;
  private List<String> unsequenceFileFolders;
  private DirectoryStrategy sequenceStrategy;
  private DirectoryStrategy unsequenceStrategy;

  private DirectoryManager() {
    sequenceFileFolders =
        new ArrayList<>(Arrays.asList(IoTDBDescriptor.getInstance().getConfig().getDataDirs()));
    for (int i = 0; i < sequenceFileFolders.size(); i++) {
      sequenceFileFolders
          .set(i, sequenceFileFolders.get(i) + File.separator + IoTDBConstant.SEQUENCE_FLODER_NAME);
    }
    mkDataDirs(sequenceFileFolders);

    unsequenceFileFolders =
        new ArrayList<>(Arrays.asList(IoTDBDescriptor.getInstance().getConfig().getDataDirs()));
    for (int i = 0; i < unsequenceFileFolders.size(); i++) {
      unsequenceFileFolders.set(i,
          unsequenceFileFolders.get(i) + File.separator + IoTDBConstant.UNSEQUENCE_FLODER_NAME);
    }
    mkDataDirs(unsequenceFileFolders);

    String strategyName = "";
    try {
      strategyName = IoTDBDescriptor.getInstance().getConfig().getMultiDirStrategyClassName();
      Class<?> clazz = Class.forName(strategyName);
      sequenceStrategy = (DirectoryStrategy) clazz.newInstance();
      sequenceStrategy.setFolders(sequenceFileFolders);
      unsequenceStrategy = (DirectoryStrategy) clazz.newInstance();
      unsequenceStrategy.setFolders(unsequenceFileFolders);
    } catch (DiskSpaceInsufficientException e) {
      logger.error("All disks of folders are full.", e);
    } catch (Exception e) {
      logger.error("Can't find strategy {} for mult-directories.", strategyName, e);
    }
  }

  public void updateFileFolders() throws LoadConfigurationException {
    try {
      List<String> sequenceFileFolders =
          new ArrayList<>(Arrays.asList(IoTDBDescriptor.getInstance().getConfig().getDataDirs()));
      mkDataDirs(sequenceFileFolders);

      List<String> unsequenceFileFolders =
          new ArrayList<>(Arrays.asList(IoTDBDescriptor.getInstance().getConfig().getDataDirs()));
      mkDataDirs(unsequenceFileFolders);
      sequenceStrategy.setFolders(sequenceFileFolders);
      unsequenceStrategy.setFolders(unsequenceFileFolders);
      this.sequenceFileFolders = sequenceFileFolders;
      this.unsequenceFileFolders = unsequenceFileFolders;
      logger.info("Success to update file folders.");
    } catch (DiskSpaceInsufficientException e) {
      logger.error(
          "Fail to update file folders, use previous folders.", e);
      throw new LoadConfigurationException(
          "Fail to update file folders because all disks of folders are full, use previous folders.");
    }
  }

  public void updateDirectoryStrategy() throws LoadConfigurationException {
    String strategyName = "";
    try {
      strategyName = IoTDBDescriptor.getInstance().getConfig().getMultiDirStrategyClassName();
      Class<?> clazz = Class.forName(strategyName);
      sequenceStrategy = (DirectoryStrategy) clazz.newInstance();
      sequenceStrategy.setFolders(sequenceFileFolders);
      unsequenceStrategy = (DirectoryStrategy) clazz.newInstance();
      unsequenceStrategy.setFolders(unsequenceFileFolders);
      logger.info("Success to update directory strategy.");
    } catch (Exception e) {
      logger.error("Fail to update directory strategy {}, use previous strategy", strategyName, e);
      throw new LoadConfigurationException(String.format(
          "Fail to update directory strategy because can't find strategy %s for mult-directories, use previous strategy",
          strategyName));
    }
  }

  public static DirectoryManager getInstance() {
    return DirectoriesHolder.INSTANCE;
  }

  private void mkDataDirs(List<String> folders) {
    for (String folder : folders) {
      File file = FSFactoryProducer.getFSFactory().getFile(folder);
      if (file.mkdirs()) {
        logger.info("folder {} doesn't exist, create it", file.getPath());
      } else {
        logger.info("create folder {} failed. Is the folder existed: {}", file.getPath(),
            file.exists());
      }
    }
  }
  
  public String getNextFolderForSequenceFile() throws DiskSpaceInsufficientException {
    return getSequenceFileFolder(getNextFolderIndexForSequenceFile());
  }

  /**
   * get next folder index for TsFile.
   *
   * @return next folder index
   */
  public int getNextFolderIndexForSequenceFile() throws DiskSpaceInsufficientException {
    return sequenceStrategy.nextFolderIndex();
  }

  public String getSequenceFileFolder(int index) {
    return sequenceFileFolders.get(index);
  }

  public List<String> getAllSequenceFileFolders() {
    return new ArrayList<>(sequenceFileFolders);
  }

  private static class DirectoriesHolder {

    private static final DirectoryManager INSTANCE = new DirectoryManager();
  }

  public String getWALFolder() {
    return IoTDBDescriptor.getInstance().getConfig().getWalDir();
  }

  public String getNextFolderForUnSequenceFile() throws DiskSpaceInsufficientException {
    return getUnSequenceFileFolder(getNextFolderIndexForUnSequenceFile());
  }

  /**
   * get next folder index for OverflowFile.
   *
   * @return next folder index
   */
  public int getNextFolderIndexForUnSequenceFile() throws DiskSpaceInsufficientException {
    return unsequenceStrategy.nextFolderIndex();
  }

  public String getUnSequenceFileFolder(int index) {
    return unsequenceFileFolders.get(index);
  }

  public int getUnSequenceFileFolderIndex(String folder) {
    return unsequenceFileFolders.indexOf(folder);
  }

  public List<String> getAllUnSequenceFileFolders() {
    return new ArrayList<>(unsequenceFileFolders);
  }

}
