/*
 * Decompiled with CFR 0.152.
 */
package org.apache.inlong.audit.source;

import com.google.common.base.Preconditions;
import com.google.gson.Gson;
import io.netty.buffer.ByteBuf;
import io.netty.channel.Channel;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.channel.group.ChannelGroup;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.List;
import org.apache.flume.Event;
import org.apache.flume.channel.ChannelProcessor;
import org.apache.flume.event.EventBuilder;
import org.apache.flume.source.AbstractSource;
import org.apache.inlong.audit.protocol.AuditApi;
import org.apache.inlong.audit.protocol.AuditData;
import org.apache.inlong.audit.protocol.Commands;
import org.apache.inlong.audit.source.ServiceDecoder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServerMessageHandler
extends ChannelInboundHandlerAdapter {
    private static final Logger LOGGER = LoggerFactory.getLogger(ServerMessageHandler.class);
    private static final Gson GSON = new Gson();
    private final ChannelGroup allChannels;
    private final ChannelProcessor processor;
    private final ServiceDecoder serviceDecoder;
    private final int maxConnections;
    private final long msgValidThresholdDays;
    private final long ONE_DAY_MS = 86400000L;

    public ServerMessageHandler(AbstractSource source, ServiceDecoder serviceDecoder, ChannelGroup allChannels, Integer maxCons, Long msgValidThresholdDays) {
        this.processor = source.getChannelProcessor();
        this.serviceDecoder = serviceDecoder;
        this.allChannels = allChannels;
        this.maxConnections = maxCons;
        this.msgValidThresholdDays = msgValidThresholdDays;
    }

    public void channelActive(ChannelHandlerContext ctx) {
        if (this.allChannels.size() - 1 >= this.maxConnections) {
            ctx.channel().disconnect();
            ctx.channel().close();
            LOGGER.warn("refuse to connect to channel: {}, connections={}, maxConnections={}", new Object[]{ctx.channel(), this.allChannels.size() - 1, this.maxConnections});
        }
        this.allChannels.add((Object)ctx.channel());
        ctx.fireChannelActive();
    }

    public void channelInactive(ChannelHandlerContext ctx) {
        ctx.fireChannelInactive();
        this.allChannels.remove((Object)ctx.channel());
    }

    public void channelRead(ChannelHandlerContext ctx, Object msg) throws Exception {
        AuditApi.BaseCommand cmd;
        if (msg == null) {
            LOGGER.warn("get null message event, just skip");
            return;
        }
        ByteBuf buf = (ByteBuf)msg;
        int len = buf.readableBytes();
        if (len == 0) {
            LOGGER.warn("receive message skip empty msg");
            buf.clear();
            return;
        }
        Channel remoteChannel = ctx.channel();
        try {
            cmd = this.serviceDecoder.extractData(buf, remoteChannel);
        }
        catch (Exception ex) {
            LOGGER.error("extract data error: ", (Throwable)ex);
            throw new IOException(ex);
        }
        finally {
            buf.release();
        }
        if (cmd == null) {
            LOGGER.warn("extract data from received msg is null");
            return;
        }
        ByteBuf channelBuffer = null;
        switch (cmd.getType()) {
            case PING: {
                Preconditions.checkArgument((boolean)cmd.hasPing());
                channelBuffer = Commands.getPongChannelBuffer();
                break;
            }
            case PONG: {
                Preconditions.checkArgument((boolean)cmd.hasPong());
                channelBuffer = Commands.getPingChannelBuffer();
                break;
            }
            case AUDIT_REQUEST: {
                Preconditions.checkArgument((boolean)cmd.hasAuditRequest());
                AuditApi.AuditReply auditReply = this.handleRequest(cmd.getAuditRequest());
                channelBuffer = Commands.getAuditReplyBuffer((AuditApi.AuditReply)auditReply);
                break;
            }
            case AUDIT_REPLY: {
                Preconditions.checkArgument((boolean)cmd.hasAuditReply());
                break;
            }
        }
        if (channelBuffer != null) {
            this.writeResponse(remoteChannel, channelBuffer);
        }
    }

    private AuditApi.AuditReply handleRequest(AuditApi.AuditRequest auditRequest) throws Exception {
        if (auditRequest == null) {
            throw new Exception("audit request cannot be null");
        }
        AuditApi.AuditReply reply = AuditApi.AuditReply.newBuilder().setRequestId(auditRequest.getRequestId()).setRspCode(AuditApi.AuditReply.RSP_CODE.SUCCESS).build();
        List bodyList = auditRequest.getMsgBodyList();
        int errorMsgBody = 0;
        for (AuditApi.AuditMessageBody auditMessageBody : bodyList) {
            long msgDays = this.messageDays(auditMessageBody.getLogTs());
            if (msgDays >= this.msgValidThresholdDays) {
                LOGGER.warn("Discard the data as it is from {} days ago, only the data with a log timestamp less than {} days is valid", (Object)msgDays, (Object)this.msgValidThresholdDays);
                continue;
            }
            AuditData auditData = new AuditData();
            auditData.setIp(auditRequest.getMsgHeader().getIp());
            auditData.setThreadId(auditRequest.getMsgHeader().getThreadId());
            auditData.setDockerId(auditRequest.getMsgHeader().getDockerId());
            auditData.setPacketId(auditRequest.getMsgHeader().getPacketId());
            auditData.setSdkTs(auditRequest.getMsgHeader().getSdkTs());
            auditData.setLogTs(auditMessageBody.getLogTs());
            auditData.setAuditId(auditMessageBody.getAuditId());
            auditData.setCount(auditMessageBody.getCount());
            auditData.setDelay(auditMessageBody.getDelay());
            auditData.setInlongGroupId(auditMessageBody.getInlongGroupId());
            auditData.setInlongStreamId(auditMessageBody.getInlongStreamId());
            auditData.setSize(auditMessageBody.getSize());
            try {
                byte[] body = GSON.toJson((Object)auditData).getBytes(StandardCharsets.UTF_8);
                Event event = EventBuilder.withBody((byte[])body, null);
                this.processor.processEvent(event);
            }
            catch (Throwable ex) {
                LOGGER.error("writing data error, discard it: ", ex);
                ++errorMsgBody;
            }
        }
        if (errorMsgBody != 0) {
            reply = reply.toBuilder().setMessage("writing data error, discard it, error body count=" + errorMsgBody).setRspCode(AuditApi.AuditReply.RSP_CODE.FAILED).build();
        }
        return reply;
    }

    public long messageDays(long logTs) {
        long currentTime = System.currentTimeMillis();
        long timeDiff = currentTime - logTs;
        return timeDiff / 86400000L;
    }

    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
        LOGGER.error("exception caught", cause);
    }

    private void writeResponse(Channel channel, ByteBuf buffer) throws Exception {
        if (channel.isWritable()) {
            channel.writeAndFlush((Object)buffer);
            return;
        }
        buffer.release();
        String msg = String.format("remote channel=%s is not writable, please check remote client!", channel);
        LOGGER.warn(msg);
        throw new Exception(msg);
    }
}

