/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.dispatcher;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collection;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.common.JobID;
import org.apache.flink.api.common.JobStatus;
import org.apache.flink.api.common.time.Time;
import org.apache.flink.runtime.concurrent.ScheduledExecutor;
import org.apache.flink.runtime.dispatcher.ExecutionGraphInfoStore;
import org.apache.flink.runtime.executiongraph.ArchivedExecutionGraph;
import org.apache.flink.runtime.messages.webmonitor.JobDetails;
import org.apache.flink.runtime.messages.webmonitor.JobsOverview;
import org.apache.flink.runtime.scheduler.ExecutionGraphInfo;
import org.apache.flink.shaded.guava18.com.google.common.base.Ticker;
import org.apache.flink.shaded.guava18.com.google.common.cache.Cache;
import org.apache.flink.shaded.guava18.com.google.common.cache.CacheBuilder;
import org.apache.flink.shaded.guava18.com.google.common.cache.CacheLoader;
import org.apache.flink.shaded.guava18.com.google.common.cache.LoadingCache;
import org.apache.flink.util.FileUtils;
import org.apache.flink.util.InstantiationUtil;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.ShutdownHookUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileExecutionGraphInfoStore
implements ExecutionGraphInfoStore {
    private static final Logger LOG = LoggerFactory.getLogger(FileExecutionGraphInfoStore.class);
    private final File storageDir;
    private final Cache<JobID, JobDetails> jobDetailsCache;
    private final LoadingCache<JobID, ExecutionGraphInfo> executionGraphInfoCache;
    private final ScheduledFuture<?> cleanupFuture;
    private final Thread shutdownHook;
    private int numFinishedJobs;
    private int numFailedJobs;
    private int numCanceledJobs;

    public FileExecutionGraphInfoStore(File rootDir, Time expirationTime, int maximumCapacity, long maximumCacheSizeBytes, ScheduledExecutor scheduledExecutor, Ticker ticker) throws IOException {
        File storageDirectory = FileExecutionGraphInfoStore.initExecutionGraphStorageDirectory(rootDir);
        LOG.info("Initializing {}: Storage directory {}, expiration time {}, maximum cache size {} bytes.", new Object[]{FileExecutionGraphInfoStore.class.getSimpleName(), storageDirectory, expirationTime.toMilliseconds(), maximumCacheSizeBytes});
        this.storageDir = (File)Preconditions.checkNotNull((Object)storageDirectory);
        Preconditions.checkArgument((storageDirectory.exists() && storageDirectory.isDirectory() ? 1 : 0) != 0, (Object)"The storage directory must exist and be a directory.");
        this.jobDetailsCache = CacheBuilder.newBuilder().expireAfterWrite(expirationTime.toMilliseconds(), TimeUnit.MILLISECONDS).maximumSize((long)maximumCapacity).removalListener(notification -> this.deleteExecutionGraphFile((JobID)notification.getKey())).ticker(ticker).build();
        this.executionGraphInfoCache = CacheBuilder.newBuilder().maximumWeight(maximumCacheSizeBytes).weigher(this::calculateSize).build((CacheLoader)new CacheLoader<JobID, ExecutionGraphInfo>(){

            public ExecutionGraphInfo load(JobID jobId) throws Exception {
                return FileExecutionGraphInfoStore.this.loadExecutionGraph(jobId);
            }
        });
        this.cleanupFuture = scheduledExecutor.scheduleWithFixedDelay(() -> this.jobDetailsCache.cleanUp(), expirationTime.toMilliseconds(), expirationTime.toMilliseconds(), TimeUnit.MILLISECONDS);
        this.shutdownHook = ShutdownHookUtil.addShutdownHook((AutoCloseable)this, (String)this.getClass().getSimpleName(), (Logger)LOG);
        this.numFinishedJobs = 0;
        this.numFailedJobs = 0;
        this.numCanceledJobs = 0;
    }

    @Override
    public int size() {
        return Math.toIntExact(this.jobDetailsCache.size());
    }

    @Override
    @Nullable
    public ExecutionGraphInfo get(JobID jobId) {
        try {
            return (ExecutionGraphInfo)this.executionGraphInfoCache.get((Object)jobId);
        }
        catch (ExecutionException e) {
            LOG.debug("Could not load archived execution graph information for job id {}.", (Object)jobId, (Object)e);
            return null;
        }
    }

    @Override
    public void put(ExecutionGraphInfo executionGraphInfo) throws IOException {
        JobID jobId = executionGraphInfo.getJobId();
        ArchivedExecutionGraph archivedExecutionGraph = executionGraphInfo.getArchivedExecutionGraph();
        JobStatus jobStatus = archivedExecutionGraph.getState();
        String jobName = archivedExecutionGraph.getJobName();
        Preconditions.checkArgument((boolean)jobStatus.isTerminalState(), (Object)("The job " + jobName + '(' + jobId + ") is not in a terminal state. Instead it is in state " + jobStatus + '.'));
        switch (jobStatus) {
            case FINISHED: {
                ++this.numFinishedJobs;
                break;
            }
            case CANCELED: {
                ++this.numCanceledJobs;
                break;
            }
            case FAILED: {
                ++this.numFailedJobs;
                break;
            }
            case SUSPENDED: {
                break;
            }
            default: {
                throw new IllegalStateException("The job " + jobName + '(' + jobId + ") should have been in a known terminal state. Instead it was in state " + jobStatus + '.');
            }
        }
        this.storeExecutionGraphInfo(executionGraphInfo);
        JobDetails detailsForJob = JobDetails.createDetailsForJob(archivedExecutionGraph);
        this.jobDetailsCache.put((Object)jobId, (Object)detailsForJob);
        this.executionGraphInfoCache.put((Object)jobId, (Object)executionGraphInfo);
    }

    @Override
    public JobsOverview getStoredJobsOverview() {
        return new JobsOverview(0, this.numFinishedJobs, this.numCanceledJobs, this.numFailedJobs);
    }

    @Override
    public Collection<JobDetails> getAvailableJobDetails() {
        return this.jobDetailsCache.asMap().values();
    }

    @Override
    @Nullable
    public JobDetails getAvailableJobDetails(JobID jobId) {
        return (JobDetails)this.jobDetailsCache.getIfPresent((Object)jobId);
    }

    @Override
    public void close() throws IOException {
        this.cleanupFuture.cancel(false);
        this.jobDetailsCache.invalidateAll();
        FileUtils.deleteFileOrDirectory((File)this.storageDir);
        ShutdownHookUtil.removeShutdownHook((Thread)this.shutdownHook, (String)this.getClass().getSimpleName(), (Logger)LOG);
    }

    private int calculateSize(JobID jobId, ExecutionGraphInfo serializableExecutionGraphInfo) {
        File executionGraphInfoFile = this.getExecutionGraphFile(jobId);
        if (executionGraphInfoFile.exists()) {
            return Math.toIntExact(executionGraphInfoFile.length());
        }
        LOG.debug("Could not find execution graph information file for {}. Estimating the size instead.", (Object)jobId);
        ArchivedExecutionGraph serializableExecutionGraph = serializableExecutionGraphInfo.getArchivedExecutionGraph();
        return serializableExecutionGraph.getAllVertices().size() * 1000 + serializableExecutionGraph.getAccumulatorsSerialized().size() * 1000;
    }

    private ExecutionGraphInfo loadExecutionGraph(JobID jobId) throws IOException, ClassNotFoundException {
        File executionGraphInfoFile = this.getExecutionGraphFile(jobId);
        if (executionGraphInfoFile.exists()) {
            try (FileInputStream fileInputStream = new FileInputStream(executionGraphInfoFile);){
                ExecutionGraphInfo executionGraphInfo = (ExecutionGraphInfo)InstantiationUtil.deserializeObject((InputStream)fileInputStream, (ClassLoader)this.getClass().getClassLoader());
                return executionGraphInfo;
            }
        }
        throw new FileNotFoundException("Could not find file for archived execution graph " + jobId + ". This indicates that the file either has been deleted or never written.");
    }

    private void storeExecutionGraphInfo(ExecutionGraphInfo executionGraphInfo) throws IOException {
        File archivedExecutionGraphFile = this.getExecutionGraphFile(executionGraphInfo.getJobId());
        try (FileOutputStream fileOutputStream = new FileOutputStream(archivedExecutionGraphFile);){
            InstantiationUtil.serializeObject((OutputStream)fileOutputStream, (Object)executionGraphInfo);
        }
    }

    private File getExecutionGraphFile(JobID jobId) {
        return new File(this.storageDir, jobId.toString());
    }

    private void deleteExecutionGraphFile(JobID jobId) {
        Preconditions.checkNotNull((Object)jobId);
        File archivedExecutionGraphFile = this.getExecutionGraphFile(jobId);
        try {
            FileUtils.deleteFileOrDirectory((File)archivedExecutionGraphFile);
        }
        catch (IOException e) {
            LOG.debug("Could not delete file {}.", (Object)archivedExecutionGraphFile, (Object)e);
        }
        this.executionGraphInfoCache.invalidate((Object)jobId);
        this.jobDetailsCache.invalidate((Object)jobId);
    }

    private static File initExecutionGraphStorageDirectory(File tmpDir) throws IOException {
        int maxAttempts = 10;
        for (int attempt = 0; attempt < 10; ++attempt) {
            File storageDirectory = new File(tmpDir, "executionGraphStore-" + UUID.randomUUID());
            if (!storageDirectory.mkdir()) continue;
            return storageDirectory;
        }
        throw new IOException("Could not create executionGraphStorage directory in " + tmpDir + '.');
    }

    @VisibleForTesting
    File getStorageDir() {
        return this.storageDir;
    }

    @VisibleForTesting
    LoadingCache<JobID, ExecutionGraphInfo> getExecutionGraphInfoCache() {
        return this.executionGraphInfoCache;
    }
}

