/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.inlong.manager.service.node;

import com.github.pagehelper.Page;
import com.github.pagehelper.PageHelper;
import org.apache.commons.lang3.StringUtils;
import org.apache.inlong.manager.common.consts.InlongConstants;
import org.apache.inlong.manager.common.enums.ErrorCodeEnum;
import org.apache.inlong.manager.common.enums.UserTypeEnum;
import org.apache.inlong.manager.common.exceptions.BusinessException;
import org.apache.inlong.manager.common.util.Preconditions;
import org.apache.inlong.manager.dao.entity.DataNodeEntity;
import org.apache.inlong.manager.dao.mapper.DataNodeEntityMapper;
import org.apache.inlong.manager.pojo.common.PageResult;
import org.apache.inlong.manager.pojo.common.UpdateResult;
import org.apache.inlong.manager.pojo.node.DataNodeInfo;
import org.apache.inlong.manager.pojo.node.DataNodePageRequest;
import org.apache.inlong.manager.pojo.node.DataNodeRequest;
import org.apache.inlong.manager.pojo.user.UserInfo;
import org.apache.inlong.manager.service.user.UserService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * Data node service layer implementation
 */
@Service
public class DataNodeServiceImpl implements DataNodeService {

    private static final Logger LOGGER = LoggerFactory.getLogger(DataNodeServiceImpl.class);

    @Autowired
    private DataNodeEntityMapper dataNodeMapper;
    @Autowired
    private DataNodeOperatorFactory operatorFactory;
    @Autowired
    private UserService userService;

    @Override
    public Integer save(DataNodeRequest request, String operator) {
        // The name cannot be modified and is automatically generated by the system
        String name = request.getName();
        if (StringUtils.isBlank(name)) {
            name = UUID.randomUUID().toString();
            request.setName(name);
        }
        String type = request.getType();

        // check if data node already exist
        DataNodeEntity existEntity = dataNodeMapper.selectByUniqueKey(name, type);
        if (existEntity != null) {
            String errMsg = String.format("data node already exist for name=%s type=%s)", name, type);
            LOGGER.error(errMsg);
            throw new BusinessException(errMsg);
        }
        // according to the data type, save sink information
        DataNodeOperator dataNodeOperator = operatorFactory.getInstance(request.getType());
        int id = dataNodeOperator.saveOpt(request, operator);
        LOGGER.debug("success to save data node={}", request);
        return id;
    }

    @Override
    public Integer save(DataNodeRequest request, UserInfo opInfo) {
        // The name cannot be modified and is automatically generated by the system
        String name = request.getName();
        if (StringUtils.isBlank(name)) {
            name = UUID.randomUUID().toString();
            request.setName(name);
        }

        // only the person in charges can query
        if (!opInfo.getAccountType().equals(UserTypeEnum.ADMIN.getCode())) {
            throw new BusinessException(ErrorCodeEnum.PERMISSION_REQUIRED);
        }
        // check if data node already exist
        DataNodeEntity existEntity =
                dataNodeMapper.selectByUniqueKey(request.getName(), request.getType());
        if (existEntity != null) {
            throw new BusinessException(ErrorCodeEnum.RECORD_DUPLICATE,
                    String.format("data node already exist for name=%s type=%s",
                            request.getName(), request.getType()));
        }
        // according to the data type, save sink information
        DataNodeOperator dataNodeOperator = operatorFactory.getInstance(request.getType());
        return dataNodeOperator.saveOpt(request, opInfo.getName());
    }

    @Override
    public DataNodeInfo get(Integer id, String currentUser) {
        DataNodeEntity entity = dataNodeMapper.selectById(id);
        if (entity == null) {
            LOGGER.error("data node not found by id={}", id);
            throw new BusinessException("data node not found");
        }
        userService.checkUser(entity.getInCharges(), currentUser,
                "Current user does not have permission to get data node info");
        String dataNodeType = entity.getType();
        DataNodeOperator dataNodeOperator = operatorFactory.getInstance(dataNodeType);
        DataNodeInfo dataNodeInfo = dataNodeOperator.getFromEntity(entity);
        LOGGER.debug("success to get data node info by id={}", id);
        return dataNodeInfo;
    }

    @Override
    public DataNodeInfo get(Integer id, UserInfo opInfo) {
        DataNodeEntity entity = dataNodeMapper.selectById(id);
        if (entity == null) {
            throw new BusinessException(ErrorCodeEnum.DATA_NODE_NOT_FOUND);
        }
        // only the person in charges can query
        userService.checkUser(entity.getInCharges(), opInfo.getName(),
                "Current user does not have permission to get data node info");
        DataNodeOperator dataNodeOperator = operatorFactory.getInstance(entity.getType());
        return dataNodeOperator.getFromEntity(entity);
    }

    @Override
    public DataNodeInfo get(String name, String type) {
        DataNodeEntity entity = dataNodeMapper.selectByUniqueKey(name, type);
        if (entity == null) {
            String errMsg = String.format("data node not found by name=%s, type=%s", name, type);
            LOGGER.error(errMsg);
            throw new BusinessException(errMsg);
        }

        DataNodeOperator dataNodeOperator = operatorFactory.getInstance(type);
        DataNodeInfo dataNodeInfo = dataNodeOperator.getFromEntity(entity);
        LOGGER.debug("success to get data node by name={} type={}", name, type);
        return dataNodeInfo;
    }

    @Override
    public PageResult<DataNodeInfo> list(DataNodePageRequest request) {
        PageHelper.startPage(request.getPageNum(), request.getPageSize());
        Page<DataNodeEntity> entityPage = (Page<DataNodeEntity>) dataNodeMapper.selectByCondition(request);
        List<DataNodeInfo> list = entityPage.stream()
                .map(entity -> {
                    DataNodeOperator dataNodeOperator = operatorFactory.getInstance(entity.getType());
                    return dataNodeOperator.getFromEntity(entity);
                }).collect(Collectors.toList());

        PageResult<DataNodeInfo> pageResult = new PageResult<>(list, entityPage.getTotal(),
                entityPage.getPageNum(), entityPage.getPageSize());

        LOGGER.debug("success to list data node by {}", request);
        return pageResult;
    }

    @Override
    public List<DataNodeInfo> list(DataNodePageRequest request, UserInfo opInfo) {
        request.setCurrentUser(opInfo.getName());
        request.setIsAdminRole(opInfo.getRoles().contains(UserTypeEnum.ADMIN.name()));
        // query result
        List<DataNodeEntity> nodeEntities = dataNodeMapper.selectByCondition(request);
        return nodeEntities.stream()
                .map(entity -> {
                    DataNodeOperator dataNodeOperator = operatorFactory.getInstance(entity.getType());
                    return dataNodeOperator.getFromEntity(entity);
                }).collect(Collectors.toList());
    }

    @Override
    @Transactional(rollbackFor = Throwable.class)
    public Boolean update(DataNodeRequest request, String operator) {
        LOGGER.info("begin to update data node by id: {}", request);
        // check whether record existed
        DataNodeEntity curEntity = dataNodeMapper.selectById(request.getId());
        if (curEntity == null) {
            throw new BusinessException(ErrorCodeEnum.RECORD_NOT_FOUND,
                    String.format("data node record not found by id=%d", request.getId()));
        }
        userService.checkUser(curEntity.getInCharges(), operator,
                "Current user does not have permission to update data node info");
        // check whether modify unmodifiable parameters
        chkUnmodifiableParams(curEntity, request);
        // Check whether the data node name exists with the same name and type
        if (request.getName() != null) {
            if (StringUtils.isBlank(request.getName())) {
                throw new BusinessException(ErrorCodeEnum.INVALID_PARAMETER,
                        "the name changed of data node is blank!");
            }
            DataNodeEntity existEntity =
                    dataNodeMapper.selectByUniqueKey(request.getName(), request.getType());
            if (existEntity != null && !existEntity.getId().equals(request.getId())) {
                throw new BusinessException(ErrorCodeEnum.RECORD_DUPLICATE,
                        String.format("data node already exist for name=%s, type=%s, required id=%s, exist id=%s",
                                request.getName(), request.getType(), request.getId(), existEntity.getId()));
            }
        }
        DataNodeOperator dataNodeOperator = operatorFactory.getInstance(request.getType());
        dataNodeOperator.updateOpt(request, operator);
        dataNodeOperator.updateRelatedStreamSource(request, curEntity, operator);
        LOGGER.info("success to update data node={}", request);
        return true;
    }

    @Override
    @Transactional(rollbackFor = Throwable.class)
    public Boolean update(DataNodeRequest request, UserInfo opInfo) {
        // only the person in charges can query
        if (!opInfo.getAccountType().equals(UserTypeEnum.ADMIN.getCode())) {
            throw new BusinessException(ErrorCodeEnum.PERMISSION_REQUIRED);
        }
        // check the record existed
        DataNodeEntity curEntity = dataNodeMapper.selectById(request.getId());
        if (curEntity == null) {
            throw new BusinessException(ErrorCodeEnum.RECORD_NOT_FOUND,
                    String.format("data node record not found by id=%d", request.getId()));
        }
        userService.checkUser(curEntity.getInCharges(), opInfo.getName(),
                "Current user does not have permission to update data node info");
        // check whether modify unmodifiable parameters
        chkUnmodifiableParams(curEntity, request);
        // Check whether the data node name exists with the same name and type
        if (request.getName() != null) {
            if (StringUtils.isBlank(request.getName())) {
                throw new BusinessException(ErrorCodeEnum.INVALID_PARAMETER,
                        "the name changed of data node is blank!");
            }
            DataNodeEntity existEntity =
                    dataNodeMapper.selectByUniqueKey(request.getName(), request.getType());
            if (existEntity != null && !existEntity.getId().equals(request.getId())) {
                throw new BusinessException(ErrorCodeEnum.RECORD_DUPLICATE,
                        String.format("data node already exist for name=%s, type=%s, required id=%s, exist id=%s",
                                request.getName(), request.getType(), request.getId(), existEntity.getId()));
            }
        }
        DataNodeOperator dataNodeOperator = operatorFactory.getInstance(request.getType());
        dataNodeOperator.updateOpt(request, opInfo.getName());
        dataNodeOperator.updateRelatedStreamSource(request, curEntity, opInfo.getName());
        return true;
    }

    @Override
    @Transactional(rollbackFor = Throwable.class)
    public UpdateResult updateByKey(DataNodeRequest request, String operator) {
        LOGGER.info("begin to update data node by key: {}", request);

        // Check whether the data node exists
        String name = request.getName();
        String type = request.getType();
        DataNodeEntity entity = dataNodeMapper.selectByUniqueKey(name, type);
        if (entity == null) {
            String errMsg = String.format("data node not found by name=%s, type=%s", name, type);
            LOGGER.error(errMsg);
            throw new BusinessException(errMsg);
        }
        userService.checkUser(entity.getInCharges(), operator,
                "Current user does not have permission to update data node info");
        request.setId(entity.getId());
        Boolean result = this.update(request, operator);
        LOGGER.info("success to update data node by key: {}", request);
        return new UpdateResult(entity.getId(), result, request.getVersion() + 1);
    }

    @Override
    public Boolean delete(Integer id, String operator) {
        DataNodeEntity entity = dataNodeMapper.selectById(id);
        Preconditions.expectNotNull(entity, ErrorCodeEnum.DATA_NODE_NOT_FOUND,
                ErrorCodeEnum.DATA_NODE_NOT_FOUND.getMessage());
        userService.checkUser(entity.getInCharges(), operator,
                "Current user does not have permission to delete data node info");
        return delete(entity, operator);
    }

    @Override
    public Boolean delete(Integer id, UserInfo opInfo) {
        // only the person in charges can query
        if (!opInfo.getAccountType().equals(UserTypeEnum.ADMIN.getCode())) {
            throw new BusinessException(ErrorCodeEnum.PERMISSION_REQUIRED);
        }
        DataNodeEntity entity = dataNodeMapper.selectById(id);
        Preconditions.expectNotNull(entity, ErrorCodeEnum.DATA_NODE_NOT_FOUND,
                ErrorCodeEnum.DATA_NODE_NOT_FOUND.getMessage());
        userService.checkUser(entity.getInCharges(), opInfo.getName(),
                "Current user does not have permission to delete data node info");
        // delete record
        entity.setIsDeleted(entity.getId());
        entity.setModifier(opInfo.getName());
        int rowCount = dataNodeMapper.updateById(entity);
        if (rowCount != InlongConstants.AFFECTED_ONE_ROW) {
            throw new BusinessException(ErrorCodeEnum.CONFIG_EXPIRED,
                    String.format("data node has already updated, data node name=%s, type=%s, current version=%s",
                            entity.getName(), entity.getType(), entity.getVersion()));
        }
        return true;
    }

    private Boolean delete(DataNodeEntity entity, String operator) {
        entity.setIsDeleted(entity.getId());
        entity.setModifier(operator);
        int rowCount = dataNodeMapper.updateById(entity);
        if (rowCount != InlongConstants.AFFECTED_ONE_ROW) {
            LOGGER.error("data node has already updated, data node name={}, type={}, current version ={}",
                    entity.getName(), entity.getType(), entity.getVersion());
            throw new BusinessException(ErrorCodeEnum.CONFIG_EXPIRED);
        }
        LOGGER.info("success to delete data node by id={}, name={}", entity.getId(), entity.getName());
        return true;
    }

    private void chkUnmodifiableParams(DataNodeEntity curEntity, DataNodeRequest request) {
        // check type
        Preconditions.expectEquals(curEntity.getType(), request.getType(),
                ErrorCodeEnum.INVALID_PARAMETER, "type not allowed modify");
        if (StringUtils.isBlank(request.getName())) {
            request.setName(curEntity.getName());
        } else {
            // check name
            Preconditions.expectEquals(curEntity.getName(), request.getName(),
                    ErrorCodeEnum.INVALID_PARAMETER, "name not allowed modify");
        }
        // check record version
        Preconditions.expectEquals(curEntity.getVersion(), request.getVersion(),
                ErrorCodeEnum.CONFIG_EXPIRED,
                String.format("record has expired with record version=%d, request version=%d",
                        curEntity.getVersion(), request.getVersion()));
    }

    @Override
    public Boolean deleteByKey(String name, String type, String operator) {
        DataNodeEntity entity = dataNodeMapper.selectByUniqueKey(name, type);
        Preconditions.expectNotNull(entity, ErrorCodeEnum.DATA_NODE_NOT_FOUND,
                ErrorCodeEnum.DATA_NODE_NOT_FOUND.getMessage());
        userService.checkUser(entity.getInCharges(), operator,
                "Current user does not have permission to delete data node info");
        return delete(entity, operator);
    }

    @Override
    public Boolean testConnection(DataNodeRequest request) {
        LOGGER.info("begin test connection for: {}", request);
        String type = request.getType();

        // according to the data node type, test connection
        DataNodeOperator dataNodeOperator = operatorFactory.getInstance(request.getType());
        Boolean result = dataNodeOperator.testConnection(request);
        LOGGER.info("connection [{}] for: {}", result ? "success" : "failed", request);
        return result;
    }

}
