/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flume.channel.file;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.io.EOFException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.Nullable;
import org.apache.flume.annotations.InterfaceAudience;
import org.apache.flume.annotations.InterfaceStability;
import org.apache.flume.channel.file.CorruptEventException;
import org.apache.flume.channel.file.FlumeEvent;
import org.apache.flume.channel.file.FlumeEventPointer;
import org.apache.flume.channel.file.LogFileFactory;
import org.apache.flume.channel.file.LogFileRetryableIOException;
import org.apache.flume.channel.file.LogRecord;
import org.apache.flume.channel.file.NoopRecordException;
import org.apache.flume.channel.file.Pair;
import org.apache.flume.channel.file.Put;
import org.apache.flume.channel.file.Take;
import org.apache.flume.channel.file.TransactionEventRecord;
import org.apache.flume.channel.file.encryption.CipherProvider;
import org.apache.flume.channel.file.encryption.KeyProvider;
import org.apache.flume.tools.DirectMemoryUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public abstract class LogFile {
    private static final Logger LOG = LoggerFactory.getLogger(LogFile.class);
    private static final ByteBuffer FILL = DirectMemoryUtils.allocate(0x100000);
    public static final byte OP_RECORD = 127;
    public static final byte OP_NOOP = 0;
    public static final byte OP_EOF = -128;

    protected static void skipRecord(RandomAccessFile fileHandle, int offset) throws IOException {
        fileHandle.seek(offset);
        int length = fileHandle.readInt();
        fileHandle.skipBytes(length);
    }

    protected static void writeDelimitedBuffer(ByteBuffer output, ByteBuffer buffer) throws IOException {
        output.putInt(buffer.limit());
        output.put(buffer);
    }

    protected static byte[] readDelimitedBuffer(RandomAccessFile fileHandle) throws IOException, CorruptEventException {
        int length = fileHandle.readInt();
        if (length < 0) {
            throw new CorruptEventException("Length of event is: " + String.valueOf(length) + ". Event must have length >= 0. Possible corruption of data or partial fsync.");
        }
        byte[] buffer = new byte[length];
        try {
            fileHandle.readFully(buffer);
        }
        catch (EOFException ex) {
            throw new CorruptEventException("Remaining data in file less than expected size of event.", ex);
        }
        return buffer;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void main(String[] args) throws EOFException, IOException, CorruptEventException {
        File file = new File(args[0]);
        try (SequentialReader reader = null;){
            LogRecord entry;
            reader = LogFileFactory.getSequentialReader(file, null, false);
            int fileId = reader.getLogFileID();
            int count = 0;
            int readCount = 0;
            int putCount = 0;
            int takeCount = 0;
            int rollbackCount = 0;
            int commitCount = 0;
            while ((entry = reader.next()) != null) {
                int offset = entry.getOffset();
                TransactionEventRecord record = entry.getEvent();
                short type = record.getRecordType();
                long trans = record.getTransactionID();
                long ts = record.getLogWriteOrderID();
                ++readCount;
                FlumeEventPointer ptr = null;
                if (type == TransactionEventRecord.Type.PUT.get()) {
                    ++putCount;
                    ptr = new FlumeEventPointer(fileId, offset);
                } else if (type == TransactionEventRecord.Type.TAKE.get()) {
                    ++takeCount;
                    Take take = (Take)record;
                    ptr = new FlumeEventPointer(take.getFileID(), take.getOffset());
                } else if (type == TransactionEventRecord.Type.ROLLBACK.get()) {
                    ++rollbackCount;
                } else if (type == TransactionEventRecord.Type.COMMIT.get()) {
                    ++commitCount;
                } else {
                    Preconditions.checkArgument(false, "Unknown record type: " + Integer.toHexString(type));
                }
                System.out.println(Joiner.on(", ").skipNulls().join(trans, ts, fileId, offset, TransactionEventRecord.getName(type), ptr));
            }
            System.out.println("Replayed " + count + " from " + file + " read: " + readCount + ", put: " + putCount + ", take: " + takeCount + ", rollback: " + rollbackCount + ", commit: " + commitCount);
        }
    }

    static {
        for (int i = 0; i < FILL.capacity(); ++i) {
            FILL.put((byte)-128);
        }
    }

    public static abstract class SequentialReader {
        private final RandomAccessFile fileHandle;
        private final FileChannel fileChannel;
        private final File file;
        private final KeyProvider encryptionKeyProvider;
        private int logFileID;
        private long lastCheckpointPosition;
        private long lastCheckpointWriteOrderID;
        private long backupCheckpointPosition;
        private long backupCheckpointWriteOrderID;

        SequentialReader(File file, @Nullable KeyProvider encryptionKeyProvider) throws IOException, EOFException {
            this.file = file;
            this.encryptionKeyProvider = encryptionKeyProvider;
            this.fileHandle = new RandomAccessFile(file, "r");
            this.fileChannel = this.fileHandle.getChannel();
        }

        abstract LogRecord doNext(int var1) throws IOException, CorruptEventException;

        abstract int getVersion();

        protected void setLastCheckpointPosition(long lastCheckpointPosition) {
            this.lastCheckpointPosition = lastCheckpointPosition;
        }

        protected void setLastCheckpointWriteOrderID(long lastCheckpointWriteOrderID) {
            this.lastCheckpointWriteOrderID = lastCheckpointWriteOrderID;
        }

        protected void setPreviousCheckpointPosition(long backupCheckpointPosition) {
            this.backupCheckpointPosition = backupCheckpointPosition;
        }

        protected void setPreviousCheckpointWriteOrderID(long backupCheckpointWriteOrderID) {
            this.backupCheckpointWriteOrderID = backupCheckpointWriteOrderID;
        }

        protected void setLogFileID(int logFileID) {
            this.logFileID = logFileID;
            Preconditions.checkArgument(logFileID >= 0, "LogFileID is not positive: " + Integer.toHexString(logFileID));
        }

        protected KeyProvider getKeyProvider() {
            return this.encryptionKeyProvider;
        }

        protected RandomAccessFile getFileHandle() {
            return this.fileHandle;
        }

        int getLogFileID() {
            return this.logFileID;
        }

        void skipToLastCheckpointPosition(long checkpointWriteOrderID) throws IOException {
            if (this.lastCheckpointPosition > 0L) {
                long position = 0L;
                if (this.lastCheckpointWriteOrderID <= checkpointWriteOrderID) {
                    position = this.lastCheckpointPosition;
                } else if (this.backupCheckpointWriteOrderID <= checkpointWriteOrderID && this.backupCheckpointPosition > 0L) {
                    position = this.backupCheckpointPosition;
                }
                this.fileChannel.position(position);
                LOG.info("fast-forward to checkpoint position: " + position);
            } else {
                LOG.info("Checkpoint for file(" + this.file.getAbsolutePath() + ") is: " + this.lastCheckpointWriteOrderID + ", which is beyond the requested checkpoint time: " + checkpointWriteOrderID + " and position " + this.lastCheckpointPosition);
            }
        }

        public LogRecord next() throws IOException, CorruptEventException {
            int offset = -1;
            try {
                byte operation;
                long position = this.fileChannel.position();
                if (position > 1623195647L) {
                    LOG.info("File position exceeds the threshold: 1623195647, position: " + position);
                }
                Preconditions.checkState((offset = (int)position) >= 0);
                while ((long)offset < this.fileHandle.length() && (operation = this.fileHandle.readByte()) != 127) {
                    if (operation == -128) {
                        LOG.info("Encountered EOF at " + offset + " in " + this.file);
                        return null;
                    }
                    if (operation == 0) {
                        LOG.info("No op event found in file: " + this.file.toString() + " at " + offset + ". Skipping event.");
                        LogFile.skipRecord(this.fileHandle, offset + 1);
                        offset = (int)this.fileHandle.getFilePointer();
                        continue;
                    }
                    LOG.error("Encountered non op-record at " + offset + " " + Integer.toHexString(operation) + " in " + this.file);
                    return null;
                }
                if ((long)offset >= this.fileHandle.length()) {
                    return null;
                }
                return this.doNext(offset);
            }
            catch (EOFException e) {
                return null;
            }
            catch (IOException e) {
                throw new IOException("Unable to read next Transaction from log file " + this.file.getCanonicalPath() + " at offset " + offset, e);
            }
        }

        public long getPosition() throws IOException {
            return this.fileChannel.position();
        }

        public void close() {
            if (this.fileHandle != null) {
                try {
                    this.fileHandle.close();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
        }
    }

    static abstract class RandomReader {
        private final File file;
        private final BlockingQueue<RandomAccessFile> readFileHandles = new ArrayBlockingQueue<RandomAccessFile>(50, true);
        private final KeyProvider encryptionKeyProvider;
        private final boolean fsyncPerTransaction;
        private volatile boolean open;

        public RandomReader(File file, @Nullable KeyProvider encryptionKeyProvider, boolean fsyncPerTransaction) throws IOException {
            this.file = file;
            this.encryptionKeyProvider = encryptionKeyProvider;
            this.readFileHandles.add(this.open());
            this.fsyncPerTransaction = fsyncPerTransaction;
            this.open = true;
        }

        protected abstract TransactionEventRecord doGet(RandomAccessFile var1) throws IOException, CorruptEventException;

        abstract int getVersion();

        File getFile() {
            return this.file;
        }

        protected KeyProvider getKeyProvider() {
            return this.encryptionKeyProvider;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        FlumeEvent get(int offset) throws IOException, InterruptedException, CorruptEventException, NoopRecordException {
            Preconditions.checkState(this.open, "File closed");
            RandomAccessFile fileHandle = this.checkOut();
            boolean error = true;
            try {
                fileHandle.seek(offset);
                byte operation = fileHandle.readByte();
                if (operation == 0) {
                    throw new NoopRecordException("No op record found. Corrupt record may have been repaired by File Channel Integrity tool");
                }
                if (operation != 127) {
                    throw new CorruptEventException("Operation code is invalid. File is corrupt. Please run File Channel Integrity tool.");
                }
                TransactionEventRecord record = this.doGet(fileHandle);
                if (!(record instanceof Put)) {
                    Preconditions.checkState(false, "Record is " + record.getClass().getSimpleName());
                }
                error = false;
                FlumeEvent flumeEvent = ((Put)record).getEvent();
                return flumeEvent;
            }
            finally {
                if (error) {
                    RandomReader.close(fileHandle, this.file);
                } else {
                    this.checkIn(fileHandle);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        synchronized void close() {
            if (this.open) {
                this.open = false;
                LOG.info("Closing RandomReader " + this.file);
                ArrayList fileHandles = Lists.newArrayList();
                while (this.readFileHandles.drainTo(fileHandles) > 0) {
                    Iterator iterator = fileHandles.iterator();
                    while (iterator.hasNext()) {
                        RandomAccessFile fileHandle;
                        RandomAccessFile randomAccessFile = fileHandle = (RandomAccessFile)iterator.next();
                        synchronized (randomAccessFile) {
                            try {
                                fileHandle.close();
                            }
                            catch (IOException e) {
                                LOG.warn("Unable to close fileHandle for " + this.file, e);
                            }
                        }
                    }
                    fileHandles.clear();
                    try {
                        Thread.sleep(5L);
                    }
                    catch (InterruptedException interruptedException) {}
                }
            }
        }

        private RandomAccessFile open() throws IOException {
            return new RandomAccessFile(this.file, "r");
        }

        private void checkIn(RandomAccessFile fileHandle) {
            if (!this.readFileHandles.offer(fileHandle)) {
                RandomReader.close(fileHandle, this.file);
            }
        }

        private RandomAccessFile checkOut() throws IOException, InterruptedException {
            RandomAccessFile fileHandle = (RandomAccessFile)this.readFileHandles.poll();
            if (fileHandle != null) {
                return fileHandle;
            }
            int remaining = this.readFileHandles.remainingCapacity();
            if (remaining > 0) {
                LOG.info("Opening " + this.file + " for read, remaining number of file handles available for reads of this file is " + remaining);
                return this.open();
            }
            return this.readFileHandles.take();
        }

        private static void close(RandomAccessFile fileHandle, File file) {
            if (fileHandle != null) {
                try {
                    fileHandle.close();
                }
                catch (IOException e) {
                    LOG.warn("Unable to close " + file, e);
                }
            }
        }
    }

    public static class OperationRecordUpdater {
        private final RandomAccessFile fileHandle;
        private final File file;

        public OperationRecordUpdater(File file) throws FileNotFoundException {
            Preconditions.checkState(file.exists(), "File to update, " + file.toString() + " does not exist.");
            this.file = file;
            this.fileHandle = new RandomAccessFile(file, "rw");
        }

        public void markRecordAsNoop(long offset) throws IOException {
            this.fileHandle.seek(offset);
            byte byteRead = this.fileHandle.readByte();
            Preconditions.checkState(byteRead == 127 || byteRead == 0, "Expected to read a record but the byte read indicates EOF");
            this.fileHandle.seek(offset);
            LOG.info("Marking event as 0 at " + offset + " for file " + this.file.toString());
            this.fileHandle.writeByte(0);
        }

        public void close() {
            try {
                this.fileHandle.getFD().sync();
                this.fileHandle.close();
            }
            catch (IOException e) {
                LOG.error("Could not close file handle to file " + this.fileHandle.toString(), e);
            }
        }
    }

    static abstract class Writer {
        private final int logFileID;
        private final File file;
        private final long maxFileSize;
        private final RandomAccessFile writeFileHandle;
        private final FileChannel writeFileChannel;
        private final CipherProvider.Encryptor encryptor;
        private final CachedFSUsableSpace usableSpace;
        private volatile boolean open;
        private long lastCommitPosition;
        private long lastSyncPosition;
        private final boolean fsyncPerTransaction;
        private final int fsyncInterval;
        private final ScheduledExecutorService syncExecutor;
        private volatile boolean dirty = false;
        private long syncCount;

        Writer(File file, int logFileID, long maxFileSize, CipherProvider.Encryptor encryptor, long usableSpaceRefreshInterval, boolean fsyncPerTransaction, int fsyncInterval) throws IOException {
            this.file = file;
            this.logFileID = logFileID;
            this.maxFileSize = Math.min(maxFileSize, 1623195647L);
            this.encryptor = encryptor;
            this.writeFileHandle = new RandomAccessFile(file, "rw");
            this.writeFileChannel = this.writeFileHandle.getChannel();
            this.fsyncPerTransaction = fsyncPerTransaction;
            this.fsyncInterval = fsyncInterval;
            if (!fsyncPerTransaction) {
                LOG.info("Sync interval = " + fsyncInterval);
                this.syncExecutor = Executors.newSingleThreadScheduledExecutor();
                this.syncExecutor.scheduleWithFixedDelay(new Runnable(){

                    @Override
                    public void run() {
                        try {
                            this.sync();
                        }
                        catch (Throwable ex) {
                            LOG.error("Data file, " + this.getFile().toString() + " could not be synced to disk due to an error.", ex);
                        }
                    }
                }, fsyncInterval, fsyncInterval, TimeUnit.SECONDS);
            } else {
                this.syncExecutor = null;
            }
            this.usableSpace = new CachedFSUsableSpace(file, usableSpaceRefreshInterval);
            LOG.info("Opened " + file);
            this.open = true;
        }

        abstract int getVersion();

        protected CipherProvider.Encryptor getEncryptor() {
            return this.encryptor;
        }

        int getLogFileID() {
            return this.logFileID;
        }

        File getFile() {
            return this.file;
        }

        String getParent() {
            return this.file.getParent();
        }

        long getUsableSpace() {
            return this.usableSpace.getUsableSpace();
        }

        long getMaxSize() {
            return this.maxFileSize;
        }

        @VisibleForTesting
        long getLastCommitPosition() {
            return this.lastCommitPosition;
        }

        @VisibleForTesting
        long getLastSyncPosition() {
            return this.lastSyncPosition;
        }

        @VisibleForTesting
        long getSyncCount() {
            return this.syncCount;
        }

        synchronized long position() throws IOException {
            return this.getFileChannel().position();
        }

        synchronized FlumeEventPointer put(ByteBuffer buffer) throws IOException {
            if (this.encryptor != null) {
                buffer = ByteBuffer.wrap(this.encryptor.encrypt(buffer.array()));
            }
            Pair<Integer, Integer> pair = this.write(buffer);
            return new FlumeEventPointer(pair.getLeft(), pair.getRight());
        }

        synchronized void take(ByteBuffer buffer) throws IOException {
            if (this.encryptor != null) {
                buffer = ByteBuffer.wrap(this.encryptor.encrypt(buffer.array()));
            }
            this.write(buffer);
        }

        synchronized void rollback(ByteBuffer buffer) throws IOException {
            if (this.encryptor != null) {
                buffer = ByteBuffer.wrap(this.encryptor.encrypt(buffer.array()));
            }
            this.write(buffer);
        }

        synchronized void commit(ByteBuffer buffer) throws IOException {
            if (this.encryptor != null) {
                buffer = ByteBuffer.wrap(this.encryptor.encrypt(buffer.array()));
            }
            this.write(buffer);
            this.dirty = true;
            this.lastCommitPosition = this.position();
        }

        private Pair<Integer, Integer> write(ByteBuffer buffer) throws IOException {
            if (!this.isOpen()) {
                throw new LogFileRetryableIOException("File closed " + this.file);
            }
            long length = this.position();
            long expectedLength = length + (long)buffer.limit();
            if (expectedLength > this.maxFileSize) {
                throw new LogFileRetryableIOException(expectedLength + " > " + this.maxFileSize);
            }
            int offset = (int)length;
            Preconditions.checkState(offset >= 0, String.valueOf(offset));
            int recordLength = 5 + buffer.limit();
            this.usableSpace.decrement(recordLength);
            this.preallocate(recordLength);
            ByteBuffer toWrite = ByteBuffer.allocate(recordLength);
            toWrite.put((byte)127);
            LogFile.writeDelimitedBuffer(toWrite, buffer);
            toWrite.position(0);
            int wrote = this.getFileChannel().write(toWrite);
            Preconditions.checkState(wrote == toWrite.limit());
            return Pair.of(this.getLogFileID(), offset);
        }

        synchronized boolean isRollRequired(ByteBuffer buffer) throws IOException {
            return this.isOpen() && this.position() + (long)buffer.limit() > this.getMaxSize();
        }

        synchronized void sync() throws IOException {
            if (!this.fsyncPerTransaction && !this.dirty) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("No events written to file, " + this.getFile().toString() + " in last " + this.fsyncInterval + " or since last commit.");
                }
                return;
            }
            if (!this.isOpen()) {
                throw new LogFileRetryableIOException("File closed " + this.file);
            }
            if (this.lastSyncPosition < this.lastCommitPosition) {
                this.getFileChannel().force(false);
                this.lastSyncPosition = this.position();
                ++this.syncCount;
                this.dirty = false;
            }
        }

        protected boolean isOpen() {
            return this.open;
        }

        protected RandomAccessFile getFileHandle() {
            return this.writeFileHandle;
        }

        protected FileChannel getFileChannel() {
            return this.writeFileChannel;
        }

        synchronized void close() {
            if (this.open) {
                this.open = false;
                if (!this.fsyncPerTransaction && this.syncExecutor != null) {
                    this.syncExecutor.shutdown();
                }
                if (this.writeFileChannel.isOpen()) {
                    LOG.info("Closing " + this.file);
                    try {
                        this.writeFileChannel.force(true);
                    }
                    catch (IOException e) {
                        LOG.warn("Unable to flush to disk " + this.file, e);
                    }
                    try {
                        this.writeFileHandle.close();
                    }
                    catch (IOException e) {
                        LOG.warn("Unable to close " + this.file, e);
                    }
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected void preallocate(int size) throws IOException {
            long position = this.position();
            if (position + (long)size > this.getFileChannel().size()) {
                LOG.debug("Preallocating at position " + position);
                ByteBuffer byteBuffer = FILL;
                synchronized (byteBuffer) {
                    FILL.position(0);
                    this.getFileChannel().write(FILL, position);
                }
            }
        }
    }

    @VisibleForTesting
    static class CachedFSUsableSpace {
        private final File fs;
        private final long interval;
        private final AtomicLong lastRefresh;
        private final AtomicLong value;

        CachedFSUsableSpace(File fs, long interval) {
            this.fs = fs;
            this.interval = interval;
            this.value = new AtomicLong(fs.getUsableSpace());
            this.lastRefresh = new AtomicLong(System.currentTimeMillis());
        }

        void decrement(long numBytes) {
            Preconditions.checkArgument(numBytes >= 0L, "numBytes less than zero");
            this.value.addAndGet(-numBytes);
        }

        long getUsableSpace() {
            long now = System.currentTimeMillis();
            if (now - this.interval > this.lastRefresh.get()) {
                this.value.set(this.fs.getUsableSpace());
                this.lastRefresh.set(now);
            }
            return Math.max(this.value.get(), 0L);
        }
    }

    static abstract class MetaDataWriter {
        private final File file;
        private final int logFileID;
        private final RandomAccessFile writeFileHandle;
        private long lastCheckpointOffset;
        private long lastCheckpointWriteOrderID;

        protected MetaDataWriter(File file, int logFileID) throws IOException {
            this.file = file;
            this.logFileID = logFileID;
            this.writeFileHandle = new RandomAccessFile(file, "rw");
        }

        protected RandomAccessFile getFileHandle() {
            return this.writeFileHandle;
        }

        protected void setLastCheckpointOffset(long lastCheckpointOffset) {
            this.lastCheckpointOffset = lastCheckpointOffset;
        }

        protected void setLastCheckpointWriteOrderID(long lastCheckpointWriteOrderID) {
            this.lastCheckpointWriteOrderID = lastCheckpointWriteOrderID;
        }

        protected long getLastCheckpointOffset() {
            return this.lastCheckpointOffset;
        }

        protected long getLastCheckpointWriteOrderID() {
            return this.lastCheckpointWriteOrderID;
        }

        protected File getFile() {
            return this.file;
        }

        protected int getLogFileID() {
            return this.logFileID;
        }

        void markCheckpoint(long logWriteOrderID) throws IOException {
            this.markCheckpoint(this.lastCheckpointOffset, logWriteOrderID);
        }

        abstract void markCheckpoint(long var1, long var3) throws IOException;

        abstract int getVersion();

        void close() {
            try {
                this.writeFileHandle.close();
            }
            catch (IOException e) {
                LOG.warn("Unable to close " + this.file, e);
            }
        }
    }
}

