/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flume.channel.file;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableSortedSet;
import com.google.common.collect.Maps;
import com.google.common.collect.SetMultimap;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.LongBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.flume.channel.file.BadCheckpointException;
import org.apache.flume.channel.file.EventQueueBackingStore;
import org.apache.flume.channel.file.EventQueueBackingStoreFactory;
import org.apache.flume.channel.file.FlumeEventQueue;
import org.apache.flume.channel.file.Log;
import org.apache.flume.channel.file.Serialization;
import org.apache.flume.channel.file.WriteOrderOracle;
import org.apache.flume.channel.file.instrumentation.FileChannelCounter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

abstract class EventQueueBackingStoreFile
extends EventQueueBackingStore {
    private static final Logger LOG = LoggerFactory.getLogger(EventQueueBackingStoreFile.class);
    private static final int MAX_ALLOC_BUFFER_SIZE = 0x200000;
    protected static final int HEADER_SIZE = 1029;
    protected static final int INDEX_VERSION = 0;
    protected static final int INDEX_WRITE_ORDER_ID = 1;
    protected static final int INDEX_CHECKPOINT_MARKER = 4;
    protected static final int CHECKPOINT_COMPLETE = 0;
    protected static final int CHECKPOINT_INCOMPLETE = 1;
    protected static final String COMPRESSED_FILE_EXTENSION = ".snappy";
    protected LongBuffer elementsBuffer;
    protected final Map<Integer, Long> overwriteMap = new HashMap<Integer, Long>();
    protected final Map<Integer, AtomicInteger> logFileIDReferenceCounts = Maps.newHashMap();
    protected final MappedByteBuffer mappedBuffer;
    protected final RandomAccessFile checkpointFileHandle;
    private final FileChannelCounter fileChannelCounter;
    protected final File checkpointFile;
    private final Semaphore backupCompletedSema = new Semaphore(1);
    protected final boolean shouldBackup;
    protected final boolean compressBackup;
    private final File backupDir;
    private final ExecutorService checkpointBackUpExecutor;

    protected EventQueueBackingStoreFile(int capacity, String name, FileChannelCounter fileChannelCounter, File checkpointFile) throws IOException, BadCheckpointException {
        this(capacity, name, fileChannelCounter, checkpointFile, null, false, false);
    }

    protected EventQueueBackingStoreFile(int capacity, String name, FileChannelCounter fileChannelCounter, File checkpointFile, File checkpointBackupDir, boolean backupCheckpoint, boolean compressBackup) throws IOException, BadCheckpointException {
        super(capacity, name);
        this.fileChannelCounter = fileChannelCounter;
        this.checkpointFile = checkpointFile;
        this.shouldBackup = backupCheckpoint;
        this.compressBackup = compressBackup;
        this.backupDir = checkpointBackupDir;
        this.checkpointFileHandle = new RandomAccessFile(checkpointFile, "rw");
        long totalBytes = (capacity + 1029) * 8;
        if (this.checkpointFileHandle.length() == 0L) {
            EventQueueBackingStoreFile.allocate(checkpointFile, totalBytes);
            this.checkpointFileHandle.seek(0L);
            this.checkpointFileHandle.writeLong(this.getVersion());
            this.checkpointFileHandle.getChannel().force(true);
            LOG.info("Preallocated " + checkpointFile + " to " + this.checkpointFileHandle.length() + " for capacity " + capacity);
        }
        if (checkpointFile.length() != totalBytes) {
            String msg = "Configured capacity is " + capacity + " but the  checkpoint file capacity is " + (checkpointFile.length() / 8L - 1029L) + ". See FileChannel documentation on how to change a channels capacity.";
            throw new BadCheckpointException(msg);
        }
        this.mappedBuffer = this.checkpointFileHandle.getChannel().map(FileChannel.MapMode.READ_WRITE, 0L, checkpointFile.length());
        this.elementsBuffer = this.mappedBuffer.asLongBuffer();
        long version = this.elementsBuffer.get(0);
        if (version != (long)this.getVersion()) {
            throw new BadCheckpointException("Invalid version: " + version + " " + name + ", expected " + this.getVersion());
        }
        long checkpointComplete = this.elementsBuffer.get(4);
        if (checkpointComplete != 0L) {
            throw new BadCheckpointException("Checkpoint was not completed correctly, probably because the agent stopped while the channel was checkpointing.");
        }
        this.checkpointBackUpExecutor = this.shouldBackup ? Executors.newSingleThreadExecutor(new ThreadFactoryBuilder().setNameFormat(this.getName() + " - CheckpointBackUpThread").build()) : null;
    }

    protected long getCheckpointLogWriteOrderID() {
        return this.elementsBuffer.get(1);
    }

    protected abstract void writeCheckpointMetaData() throws IOException;

    protected void backupCheckpoint(File backupDirectory) throws IOException {
        int availablePermits = this.backupCompletedSema.drainPermits();
        Preconditions.checkState(availablePermits == 0, "Expected no permits to be available in the backup semaphore, but " + availablePermits + " permits were available.");
        if (this.slowdownBackup.booleanValue()) {
            try {
                TimeUnit.SECONDS.sleep(10L);
            }
            catch (Exception ex) {
                Throwables.propagate(ex);
            }
        }
        File backupFile = new File(backupDirectory, "backupComplete");
        if (EventQueueBackingStoreFile.backupExists(backupDirectory) && !backupFile.delete()) {
            throw new IOException("Error while doing backup of checkpoint. Could not remove" + backupFile.toString() + ".");
        }
        Serialization.deleteAllFiles(backupDirectory, Log.EXCLUDES);
        File checkpointDir = this.checkpointFile.getParentFile();
        File[] checkpointFiles = checkpointDir.listFiles();
        Preconditions.checkNotNull(checkpointFiles, "Could not retrieve files from the checkpoint directory. Cannot complete backup of the checkpoint.");
        for (File origFile : checkpointFiles) {
            if (Log.EXCLUDES.contains(origFile.getName())) continue;
            if (this.compressBackup && origFile.equals(this.checkpointFile)) {
                Serialization.compressFile(origFile, new File(backupDirectory, origFile.getName() + COMPRESSED_FILE_EXTENSION));
                continue;
            }
            Serialization.copyFile(origFile, new File(backupDirectory, origFile.getName()));
        }
        Preconditions.checkState(!backupFile.exists(), "The backup file exists while it is not supposed to. Are multiple channels configured to use this directory: " + backupDirectory.toString() + " as backup?");
        if (!backupFile.createNewFile()) {
            LOG.error("Could not create backup file. Backup of checkpoint will not be used during replay even if checkpoint is bad.");
        }
    }

    public static boolean restoreBackup(File checkpointDir, File backupDir) throws IOException {
        if (!EventQueueBackingStoreFile.backupExists(backupDir)) {
            return false;
        }
        Serialization.deleteAllFiles(checkpointDir, Log.EXCLUDES);
        File[] backupFiles = backupDir.listFiles();
        if (backupFiles == null) {
            return false;
        }
        for (File backupFile : backupFiles) {
            String fileName = backupFile.getName();
            if (fileName.equals("backupComplete") || fileName.equals("in_use.lock")) continue;
            if (fileName.endsWith(COMPRESSED_FILE_EXTENSION)) {
                Serialization.decompressFile(backupFile, new File(checkpointDir, fileName.substring(0, fileName.lastIndexOf("."))));
                continue;
            }
            Serialization.copyFile(backupFile, new File(checkpointDir, fileName));
        }
        return true;
    }

    @Override
    void beginCheckpoint() throws IOException {
        LOG.info("Start checkpoint for " + this.checkpointFile + ", elements to sync = " + this.overwriteMap.size());
        if (this.shouldBackup) {
            int permits = this.backupCompletedSema.drainPermits();
            Preconditions.checkState(permits <= 1, "Expected only one or less permits to checkpoint, but got " + String.valueOf(permits) + " permits");
            if (permits < 1) {
                throw new IOException("Previous backup of checkpoint files is still in progress. Will attempt to checkpoint only at the end of the next checkpoint interval. Try increasing the checkpoint interval if this error happens often.");
            }
        }
        this.elementsBuffer.put(4, 1L);
        this.mappedBuffer.force();
    }

    @Override
    void checkpoint() throws IOException {
        this.setLogWriteOrderID(WriteOrderOracle.next());
        LOG.info("Updating checkpoint metadata: logWriteOrderID: " + this.getLogWriteOrderID() + ", queueSize: " + this.getSize() + ", queueHead: " + this.getHead());
        this.elementsBuffer.put(1, this.getLogWriteOrderID());
        try {
            this.writeCheckpointMetaData();
        }
        catch (IOException e) {
            throw new IOException("Error writing metadata", e);
        }
        Iterator<Integer> it = this.overwriteMap.keySet().iterator();
        while (it.hasNext()) {
            int index = it.next();
            long value = this.overwriteMap.get(index);
            this.elementsBuffer.put(index, value);
            it.remove();
        }
        Preconditions.checkState(this.overwriteMap.isEmpty(), "concurrent update detected ");
        this.elementsBuffer.put(4, 0L);
        this.mappedBuffer.force();
        if (this.shouldBackup) {
            this.startBackupThread();
        }
    }

    private void startBackupThread() {
        Preconditions.checkNotNull(this.checkpointBackUpExecutor, "Expected the checkpoint backup exector to be non-null, but it is null. Checkpoint will not be backed up.");
        LOG.info("Attempting to back up checkpoint.");
        this.checkpointBackUpExecutor.submit(new Runnable(){

            @Override
            public void run() {
                boolean error = false;
                try {
                    EventQueueBackingStoreFile.this.backupCheckpoint(EventQueueBackingStoreFile.this.backupDir);
                }
                catch (Throwable throwable) {
                    EventQueueBackingStoreFile.this.fileChannelCounter.incrementCheckpointBackupWriteErrorCount();
                    error = true;
                    LOG.error("Backing up of checkpoint directory failed.", throwable);
                }
                finally {
                    EventQueueBackingStoreFile.this.backupCompletedSema.release();
                }
                if (!error) {
                    LOG.info("Checkpoint backup completed.");
                }
            }
        });
    }

    @Override
    void close() {
        this.mappedBuffer.force();
        try {
            this.checkpointFileHandle.close();
        }
        catch (IOException e) {
            LOG.info("Error closing " + this.checkpointFile, e);
        }
        if (this.checkpointBackUpExecutor != null && !this.checkpointBackUpExecutor.isShutdown()) {
            this.checkpointBackUpExecutor.shutdown();
            try {
                while (!this.checkpointBackUpExecutor.awaitTermination(1L, TimeUnit.SECONDS)) {
                }
            }
            catch (InterruptedException ex) {
                LOG.warn("Interrupted while waiting for checkpoint backup to complete");
            }
        }
    }

    @Override
    long get(int index) {
        int realIndex = this.getPhysicalIndex(index);
        long result = 0L;
        result = this.overwriteMap.containsKey(realIndex) ? this.overwriteMap.get(realIndex).longValue() : this.elementsBuffer.get(realIndex);
        return result;
    }

    @Override
    ImmutableSortedSet<Integer> getReferenceCounts() {
        return ImmutableSortedSet.copyOf(this.logFileIDReferenceCounts.keySet());
    }

    @Override
    void put(int index, long value) {
        int realIndex = this.getPhysicalIndex(index);
        this.overwriteMap.put(realIndex, value);
    }

    @Override
    boolean syncRequired() {
        return this.overwriteMap.size() > 0;
    }

    @Override
    protected void incrementFileID(int fileID) {
        AtomicInteger counter = this.logFileIDReferenceCounts.get(fileID);
        if (counter == null) {
            counter = new AtomicInteger(0);
            this.logFileIDReferenceCounts.put(fileID, counter);
        }
        counter.incrementAndGet();
    }

    @Override
    protected void decrementFileID(int fileID) {
        AtomicInteger counter = this.logFileIDReferenceCounts.get(fileID);
        Preconditions.checkState(counter != null, "null counter ");
        int count = counter.decrementAndGet();
        if (count == 0) {
            this.logFileIDReferenceCounts.remove(fileID);
        }
    }

    protected int getPhysicalIndex(int index) {
        return 1029 + (this.getHead() + index) % this.getCapacity();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected static void allocate(File file, long totalBytes) throws IOException {
        RandomAccessFile checkpointFile = new RandomAccessFile(file, "rw");
        boolean success = false;
        try {
            if (totalBytes <= 0x200000L) {
                checkpointFile.write(new byte[(int)totalBytes]);
            } else {
                long remainingBytes;
                byte[] initBuffer = new byte[0x200000];
                for (remainingBytes = totalBytes; remainingBytes >= 0x200000L; remainingBytes -= 0x200000L) {
                    checkpointFile.write(initBuffer);
                }
                if (remainingBytes > 0L) {
                    checkpointFile.write(initBuffer, 0, (int)remainingBytes);
                }
            }
            success = true;
        }
        finally {
            block12: {
                try {
                    checkpointFile.close();
                }
                catch (IOException e) {
                    if (!success) break block12;
                    throw e;
                }
            }
        }
    }

    public static boolean backupExists(File backupDir) {
        return new File(backupDir, "backupComplete").exists();
    }

    public static void main(String[] args) throws Exception {
        File file = new File(args[0]);
        File inflightTakesFile = new File(args[1]);
        File inflightPutsFile = new File(args[2]);
        File queueSetDir = new File(args[3]);
        if (!file.exists()) {
            throw new IOException("File " + file + " does not exist");
        }
        if (file.length() == 0L) {
            throw new IOException("File " + file + " is empty");
        }
        int capacity = (int)((file.length() - 8232L) / 8L);
        EventQueueBackingStoreFile backingStore = (EventQueueBackingStoreFile)EventQueueBackingStoreFactory.get(file, capacity, "debug", new FileChannelCounter("Main"), false);
        System.out.println("File Reference Counts" + backingStore.logFileIDReferenceCounts);
        System.out.println("Queue Capacity " + backingStore.getCapacity());
        System.out.println("Queue Size " + backingStore.getSize());
        System.out.println("Queue Head " + backingStore.getHead());
        for (int index = 0; index < backingStore.getCapacity(); ++index) {
            long value = backingStore.get(backingStore.getPhysicalIndex(index));
            int fileID = (int)(value >>> 32);
            int offset = (int)value;
            System.out.println(index + ":" + Long.toHexString(value) + " fileID = " + fileID + ", offset = " + offset);
        }
        FlumeEventQueue queue = new FlumeEventQueue(backingStore, inflightTakesFile, inflightPutsFile, queueSetDir);
        SetMultimap<Long, Long> putMap = queue.deserializeInflightPuts();
        System.out.println("Inflight Puts:");
        for (Long txnID : putMap.keySet()) {
            Collection puts = putMap.get((Object)txnID);
            System.out.println("Transaction ID: " + String.valueOf(txnID));
            Iterator iterator = puts.iterator();
            while (iterator.hasNext()) {
                long value = (Long)iterator.next();
                int fileID = (int)(value >>> 32);
                int offset = (int)value;
                System.out.println(Long.toHexString(value) + " fileID = " + fileID + ", offset = " + offset);
            }
        }
        SetMultimap<Long, Long> takeMap = queue.deserializeInflightTakes();
        System.out.println("Inflight takes:");
        for (Long txnID : takeMap.keySet()) {
            Collection takes = takeMap.get((Object)txnID);
            System.out.println("Transaction ID: " + String.valueOf(txnID));
            Iterator iterator = takes.iterator();
            while (iterator.hasNext()) {
                long value = (Long)iterator.next();
                int fileID = (int)(value >>> 32);
                int offset = (int)value;
                System.out.println(Long.toHexString(value) + " fileID = " + fileID + ", offset = " + offset);
            }
        }
    }
}

