/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flume.channel.jdbc.impl;

import com.google.common.collect.ImmutableMap;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Properties;
import java.util.concurrent.atomic.AtomicLong;
import javax.sql.DataSource;
import org.apache.flink.elasticsearch7.shaded.org.apache.commons.dbcp.ConnectionFactory;
import org.apache.flink.elasticsearch7.shaded.org.apache.commons.dbcp.DriverManagerConnectionFactory;
import org.apache.flink.elasticsearch7.shaded.org.apache.commons.dbcp.PoolableConnectionFactory;
import org.apache.flink.elasticsearch7.shaded.org.apache.commons.dbcp.PoolingDataSource;
import org.apache.flink.elasticsearch7.shaded.org.apache.commons.pool.KeyedObjectPoolFactory;
import org.apache.flink.elasticsearch7.shaded.org.apache.commons.pool.ObjectPool;
import org.apache.flink.elasticsearch7.shaded.org.apache.commons.pool.impl.GenericKeyedObjectPoolFactory;
import org.apache.flink.elasticsearch7.shaded.org.apache.commons.pool.impl.GenericObjectPool;
import org.apache.flume.Context;
import org.apache.flume.Event;
import org.apache.flume.channel.jdbc.DatabaseType;
import org.apache.flume.channel.jdbc.JdbcChannelException;
import org.apache.flume.channel.jdbc.JdbcChannelProvider;
import org.apache.flume.channel.jdbc.TransactionIsolation;
import org.apache.flume.channel.jdbc.impl.JdbcTransactionFactory;
import org.apache.flume.channel.jdbc.impl.JdbcTransactionImpl;
import org.apache.flume.channel.jdbc.impl.PersistableEvent;
import org.apache.flume.channel.jdbc.impl.SchemaHandler;
import org.apache.flume.channel.jdbc.impl.SchemaHandlerFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JdbcChannelProviderImpl
implements JdbcChannelProvider {
    private static final Logger LOGGER = LoggerFactory.getLogger(JdbcChannelProviderImpl.class);
    private static final String EMBEDDED_DERBY_DRIVER_CLASSNAME = "org.apache.derby.jdbc.EmbeddedDriver";
    private static final String DEFAULT_DRIVER_CLASSNAME = "org.apache.derby.jdbc.EmbeddedDriver";
    private static final String DEFAULT_USERNAME = "sa";
    private static final String DEFAULT_PASSWORD = "";
    private static final String DEFAULT_DBTYPE = "DERBY";
    private GenericObjectPool connectionPool;
    private KeyedObjectPoolFactory statementPool;
    private DataSource dataSource;
    private DatabaseType databaseType;
    private SchemaHandler schemaHandler;
    private JdbcTransactionFactory txFactory;
    private String connectUrl;
    private String driverClassName;
    private long maxCapacity = 0L;
    private AtomicLong currentSize = new AtomicLong(0L);

    @Override
    public void initialize(Context context) {
        LOGGER.debug("Initializing JDBC Channel provider");
        this.initializeSystemProperties(context);
        this.initializeDataSource(context);
        this.initializeSchema(context);
        this.initializeChannelState(context);
    }

    private void initializeSystemProperties(Context context) {
        ImmutableMap<String, String> sysPropsNew;
        HashMap<String, String> sysProps = new HashMap<String, String>();
        ImmutableMap<String, String> sysPropsOld = context.getSubProperties("org.apache.flume.channel.jdbc.sysprop.");
        if (sysPropsOld.size() > 0) {
            LOGGER.warn("Long form configuration prefix \"org.apache.flume.channel.jdbc.sysprop.\" is deprecated. Please use the short form prefix \"sysprop.\" instead.");
            sysProps.putAll(sysPropsOld);
        }
        if ((sysPropsNew = context.getSubProperties("sysprop.")).size() > 0) {
            sysProps.putAll(sysPropsNew);
        }
        for (String key : sysProps.keySet()) {
            String value = (String)sysProps.get(key);
            if (key == null || value == null) continue;
            System.setProperty(key, value);
        }
    }

    private void initializeChannelState(Context context) {
        String maxCapacityStr = this.getConfigurationString(context, "maximum.capacity", "org.apache.flume.channel.jdbc.maximum.capacity", "0");
        long maxCapacitySpecified = 0L;
        try {
            maxCapacitySpecified = Long.parseLong(maxCapacityStr);
        }
        catch (NumberFormatException nfe) {
            LOGGER.warn("Invalid value specified for maximum channel capacity: " + maxCapacityStr, nfe);
        }
        if (maxCapacitySpecified > 0L) {
            this.maxCapacity = maxCapacitySpecified;
            LOGGER.info("Maximum channel capacity: {}", (Object)this.maxCapacity);
        } else {
            LOGGER.warn("JDBC channel will operate without a capacity limit.");
        }
        if (this.maxCapacity > 0L) {
            try (JdbcTransactionImpl tx = null;){
                tx = this.getTransaction();
                tx.begin();
                Connection conn = tx.getConnection();
                this.currentSize.set(this.schemaHandler.getChannelSize(conn));
                tx.commit();
            }
            long currentSizeLong = this.currentSize.get();
            if (currentSizeLong > this.maxCapacity) {
                LOGGER.warn("The current size of channel (" + currentSizeLong + ") is more than the specified maximum capacity (" + this.maxCapacity + "). If this situation persists, it may require resizing and replanning of your deployment.");
            }
            LOGGER.info("Current channel size: {}", (Object)currentSizeLong);
        }
    }

    private void initializeSchema(Context context) {
        String createSchemaFlag = this.getConfigurationString(context, "create.schema", "org.apache.flume.channel.jdbc.create.schema", "true");
        boolean createSchema = Boolean.valueOf(createSchemaFlag);
        LOGGER.debug("Create schema flag set to: " + createSchema);
        this.schemaHandler = SchemaHandlerFactory.getHandler(this.databaseType, this.dataSource);
        if (!this.schemaHandler.schemaExists()) {
            boolean createForeignKeys;
            if (!createSchema) {
                throw new JdbcChannelException("Schema does not exist and auto-generation is disabled. Please enable auto-generation of schema and try again.");
            }
            String createIndexFlag = this.getConfigurationString(context, "create.index", "org.apache.flume.channel.jdbc.create.index", "true");
            String createForeignKeysFlag = this.getConfigurationString(context, "create.foreignkey", "org.apache.flume.channel.jdbc.create.foreignkey", "true");
            boolean createIndex = Boolean.valueOf(createIndexFlag);
            if (!createIndex) {
                LOGGER.warn("Index creation is disabled, indexes will not be created.");
            }
            if (createForeignKeys = Boolean.valueOf(createForeignKeysFlag).booleanValue()) {
                LOGGER.info("Foreign Key Constraints will be enabled.");
            } else {
                LOGGER.info("Foreign Key Constratins will be disabled.");
            }
            this.schemaHandler.createSchemaObjects(createForeignKeys, createIndex);
        }
        this.schemaHandler.validateSchema();
    }

    @Override
    public void close() {
        try {
            this.connectionPool.close();
        }
        catch (Exception ex) {
            throw new JdbcChannelException("Unable to close connection pool", ex);
        }
        if (this.databaseType.equals((Object)DatabaseType.DERBY) && this.driverClassName.equals("org.apache.derby.jdbc.EmbeddedDriver")) {
            if (this.connectUrl.startsWith("jdbc:derby:")) {
                int index = this.connectUrl.indexOf(";");
                String baseUrl = null;
                baseUrl = index != -1 ? this.connectUrl.substring(0, index + 1) : this.connectUrl + ";";
                String shutDownUrl = baseUrl + "shutdown=true";
                LOGGER.debug("Attempting to shutdown embedded Derby using URL: " + shutDownUrl);
                try {
                    DriverManager.getConnection(shutDownUrl);
                }
                catch (SQLException ex) {
                    if (ex.getErrorCode() != 45000) {
                        throw new JdbcChannelException("Unable to shutdown embedded Derby: " + shutDownUrl + " Error Code: " + ex.getErrorCode(), ex);
                    }
                    LOGGER.info("Embedded Derby shutdown raised SQL STATE 45000 as expected.");
                }
            } else {
                LOGGER.warn("Even though embedded Derby drvier was loaded, the connect URL is of an unexpected form: " + this.connectUrl + ". Therfore no attempt will be made to shutdown embedded Derby instance.");
            }
        }
        this.dataSource = null;
        this.txFactory = null;
        this.schemaHandler = null;
    }

    @Override
    public void persistEvent(String channel, Event event) {
        PersistableEvent persistableEvent = new PersistableEvent(channel, event);
        try (JdbcTransactionImpl tx = null;){
            long currentSizeLong;
            tx = this.getTransaction();
            tx.begin();
            if (this.maxCapacity > 0L && (currentSizeLong = this.currentSize.get()) >= this.maxCapacity) {
                throw new JdbcChannelException("Channel capacity reached: maxCapacity: " + this.maxCapacity + ", currentSize: " + currentSizeLong);
            }
            this.schemaHandler.storeEvent(persistableEvent, tx.getConnection());
            tx.incrementPersistedEventCount();
            tx.commit();
        }
        LOGGER.debug("Persisted event: {}", (Object)persistableEvent.getEventId());
    }

    @Override
    public Event removeEvent(String channelName) {
        PersistableEvent result = null;
        try (JdbcTransactionImpl tx = null;){
            tx = this.getTransaction();
            tx.begin();
            result = this.schemaHandler.fetchAndDeleteEvent(channelName, tx.getConnection());
            if (result != null) {
                tx.incrementRemovedEventCount();
            }
            tx.commit();
        }
        if (result != null) {
            LOGGER.debug("Removed event: {}", (Object)result.getEventId());
        } else {
            LOGGER.debug("No event found for removal");
        }
        return result;
    }

    @Override
    public JdbcTransactionImpl getTransaction() {
        return (JdbcTransactionImpl)this.txFactory.get();
    }

    private void initializeDataSource(Context context) {
        Object oldPass;
        Object oldUser;
        this.driverClassName = this.getConfigurationString(context, "driver.class", "org.apache.flume.channel.jdbc.driver.class", null);
        this.connectUrl = this.getConfigurationString(context, "driver.url", "org.apache.flume.channel.jdbc.driver.url", null);
        String userName = this.getConfigurationString(context, "db.username", "org.apache.flume.channel.jdbc.db.username", null);
        String password = this.getConfigurationString(context, "db.password", "org.apache.flume.channel.jdbc.db.password", null);
        String jdbcPropertiesFile = this.getConfigurationString(context, "connection.properties.file", "org.apache.flume.channel.jdbc.connection.properties.file", null);
        String dbTypeName = this.getConfigurationString(context, "db.type", "org.apache.flume.channel.jdbc.db.type", null);
        if (this.connectUrl == null || this.connectUrl.trim().length() == 0) {
            LOGGER.warn("No connection URL specified. Using embedded derby database instance.");
            this.driverClassName = "org.apache.derby.jdbc.EmbeddedDriver";
            userName = DEFAULT_USERNAME;
            password = DEFAULT_PASSWORD;
            dbTypeName = DEFAULT_DBTYPE;
            String homePath = System.getProperty("user.home").replace('\\', '/');
            String defaultDbDir = homePath + "/.flume/jdbc-channel";
            File dbDir = new File(defaultDbDir);
            String canonicalDbDirPath = null;
            try {
                canonicalDbDirPath = dbDir.getCanonicalPath();
            }
            catch (IOException ex) {
                throw new JdbcChannelException("Unable to find canonical path of dir: " + defaultDbDir, ex);
            }
            if (!dbDir.exists() && !dbDir.mkdirs()) {
                throw new JdbcChannelException("unable to create directory: " + canonicalDbDirPath);
            }
            this.connectUrl = "jdbc:derby:" + canonicalDbDirPath + "/db;create=true";
            jdbcPropertiesFile = null;
            LOGGER.warn("Overriding values for - driver: " + this.driverClassName + ", user: " + userName + "connectUrl: " + this.connectUrl + ", jdbc properties file: " + jdbcPropertiesFile + ", dbtype: " + dbTypeName);
        }
        this.databaseType = DatabaseType.getByName(dbTypeName);
        switch (this.databaseType) {
            case DERBY: 
            case MYSQL: {
                break;
            }
            default: {
                throw new JdbcChannelException("Database " + (Object)((Object)this.databaseType) + " not supported at this time");
            }
        }
        if (this.driverClassName == null || this.driverClassName.trim().length() == 0) {
            throw new JdbcChannelException("No jdbc driver specified");
        }
        try {
            Class.forName(this.driverClassName);
        }
        catch (ClassNotFoundException ex) {
            throw new JdbcChannelException("Unable to load driver: " + this.driverClassName, ex);
        }
        Properties jdbcProps = new Properties();
        if (jdbcPropertiesFile != null && jdbcPropertiesFile.trim().length() > 0) {
            File jdbcPropsFile = new File(jdbcPropertiesFile.trim());
            if (!jdbcPropsFile.exists()) {
                throw new JdbcChannelException("Jdbc properties file does not exist: " + jdbcPropertiesFile);
            }
            FileInputStream inStream = null;
            try {
                inStream = new FileInputStream(jdbcPropsFile);
                jdbcProps.load(inStream);
            }
            catch (IOException ex) {
                throw new JdbcChannelException("Unable to load jdbc properties from file: " + jdbcPropertiesFile, ex);
            }
            finally {
                if (inStream != null) {
                    try {
                        ((InputStream)inStream).close();
                    }
                    catch (IOException ex) {
                        LOGGER.error("Unable to close file: " + jdbcPropertiesFile, ex);
                    }
                }
            }
        }
        if (userName != null && (oldUser = jdbcProps.put("user", userName)) != null) {
            LOGGER.warn("Overriding user from: " + oldUser + " to: " + userName);
        }
        if (password != null && (oldPass = jdbcProps.put("password", password)) != null) {
            LOGGER.warn("Overriding password from the jdbc properties with  the one specified explicitly.");
        }
        if (LOGGER.isDebugEnabled()) {
            StringBuilder sb = new StringBuilder("JDBC Properties {");
            boolean first = true;
            Enumeration<?> propertyKeys = jdbcProps.propertyNames();
            while (propertyKeys.hasMoreElements()) {
                if (first) {
                    first = false;
                } else {
                    sb.append(", ");
                }
                String key = (String)propertyKeys.nextElement();
                sb.append(key).append("=");
                if (key.equalsIgnoreCase("password")) {
                    sb.append("*******");
                    continue;
                }
                sb.append(jdbcProps.get(key));
            }
            sb.append("}");
            LOGGER.debug(sb.toString());
        }
        String txIsolation = this.getConfigurationString(context, "transaction.isolation", "org.apache.flume.channel.jdbc.transaction.isolation", TransactionIsolation.READ_COMMITTED.getName());
        TransactionIsolation txIsolationLevel = TransactionIsolation.getByName(txIsolation);
        LOGGER.debug("Transaction isolation will be set to: " + (Object)((Object)txIsolationLevel));
        DriverManagerConnectionFactory connFactory = new DriverManagerConnectionFactory(this.connectUrl, jdbcProps);
        this.connectionPool = new GenericObjectPool();
        String maxActiveConnections = this.getConfigurationString(context, "maximum.connections", "org.apache.flume.channel.jdbc.maximum.connections", "10");
        int maxActive = 10;
        if (maxActiveConnections != null && maxActiveConnections.length() > 0) {
            try {
                maxActive = Integer.parseInt(maxActiveConnections);
            }
            catch (NumberFormatException nfe) {
                LOGGER.warn("Max active connections has invalid value: " + maxActiveConnections + ", Using default: " + maxActive);
            }
        }
        LOGGER.debug("Max active connections for the pool: " + maxActive);
        this.connectionPool.setMaxActive(maxActive);
        this.statementPool = new GenericKeyedObjectPoolFactory(null);
        new PoolableConnectionFactory((ConnectionFactory)connFactory, (ObjectPool)this.connectionPool, this.statementPool, this.databaseType.getValidationQuery(), false, false, txIsolationLevel.getCode());
        this.dataSource = new PoolingDataSource(this.connectionPool);
        this.txFactory = new JdbcTransactionFactory(this.dataSource, this);
    }

    protected void updateCurrentChannelSize(long delta) {
        long currentSizeLong = this.currentSize.addAndGet(delta);
        LOGGER.debug("channel size updated to: " + currentSizeLong);
    }

    private String getConfigurationString(Context context, String key, String oldKey, String defaultValue) {
        String value;
        String oldValue = context.getString(oldKey);
        if (oldValue != null && oldValue.length() > 0) {
            LOGGER.warn("Long form configuration key \"" + oldKey + "\" is deprecated. Please use the short form key \"" + key + "\" instead.");
        }
        if ((value = context.getString(key)) == null) {
            value = oldValue != null ? oldValue : defaultValue;
        }
        return value;
    }
}

