/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringWriter;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import org.apache.iceberg.HistoryEntry;
import org.apache.iceberg.MetadataUpdate;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.PartitionSpecParser;
import org.apache.iceberg.Schema;
import org.apache.iceberg.SchemaParser;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.SnapshotParser;
import org.apache.iceberg.SortOrder;
import org.apache.iceberg.SortOrderParser;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.exceptions.RuntimeIOException;
import org.apache.iceberg.io.FileIO;
import org.apache.iceberg.io.InputFile;
import org.apache.iceberg.io.OutputFile;
import org.apache.iceberg.io.PositionOutputStream;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableCollection;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableList;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.shaded.com.fasterxml.jackson.core.JsonGenerator;
import org.apache.iceberg.shaded.com.fasterxml.jackson.databind.JsonNode;
import org.apache.iceberg.util.JsonUtil;

public class TableMetadataParser {
    static final String FORMAT_VERSION = "format-version";
    static final String TABLE_UUID = "table-uuid";
    static final String LOCATION = "location";
    static final String LAST_SEQUENCE_NUMBER = "last-sequence-number";
    static final String LAST_UPDATED_MILLIS = "last-updated-ms";
    static final String LAST_COLUMN_ID = "last-column-id";
    static final String SCHEMA = "schema";
    static final String SCHEMAS = "schemas";
    static final String CURRENT_SCHEMA_ID = "current-schema-id";
    static final String PARTITION_SPEC = "partition-spec";
    static final String PARTITION_SPECS = "partition-specs";
    static final String DEFAULT_SPEC_ID = "default-spec-id";
    static final String LAST_PARTITION_ID = "last-partition-id";
    static final String DEFAULT_SORT_ORDER_ID = "default-sort-order-id";
    static final String SORT_ORDERS = "sort-orders";
    static final String PROPERTIES = "properties";
    static final String CURRENT_SNAPSHOT_ID = "current-snapshot-id";
    static final String SNAPSHOTS = "snapshots";
    static final String SNAPSHOT_ID = "snapshot-id";
    static final String TIMESTAMP_MS = "timestamp-ms";
    static final String SNAPSHOT_LOG = "snapshot-log";
    static final String METADATA_FILE = "metadata-file";
    static final String METADATA_LOG = "metadata-log";

    private TableMetadataParser() {
    }

    public static void overwrite(TableMetadata metadata, OutputFile outputFile) {
        TableMetadataParser.internalWrite(metadata, outputFile, true);
    }

    public static void write(TableMetadata metadata, OutputFile outputFile) {
        TableMetadataParser.internalWrite(metadata, outputFile, false);
    }

    public static void internalWrite(TableMetadata metadata, OutputFile outputFile, boolean overwrite) {
        boolean isGzip = Codec.fromFileName(outputFile.location()) == Codec.GZIP;
        PositionOutputStream stream = overwrite ? outputFile.createOrOverwrite() : outputFile.create();
        try (OutputStream ou = isGzip ? new GZIPOutputStream(stream) : stream;
             OutputStreamWriter writer = new OutputStreamWriter(ou, StandardCharsets.UTF_8);){
            JsonGenerator generator = JsonUtil.factory().createGenerator(writer);
            generator.useDefaultPrettyPrinter();
            TableMetadataParser.toJson(metadata, generator);
            generator.flush();
        }
        catch (IOException e) {
            throw new RuntimeIOException(e, "Failed to write json to file: %s", outputFile);
        }
    }

    public static String getFileExtension(String codecName) {
        return TableMetadataParser.getFileExtension(Codec.fromName(codecName));
    }

    public static String getFileExtension(Codec codec) {
        return codec.extension + ".metadata.json";
    }

    public static String getOldFileExtension(Codec codec) {
        return ".metadata.json" + codec.extension;
    }

    public static String toJson(TableMetadata metadata) {
        String string;
        StringWriter writer = new StringWriter();
        Throwable throwable = null;
        try {
            JsonGenerator generator = JsonUtil.factory().createGenerator(writer);
            TableMetadataParser.toJson(metadata, generator);
            generator.flush();
            string = writer.toString();
        }
        catch (Throwable throwable2) {
            try {
                try {
                    throwable = throwable2;
                    throw throwable2;
                }
                catch (Throwable throwable3) {
                    TableMetadataParser.$closeResource(throwable, writer);
                    throw throwable3;
                }
            }
            catch (IOException e) {
                throw new RuntimeIOException(e, "Failed to write json for: %s", metadata);
            }
        }
        TableMetadataParser.$closeResource(throwable, writer);
        return string;
    }

    private static void toJson(TableMetadata metadata, JsonGenerator generator) throws IOException {
        generator.writeStartObject();
        generator.writeNumberField(FORMAT_VERSION, metadata.formatVersion());
        generator.writeStringField(TABLE_UUID, metadata.uuid());
        generator.writeStringField(LOCATION, metadata.location());
        if (metadata.formatVersion() > 1) {
            generator.writeNumberField(LAST_SEQUENCE_NUMBER, metadata.lastSequenceNumber());
        }
        generator.writeNumberField(LAST_UPDATED_MILLIS, metadata.lastUpdatedMillis());
        generator.writeNumberField(LAST_COLUMN_ID, metadata.lastColumnId());
        if (metadata.formatVersion() == 1) {
            generator.writeFieldName(SCHEMA);
            SchemaParser.toJson(metadata.schema(), generator);
        }
        generator.writeNumberField(CURRENT_SCHEMA_ID, metadata.currentSchemaId());
        generator.writeArrayFieldStart(SCHEMAS);
        for (Schema schema : metadata.schemas()) {
            SchemaParser.toJson(schema, generator);
        }
        generator.writeEndArray();
        if (metadata.formatVersion() == 1) {
            generator.writeFieldName(PARTITION_SPEC);
            PartitionSpecParser.toJsonFields(metadata.spec(), generator);
        }
        generator.writeNumberField(DEFAULT_SPEC_ID, metadata.defaultSpecId());
        generator.writeArrayFieldStart(PARTITION_SPECS);
        for (PartitionSpec partitionSpec : metadata.specs()) {
            PartitionSpecParser.toJson(partitionSpec, generator);
        }
        generator.writeEndArray();
        generator.writeNumberField(LAST_PARTITION_ID, metadata.lastAssignedPartitionId());
        generator.writeNumberField(DEFAULT_SORT_ORDER_ID, metadata.defaultSortOrderId());
        generator.writeArrayFieldStart(SORT_ORDERS);
        for (SortOrder sortOrder : metadata.sortOrders()) {
            SortOrderParser.toJson(sortOrder, generator);
        }
        generator.writeEndArray();
        generator.writeObjectFieldStart(PROPERTIES);
        for (Map.Entry entry : metadata.properties().entrySet()) {
            generator.writeStringField((String)entry.getKey(), (String)entry.getValue());
        }
        generator.writeEndObject();
        generator.writeNumberField(CURRENT_SNAPSHOT_ID, metadata.currentSnapshot() != null ? metadata.currentSnapshot().snapshotId() : -1L);
        generator.writeArrayFieldStart(SNAPSHOTS);
        for (Snapshot snapshot : metadata.snapshots()) {
            SnapshotParser.toJson(snapshot, generator);
        }
        generator.writeEndArray();
        generator.writeArrayFieldStart(SNAPSHOT_LOG);
        for (HistoryEntry historyEntry : metadata.snapshotLog()) {
            generator.writeStartObject();
            generator.writeNumberField(TIMESTAMP_MS, historyEntry.timestampMillis());
            generator.writeNumberField(SNAPSHOT_ID, historyEntry.snapshotId());
            generator.writeEndObject();
        }
        generator.writeEndArray();
        generator.writeArrayFieldStart(METADATA_LOG);
        for (TableMetadata.MetadataLogEntry metadataLogEntry : metadata.previousFiles()) {
            generator.writeStartObject();
            generator.writeNumberField(TIMESTAMP_MS, metadataLogEntry.timestampMillis());
            generator.writeStringField(METADATA_FILE, metadataLogEntry.file());
            generator.writeEndObject();
        }
        generator.writeEndArray();
        generator.writeEndObject();
    }

    public static TableMetadata read(FileIO io, String path) {
        return TableMetadataParser.read(io, io.newInputFile(path));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static TableMetadata read(FileIO io, InputFile file) {
        Codec codec = Codec.fromFileName(file.location());
        try (InputStream inputStream = codec == Codec.GZIP ? new GZIPInputStream(file.newStream()) : file.newStream();){
            TableMetadata tableMetadata = TableMetadataParser.fromJson(io, file, JsonUtil.mapper().readValue(inputStream, JsonNode.class));
            return tableMetadata;
        }
        catch (IOException iOException) {
            throw new RuntimeIOException(iOException, "Failed to read file: %s", file);
        }
    }

    public static TableMetadata fromJson(FileIO io, String json) {
        return TableMetadataParser.fromJson(io, null, json);
    }

    public static TableMetadata fromJson(FileIO io, String metadataLocation, String json) {
        try {
            JsonNode node = JsonUtil.mapper().readValue(json, JsonNode.class);
            return TableMetadataParser.fromJson(io, metadataLocation, node);
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed to read JSON string: " + json, e);
        }
    }

    static TableMetadata fromJson(FileIO io, InputFile file, JsonNode node) {
        return TableMetadataParser.fromJson(io, file.location(), node);
    }

    static TableMetadata fromJson(FileIO io, String metadataLocation, JsonNode node) {
        ImmutableCollection sortOrders;
        int defaultSortOrderId;
        JsonNode sortOrderArray;
        ImmutableCollection specs;
        int defaultSpecId;
        ImmutableCollection schemas;
        int currentSchemaId;
        Preconditions.checkArgument(node.isObject(), "Cannot parse metadata from a non-object: %s", (Object)node);
        int formatVersion = JsonUtil.getInt(FORMAT_VERSION, node);
        Preconditions.checkArgument(formatVersion <= 2, "Cannot read unsupported version %s", formatVersion);
        String uuid = JsonUtil.getStringOrNull(TABLE_UUID, node);
        String location = JsonUtil.getString(LOCATION, node);
        long lastSequenceNumber = formatVersion > 1 ? JsonUtil.getLong(LAST_SEQUENCE_NUMBER, node) : 0L;
        int lastAssignedColumnId = JsonUtil.getInt(LAST_COLUMN_ID, node);
        Schema schema = null;
        JsonNode schemaArray = node.get(SCHEMAS);
        if (schemaArray != null) {
            Preconditions.checkArgument(schemaArray.isArray(), "Cannot parse schemas from non-array: %s", (Object)schemaArray);
            currentSchemaId = JsonUtil.getInt(CURRENT_SCHEMA_ID, node);
            ImmutableList.Builder builder = ImmutableList.builder();
            for (JsonNode schemaNode : schemaArray) {
                Schema current = SchemaParser.fromJson(schemaNode);
                if (current.schemaId() == currentSchemaId) {
                    schema = current;
                }
                builder.add(current);
            }
            Preconditions.checkArgument(schema != null, "Cannot find schema with %s=%s from %s", (Object)CURRENT_SCHEMA_ID, (Object)currentSchemaId, (Object)SCHEMAS);
            schemas = builder.build();
        } else {
            Preconditions.checkArgument(formatVersion == 1, "%s must exist in format v%s", (Object)SCHEMAS, formatVersion);
            schema = SchemaParser.fromJson(node.get(SCHEMA));
            currentSchemaId = schema.schemaId();
            schemas = ImmutableList.of(schema);
        }
        JsonNode specArray = node.get(PARTITION_SPECS);
        if (specArray != null) {
            Preconditions.checkArgument(specArray.isArray(), "Cannot parse partition specs from non-array: %s", (Object)specArray);
            defaultSpecId = JsonUtil.getInt(DEFAULT_SPEC_ID, node);
            ImmutableList.Builder builder = ImmutableList.builder();
            for (JsonNode spec : specArray) {
                builder.add(PartitionSpecParser.fromJson(schema, spec));
            }
            specs = builder.build();
        } else {
            Preconditions.checkArgument(formatVersion == 1, "%s must exist in format v%s", (Object)PARTITION_SPECS, formatVersion);
            defaultSpecId = 0;
            specs = ImmutableList.of(PartitionSpecParser.fromJsonFields(schema, 0, node.get(PARTITION_SPEC)));
        }
        Integer lastAssignedPartitionId = JsonUtil.getIntOrNull(LAST_PARTITION_ID, node);
        if (lastAssignedPartitionId == null) {
            Preconditions.checkArgument(formatVersion == 1, "%s must exist in format v%s", (Object)LAST_PARTITION_ID, formatVersion);
            lastAssignedPartitionId = specs.stream().mapToInt(PartitionSpec::lastAssignedFieldId).max().orElse(999);
        }
        if ((sortOrderArray = node.get(SORT_ORDERS)) != null) {
            defaultSortOrderId = JsonUtil.getInt(DEFAULT_SORT_ORDER_ID, node);
            ImmutableList.Builder sortOrdersBuilder = ImmutableList.builder();
            for (JsonNode sortOrder : sortOrderArray) {
                sortOrdersBuilder.add(SortOrderParser.fromJson(schema, sortOrder));
            }
            sortOrders = sortOrdersBuilder.build();
        } else {
            Preconditions.checkArgument(formatVersion == 1, "%s must exist in format v%s", (Object)SORT_ORDERS, formatVersion);
            SortOrder defaultSortOrder = SortOrder.unsorted();
            sortOrders = ImmutableList.of(defaultSortOrder);
            defaultSortOrderId = defaultSortOrder.orderId();
        }
        Map<String, String> properties = JsonUtil.getStringMap(PROPERTIES, node);
        long currentVersionId = JsonUtil.getLong(CURRENT_SNAPSHOT_ID, node);
        long lastUpdatedMillis = JsonUtil.getLong(LAST_UPDATED_MILLIS, node);
        JsonNode snapshotArray = node.get(SNAPSHOTS);
        Preconditions.checkArgument(snapshotArray.isArray(), "Cannot parse snapshots from non-array: %s", (Object)snapshotArray);
        ArrayList<Snapshot> snapshots = Lists.newArrayListWithExpectedSize(snapshotArray.size());
        Iterator<JsonNode> iterator = snapshotArray.elements();
        while (iterator.hasNext()) {
            snapshots.add(SnapshotParser.fromJson(io, iterator.next()));
        }
        ImmutableList.Builder entries = ImmutableList.builder();
        if (node.has(SNAPSHOT_LOG)) {
            Iterator<JsonNode> logIterator = node.get(SNAPSHOT_LOG).elements();
            while (logIterator.hasNext()) {
                JsonNode entryNode = logIterator.next();
                entries.add(new TableMetadata.SnapshotLogEntry(JsonUtil.getLong(TIMESTAMP_MS, entryNode), JsonUtil.getLong(SNAPSHOT_ID, entryNode)));
            }
        }
        ImmutableList.Builder metadataEntries = ImmutableList.builder();
        if (node.has(METADATA_LOG)) {
            Iterator<JsonNode> logIterator = node.get(METADATA_LOG).elements();
            while (logIterator.hasNext()) {
                JsonNode entryNode = logIterator.next();
                metadataEntries.add(new TableMetadata.MetadataLogEntry(JsonUtil.getLong(TIMESTAMP_MS, entryNode), JsonUtil.getString(METADATA_FILE, entryNode)));
            }
        }
        return new TableMetadata(metadataLocation, formatVersion, uuid, location, lastSequenceNumber, lastUpdatedMillis, lastAssignedColumnId, currentSchemaId, (List<Schema>)((Object)schemas), defaultSpecId, (List<PartitionSpec>)((Object)specs), lastAssignedPartitionId, defaultSortOrderId, (List<SortOrder>)((Object)sortOrders), properties, currentVersionId, (List<Snapshot>)snapshots, (List<HistoryEntry>)((Object)entries.build()), (List<TableMetadata.MetadataLogEntry>)((Object)metadataEntries.build()), (List<MetadataUpdate>)ImmutableList.of());
    }

    public static enum Codec {
        NONE(""),
        GZIP(".gz");

        private final String extension;

        private Codec(String extension) {
            this.extension = extension;
        }

        public static Codec fromName(String codecName) {
            Preconditions.checkArgument(codecName != null, "Codec name is null");
            return Codec.valueOf(codecName.toUpperCase(Locale.ENGLISH));
        }

        public static Codec fromFileName(String fileName) {
            Preconditions.checkArgument(fileName.contains(".metadata.json"), "%s is not a valid metadata file", (Object)fileName);
            if (fileName.endsWith(".metadata.json.gz")) {
                return GZIP;
            }
            String fileNameWithoutSuffix = fileName.substring(0, fileName.lastIndexOf(".metadata.json"));
            if (fileNameWithoutSuffix.endsWith(Codec.GZIP.extension)) {
                return GZIP;
            }
            return NONE;
        }
    }
}

