/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.oracle.logminer;

import com.ververica.cdc.connectors.shaded.org.apache.kafka.connect.errors.DataException;
import io.debezium.annotation.NotThreadSafe;
import io.debezium.connector.oracle.OracleDatabaseSchema;
import io.debezium.connector.oracle.OracleOffsetContext;
import io.debezium.connector.oracle.OracleStreamingChangeEventSourceMetrics;
import io.debezium.connector.oracle.Scn;
import io.debezium.connector.oracle.logminer.LogMinerChangeRecordEmitter;
import io.debezium.connector.oracle.logminer.LogMinerHelper;
import io.debezium.connector.oracle.logminer.valueholder.LogMinerDmlEntry;
import io.debezium.pipeline.ErrorHandler;
import io.debezium.pipeline.EventDispatcher;
import io.debezium.pipeline.source.spi.ChangeEventSource;
import io.debezium.relational.TableId;
import io.debezium.util.Clock;
import java.sql.Timestamp;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public final class TransactionalBuffer
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(TransactionalBuffer.class);
    private final Map<String, Transaction> transactions = new HashMap<String, Transaction>();
    private final OracleDatabaseSchema schema;
    private final Clock clock;
    private final ErrorHandler errorHandler;
    private final Set<String> abandonedTransactionIds;
    private final Set<String> rolledBackTransactionIds;
    private final OracleStreamingChangeEventSourceMetrics streamingMetrics;
    private Scn lastCommittedScn;

    TransactionalBuffer(OracleDatabaseSchema schema, Clock clock, ErrorHandler errorHandler, OracleStreamingChangeEventSourceMetrics streamingMetrics) {
        this.schema = schema;
        this.clock = clock;
        this.errorHandler = errorHandler;
        this.lastCommittedScn = Scn.NULL;
        this.abandonedTransactionIds = new HashSet<String>();
        this.rolledBackTransactionIds = new HashSet<String>();
        this.streamingMetrics = streamingMetrics;
    }

    Set<String> getRolledBackTransactionIds() {
        return new HashSet<String>(this.rolledBackTransactionIds);
    }

    void registerDmlOperation(int operation, String transactionId, Scn scn, TableId tableId, LogMinerDmlEntry parseEntry, Instant changeTime, String rowId) {
        if (this.abandonedTransactionIds.contains(transactionId)) {
            LogMinerHelper.logWarn(this.streamingMetrics, "Captured DML for abandoned transaction {}, ignored.", transactionId);
            return;
        }
        if (this.rolledBackTransactionIds.contains(transactionId)) {
            LogMinerHelper.logWarn(this.streamingMetrics, "Captured DML for rolled back transaction {}, ignored.", transactionId);
            return;
        }
        Transaction transaction = this.transactions.computeIfAbsent(transactionId, s -> new Transaction(transactionId, scn));
        transaction.events.add(new DmlEvent(operation, parseEntry, scn, tableId, rowId));
        this.streamingMetrics.setActiveTransactions(this.transactions.size());
        this.streamingMetrics.incrementRegisteredDmlCount();
        this.streamingMetrics.calculateLagMetrics(changeTime);
    }

    void undoDmlOperation(String transactionId, String undoRowId, TableId tableId) {
        Transaction transaction = this.transactions.get(transactionId);
        if (transaction == null) {
            LOGGER.warn("Cannot undo changes to {} with row id {} as transaction {} not found.", new Object[]{tableId, undoRowId, transactionId});
            return;
        }
        transaction.events.removeIf(o -> {
            if (o.getRowId().equals(undoRowId)) {
                LOGGER.trace("Undoing change to {} with row id {} in transaction {}", new Object[]{tableId, undoRowId, transactionId});
                return true;
            }
            return false;
        });
    }

    boolean commit(String transactionId, Scn scn, OracleOffsetContext offsetContext, Timestamp timestamp, ChangeEventSource.ChangeEventSourceContext context, String debugMessage, EventDispatcher<TableId> dispatcher) {
        Instant start = Instant.now();
        Transaction transaction = this.transactions.remove(transactionId);
        if (transaction == null) {
            return false;
        }
        Scn smallestScn = this.calculateSmallestScn();
        this.abandonedTransactionIds.remove(transactionId);
        if (offsetContext.getCommitScn() != null && offsetContext.getCommitScn().compareTo(scn) > 0 || this.lastCommittedScn.compareTo(scn) > 0) {
            LogMinerHelper.logWarn(this.streamingMetrics, "Transaction {} was already processed, ignore. Committed SCN in offset is {}, commit SCN of the transaction is {}, last committed SCN is {}", transactionId, offsetContext.getCommitScn(), scn, this.lastCommittedScn);
            this.streamingMetrics.setActiveTransactions(this.transactions.size());
            return false;
        }
        LOGGER.trace("COMMIT, {}, smallest SCN: {}", (Object)debugMessage, (Object)smallestScn);
        this.commit(context, offsetContext, start, transaction, timestamp, smallestScn, scn, dispatcher);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void commit(ChangeEventSource.ChangeEventSourceContext context, OracleOffsetContext offsetContext, Instant start, Transaction transaction, Timestamp timestamp, Scn smallestScn, Scn scn, EventDispatcher<TableId> dispatcher) {
        try {
            int counter = transaction.events.size();
            for (DmlEvent event : transaction.events) {
                if (!context.isRunning()) {
                    return;
                }
                if (smallestScn == null || scn.compareTo(smallestScn) < 0) {
                    offsetContext.setScn(event.getScn());
                    this.streamingMetrics.setOldestScn(event.getScn());
                }
                offsetContext.setTransactionId(transaction.transactionId);
                offsetContext.setSourceTime(timestamp.toInstant());
                offsetContext.setTableId(event.getTableId());
                if (--counter == 0) {
                    offsetContext.setCommitScn(scn);
                }
                LOGGER.trace("Processing DML event {} with SCN {}", (Object)event.getEntry(), (Object)event.getScn());
                dispatcher.dispatchDataChangeEvent(event.getTableId(), new LogMinerChangeRecordEmitter(offsetContext, event.getEntry(), this.schema.tableFor(event.getTableId()), this.clock));
            }
            this.lastCommittedScn = Scn.valueOf(scn.longValue());
            if (!transaction.events.isEmpty()) {
                dispatcher.dispatchTransactionCommittedEvent(offsetContext);
            }
        }
        catch (InterruptedException e) {
            LogMinerHelper.logError(this.streamingMetrics, "Thread interrupted during running", e);
            Thread.currentThread().interrupt();
        }
        catch (Exception e) {
            this.errorHandler.setProducerThrowable(e);
        }
        finally {
            this.streamingMetrics.incrementCommittedTransactions();
            this.streamingMetrics.setActiveTransactions(this.transactions.size());
            this.streamingMetrics.incrementCommittedDmlCount(transaction.events.size());
            this.streamingMetrics.setCommittedScn(scn);
            this.streamingMetrics.setOffsetScn(offsetContext.getScn());
            this.streamingMetrics.setLastCommitDuration(Duration.between(start, Instant.now()));
        }
    }

    boolean rollback(String transactionId, String debugMessage) {
        Transaction transaction = this.transactions.get(transactionId);
        if (transaction != null) {
            LOGGER.debug("Transaction rolled back: {}", (Object)debugMessage);
            this.transactions.remove(transactionId);
            this.abandonedTransactionIds.remove(transactionId);
            this.rolledBackTransactionIds.add(transactionId);
            this.streamingMetrics.setActiveTransactions(this.transactions.size());
            this.streamingMetrics.incrementRolledBackTransactions();
            this.streamingMetrics.addRolledBackTransactionId(transactionId);
            return true;
        }
        return false;
    }

    void abandonLongTransactions(Scn thresholdScn, OracleOffsetContext offsetContext) {
        LogMinerHelper.logWarn(this.streamingMetrics, "All transactions with first SCN <= {} will be abandoned, offset: {}", thresholdScn, offsetContext.getScn());
        Scn threshold = Scn.valueOf(thresholdScn.toString());
        Scn smallestScn = this.calculateSmallestScn();
        if (smallestScn == null) {
            return;
        }
        if (threshold.compareTo(smallestScn) < 0) {
            threshold = smallestScn;
        }
        Iterator<Map.Entry<String, Transaction>> iter = this.transactions.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry<String, Transaction> transaction = iter.next();
            if (transaction.getValue().firstScn.compareTo(threshold) > 0) continue;
            LogMinerHelper.logWarn(this.streamingMetrics, "Following long running transaction {} will be abandoned and ignored: {} ", transaction.getKey(), transaction.getValue().toString());
            this.abandonedTransactionIds.add(transaction.getKey());
            iter.remove();
            this.streamingMetrics.addAbandonedTransactionId(transaction.getKey());
            this.streamingMetrics.setActiveTransactions(this.transactions.size());
        }
    }

    boolean isTransactionRegistered(String txId) {
        return this.transactions.get(txId) != null;
    }

    private Scn calculateSmallestScn() {
        Scn scn = this.transactions.isEmpty() ? null : this.transactions.values().stream().map(transaction -> ((Transaction)transaction).firstScn).min(Scn::compareTo).orElseThrow(() -> new DataException("Cannot calculate smallest SCN"));
        this.streamingMetrics.setOldestScn(scn == null ? Scn.valueOf(-1) : scn);
        return scn;
    }

    boolean isEmpty() {
        return this.transactions.isEmpty();
    }

    public String toString() {
        StringBuilder result = new StringBuilder();
        this.transactions.values().forEach(t -> result.append(t.toString()));
        return result.toString();
    }

    @Override
    public void close() {
        this.transactions.clear();
        if (this.streamingMetrics != null) {
            this.streamingMetrics.unregister(LOGGER);
        }
    }

    private static class DmlEvent {
        private final int operation;
        private final LogMinerDmlEntry entry;
        private final Scn scn;
        private final TableId tableId;
        private final String rowId;

        public DmlEvent(int operation, LogMinerDmlEntry entry, Scn scn, TableId tableId, String rowId) {
            this.operation = operation;
            this.scn = scn;
            this.tableId = tableId;
            this.rowId = rowId;
            this.entry = entry;
        }

        public int getOperation() {
            return this.operation;
        }

        public LogMinerDmlEntry getEntry() {
            return this.entry;
        }

        public Scn getScn() {
            return this.scn;
        }

        public TableId getTableId() {
            return this.tableId;
        }

        public String getRowId() {
            return this.rowId;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            DmlEvent dmlEvent = (DmlEvent)o;
            return this.operation == dmlEvent.operation && Objects.equals(this.entry, dmlEvent.entry) && Objects.equals(this.scn, dmlEvent.scn) && Objects.equals(this.tableId, dmlEvent.tableId) && Objects.equals(this.rowId, dmlEvent.rowId);
        }

        public int hashCode() {
            return Objects.hash(this.operation, this.entry, this.scn, this.tableId, this.rowId);
        }
    }

    private static final class Transaction {
        private final String transactionId;
        private final Scn firstScn;
        private Scn lastScn;
        private final List<DmlEvent> events;

        private Transaction(String transactionId, Scn firstScn) {
            this.transactionId = transactionId;
            this.firstScn = firstScn;
            this.events = new ArrayList<DmlEvent>();
            this.lastScn = firstScn;
        }

        public String toString() {
            return "Transaction{transactionId=" + this.transactionId + ", firstScn=" + this.firstScn + ", lastScn=" + this.lastScn + '}';
        }
    }
}

