/*
 * Decompiled with CFR 0.152.
 */
package org.apache.inlong.agent.core.task.file;

import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.inlong.agent.common.AbstractDaemon;
import org.apache.inlong.agent.common.AgentThreadFactory;
import org.apache.inlong.agent.conf.AgentConfiguration;
import org.apache.inlong.agent.conf.TaskProfile;
import org.apache.inlong.agent.core.task.OffsetManager;
import org.apache.inlong.agent.core.task.TaskAction;
import org.apache.inlong.agent.db.Db;
import org.apache.inlong.agent.db.RocksDbImp;
import org.apache.inlong.agent.db.TaskProfileDb;
import org.apache.inlong.agent.metrics.audit.AuditUtils;
import org.apache.inlong.agent.plugin.file.Task;
import org.apache.inlong.agent.utils.AgentUtils;
import org.apache.inlong.agent.utils.ThreadUtils;
import org.apache.inlong.common.enums.TaskStateEnum;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TaskManager
extends AbstractDaemon {
    private static final Logger LOGGER = LoggerFactory.getLogger(TaskManager.class);
    public static final int CONFIG_QUEUE_CAPACITY = 1;
    public static final int CORE_THREAD_SLEEP_TIME = 1000;
    public static final int CORE_THREAD_PRINT_TIME = 10000;
    private static final int ACTION_QUEUE_CAPACITY = 100000;
    private long lastPrintTime = 0L;
    private final Db taskBasicDb;
    private final Db instanceBasicDb;
    private final Db offsetBasicDb;
    private final TaskProfileDb taskDb;
    private final ConcurrentHashMap<String, Task> taskMap;
    private final BlockingQueue<List<TaskProfile>> configQueue;
    private final ThreadPoolExecutor runningPool;
    private final BlockingQueue<Task> pendingTasks;
    private final int taskMaxLimit;
    private final AgentConfiguration agentConf = AgentConfiguration.getAgentConf();
    private final BlockingQueue<TaskAction> actionQueue;

    public TaskManager() {
        this.taskBasicDb = TaskManager.initDb(this.agentConf.get("agent.rocks.db.path", ".localdb/task"));
        this.taskDb = new TaskProfileDb(this.taskBasicDb);
        this.instanceBasicDb = TaskManager.initDb(this.agentConf.get("agent.rocks.db.path", ".localdb/instance"));
        this.offsetBasicDb = TaskManager.initDb(this.agentConf.get("agent.rocks.db.path", ".localdb/offset"));
        OffsetManager.init(this.offsetBasicDb, this.instanceBasicDb);
        this.runningPool = new ThreadPoolExecutor(0, Integer.MAX_VALUE, 60L, TimeUnit.SECONDS, new SynchronousQueue<Runnable>(), (ThreadFactory)new AgentThreadFactory("task-manager-running-pool"));
        this.taskMap = new ConcurrentHashMap();
        this.taskMaxLimit = this.agentConf.getInt("job.number.limit", 15);
        this.pendingTasks = new LinkedBlockingQueue<Task>(this.taskMaxLimit);
        this.configQueue = new LinkedBlockingQueue<List<TaskProfile>>(1);
        this.actionQueue = new LinkedBlockingQueue<TaskAction>(100000);
    }

    public TaskProfileDb getTaskDb() {
        return this.taskDb;
    }

    public static Db initDb(String childPath) {
        try {
            return new RocksDbImp(childPath);
        }
        catch (Exception ex) {
            throw new UnsupportedClassVersionError(ex.getMessage());
        }
    }

    public void submitTaskProfiles(List<TaskProfile> taskProfiles) {
        if (taskProfiles == null) {
            return;
        }
        while (this.configQueue.size() != 0) {
            this.configQueue.poll();
        }
        for (int i = 0; i < taskProfiles.size(); ++i) {
            LOGGER.info("submitTaskProfiles index {} total {} {}", new Object[]{i, taskProfiles.size(), taskProfiles.get(i).toJsonStr()});
        }
        this.configQueue.add(taskProfiles);
    }

    public boolean submitAction(TaskAction action) {
        if (action == null) {
            return false;
        }
        return this.actionQueue.offer(action);
    }

    private Runnable coreThread() {
        return () -> {
            Thread.currentThread().setName("task-manager-core");
            while (this.isRunnable()) {
                try {
                    AgentUtils.silenceSleepInMs((long)1000L);
                    this.printTaskDetail();
                    this.dealWithConfigQueue(this.configQueue);
                    this.dealWithActionQueue(this.actionQueue);
                    AuditUtils.add((int)30007, (String)"", (String)"", (long)AgentUtils.getCurrentTime(), (int)1, (long)1L);
                }
                catch (Throwable ex) {
                    LOGGER.error("exception caught", ex);
                    ThreadUtils.threadThrowableHandler((Thread)Thread.currentThread(), (Throwable)ex);
                }
            }
        };
    }

    private void printTaskDetail() {
        if (AgentUtils.getCurrentTime() - this.lastPrintTime > 10000L) {
            List tasksInDb = this.taskDb.getTasks();
            TaskPrintStat stat = new TaskPrintStat();
            for (int i = 0; i < tasksInDb.size(); ++i) {
                TaskProfile task = (TaskProfile)tasksInDb.get(i);
                stat.stat(task.getState());
            }
            LOGGER.info("taskManager running! mem {} db total {} {} ", new Object[]{this.taskMap.size(), tasksInDb.size(), stat});
            this.lastPrintTime = AgentUtils.getCurrentTime();
        }
    }

    private void dealWithConfigQueue(BlockingQueue<List<TaskProfile>> queue) {
        List dataConfigs = (List)queue.poll();
        if (dataConfigs == null) {
            return;
        }
        this.keepPaceWithManager(dataConfigs);
        this.keepPaceWithDb();
    }

    private void dealWithActionQueue(BlockingQueue<TaskAction> queue) {
        block5: while (this.isRunnable()) {
            try {
                TaskAction action = (TaskAction)queue.poll();
                if (action == null) break;
                TaskProfile profile = action.getProfile();
                switch (action.getActionType()) {
                    case FINISH: {
                        LOGGER.info("deal finish action, taskId {}", (Object)profile.getTaskId());
                        this.finishTask(profile);
                        continue block5;
                    }
                }
                LOGGER.error("invalid action type for action queue: taskId {} type {}", (Object)profile.getTaskId(), (Object)action.getActionType());
            }
            catch (Throwable ex) {
                LOGGER.error("dealWithActionQueue", ex);
                ThreadUtils.threadThrowableHandler((Thread)Thread.currentThread(), (Throwable)ex);
            }
        }
    }

    private void keepPaceWithManager(List<TaskProfile> taskProfiles) {
        ConcurrentHashMap<String, TaskProfile> tasksFromManager = new ConcurrentHashMap<String, TaskProfile>();
        taskProfiles.forEach(profile -> {
            TaskStateEnum state = profile.getState();
            if (state == TaskStateEnum.RUNNING || state == TaskStateEnum.FROZEN) {
                tasksFromManager.put(profile.getTaskId(), (TaskProfile)profile);
            } else {
                LOGGER.error("task {} invalid task state {}", profile, (Object)state);
            }
        });
        this.traverseManagerTasksToDb(tasksFromManager);
        this.traverseDbTasksToManager(tasksFromManager);
    }

    private void keepPaceWithDb() {
        this.traverseDbTasksToMemory();
        this.traverseMemoryTasksToDb();
    }

    private void traverseManagerTasksToDb(Map<String, TaskProfile> tasksFromManager) {
        tasksFromManager.values().forEach(profileFromManager -> {
            TaskProfile taskFromDb = this.taskDb.getTask(profileFromManager.getTaskId());
            if (taskFromDb == null) {
                LOGGER.info("traverseManagerTasksToDb task {} not found in db retry {} state {}, add it", new Object[]{profileFromManager.getTaskId(), profileFromManager.isRetry(), profileFromManager.getState()});
                this.addTask((TaskProfile)profileFromManager);
            } else {
                TaskStateEnum dbState;
                TaskStateEnum managerState = profileFromManager.getState();
                if (managerState == (dbState = taskFromDb.getState())) {
                    return;
                }
                if (dbState == TaskStateEnum.RETRY_FINISH) {
                    LOGGER.info("traverseManagerTasksToDb task {} dbState {} retry {}, do nothing", new Object[]{taskFromDb.getTaskId(), dbState, taskFromDb.isRetry()});
                    return;
                }
                if (managerState == TaskStateEnum.RUNNING) {
                    LOGGER.info("traverseManagerTasksToDb task {} dbState {} retry {}, active it", new Object[]{taskFromDb.getTaskId(), dbState, taskFromDb.isRetry()});
                    this.activeTask((TaskProfile)profileFromManager);
                } else {
                    LOGGER.info("traverseManagerTasksToDb task {} dbState {} retry {}, freeze it", new Object[]{taskFromDb.getTaskId(), dbState, taskFromDb.isRetry()});
                    this.freezeTask((TaskProfile)profileFromManager);
                }
            }
        });
    }

    private void traverseDbTasksToManager(Map<String, TaskProfile> tasksFromManager) {
        this.taskDb.getTasks().forEach(profileFromDb -> {
            if (!tasksFromManager.containsKey(profileFromDb.getTaskId())) {
                LOGGER.info("traverseDbTasksToManager try to delete task {}", (Object)profileFromDb.getTaskId());
                this.deleteTask((TaskProfile)profileFromDb);
            }
        });
    }

    private void traverseDbTasksToMemory() {
        this.taskDb.getTasks().forEach(profileFromDb -> {
            TaskStateEnum dbState = profileFromDb.getState();
            Task task = this.taskMap.get(profileFromDb.getTaskId());
            if (dbState == TaskStateEnum.RUNNING) {
                if (task == null) {
                    LOGGER.info("traverseDbTasksToMemory add task to mem taskId {}", (Object)profileFromDb.getTaskId());
                    this.addToMemory((TaskProfile)profileFromDb);
                }
            } else if (dbState == TaskStateEnum.FROZEN) {
                if (task != null) {
                    LOGGER.info("traverseDbTasksToMemory delete task from mem taskId {}", (Object)profileFromDb.getTaskId());
                    this.deleteFromMemory(profileFromDb.getTaskId());
                }
            } else if (dbState != TaskStateEnum.RETRY_FINISH) {
                LOGGER.error("task {} invalid state {}", (Object)profileFromDb.getTaskId(), (Object)dbState);
            }
        });
    }

    private void traverseMemoryTasksToDb() {
        this.taskMap.values().forEach(task -> {
            TaskProfile profileFromDb = this.taskDb.getTask(task.getTaskId());
            if (profileFromDb == null) {
                this.deleteFromMemory(task.getTaskId());
                return;
            }
            TaskStateEnum stateFromDb = profileFromDb.getState();
            if (stateFromDb != TaskStateEnum.RUNNING) {
                this.deleteFromMemory(task.getTaskId());
            }
        });
    }

    private void addTask(TaskProfile taskProfile) {
        if (this.taskMap.size() >= this.taskMaxLimit) {
            LOGGER.error("taskMap size {} over limit {}", (Object)this.taskMap.size(), (Object)this.taskMaxLimit);
            return;
        }
        if (!this.isProfileValid(taskProfile)) {
            LOGGER.error("task profile invalid {}", (Object)taskProfile.toJsonStr());
            return;
        }
        this.addToDb(taskProfile);
        TaskStateEnum state = TaskStateEnum.getTaskState((int)taskProfile.getInt("task.state"));
        if (state == TaskStateEnum.RUNNING) {
            this.addToMemory(taskProfile);
        } else {
            LOGGER.info("taskId {} state {} no need to add to memory", (Object)taskProfile.getTaskId(), (Object)taskProfile.getState());
        }
    }

    private void deleteTask(TaskProfile taskProfile) {
        this.deleteFromDb(taskProfile);
        this.deleteFromMemory(taskProfile.getTaskId());
    }

    private void freezeTask(TaskProfile taskProfile) {
        this.updateToDb(taskProfile);
        this.deleteFromMemory(taskProfile.getTaskId());
    }

    private void finishTask(TaskProfile taskProfile) {
        taskProfile.setState(TaskStateEnum.RETRY_FINISH);
        this.updateToDb(taskProfile);
        this.deleteFromMemory(taskProfile.getTaskId());
    }

    private void activeTask(TaskProfile taskProfile) {
        this.updateToDb(taskProfile);
        this.addToMemory(taskProfile);
    }

    private void restoreFromDb() {
        List taskProfileList = this.taskDb.getTasks();
        taskProfileList.forEach(profile -> {
            if (profile.getState() == TaskStateEnum.RUNNING) {
                LOGGER.info("restoreFromDb taskId {}", (Object)profile.getTaskId());
                this.addToMemory((TaskProfile)profile);
            }
        });
    }

    private void stopAllTasks() {
        this.taskMap.values().forEach(task -> task.destroy());
        this.taskMap.clear();
    }

    private boolean isProfileValid(TaskProfile profile) {
        try {
            Class<?> taskClass = Class.forName(profile.getTaskClass());
            Task task = (Task)taskClass.newInstance();
            return task.isProfileValid(profile);
        }
        catch (Throwable t) {
            LOGGER.error("isProfileValid error: ", t);
            return false;
        }
    }

    private void addToDb(TaskProfile taskProfile) {
        if (this.taskDb.getTask(taskProfile.getTaskId()) != null) {
            LOGGER.error("task {} should not exist", (Object)taskProfile.getTaskId());
        }
        this.taskDb.storeTask(taskProfile);
    }

    private void deleteFromDb(TaskProfile taskProfile) {
        if (this.taskDb.getTask(taskProfile.getTaskId()) == null) {
            LOGGER.error("try to delete task {} but not found in db", (Object)taskProfile);
            return;
        }
        this.taskDb.deleteTask(taskProfile.getTaskId());
    }

    private void updateToDb(TaskProfile taskProfile) {
        if (this.taskDb.getTask(taskProfile.getTaskId()) == null) {
            LOGGER.error("task {} not found, agent may have been reinstalled", (Object)taskProfile);
        }
        this.taskDb.storeTask(taskProfile);
    }

    private void addToMemory(TaskProfile taskProfile) {
        Task oldTask = this.taskMap.get(taskProfile.getTaskId());
        if (oldTask != null) {
            oldTask.destroy();
            this.taskMap.remove(taskProfile.getTaskId());
            LOGGER.error("old task {} should not exist, try stop it first", (Object)taskProfile.getTaskId());
        }
        try {
            Class<?> taskClass = Class.forName(taskProfile.getTaskClass());
            Task task = (Task)taskClass.newInstance();
            task.init((Object)this, taskProfile, this.instanceBasicDb);
            this.taskMap.put(taskProfile.getTaskId(), task);
            this.runningPool.submit((Runnable)task);
            LOGGER.info("add task {} into memory, taskMap size {}, runningPool task total {}, runningPool task active {}", new Object[]{task.getTaskId(), this.taskMap.size(), this.runningPool.getTaskCount(), this.runningPool.getActiveCount()});
        }
        catch (Throwable t) {
            LOGGER.error("add task error: ", t);
        }
    }

    private void deleteFromMemory(String taskId) {
        Task oldTask = this.taskMap.get(taskId);
        if (oldTask == null) {
            LOGGER.error("old task {} not found", (Object)taskId);
            return;
        }
        oldTask.destroy();
        this.taskMap.remove(oldTask.getTaskId());
        LOGGER.info("delete task {} from memory, taskMap size {}, runningPool task total {}, runningPool task active {}", new Object[]{oldTask.getTaskId(), this.taskMap.size(), this.runningPool.getTaskCount(), this.runningPool.getActiveCount()});
    }

    public Task getTask(String taskId) {
        return this.taskMap.get(taskId);
    }

    public TaskProfile getTaskProfile(String taskId) {
        return this.taskDb.getTask(taskId);
    }

    public void start() throws Exception {
        this.restoreFromDb();
        this.submitWorker(this.coreThread());
        OffsetManager.getInstance().start();
    }

    public void stop() throws Exception {
        this.stopAllTasks();
        this.waitForTerminate();
        this.runningPool.shutdown();
    }

    private class TaskPrintStat {
        public int newCount = 0;
        public int runningCont = 0;
        public int frozenCount = 0;
        public int finishedCount = 0;
        public int otherCount = 0;

        private TaskPrintStat() {
        }

        private void stat(TaskStateEnum state) {
            switch (state) {
                case NEW: {
                    ++this.newCount;
                    break;
                }
                case RUNNING: {
                    ++this.runningCont;
                    break;
                }
                case FROZEN: {
                    ++this.frozenCount;
                    break;
                }
                case RETRY_FINISH: {
                    ++this.finishedCount;
                    break;
                }
                default: {
                    ++this.otherCount;
                }
            }
        }

        public String toString() {
            return String.format("new %d running %d frozen %d finished %d other %d", this.newCount, this.runningCont, this.frozenCount, this.finishedCount, this.otherCount);
        }
    }
}

