/*
 * Decompiled with CFR 0.152.
 */
package org.apache.inlong.agent.core.instance;

import java.util.Iterator;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.inlong.agent.common.AbstractDaemon;
import org.apache.inlong.agent.common.AgentThreadFactory;
import org.apache.inlong.agent.conf.AgentConfiguration;
import org.apache.inlong.agent.conf.InstanceProfile;
import org.apache.inlong.agent.conf.TaskProfile;
import org.apache.inlong.agent.core.instance.InstanceAction;
import org.apache.inlong.agent.db.Db;
import org.apache.inlong.agent.db.InstanceDb;
import org.apache.inlong.agent.db.TaskProfileDb;
import org.apache.inlong.agent.metrics.audit.AuditUtils;
import org.apache.inlong.agent.plugin.Instance;
import org.apache.inlong.agent.utils.AgentUtils;
import org.apache.inlong.agent.utils.DateTransUtils;
import org.apache.inlong.agent.utils.ThreadUtils;
import org.apache.inlong.common.enums.InstanceStateEnum;
import org.apache.inlong.common.enums.TaskStateEnum;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InstanceManager
extends AbstractDaemon {
    private static final Logger LOGGER = LoggerFactory.getLogger(InstanceManager.class);
    private static final int ACTION_QUEUE_CAPACITY = 100;
    public static final int CLEAN_INSTANCE_ONCE_LIMIT = 10;
    public volatile int CORE_THREAD_SLEEP_TIME_MS = 1000;
    public static final int INSTANCE_DB_CLEAN_INTERVAL_MS = 10000;
    private long lastCleanTime = 0L;
    public static final String DB_INSTANCE_EXPIRE_CYCLE_COUNT = "3";
    private final InstanceDb instanceDb;
    private TaskProfileDb taskProfileDb;
    private TaskProfile taskFromDb;
    private final ConcurrentHashMap<String, Instance> instanceMap;
    private final BlockingQueue<InstanceAction> actionQueue;
    private static final ThreadPoolExecutor EXECUTOR_SERVICE = new ThreadPoolExecutor(0, Integer.MAX_VALUE, 1L, TimeUnit.SECONDS, new SynchronousQueue<Runnable>(), (ThreadFactory)new AgentThreadFactory("instance-manager"));
    private final int instanceLimit;
    private final AgentConfiguration agentConf;
    private final String taskId;
    private volatile boolean runAtLeastOneTime = false;
    private volatile boolean running = false;
    private final double reserveCoefficient = 0.8;

    public InstanceManager(String taskId, int instanceLimit, Db basicDb, TaskProfileDb taskProfileDb) {
        this.taskId = taskId;
        this.instanceDb = new InstanceDb(basicDb);
        this.taskProfileDb = taskProfileDb;
        this.agentConf = AgentConfiguration.getAgentConf();
        this.instanceMap = new ConcurrentHashMap();
        this.instanceLimit = instanceLimit;
        this.actionQueue = new LinkedBlockingQueue<InstanceAction>(100);
    }

    public String getTaskId() {
        return this.taskId;
    }

    public Instance getInstance(String instanceId) {
        return this.instanceMap.get(instanceId);
    }

    public InstanceProfile getInstanceProfile(String instanceId) {
        return this.instanceDb.getInstance(this.taskId, instanceId);
    }

    public boolean submitAction(InstanceAction action) {
        if (action == null) {
            return false;
        }
        return this.actionQueue.offer(action);
    }

    private Runnable coreThread() {
        return () -> {
            Thread.currentThread().setName("instance-manager-core-" + this.taskId);
            this.running = true;
            while (this.isRunnable()) {
                try {
                    AgentUtils.silenceSleepInMs((long)this.CORE_THREAD_SLEEP_TIME_MS);
                    this.cleanDbInstance();
                    this.dealWithActionQueue(this.actionQueue);
                    this.keepPaceWithDb();
                    String inlongGroupId = this.taskFromDb.getInlongGroupId();
                    String inlongStreamId = this.taskFromDb.getInlongStreamId();
                    AuditUtils.add((int)30009, (String)inlongGroupId, (String)inlongStreamId, (long)AgentUtils.getCurrentTime(), (int)1, (long)1L);
                }
                catch (Throwable ex) {
                    LOGGER.error("coreThread {}", ex);
                    ThreadUtils.threadThrowableHandler((Thread)Thread.currentThread(), (Throwable)ex);
                }
                this.runAtLeastOneTime = true;
            }
            this.running = false;
        };
    }

    private void cleanDbInstance() {
        if (AgentUtils.getCurrentTime() - this.lastCleanTime > 10000L) {
            List instances = this.instanceDb.getInstances(this.taskId);
            this.doCleanDbInstance(instances);
            InstancePrintStat stat = new InstancePrintStat();
            for (int i = 0; i < instances.size(); ++i) {
                InstanceProfile instance = (InstanceProfile)instances.get(i);
                stat.stat(instance.getState());
            }
            LOGGER.info("instanceManager running! taskId {} mem {} db total {} {} action count {}", new Object[]{this.taskId, this.instanceMap.size(), instances.size(), stat, this.actionQueue.size()});
            this.lastCleanTime = AgentUtils.getCurrentTime();
        }
    }

    private void doCleanDbInstance(List<InstanceProfile> instances) {
        AtomicInteger cleanCount = new AtomicInteger();
        Iterator<InstanceProfile> iterator = instances.iterator();
        while (iterator.hasNext()) {
            if (cleanCount.get() > 10) {
                return;
            }
            InstanceProfile instanceFromDb = iterator.next();
            if (instanceFromDb.getState() != InstanceStateEnum.FINISHED) {
                return;
            }
            TaskProfile taskFromDb = this.taskProfileDb.getTask(this.taskId);
            if (taskFromDb != null) {
                if (taskFromDb.getCycleUnit().compareToIgnoreCase("R") == 0) {
                    return;
                }
                if (taskFromDb.isRetry() ? taskFromDb.getState() != TaskStateEnum.RETRY_FINISH : instanceFromDb.getState() != InstanceStateEnum.FINISHED) {
                    return;
                }
            }
            long expireTime = DateTransUtils.calcOffset((String)(DB_INSTANCE_EXPIRE_CYCLE_COUNT + taskFromDb.getCycleUnit()));
            if (AgentUtils.getCurrentTime() - instanceFromDb.getModifyTime() <= expireTime) continue;
            cleanCount.getAndIncrement();
            LOGGER.info("instance has expired, delete from db dataTime {} taskId {} instanceId {}", new Object[]{instanceFromDb.getSourceDataTime(), instanceFromDb.getTaskId(), instanceFromDb.getInstanceId()});
            this.deleteFromDb(instanceFromDb.getInstanceId());
            iterator.remove();
        }
    }

    private void keepPaceWithDb() {
        this.traverseDbTasksToMemory();
        this.traverseMemoryTasksToDb();
    }

    private void traverseDbTasksToMemory() {
        this.instanceDb.getInstances(this.taskId).forEach(profileFromDb -> {
            InstanceStateEnum dbState = profileFromDb.getState();
            Instance instance = this.instanceMap.get(profileFromDb.getInstanceId());
            switch (dbState) {
                case DEFAULT: {
                    if (instance != null) break;
                    LOGGER.info("traverseDbTasksToMemory add instance to mem taskId {} instanceId {}", (Object)profileFromDb.getTaskId(), (Object)profileFromDb.getInstanceId());
                    this.addToMemory((InstanceProfile)profileFromDb);
                    break;
                }
                case FINISHED: {
                    if (instance == null) break;
                    LOGGER.info("traverseDbTasksToMemory delete instance from mem taskId {} instanceId {}", (Object)profileFromDb.getTaskId(), (Object)profileFromDb.getInstanceId());
                    this.deleteFromMemory(profileFromDb.getInstanceId());
                    break;
                }
                default: {
                    LOGGER.error("instance invalid state {} taskId {} instanceId {}", new Object[]{dbState, profileFromDb.getTaskId(), profileFromDb.getInstanceId()});
                }
            }
        });
    }

    private void traverseMemoryTasksToDb() {
        this.instanceMap.values().forEach(instance -> {
            InstanceProfile profileFromDb = this.instanceDb.getInstance(instance.getTaskId(), instance.getInstanceId());
            if (profileFromDb == null) {
                this.deleteFromMemory(instance.getInstanceId());
                return;
            }
            InstanceStateEnum stateFromDb = profileFromDb.getState();
            if (stateFromDb != InstanceStateEnum.DEFAULT) {
                this.deleteFromMemory(instance.getInstanceId());
            }
        });
    }

    private void dealWithActionQueue(BlockingQueue<InstanceAction> queue) {
        block7: while (this.isRunnable()) {
            try {
                InstanceAction action = (InstanceAction)queue.poll();
                if (action == null) break;
                switch (action.getActionType()) {
                    case ADD: {
                        this.addInstance(action.getProfile());
                        continue block7;
                    }
                    case FINISH: {
                        this.finishInstance(action.getProfile());
                        continue block7;
                    }
                    case DELETE: {
                        this.deleteInstance(action.getProfile().getInstanceId());
                        continue block7;
                    }
                }
                LOGGER.error("invalid action type for instance manager: taskId {} type {}", (Object)this.taskId, (Object)action.getActionType());
            }
            catch (Throwable ex) {
                LOGGER.error("dealWithActionQueue", ex);
                ThreadUtils.threadThrowableHandler((Thread)Thread.currentThread(), (Throwable)ex);
            }
        }
    }

    public void start() {
        this.restoreFromDb();
        this.submitWorker(this.coreThread());
    }

    public void stop() {
        this.waitForTerminate();
        this.stopAllInstances();
    }

    public void waitForTerminate() {
        super.waitForTerminate();
        while (this.running) {
            AgentUtils.silenceSleepInMs((long)this.CORE_THREAD_SLEEP_TIME_MS);
        }
    }

    private void restoreFromDb() {
        this.taskFromDb = this.taskProfileDb.getTask(this.taskId);
        List profileList = this.instanceDb.getInstances(this.taskId);
        profileList.forEach(profile -> {
            InstanceStateEnum state = profile.getState();
            if (state == InstanceStateEnum.DEFAULT) {
                LOGGER.info("instance restoreFromDb addToMem state {} taskId {} instanceId {}", new Object[]{state, this.taskId, profile.getInstanceId()});
                this.addToMemory((InstanceProfile)profile);
            } else {
                LOGGER.info("instance restoreFromDb ignore state {} taskId {} instanceId {}", new Object[]{state, this.taskId, profile.getInstanceId()});
            }
        });
    }

    private void addInstance(InstanceProfile profile) {
        if (this.instanceMap.size() >= this.instanceLimit) {
            LOGGER.error("instanceMap size {} over limit {}", (Object)this.instanceMap.size(), (Object)this.instanceLimit);
            return;
        }
        LOGGER.info("add instance taskId {} instanceId {}", (Object)this.taskId, (Object)profile.getInstanceId());
        if (!this.shouldAddAgain(profile.getInstanceId(), profile.getFileUpdateTime())) {
            LOGGER.info("addInstance shouldAddAgain returns false skip taskId {} instanceId {}", (Object)this.taskId, (Object)profile.getInstanceId());
            return;
        }
        this.addToDb(profile, true);
        this.addToMemory(profile);
    }

    private void finishInstance(InstanceProfile profile) {
        profile.setState(InstanceStateEnum.FINISHED);
        profile.setModifyTime(Long.valueOf(AgentUtils.getCurrentTime()));
        this.addToDb(profile, false);
        this.deleteFromMemory(profile.getInstanceId());
        LOGGER.info("finished instance state {} taskId {} instanceId {}", new Object[]{profile.getState(), profile.getTaskId(), profile.getInstanceId()});
    }

    private void deleteInstance(String instanceId) {
        this.deleteFromDb(instanceId);
        this.deleteFromMemory(instanceId);
    }

    private void deleteFromDb(String instanceId) {
        InstanceProfile profile = this.instanceDb.getInstance(this.taskId, instanceId);
        String inlongGroupId = profile.getInlongGroupId();
        String inlongStreamId = profile.getInlongStreamId();
        this.instanceDb.deleteInstance(this.taskId, instanceId);
        LOGGER.info("delete instance from db: taskId {} instanceId {} result {}", new Object[]{this.taskId, instanceId, this.instanceDb.getInstance(this.taskId, instanceId)});
        AuditUtils.add((int)30006, (String)inlongGroupId, (String)inlongStreamId, (long)profile.getSinkDataTime(), (int)1, (long)1L);
    }

    private void deleteFromMemory(String instanceId) {
        Instance instance = this.instanceMap.get(instanceId);
        if (instance == null) {
            LOGGER.error("try to delete instance from memory but not found: taskId {} instanceId {}", (Object)this.taskId, (Object)instanceId);
            return;
        }
        String inlongGroupId = instance.getProfile().getInlongGroupId();
        String inlongStreamId = instance.getProfile().getInlongStreamId();
        instance.destroy();
        this.instanceMap.remove(instanceId);
        LOGGER.info("delete instance from memory: taskId {} instanceId {}", (Object)this.taskId, (Object)instance.getInstanceId());
        AuditUtils.add((int)30004, (String)inlongGroupId, (String)inlongStreamId, (long)instance.getProfile().getSinkDataTime(), (int)1, (long)1L);
    }

    private void addToDb(InstanceProfile profile, boolean addNew) {
        LOGGER.info("add instance to db state {} instanceId {}", (Object)profile.getState(), (Object)profile.getInstanceId());
        this.instanceDb.storeInstance(profile);
        if (addNew) {
            String inlongGroupId = profile.getInlongGroupId();
            String inlongStreamId = profile.getInlongStreamId();
            AuditUtils.add((int)30005, (String)inlongGroupId, (String)inlongStreamId, (long)profile.getSinkDataTime(), (int)1, (long)1L);
        }
    }

    private void addToMemory(InstanceProfile instanceProfile) {
        String inlongGroupId = instanceProfile.getInlongGroupId();
        String inlongStreamId = instanceProfile.getInlongStreamId();
        Instance oldInstance = this.instanceMap.get(instanceProfile.getInstanceId());
        if (oldInstance != null) {
            oldInstance.destroy();
            this.instanceMap.remove(instanceProfile.getInstanceId());
            LOGGER.error("old instance {} should not exist, try stop it first", (Object)instanceProfile.getInstanceId());
            AuditUtils.add((int)30014, (String)inlongGroupId, (String)inlongStreamId, (long)instanceProfile.getSinkDataTime(), (int)1, (long)1L);
        }
        LOGGER.info("instanceProfile {}", (Object)instanceProfile.toJsonStr());
        try {
            Class<?> taskClass = Class.forName(instanceProfile.getInstanceClass());
            Instance instance = (Instance)taskClass.newInstance();
            instance.init((Object)this, instanceProfile);
            this.instanceMap.put(instanceProfile.getInstanceId(), instance);
            EXECUTOR_SERVICE.submit((Runnable)instance);
            LOGGER.info("add instance to memory instanceId {} instanceMap size {}, runningPool instance total {}, runningPool instance active {}", new Object[]{instance.getInstanceId(), this.instanceMap.size(), EXECUTOR_SERVICE.getTaskCount(), EXECUTOR_SERVICE.getActiveCount()});
            AuditUtils.add((int)30003, (String)inlongGroupId, (String)inlongStreamId, (long)instanceProfile.getSinkDataTime(), (int)1, (long)1L);
        }
        catch (Throwable t) {
            LOGGER.error("add instance error {}", (Object)t.getMessage());
        }
    }

    private void stopAllInstances() {
        this.instanceMap.values().forEach(instance -> this.deleteFromMemory(instance.getInstanceId()));
        this.instanceMap.clear();
    }

    public boolean shouldAddAgain(String fileName, long lastModifyTime) {
        InstanceProfile profileFromDb = this.instanceDb.getInstance(this.taskId, fileName);
        if (profileFromDb == null) {
            LOGGER.debug("not in db should add {}", (Object)fileName);
            return true;
        }
        InstanceStateEnum state = profileFromDb.getState();
        if (state == InstanceStateEnum.FINISHED && lastModifyTime > profileFromDb.getModifyTime()) {
            LOGGER.debug("finished but file update again {}", (Object)fileName);
            return true;
        }
        if (state == InstanceStateEnum.DELETE) {
            LOGGER.debug("delete and add again {}", (Object)fileName);
            return true;
        }
        return false;
    }

    public boolean isFull() {
        return (double)(this.instanceMap.size() + this.actionQueue.size()) >= (double)this.instanceLimit * 0.8;
    }

    public boolean allInstanceFinished() {
        if (!this.runAtLeastOneTime) {
            return false;
        }
        if (!this.instanceMap.isEmpty()) {
            return false;
        }
        if (!this.actionQueue.isEmpty()) {
            return false;
        }
        List instances = this.instanceDb.getInstances(this.taskId);
        for (int i = 0; i < instances.size(); ++i) {
            InstanceProfile profile = (InstanceProfile)instances.get(i);
            if (profile.getState() == InstanceStateEnum.FINISHED) continue;
            return false;
        }
        return true;
    }

    private class InstancePrintStat {
        public int defaultCount = 0;
        public int finishedCount = 0;
        public int deleteCount = 0;
        public int otherCount = 0;

        private InstancePrintStat() {
        }

        private void stat(InstanceStateEnum state) {
            switch (state) {
                case DEFAULT: {
                    ++this.defaultCount;
                    break;
                }
                case FINISHED: {
                    ++this.finishedCount;
                    break;
                }
                case DELETE: {
                    ++this.deleteCount;
                }
                default: {
                    ++this.otherCount;
                }
            }
        }

        public String toString() {
            return String.format("default %d finished %d delete %d other %d", this.defaultCount, this.finishedCount, this.deleteCount, this.otherCount);
        }
    }
}

