/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tuweni.rlpx.vertx;

import io.vertx.core.Handler;
import io.vertx.core.Vertx;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.net.NetClient;
import io.vertx.core.net.NetClientOptions;
import io.vertx.core.net.NetServer;
import io.vertx.core.net.NetServerOptions;
import io.vertx.core.net.NetSocket;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.apache.tuweni.bytes.Bytes;
import org.apache.tuweni.bytes.Bytes32;
import org.apache.tuweni.concurrent.AsyncCompletion;
import org.apache.tuweni.concurrent.AsyncResult;
import org.apache.tuweni.concurrent.CompletableAsyncCompletion;
import org.apache.tuweni.concurrent.CompletableAsyncResult;
import org.apache.tuweni.crypto.SECP256K1;
import org.apache.tuweni.rlpx.HandshakeMessage;
import org.apache.tuweni.rlpx.MemoryWireConnectionsRepository;
import org.apache.tuweni.rlpx.RLPxConnection;
import org.apache.tuweni.rlpx.RLPxConnectionFactory;
import org.apache.tuweni.rlpx.RLPxMessage;
import org.apache.tuweni.rlpx.RLPxService;
import org.apache.tuweni.rlpx.WireConnectionRepository;
import org.apache.tuweni.rlpx.wire.DefaultWireConnection;
import org.apache.tuweni.rlpx.wire.DisconnectReason;
import org.apache.tuweni.rlpx.wire.SubProtocol;
import org.apache.tuweni.rlpx.wire.SubProtocolClient;
import org.apache.tuweni.rlpx.wire.SubProtocolHandler;
import org.apache.tuweni.rlpx.wire.SubProtocolIdentifier;
import org.apache.tuweni.rlpx.wire.WireConnection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class VertxRLPxService
implements RLPxService {
    private static final int DEVP2P_VERSION = 5;
    private static final Logger logger = LoggerFactory.getLogger(VertxRLPxService.class);
    private final AtomicBoolean started = new AtomicBoolean(false);
    private final Vertx vertx;
    private final int listenPort;
    private final String networkInterface;
    private final int advertisedPort;
    private final SECP256K1.KeyPair keyPair;
    private final List<SubProtocol> subProtocols;
    private final String clientId;
    private final WireConnectionRepository repository;
    private LinkedHashMap<SubProtocolIdentifier, SubProtocolHandler> handlers;
    private LinkedHashMap<SubProtocolIdentifier, SubProtocolClient> clients;
    private NetClient client;
    private NetServer server;
    private List<SECP256K1.PublicKey> keepAliveList = new ArrayList<SECP256K1.PublicKey>();

    private static void checkPort(int port) {
        if (port < 0 || port > 65536) {
            throw new IllegalArgumentException("Invalid port: " + port);
        }
    }

    public VertxRLPxService(Vertx vertx, int listenPort, String networkInterface, int advertisedPort, SECP256K1.KeyPair identityKeyPair, List<SubProtocol> subProtocols, String clientId) {
        this(vertx, listenPort, networkInterface, advertisedPort, identityKeyPair, subProtocols, clientId, new MemoryWireConnectionsRepository());
    }

    public VertxRLPxService(Vertx vertx, int listenPort, String networkInterface, int advertisedPort, SECP256K1.KeyPair identityKeyPair, List<SubProtocol> subProtocols, String clientId, WireConnectionRepository repository) {
        VertxRLPxService.checkPort(listenPort);
        VertxRLPxService.checkPort(advertisedPort);
        if (clientId == null || clientId.trim().isEmpty()) {
            throw new IllegalArgumentException("Client ID must contain a valid identifier");
        }
        this.vertx = vertx;
        this.listenPort = listenPort;
        this.networkInterface = networkInterface;
        this.advertisedPort = advertisedPort;
        this.keyPair = identityKeyPair;
        this.subProtocols = subProtocols;
        this.clientId = clientId;
        this.repository = repository;
        repository.addDisconnectionListener(c -> {
            if (this.keepAliveList.contains(c.peerPublicKey())) {
                this.tryConnect(c.peerPublicKey(), new InetSocketAddress(c.peerHost(), c.peerPort()));
            }
        });
    }

    private void tryConnect(SECP256K1.PublicKey peerPublicKey, InetSocketAddress inetSocketAddress) {
        this.vertx.runOnContext(event -> this.connectTo(peerPublicKey, inetSocketAddress).whenComplete((result, e) -> {
            if (e != null) {
                logger.warn("Error reconnecting to peer {}@{}: {}", new Object[]{peerPublicKey, inetSocketAddress, e});
                this.tryConnect(peerPublicKey, inetSocketAddress);
            } else {
                logger.info("Connected successfully to keep alive peer {}@{}", (Object)peerPublicKey, (Object)inetSocketAddress);
            }
        }));
    }

    @Override
    public AsyncCompletion start() {
        if (this.started.compareAndSet(false, true)) {
            this.handlers = new LinkedHashMap();
            this.clients = new LinkedHashMap();
            for (SubProtocol subProtocol : this.subProtocols) {
                for (SubProtocolIdentifier identifier : subProtocol.getCapabilities()) {
                    if (identifier.versionRange() == 0) {
                        throw new IllegalArgumentException("Invalid subprotocol " + identifier.toString());
                    }
                    SubProtocolClient client = subProtocol.createClient(this, identifier);
                    this.clients.put(identifier, client);
                    this.handlers.put(identifier, subProtocol.createHandler(this, client));
                }
            }
            this.client = this.vertx.createNetClient(new NetClientOptions());
            this.server = this.vertx.createNetServer(new NetServerOptions().setPort(this.listenPort).setHost(this.networkInterface).setTcpKeepAlive(true)).connectHandler(this::receiveMessage);
            CompletableAsyncCompletion complete = AsyncCompletion.incomplete();
            this.server.listen(res -> {
                if (res.succeeded()) {
                    complete.complete();
                } else {
                    complete.completeExceptionally(res.cause());
                }
            });
            return complete;
        }
        return AsyncCompletion.completed();
    }

    @Override
    public void send(SubProtocolIdentifier subProtocolIdentifier, int messageType, WireConnection connection, Bytes message) {
        if (!this.started.get()) {
            throw new IllegalStateException("The RLPx service is not active");
        }
        ((DefaultWireConnection)connection).sendMessage(subProtocolIdentifier, messageType, message);
    }

    @Override
    public void disconnect(WireConnection connection, DisconnectReason disconnectReason) {
        if (!this.started.get()) {
            throw new IllegalStateException("The RLPx service is not active");
        }
        ((DefaultWireConnection)connection).disconnect(disconnectReason);
    }

    private void receiveMessage(final NetSocket netSocket) {
        netSocket.handler((Handler)new Handler<Buffer>(){
            private RLPxConnection conn;
            private DefaultWireConnection wireConnection;

            public void handle(Buffer buffer) {
                if (this.conn == null) {
                    this.conn = RLPxConnectionFactory.respondToHandshake(Bytes.wrapBuffer((Buffer)buffer), VertxRLPxService.this.keyPair, bytes -> netSocket.write(Buffer.buffer((byte[])bytes.toArrayUnsafe())));
                    if (this.wireConnection == null) {
                        this.wireConnection = VertxRLPxService.this.createConnection(this.conn, netSocket, (CompletableAsyncResult<WireConnection>)AsyncResult.incomplete());
                    }
                } else {
                    this.conn.stream(Bytes.wrapBuffer((Buffer)buffer), this.wireConnection::messageReceived);
                }
            }
        });
    }

    @Override
    public AsyncCompletion stop() {
        if (this.started.compareAndSet(true, false)) {
            for (WireConnection conn : this.repository.asIterable()) {
                ((DefaultWireConnection)conn).disconnect(DisconnectReason.CLIENT_QUITTING);
            }
            this.repository.close();
            this.client.close();
            AsyncCompletion handlersCompletion = AsyncCompletion.allOf((Collection)this.handlers.values().stream().map(SubProtocolHandler::stop).collect(Collectors.toList()));
            CompletableAsyncCompletion completableAsyncCompletion = AsyncCompletion.incomplete();
            this.server.close(res -> {
                if (res.succeeded()) {
                    completableAsyncCompletion.complete();
                } else {
                    completableAsyncCompletion.completeExceptionally(res.cause());
                }
            });
            return handlersCompletion.thenCombine((AsyncCompletion)completableAsyncCompletion);
        }
        return AsyncCompletion.completed();
    }

    @Override
    public int actualPort() {
        if (!this.started.get()) {
            throw new IllegalStateException("The RLPx service is not active");
        }
        return this.server.actualPort();
    }

    @Override
    public InetSocketAddress actualSocketAddress() {
        if (!this.started.get()) {
            throw new IllegalStateException("The RLPx service is not active");
        }
        return new InetSocketAddress(this.networkInterface, this.server.actualPort());
    }

    @Override
    public int advertisedPort() {
        if (!this.started.get()) {
            throw new IllegalStateException("The RLPx service is not active");
        }
        return this.listenPort == 0 ? this.actualPort() : this.advertisedPort;
    }

    @Override
    public WireConnectionRepository repository() {
        return this.repository;
    }

    @Override
    public SubProtocolClient getClient(SubProtocolIdentifier subProtocolIdentifier) {
        if (!this.started.get()) {
            throw new IllegalStateException("The RLPx service is not active");
        }
        return this.clients.get(subProtocolIdentifier);
    }

    @Override
    public AsyncResult<WireConnection> connectTo(final SECP256K1.PublicKey peerPublicKey, InetSocketAddress peerAddress) {
        if (!this.started.get()) {
            throw new IllegalStateException("The RLPx service is not active");
        }
        final CompletableAsyncResult connected = AsyncResult.incomplete();
        logger.debug("Connecting to {} with public key {}", (Object)peerAddress, (Object)peerPublicKey);
        this.client.connect(peerAddress.getPort(), peerAddress.getHostString(), netSocketFuture -> netSocketFuture.map(netSocket -> {
            final Bytes32 nonce = RLPxConnectionFactory.generateRandomBytes32();
            final SECP256K1.KeyPair ephemeralKeyPair = SECP256K1.KeyPair.random();
            final Bytes initHandshakeMessage = RLPxConnectionFactory.init(this.keyPair, peerPublicKey, ephemeralKeyPair, nonce);
            logger.debug("Initiating handshake to {}", (Object)peerAddress);
            netSocket.write(Buffer.buffer((byte[])initHandshakeMessage.toArrayUnsafe()));
            netSocket.closeHandler(event -> {
                logger.debug("Connection {} closed", (Object)peerAddress);
                if (!connected.isDone()) {
                    connected.cancel();
                }
            });
            netSocket.handler((Handler)new Handler<Buffer>(){
                private RLPxConnection conn;
                private DefaultWireConnection wireConnection;

                public void handle(Buffer buffer) {
                    try {
                        Bytes messageBytes = Bytes.wrapBuffer((Buffer)buffer);
                        if (this.conn == null) {
                            int messageSize = RLPxConnectionFactory.messageSize(messageBytes);
                            Bytes responseBytes = messageBytes;
                            if (messageBytes.size() > messageSize) {
                                responseBytes = responseBytes.slice(0, messageSize);
                            }
                            messageBytes = messageBytes.slice(messageSize);
                            HandshakeMessage responseMessage = RLPxConnectionFactory.readResponse(responseBytes, VertxRLPxService.this.keyPair.secretKey());
                            this.conn = RLPxConnectionFactory.createConnection(true, initHandshakeMessage, responseBytes, ephemeralKeyPair.secretKey(), responseMessage.ephemeralPublicKey(), nonce, responseMessage.nonce(), VertxRLPxService.this.keyPair.publicKey(), peerPublicKey);
                            this.wireConnection = VertxRLPxService.this.createConnection(this.conn, netSocket, (CompletableAsyncResult<WireConnection>)connected);
                            if (messageBytes.isEmpty()) {
                                return;
                            }
                        }
                        if (this.conn != null) {
                            this.conn.stream(messageBytes, this.wireConnection::messageReceived);
                        }
                    }
                    catch (Exception e) {
                        logger.error(e.getMessage(), (Throwable)e);
                        connected.completeExceptionally((Throwable)e);
                        netSocket.close();
                    }
                }
            });
            return null;
        }));
        return connected;
    }

    private DefaultWireConnection createConnection(RLPxConnection conn, NetSocket netSocket, CompletableAsyncResult<WireConnection> ready) {
        String host = netSocket.remoteAddress().host();
        int port = netSocket.remoteAddress().port();
        DefaultWireConnection wireConnection = new DefaultWireConnection(conn.publicKey().bytes(), conn.peerPublicKey().bytes(), message -> {
            RLPxConnection rLPxConnection = conn;
            synchronized (rLPxConnection) {
                Bytes bytes = conn.write((RLPxMessage)message);
                this.vertx.eventBus().send(netSocket.writeHandlerID(), (Object)Buffer.buffer((byte[])bytes.toArrayUnsafe()));
            }
        }, conn::configureAfterHandshake, () -> ((NetSocket)netSocket).end(), this.handlers, 5, this.clientId, this.advertisedPort(), ready, host, port);
        this.repository.add(wireConnection);
        wireConnection.handleConnectionStart();
        return wireConnection;
    }

    @Override
    public boolean addToKeepAliveList(SECP256K1.PublicKey peerPublicKey) {
        return this.keepAliveList.add(peerPublicKey);
    }
}

