/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

import {ShepherdService} from "../../services/tour/shepherd.service";
import {EventEmitter, Injectable} from "@angular/core";
import {PipelineService} from "../../platform-services/apis/pipeline.service";
import {PanelType} from "../../core-ui/dialog/base-dialog/base-dialog.model";
import {DialogService} from "../../core-ui/dialog/base-dialog/base-dialog.service";
import {PipelineStatusDialogComponent} from "../dialog/pipeline-status/pipeline-status-dialog.component";
import {Pipeline} from "../../core-model/gen/streampipes-model";
import {DeletePipelineDialogComponent} from "../dialog/delete-pipeline/delete-pipeline-dialog.component";
import {DialogRef} from "../../core-ui/dialog/base-dialog/dialog-ref";
import {Router} from "@angular/router";
import {PipelineAction} from "../model/pipeline-model";
import {PipelineNotificationsComponent} from "../dialog/pipeline-notifications/pipeline-notifications.component";

@Injectable()
export class PipelineOperationsService {

  starting: any;
  stopping: any;

  constructor(
      private ShepherdService: ShepherdService,
      private PipelineService: PipelineService,
      private DialogService: DialogService,
      private Router: Router) {
  }

  startPipeline(pipelineId: string,
                refreshPipelinesEmitter: EventEmitter<boolean>,
                toggleRunningOperation?,) {
    if (toggleRunningOperation) {
      toggleRunningOperation('starting');
    }
    let dialogRef = this.showPipelineOperationsDialog(pipelineId, PipelineAction.Start);
    this.afterPipelineOperationsDialogClosed(dialogRef, refreshPipelinesEmitter, "starting", toggleRunningOperation);
  }

  stopPipeline(pipelineId: string,
               refreshPipelinesEmitter: EventEmitter<boolean>,
               toggleRunningOperation?) {
    if (toggleRunningOperation) {
      toggleRunningOperation('stopping');
    }
    let dialogRef = this.showPipelineOperationsDialog(pipelineId, PipelineAction.Stop);
    this.afterPipelineOperationsDialogClosed(dialogRef, refreshPipelinesEmitter, "stopping", toggleRunningOperation);
  }

  afterPipelineOperationsDialogClosed(dialogRef: DialogRef<PipelineStatusDialogComponent>,
                                      refreshPipelinesEmitter: EventEmitter<boolean>,
                                      toggleAction: string,
                                      toggleRunningOperation?) {
    dialogRef.afterClosed().subscribe(msg => {
      refreshPipelinesEmitter.emit(true);
      if (toggleRunningOperation) {
        toggleRunningOperation(toggleAction);
      }
    });
  }

  showDeleteDialog(pipeline: Pipeline,
                   refreshPipelinesEmitter: EventEmitter<boolean>,
                   switchToPipelineView?: any) {
    let dialogRef: DialogRef<DeletePipelineDialogComponent> = this.DialogService.open(DeletePipelineDialogComponent, {
      panelType: PanelType.STANDARD_PANEL,
      title: "Delete Pipeline",
      width: "70vw",
      data: {
        "pipeline": pipeline,
      }
    });

    dialogRef.afterClosed().subscribe(data => {
      if (data) {
        if (!switchToPipelineView) {
          refreshPipelinesEmitter.emit(true);
        } else {
          switchToPipelineView();
        }
      }
    })
  };

  showPipelineOperationsDialog(pipelineId: string,
                               action: PipelineAction): DialogRef<PipelineStatusDialogComponent> {
    return this.DialogService.open(PipelineStatusDialogComponent, {
      panelType: PanelType.STANDARD_PANEL,
      title: "Pipeline Status",
      width: "70vw",
      data: {
        "pipelineId": pipelineId,
        "action": action
      }
    });
  };

  showPipelineNotificationsDialog(pipeline: Pipeline,
                                  refreshPipelinesEmitter: EventEmitter<boolean>) {
    let dialogRef: DialogRef<PipelineNotificationsComponent> = this.DialogService.open(PipelineNotificationsComponent, {
      panelType: PanelType.STANDARD_PANEL,
      title: "Pipeline Notifications",
      width: "70vw",
      data: {
        "pipeline": pipeline,
      }
    });

    dialogRef.afterClosed().subscribe(close => {
      refreshPipelinesEmitter.emit(true);
    });
  }

  showPipelineInEditor(id) {
    this.Router.navigate(["editor"], { queryParams: { pipeline: id }});
  }

  showPipelineDetails(id) {
    this.Router.navigate(["pipeline-details"], { queryParams: { pipeline: id }});
  }

  modifyPipeline(pipeline) {
    this.showPipelineInEditor(pipeline);
  }

  showLogs(id) {
    //this.$state.go("streampipes.pipelinelogs", {pipeline: id});
  }
}
